/* normalmode_chown.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2011 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "normalmode.h"
#include "worker_locale.h"
#include "fileentry.hh"
#include "nmcopyopdir.hh"
#include "nmfieldlistviewdnd.hh"
#include "verzeichnis.hh"
#include "simplelist.hh"
#include <aguix/button.h>

void NormalMode::chownf(struct NM_chownorder *coorder)
{
    NM_specialsourceInt *ss1;
    changeown_info_t *coinfo;
    std::list<NM_specialsourceInt*> *colist;
    std::list<NM_specialsourceInt*>::iterator iti1;
  
    if ( coorder == NULL ) return;
    if ( getCurrentDir() == NULL ) return;

    finishsearchmode();
  
    colist = new std::list<NM_specialsourceInt*>;
    switch ( coorder->source ) {
        case NM_chownorder::NM_SPECIAL:
            buildSpecialSourceIntList( *( coorder->sources ),
                                       *colist );
            break;
        case NM_chownorder::NM_ONLYACTIVE:
            getSelFiles( colist, NM_GETFILES_ONLYACTIVE, false );
            break;
        default:  // all selected entries
            getSelFiles( colist, NM_GETFILES_SELORACT, false );
            break;
    }

    buildOwnerRequestInfos();

    coinfo = new changeown_info_t;
    for ( iti1 = colist->begin(); iti1 != colist->end(); iti1++ ) {
        ss1 = *iti1;
        if ( worker_changeown( ss1, coinfo, coorder ) != 0 ) {
            break;
        }
    }

    for ( iti1 = colist->begin(); iti1 != colist->end(); iti1++ ) {
        ss1 = *iti1;
        delete ss1;
    }
    delete coinfo;
    delete colist;
    freeOwnerRequestInfos();
    update(false);
}

int NormalMode::worker_changeown( const struct NM_specialsourceInt *ss1,
                                  changeown_info_t *coinfo,
                                  const NM_chownorder *coorder )
{
    bool enter, dontChange, skip, cancel;
    int erg;
    uid_t newuid = (uid_t)-1;
    gid_t newgid = (gid_t)-1;
    FileEntry *subfe;
    NM_specialsourceInt *ss2;
    NM_CopyOp_Dir *cod;
  
    if ( ( ss1 == NULL ) || ( coinfo == NULL ) || ( coorder == NULL ) ) return -1;

    // do we have to enter this entry?
    enter = false;
    if ( ( ss1->entry()->isDir() == true ) && ( coorder->recursive == true ) ) {
        if ( ss1->entry()->isLink == false ) enter = true;
    }
  
    // check operation applies to this entry
    dontChange = false;
    if ( ( ss1->entry()->isDir() == true ) && ( coorder->ondirs == false ) ) dontChange = true;
    else if ( ( ss1->entry()->isDir() == false ) && ( coorder->onfiles == false ) ) dontChange = true;

    /* skip means skip entry AND all sub entries
     * cancel is clear
     * dontchange means to normally handle this entry
     *   but actually don't change mod
     */
    skip = cancel = false;
    if ( dontChange == false ) {
        // ask for new owner
        if ( coinfo->forAll == true ) {
            newuid = coinfo->newuid;
            newgid = coinfo->newgid;
        } else {
            erg = requestNewOwner( ss1->entry(), &newuid, &newgid );
            if ( erg == 1 ) {
                coinfo->forAll = true;
                coinfo->newuid = newuid;
                coinfo->newgid = newgid;
            } else if ( erg == 2 ) {
                skip = true;
            } else if ( erg == 3 ) {
                cancel = true;
            }
        }
    }
    if ( skip == true ) return 0;
    if ( cancel == true ) return 1;

    if ( dontChange == false ) {
        if ( applyNewOwner( ss1, newuid, newgid ) != 0 ) cancel = true;
    }
    if ( ( enter == true ) && ( cancel == false ) ) {
        cod = new NM_CopyOp_Dir( ss1->entry() );
        if ( cod->user_abort == true ) {
            cancel = true;
        } else if ( cod->ok == true ) {
            for ( Verzeichnis::verz_it subfe_it1 = cod->verz->begin();
                  subfe_it1 != cod->verz->end() && cancel == false;
                  subfe_it1++ ) {
                subfe = *subfe_it1;
                if ( strcmp( subfe->name, ".." ) != 0 ) {
                    ss2 = new NM_specialsourceInt( subfe );
                    ss2->row = -1;
                    if ( worker_changeown( ss2, coinfo, coorder ) != 0 ) {
                        cancel = true;
                    }
                    delete ss2;
                }
            }
        }
        delete cod;
    }
    return ( cancel == true ) ? 1 : 0;
}

int NormalMode::applyNewOwner( const NM_specialsourceInt *ss1, uid_t newuid, gid_t newgid )
{
    char *textstr, *buttonstr;
    int erg;
    bool cancel = false;
  
    if ( ss1 == NULL ) return -1;

    if ( worker_chown( ss1->entry()->fullname, newuid, newgid ) != 0 ) {
        if ( errno == EPERM ) {
            buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                           strlen( catalog.getLocale( 8 ) ) + 1 );
            sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                     catalog.getLocale( 8 ) );
            textstr = (char*)_allocsafe( strlen( catalog.getLocale( 446 ) ) + strlen( ss1->entry()->fullname ) + 1 );
            sprintf( textstr, catalog.getLocale( 446 ), ss1->entry()->fullname );
            erg = request( catalog.getLocale( 347 ), textstr, buttonstr );
            _freesafe( buttonstr );
            _freesafe( textstr );
            if ( erg == 1 ) cancel = true;
        } else {
            // error
            buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                           strlen( catalog.getLocale( 8 ) ) + 1 );
            sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                     catalog.getLocale( 8 ) );
            textstr = (char*)_allocsafe( strlen( catalog.getLocale( 447 ) ) + strlen( ss1->entry()->fullname ) + 1 );
            sprintf( textstr, catalog.getLocale( 447 ), ss1->entry()->fullname );
            erg = request( catalog.getLocale( 347 ), textstr, buttonstr );
            _freesafe( buttonstr );
            _freesafe( textstr );
            if ( erg == 1 ) cancel = true;
        }
    } else {
        if ( lv->isValidRow( ss1->row ) == true ) deselect( ss1->entry(), ss1->row );
    }
    return ( cancel == true ) ? 1 : 0;
}

void NormalMode::freeOwnerRequestInfos()
{
    chownrequest_id_name_t *elem;

    if ( chownUserList != NULL ) {
        elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
        while ( elem != NULL ) {
            _freesafe( elem->name );
            _freesafe( elem );
            chownUserList->removeFirstElement();
            elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
        }
        delete chownUserList;
        chownUserList = NULL;
    }
  
    if ( chownGroupList != NULL ) {
        elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
        while ( elem != NULL ) {
            _freesafe( elem->name );
            _freesafe( elem );
            chownGroupList->removeFirstElement();
            elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
        }
        delete chownGroupList;
        chownGroupList = NULL;
    }
}

void NormalMode::buildOwnerRequestInfos()
{
    struct passwd *pwdp;
    struct group *grpp;
    chownrequest_id_name_t *elem;

    freeOwnerRequestInfos();

    chownUserList = new List();
    chownGroupList = new List();
  
    //setpwent();   // is this needed?
    pwdp = getpwent();
    while ( pwdp != NULL ) {
        elem = (chownrequest_id_name_t*)_allocsafe( sizeof( chownrequest_id_name_t ) );
        elem->name = dupstring( pwdp->pw_name );
        elem->id.uid = pwdp->pw_uid;
        chownUserList->addElement( elem );
        pwdp = getpwent();
    }
    endpwent();

    //setgrent();  // is this needed?
    grpp = getgrent();
    while ( grpp != NULL ) {
        elem = (chownrequest_id_name_t*)_allocsafe( sizeof( chownrequest_id_name_t ) );
        elem->name = dupstring( grpp->gr_name );
        elem->id.gid = grpp->gr_gid;
        chownGroupList->addElement( elem );
        grpp = getgrent();
    }
    endgrent();
}

int NormalMode::requestNewOwner( FileEntry *fe, uid_t *return_owner, gid_t *return_group )
{
    uid_t towner;
    gid_t tgroup;
    Button *okb, *cb, *ok2allb, *skipb;
    AWindow *win;
    Text *ttext, *utext, *gtext;
    int tw, ttw, tth, ttx, tty;
    AGMessage *msg;
    int endmode = -1;
    char *tstr;
    GUIElement *ba[4];
    FieldListView *lvu, *lvg;
    chownrequest_id_name_t *elem;
    int row, pos, i;
    int w;
  
    if ( ( chownUserList == NULL ) || ( chownGroupList == NULL ) )
        buildOwnerRequestInfos();

    towner = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->duserid() : fe->userid();
    tgroup = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dgroupid() : fe->groupid();

    ttw = tth = 10;
    ttx = tty = 5;
    win = new AWindow( aguix, 10, 10, ttw, tth, 0, catalog.getLocaleCom( 43 ) );
    win->create();
    tstr = (char*)_allocsafe( strlen( catalog.getLocale( 448 ) ) + strlen( fe->fullname ) + 1 );
    sprintf( tstr, catalog.getLocale( 448 ), fe->fullname );

    ttext = (Text*)win->add( new Text( aguix, ttx, tty, tstr, 1 ) );
    _freesafe( tstr );
  
    tty += ttext->getHeight() + 5;
  
    utext = (Text*)win->add( new Text( aguix, ttx, tty, catalog.getLocale( 214 ), 1 ) );
    gtext = (Text*)win->add( new Text( aguix, ttx, tty, catalog.getLocale( 215 ), 1 ) );
    tty += utext->getHeight() + 5;

    lvu = (FieldListView*)win->add( new FieldListView( aguix, ttx, tty, 50, 10 * aguix->getCharHeight(), 0 ) );
    lvg = (FieldListView*)win->add( new FieldListView( aguix, ttx, tty, 50, 10 * aguix->getCharHeight(), 0 ) );
  
    lvu->setNrOfFields( 1 );
    lvg->setNrOfFields( 1 );

    elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
    pos = 0;
    while ( elem != NULL ) {
        row = lvu->addRow();
        lvu->setText( row, 0, elem->name );
        lvu->setData( row, pos );
        lvu->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
        if ( towner == elem->id.uid ) {
            lvu->setActiveRow( row );
        }
        elem = (chownrequest_id_name_t*)chownUserList->getNextElement();
        pos++;
    }
    elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
    pos = 0;
    while ( elem != NULL ) {
        row = lvg->addRow();
        lvg->setText( row, 0, elem->name );
        lvg->setData( row, pos );
        lvg->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
        if ( tgroup == elem->id.gid ) {
            lvg->setActiveRow( row );
        }
        elem = (chownrequest_id_name_t*)chownGroupList->getNextElement();
        pos++;
    }
  
    lvu->setVBarState( 2 );
    lvu->setHBarState( 0 );
    lvg->setVBarState( 2 );
    lvg->setHBarState( 0 );
    lvu->maximizeX();
    lvg->maximizeX();
    lvu->setDisplayFocus( true );
    lvg->setDisplayFocus( true );
    lvu->setAcceptFocus( true );
    lvg->setAcceptFocus( true );
  
    tw = a_max( utext->getWidth(), lvu->getWidth() );
    lvu->resize( a_max( tw, 20 * aguix->getTextWidth( "x" ) ), lvu->getHeight() );
    tw = a_max( gtext->getWidth(), lvg->getWidth() );
    lvg->resize( a_max( tw, 20 * aguix->getTextWidth( "x" ) ), lvg->getHeight() );
    lvg->move( lvu->getX() + lvu->getWidth() + 10, lvg->getY() );
    lvu->showActive();
    lvg->showActive();
  
    gtext->move( lvg->getX(), gtext->getY() );
  
    lvu->takeFocus();

    tty += lvu->getHeight() +5;
  
    win->maximizeX();
    w = win->getWidth();

    okb = (Button*)win->add( new Button( aguix,
                                         5,
                                         tty,
                                         catalog.getLocale( 11 ),
                                         1,
                                         0,
                                         0 ) );
    ok2allb = (Button*)win->add( new Button( aguix,
                                             0,
                                             tty,
                                             catalog.getLocale( 224 ),
                                             1,
                                             0,
                                             0 ) );
    skipb = (Button*)win->add( new Button( aguix,
                                           0,
                                           tty,
                                           catalog.getLocale( 225 ),
                                           1,
                                           0,
                                           0 ) );
    cb = (Button*)win->add( new Button( aguix,
                                        0,
                                        tty,
                                        catalog.getLocale( 8 ),
                                        1,
                                        0,
                                        0 ) );

    ba[0] = okb;
    ba[1] = ok2allb;
    ba[2] = skipb;
    ba[3] = cb;
    tw = AGUIX::scaleElementsW( w, 5, 5, -1, false, false, ba, NULL, 4 );
    if ( tw > w ) {
        w = tw;
        win->resize( w, win->getHeight() );
    }
  
    tty += okb->getHeight() + 5;
  
    for ( i = 0; i < 4; i++ ) {
        ba[i]->setAcceptFocus( true );
    }
  
    tth = tty;
    okb->takeFocus();
    win->setDoTabCycling( true );
    win->resize( w, tth );
    win->setMaxSize( w, tth );
    win->setMinSize( w, tth );
    win->show();
    for( ; endmode == -1; ) {
        msg = aguix->WaitMessage( win );
        if ( msg != NULL ) {
            switch ( msg->type ) {
                case AG_CLOSEWINDOW:
                    if ( msg->closewindow.window == win->getWindow() ) endmode = 3;
                    break;
                case AG_BUTTONCLICKED:
                    if ( msg->button.button == okb ) endmode = 0;
                    else if ( msg->button.button == ok2allb ) endmode = 1;
                    else if ( msg->button.button == skipb ) endmode = 2;
                    else if ( msg->button.button == cb ) endmode = 3;
                    break;
                case AG_KEYPRESSED:
                    if ( win->isParent( msg->key.window, false ) == true ) {
                        switch ( msg->key.key ) {
                            case XK_Return:
                            case XK_KP_Enter:
                                if ( ( ok2allb->getHasFocus() == false ) &&
                                     ( skipb->getHasFocus() == false ) &&
                                     ( cb->getHasFocus() == false ) ) {
                                    endmode = 0;
                                }
                                break;
                            case XK_F1:
                                endmode = 0;
                                break;
                            case XK_Escape:
                            case XK_F4:
                                endmode = 3;
                                break;
                            case XK_F2:
                                endmode = 1;
                                break;
                            case XK_F3:
                                endmode = 2;
                                break;
                        }
                    }
                    break;
                case AG_FIELDLV_PRESSED:
                    if ( msg->fieldlv.lv == lvu ) {
                        lvu->takeFocus();
                    } else if ( msg->fieldlv.lv == lvg ) {
                        lvg->takeFocus();
                    }
            }
            aguix->ReplyMessage( msg );
        }
    }
  
    if ( ( endmode == 0 ) || ( endmode == 1 ) ) {
        // ok
        if ( return_owner != NULL )
            *return_owner = (uid_t)-1;  // -1 means no change for chown
        if ( return_group != NULL )
            *return_group = (gid_t)-1;

        row = lvu->getActiveRow();
        if ( lvu->isValidRow( row ) == true ) {
            elem = (chownrequest_id_name_t*)chownUserList->getElementAt( lvu->getData( row ) );
            if ( elem != NULL ) {
                if ( return_owner != NULL )
                    *return_owner = elem->id.uid;
            }
        }

        row = lvg->getActiveRow();
        if ( lvg->isValidRow( row ) == true ) {
            elem = (chownrequest_id_name_t*)chownGroupList->getElementAt( lvg->getData( row ) );
            if ( elem != NULL ) {
                if ( return_group != NULL )
                    *return_group = elem->id.gid;
            }
        }
    }
  
    delete win;

    return endmode;
}

/*
 * tryApplyOwnerPerm will try to set owner and permission
 *   it will apply SUID/SGID only if owner or root (as root only
 *   if chown don't fail)
 * this function don't care if chmod/chown fails
 *
 * returnvalue:
 *   0 okay
 *   -1 wrong args
 *   Bit 0 set: SUID cleared
 *   Bit 1 set: SGID cleared
 *   Bit 2 set: chown failed
 *   Bit 3 set: chmod failed
 */
int NormalMode::tryApplyOwnerPerm( const char *filename,
                                   const uid_t wanted_user,
                                   const gid_t wanted_group,
                                   const mode_t wanted_mode,
                                   struct NM_copyorder *copyorder )
{
    mode_t use_mode;
    int erg = 0, chownres, chmodres;

    if ( filename == NULL ) return -1;
  
    chownres = worker_chown( filename, wanted_user, wanted_group );
    if ( chownres != 0 ) {
        erg |= 1<<2;
    }
    use_mode = wanted_mode;
    if ( ( use_mode & S_ISUID ) != 0 ) {
        // SUID bit set, check if we still want it
        // root will apply it (when chown didn't failed and)
        // and euid/egid == ower
        if ( ! ( ( geteuid() == wanted_user ) ||
                 ( ( geteuid() == 0 ) && ( chownres == 0 ) ) ) ) {
            use_mode &= ~S_ISUID;
            erg |= 1<<0;
        }
    }
    if ( ( use_mode & S_ISGID ) != 0 ) {
        // SGID bit set, check if we still want it
        // root will apply it (when chown didn't failed and)
        // and euid/egid == ower
        if ( ! ( ( getegid() == wanted_group ) ||
                 ( ( geteuid() == 0 ) && ( chownres == 0 ) ) ) ) {
            use_mode &= ~S_ISGID;
            erg |= 1<<1;
        }
    }
    chmodres = chmod ( filename, use_mode );
    if ( chmodres != 0 ) {
        erg |= 1<<3;
    }
    if ( ( ( erg & 3 ) != 0 ) && ( copyorder != NULL ) ) {
        // removed SUID/SGUID bit, let the user know this
        if ( copyorder->ignoreLosedAttr != true ) {
            int choose;
            char *textstr;
            std::string str2;

            textstr = (char*)_allocsafe( strlen( catalog.getLocale( 536 ) ) + strlen( filename ) + 1 );
            sprintf( textstr, catalog.getLocale( 536 ), filename );
            str2 = catalog.getLocale( 11 );
            str2 += "|";
            str2 += catalog.getLocale( 537 );
            choose = request( copyorder->cowin,
                              catalog.getLocale( 125 ),
                              textstr,
                              str2.c_str(),
                              Requester::REQUEST_CANCELWITHLEFT );
            _freesafe( textstr );
            if ( choose == 1 ) {
                copyorder->ignoreLosedAttr = true;
            }
        }
    }
    return erg;
}

NormalMode::changeown_info::changeown_info()
{
    forAll = false;
    newuid = (uid_t)-1;
    newgid = (gid_t)-1;
}
