/* pathjumpui.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "pathjumpui.hh"
#include <aguix/aguix.h>
#include <aguix/awindow.h>
#include <aguix/text.h>
#include <aguix/fieldlistview.h>
#include <aguix/button.h>
#include <aguix/solidbutton.h>
#include <aguix/util.h>
#include <aguix/choosebutton.h>
#include "worker_locale.h"
#include "nwc_path.hh"
#include <algorithm>
#include "worker.h"
#include "deeppathstore.hh"
#include "stringmatcher_fnmatch.hh"
#include "bookmarkdbproxy.hh"
#include "persdeeppathstore.hh"
#include "prefixdb.hh"
#include "wconfig.h"
#include "stringmatcher_flexiblematch.hh"

PathJumpUI::PathJumpUI( AGUIX &aguix, DeepPathStore &path_store, Worker &worker )
  : m_aguix( aguix ),
    m_path_store( path_store ),
    m_current_depth( -1 ),
    m_show_hidden_entries( false )
{
    std::list< std::string > paths = m_path_store.getPaths();
    int max_path_components = 0;
    std::list< std::string > pers_paths = worker.getPathsPers();

    m_bookmarks = Worker::getBookmarkDBInstance().getNamesOfEntries( "" );

    paths.sort();

    for ( std::list< std::string >::const_iterator it1 = paths.begin();
          it1 != paths.end();
          it1++ ) {
        m_entries.push_back( pathjump_entry( PATH_SHOW, -1, *it1 ) );

        int slashes = std::count( it1->begin(),
                                  it1->end(),
                                  '/' );

        if ( slashes + 1 > max_path_components ) {
            max_path_components = slashes + 1;
        }
    }

    // currently it will not be checked if there are entries in both
    // groups (visited paths and bookmarks+persistent paths since only
    // one group is shown at once

    DeepPathStore temp_store;

    for ( std::list< std::string >::const_iterator it1 = m_bookmarks.begin();
          it1 != m_bookmarks.end();
          it1++ ) {
        temp_store.storePath( *it1 );
    }

    for ( std::list< std::string >::const_iterator it1 = pers_paths.begin();
          it1 != pers_paths.end();
          it1++ ) {
        temp_store.storePath( *it1 );
    }

    std::list< std::string > temp_paths = temp_store.getPaths();

    for ( std::list< std::string >::const_iterator it1 = temp_paths.begin();
          it1 != temp_paths.end();
          it1++ ) {
        m_entries.push_back( pathjump_entry( BOOKMARK_HIDE, -1, *it1 ) );

        int slashes = std::count( it1->begin(),
                                  it1->end(),
                                  '/' );

        if ( slashes + 1 > max_path_components ) {
            max_path_components = slashes + 1;
        }
    }

    m_win = std::auto_ptr<AWindow>( new AWindow( &m_aguix,
                                                 0, 0,
                                                 400, 400,
                                                 0, catalog.getLocaleCom( 60 ) ) );
    m_win->create();

    m_co1 = m_win->setContainer( new AContainer( m_win.get(), 1, 5 ), true );
    m_co1->setMaxSpace( 5 );
    
    AContainer *co1_2 = m_co1->add( new AContainer( m_win.get(), 2, 1 ), 0, 0 );
    co1_2->setMaxSpace( 5 );
    co1_2->setBorderWidth( 0 );
    co1_2->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 758 ), 1 ),
                0, 0, AContainer::CO_FIX );
    m_infixtext = static_cast<Text*>( co1_2->add( new Text( &m_aguix, 0, 0, "", 1 ),
                                                  1, 0, AContainer::CO_INCW ) );

    m_show_all_cb = (ChooseButton*)m_co1->add( new ChooseButton( &m_aguix, 0, 0,
                                                                 false,
                                                                 catalog.getLocale( 977 ),
                                                                 LABEL_RIGHT, 1, 0 ),
                                               0, 1, AContainer::CO_INCWNR );

    m_breadcrumb_co = m_co1->add( new AContainer( m_win.get(), max_path_components + 1, 1 ), 0, 2 );
    m_breadcrumb_co->setMaxSpace( 5 );

    for ( int i = 0; i < max_path_components; i++ ) {
        Button *b = (Button*)m_breadcrumb_co->add( new Button( &m_aguix,
                                                               0,
                                                               0,
                                                               "",
                                                               1,
                                                               0,
                                                               0 ), i, 0, AContainer::CO_INCW );
        m_breadcrumb_buttons.push_back( b );
        b->setAcceptFocus( false );
        b->hide();
    }

    m_lv = static_cast<FieldListView*>( m_co1->add( new FieldListView( &m_aguix, 
                                                                     0, 0, 
                                                                     400, 300, 0 ),
                                                  0, 3, AContainer::CO_MIN ) );
    m_lv->setNrOfFields( 2 );
    m_lv->setShowHeader( true );
    m_lv->setFieldText( 1, catalog.getLocale( 809 ) );
    m_lv->setHBarState( 2 );
    m_lv->setVBarState( 2 );
    m_lv->setAcceptFocus( true );
    m_lv->setDisplayFocus( true );
    m_lv->setFieldText( 0, catalog.getLocale( 978 ) );
    m_lv->setGlobalFieldSpace( 5 );

    AContainer *co1_3 = m_co1->add( new AContainer( m_win.get(), 2, 1 ), 0, 4 );
    co1_3->setMinSpace( 5 );
    co1_3->setMaxSpace( -1 );
    co1_3->setBorderWidth( 0 );
    m_okb = (Button*)co1_3->add( new Button( &m_aguix,
                                             0,
                                             0,
                                             catalog.getLocale( 761 ),
                                             1,
                                             0,
                                             0 ), 0, 0, AContainer::CO_FIX );
    m_cancelb = (Button*)co1_3->add( new Button( &m_aguix,
                                                 0,
                                                 0,
                                                 catalog.getLocale( 633 ),
                                                 1,
                                                 0,
                                                 0 ), 1, 0, AContainer::CO_FIX );

    m_win->contMaximize( true );
    m_win->setDoTabCycling( true );

    std::string cfgfile = Worker::getWorkerConfigDir();
#ifdef USEOWNCONFIGFILES
    cfgfile = NWC::Path::join( cfgfile, "directory-history.prob2" );
#else
    cfgfile = NWC::Path::join( cfgfile, "directory-history.prob" );
#endif

    // make this instance static in this class or worker to avoid
    // reparsing it
    m_pdb = std::auto_ptr< PrefixDB >( new PrefixDB( cfgfile ) );
}

PathJumpUI::~PathJumpUI()
{
}

int PathJumpUI::mainLoop()
{
    showData();

    highlight_best_hit( m_dirname + "*" );

    maximizeWin();
    m_win->show();

    m_lv->takeFocus();

    AGMessage *msg;
    int endmode = 0;

    for ( ; endmode == 0; ) {
        msg = m_aguix.WaitMessage( NULL );
        if ( msg != NULL ) {
            switch ( msg->type ) {
              case AG_CLOSEWINDOW:
                  endmode = -1;
                  break;
              case AG_BUTTONCLICKED:
                  if ( msg->button.button == m_okb ) {
                      endmode = 1;
                  } else if ( msg->button.button == m_cancelb ) {
                      endmode = -1;
                  } else {
                      for ( unsigned int i = 0; i < m_breadcrumb_buttons.size(); i++ ) {
                          if ( m_breadcrumb_buttons[i] == msg->button.button ) {
                              m_current_depth = i;
                              endmode = 1;
                          }
                      }
                  }
                  break;
              case AG_KEYPRESSED:
                  if ( msg->key.key == XK_BackSpace ) {
                      if ( KEYSTATEMASK( msg->key.keystate ) == ShiftMask ) {
                          m_filter = "";
                          apply_filter( m_filter );
                          showData();
                      } else {
                          std::string cur_infix = m_filter;
                          
                          if ( cur_infix.length() > 0 ) {
                              cur_infix.resize( cur_infix.length() - 1 );
                              m_filter = cur_infix;
                              apply_filter( m_filter );
                              showData();
                          }
                      }
                  } else if ( msg->key.key == XK_Return ) {
                      endmode = 1;
                  } else if ( msg->key.key == XK_Escape ) {
                      endmode = -1;
                  } else if ( msg->key.key == XK_Left ) {
                      if ( ! m_current_components.empty() ) {
                          if ( m_current_depth < 0 ) {
                              m_current_depth = m_current_components.size() - 1;
                              update_breadcrumb();
                          } else if ( m_current_depth > 0 ) {
                              m_current_depth--;
                              update_breadcrumb();
                          }
                      }
                  } else if ( msg->key.key == XK_Right ) {
                      if ( ! m_current_components.empty() ) {
                          if ( m_current_depth < 0 ) {
                              m_current_depth = 0;
                              update_breadcrumb();
                          } else if ( m_current_depth < (int)( m_current_components.size() - 1 ) ) {
                              m_current_depth++;
                              update_breadcrumb();
                          }
                      }
                  } else if ( msg->key.key == XK_Up ) {
                  } else if ( msg->key.key == XK_Down ) {
                  } else if ( msg->key.key == XK_Next ) {
                  } else if ( msg->key.key == XK_Prior ) {
                  } else if ( msg->key.key == XK_Home ) {
                      // key is grabbed by listview so it has no effect at the moment
                      if ( ! m_current_components.empty() ) {
                          m_current_depth = 0;
                          update_breadcrumb();
                      }
                  } else if ( msg->key.key == XK_End ) {
                      // key is grabbed by listview so it has no effect at the moment
                      if ( ! m_current_components.empty() ) {
                          m_current_depth = m_current_components.size() - 1;
                          update_breadcrumb();
                      }
                  } else if ( IsModifierKey( msg->key.key ) ||
                              IsCursorKey( msg->key.key ) ||
                              IsPFKey( msg->key.key ) ||
                              IsFunctionKey( msg->key.key ) ||
                              IsMiscFunctionKey( msg->key.key ) ||
                              ( ( msg->key.key >= XK_BackSpace ) && ( msg->key.key <= XK_Escape ) ) ) {
                  } else if ( strlen( msg->key.keybuf ) > 0 ) {
                      if ( msg->key.key == XK_a &&
                           KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                          m_show_hidden_entries = ! m_show_hidden_entries;
                          m_show_all_cb->setState( m_show_hidden_entries );
                          apply_filter( m_filter );
                          showData();
                      } else {
                          std::string cur_infix = m_filter;
                          cur_infix += msg->key.keybuf;
                          if ( apply_filter( cur_infix ) > 0 ) {
                              m_filter = cur_infix;
                          } else {
                              apply_filter( m_filter );
                          }
                          showData();
                      }
                  }
                  break;
              case AG_FIELDLV_DOUBLECLICK:
                  if ( msg->fieldlv.lv == m_lv ) {
                      // double click in lv, actual element is unimportant here
                      endmode = 1;
                  }
                  break;
                case AG_FIELDLV_ONESELECT:
                case AG_FIELDLV_MULTISELECT:
                    if ( msg->fieldlv.lv == m_lv ) {
                        update_breadcrumb( m_filter );
                    }
                    break;
                case AG_CHOOSECLICKED:
                    if ( msg->choose.button == m_show_all_cb ) {
                        m_show_hidden_entries = msg->choose.state;
                        apply_filter( m_filter );
                        showData();
                    }
                    break;
            }
            m_aguix.ReplyMessage( msg );
        }
    }

    m_selected_path = "";

    if ( endmode == 1 ) {
        std::string res = "";

        for ( unsigned int i = 0; i < m_current_components.size(); i++ ) {
            if ( m_current_depth >= 0 && (int)i > m_current_depth ) break;

            res += "/";
            res += m_current_components[i];
        }

        m_selected_path = res;

        // check if this path is allowed to be tracked, than put its
        // access into the prefixdb
        if ( ! m_filter.empty() && ! wconfig->getPathJumpAllowDirs().empty() ) {
            const std::list< std::string > &l = wconfig->getPathJumpAllowDirs();

            for ( std::list< std::string >::const_iterator it1 = l.begin();
                  it1 != l.end();
                  it1++ ) {
                if ( AGUIXUtils::starts_with( m_selected_path, *it1 ) ) {
                    m_pdb->pushAccess( m_filter, m_selected_path );
                    break;
                }
            }
        }
    }

    m_win->hide();
    
    return endmode;
}

void PathJumpUI::showData()
{
    m_infixtext->setText( m_filter.c_str() );

    int current_row = m_lv->getActiveRow();
    std::string current_path;

    if ( m_lv->isValidRow( current_row ) ) {
        current_path = m_lv->getText( current_row, 1 );
    }

    std::string best_hit;

    if ( ! m_filter.empty() ) {
        best_hit = m_pdb->getBestHit( m_filter );
    }

    m_lv->setSize( 0 );
    
    int size = 0;
    bool best_hit_found = false;

    for ( std::list< pathjump_entry >::const_iterator it1 = m_entries.begin();
          it1 != m_entries.end();
          it1++ ) {
        if ( it1->m_type == PATH_SHOW ||
             it1->m_type == BOOKMARK_SHOW ) {
            int row = m_lv->addRow();

            m_lv->setText( row, 1, it1->m_path );

            if ( it1->m_blockcount >= 0 && ! m_filter.empty() ) {
                std::string qual = AGUIXUtils::formatStringToString( "%3.0f%%",
                                                                     100.0 / (double)( it1->m_blockcount + 1 ) );
                m_lv->setText( row, 0, qual );
            }

            m_lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );

            if ( ! best_hit_found &&
                 ! best_hit.empty() &&
                 AGUIXUtils::starts_with( it1->m_path, best_hit ) ) {
                m_lv->setActiveRow( row );
                best_hit_found = true;
            }

            size++;
        }
    }

    if ( ! best_hit_found ) {
        for ( int row = 0; row < m_lv->getElements(); row++ ) {
            if ( AGUIXUtils::starts_with( m_lv->getText( row, 1 ),
                                          current_path ) ) {
                m_lv->setActiveRow( row );
                break;
            }
        }
    }

    if ( m_lv->isValidRow( m_lv->getActiveRow() ) == false ) {
        if ( m_lv->isValidRow( current_row ) ) {
            m_lv->setActiveRow( current_row );
        } else if ( current_row < 0 ) {
            m_lv->setActiveRow( 0 );
        } else {
            m_lv->setActiveRow( size - 1 );
        }
    }

    m_lv->showActive();
    m_lv->redraw();

    if ( best_hit_found ) {
        std::string f = best_hit;
        f += "*";
        update_breadcrumb( f );
    } else {
        update_breadcrumb( m_filter );
    }
}

std::string PathJumpUI::getSelectedPath()
{
    return m_selected_path;
}

void PathJumpUI::setCurrentDirname( const std::string &dirname )
{
    m_dirname = dirname;
}

void PathJumpUI::setCurrentBasename( const std::string &basename )
{
    m_basename = basename;
}

void PathJumpUI::maximizeWin()
{
    m_lv->maximizeX();
    m_lv->maximizeY();
    int my_w = m_lv->getWidth() + 10;
    int my_h = m_lv->getHeight() + 10;

    if ( my_w < 400 ) my_w = 400;
    if ( my_h < 300 ) my_h = 300;

    int mw = m_aguix.getRootWindowWidth() * 80 / 100;
    int mh = m_aguix.getRootWindowHeight() * 80 / 100;
    m_co1->resize( mw, mh );
    m_co1->rearrange();

    if ( my_w < m_lv->getWidth() ) {
        m_co1->setMinWidth( my_w, 0, 3 );
    } else {
        m_co1->setMinWidth( m_lv->getWidth(), 0, 3 );
    }
    if ( my_h < m_lv->getHeight() ) {
        m_co1->setMinHeight( my_h, 0, 3 );
    } else {
        m_co1->setMinHeight( m_lv->getHeight(), 0, 3 );
    }
    m_win->contMaximize( true );
}

void PathJumpUI::highlight_best_hit( const std::string &filter )
{
    StringMatcherFNMatch matcher;
    int matches = 0;

    matcher.setMatchString( filter );
    matcher.setMatchCaseSensitive( false );

    for ( int i = 0; i < m_lv->getElements(); i++ ) {
        if ( matcher.match( m_lv->getText( i, 1 ) ) ) {
            m_lv->setActiveRow( i );
	    m_lv->showActive();
            matches++;
            break;
        }
    }

    if ( ! matches ) {
        std::string flexible_filter;

        if ( ! filter.empty() && filter[0] == '*' ) {
            flexible_filter = "*";
        }

        flexible_filter += StringMatcherFNMatch::convertMatchStringToFlexibleMatch( filter ) + "*";

        matcher.setMatchString( flexible_filter );
        matcher.setMatchCaseSensitive( false );

        for ( int i = 0; i < m_lv->getElements(); i++ ) {
            if ( matcher.match( m_lv->getText( i, 1 ) ) ) {
                m_lv->setActiveRow( i );
                m_lv->showActive();
                matches++;
                break;
            }
        }
    }

    if ( m_lv->isValidRow( m_lv->getActiveRow() ) == true ) {
        update_breadcrumb( m_lv->getText( m_lv->getActiveRow(), 1 ),
                           filter );
    }
}

int PathJumpUI::apply_filter( const std::string &filter )
{
    int matches = 0;

    StringMatcherFlexibleMatch flmatch;
    StringMatcherFNMatch match;
    bool use_flexible = true;

    flmatch.setMatchString( filter );
    flmatch.setMatchCaseSensitive( false );

    if ( filter.find( "*" ) != std::string::npos ) {
        std::string real_filter = "*";
        real_filter += filter;
        real_filter += "*";

        match.setMatchString( real_filter );
        match.setMatchCaseSensitive( false );

        use_flexible = false;
    }

    int exact_path_matches = 0, exact_bookmark_matches = 0,
        path_matches = 0, bookmark_matches = 0;

    for ( std::list< pathjump_entry >::iterator it1 = m_entries.begin();
          it1 != m_entries.end();
          it1++ ) {
        if ( use_flexible ) {
            it1->m_blockcount = flmatch.countNonMatchingBlocks( it1->m_path );
        } else {
            it1->m_blockcount = match.match( it1->m_path ) ? 0 : -1;
        }

        if ( it1->m_blockcount == 0 ) {
            if ( it1->m_type == PATH_SHOW ||
                 it1->m_type == PATH_HIDE ) {
                exact_path_matches++;
                path_matches++;
            } else {
                exact_bookmark_matches++;
                bookmark_matches++;
            }
        } else if ( it1->m_blockcount > 0 ) {
            if ( it1->m_type == PATH_SHOW ||
                 it1->m_type == PATH_HIDE ) {
                path_matches++;
            } else {
                bookmark_matches++;
            }
        }
    }

    m_entries.sort();

    for ( std::list< pathjump_entry >::iterator it1 = m_entries.begin();
          it1 != m_entries.end();
          it1++ ) {
        if ( it1->m_type == PATH_SHOW ||
             it1->m_type == PATH_HIDE ) {
            if ( it1->m_blockcount == 0 ||
                 ( it1->m_blockcount > 0 && exact_path_matches == 0 ) ) {
                it1->m_type = PATH_SHOW;
                matches++;
            } else {
                it1->m_type = PATH_HIDE;
            }
        } else {
            if ( ( path_matches == 0 && ! filter.empty() ) ||
                 m_show_hidden_entries == true ) {
                if ( it1->m_blockcount == 0 ||
                     ( it1->m_blockcount > 0 && exact_bookmark_matches == 0 ) ) {
                    it1->m_type = BOOKMARK_SHOW;
                    matches++;
                } else {
                    it1->m_type = BOOKMARK_HIDE;
                }
            } else {
                    it1->m_type = BOOKMARK_HIDE;
            }
        }
    }

    return matches;
}

int PathJumpUI::update_breadcrumb( const std::string &path,
                                   const std::string &highlight_filter )
{
    std::vector< std::string > components;
    StringMatcherFNMatch matcher;

    m_current_components.clear();

    AGUIXUtils::split_string( components, path, '/' );

    if ( components.size() > m_breadcrumb_buttons.size() ) {
        // something wrong here, shouldn't happen
        return 1;
    }

    if ( ! highlight_filter.empty() ) {
        matcher.setMatchString( highlight_filter );
        matcher.setMatchCaseSensitive( false );
        m_current_depth = -1;

        if ( ! matcher.match( path ) ) {
            std::string flexible_filter;

            if ( ! highlight_filter.empty() && highlight_filter[0] == '*' ) {
                flexible_filter = "*";
            }

            flexible_filter += StringMatcherFNMatch::convertMatchStringToFlexibleMatch( highlight_filter ) + "*";

            matcher.setMatchString( flexible_filter );
            matcher.setMatchCaseSensitive( false );
        }
    }

    unsigned int i = 0;
    std::string prefix_path = "";
    bool smallest_match_found = false;

    for ( std::vector< std::string >::const_iterator it1 = components.begin();
          it1 != components.end();
          it1++ ) {
        if ( ! it1->empty() ) {
            m_breadcrumb_buttons[i]->setText( 0, it1->c_str() );
            m_current_components.push_back( *it1 );
            prefix_path += "/" + *it1;

            if ( ! smallest_match_found &&
                 ! highlight_filter.empty() &&
                 matcher.match( prefix_path ) ) {
                m_current_depth = i;
                smallest_match_found = true;
            }

            if ( m_current_depth == (int)i ) {
                m_breadcrumb_buttons[i]->setFG( 0, 2 );
                m_breadcrumb_buttons[i]->setBG( 0, 3 );
            } else {
                m_breadcrumb_buttons[i]->setFG( 0, 1 );
                m_breadcrumb_buttons[i]->setBG( 0, 0 );
            }

            m_breadcrumb_buttons[i]->show();

            m_breadcrumb_co->setMaxWidth( m_breadcrumb_buttons[i]->getMaximumWidth(),
                                          i, 0 );
            i++;
        }
    }

    for ( ; i < m_breadcrumb_buttons.size(); i++ ) {
        m_breadcrumb_buttons[i]->hide();
        m_breadcrumb_co->setMaxWidth( 0,
                                      i, 0 );
    }

    m_win->updateCont();

    return 0;
}

int PathJumpUI::update_breadcrumb( const std::string &highlight_filter )
{
    if ( m_lv->isValidRow( m_lv->getActiveRow() ) == true ) {
        if ( ! highlight_filter.empty() &&
             highlight_filter[0] != '*' ) {
            return update_breadcrumb( m_lv->getText( m_lv->getActiveRow(), 1 ),
                                      "*" + highlight_filter );
        } else {
            return update_breadcrumb( m_lv->getText( m_lv->getActiveRow(), 1 ),
                                      highlight_filter );
        }
    }
    return 1;
}

int PathJumpUI::update_breadcrumb()
{
    if ( m_lv->isValidRow( m_lv->getActiveRow() ) == true ) {
        return update_breadcrumb( m_lv->getText( m_lv->getActiveRow(), 1 ),
                                  "" );
    }
    return 1;
}
