/* deeppathstore.hh
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012-2015 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DEEPPATHSTORE_HH
#define DEEPPATHSTORE_HH

#include "wdefines.h"
#include <list>
#include <string>

class DeepPathNode
{
public:
    DeepPathNode( const std::string &name, time_t ts ) : m_name( name ),
                                                         m_ts( ts )
    {
    }

    DeepPathNode()
    {
    }

    const std::string &get_name() const
    {
        return m_name;
    }

    const time_t get_ts() const
    {
        return m_ts;
    }

    void set_ts( time_t ts )
    {
        m_ts = ts;
    }

    const std::list< DeepPathNode > &get_children() const
    {
        return m_children;
    }

    typedef enum {
        NODE_UNCHANGED,
        NODE_INSERTED,
        NODE_UPDATED,
    } node_change_t;

    DeepPathNode &lookup_or_insert( const std::string &name, time_t ts, node_change_t *changed )
    {
        for ( std::list< DeepPathNode >::iterator it1 = m_children.begin();
              it1 != m_children.end();
              it1++ ) {
            if ( it1->get_name() == name ) return *it1;
        }

        if ( changed != NULL ) {
            *changed = NODE_INSERTED;
        }

        return *m_children.insert( m_children.end(), DeepPathNode( name, ts ) );
    }

    DeepPathNode *lookup( const std::string &name )
    {
        for ( std::list< DeepPathNode >::iterator it1 = m_children.begin();
              it1 != m_children.end();
              it1++ ) {
            if ( it1->get_name() == name ) return &(*it1);
        }

        return NULL;
    }

    void clear()
    {
        m_children.clear();
    }
private:
    std::list< DeepPathNode > m_children;
    std::string m_name;
    time_t m_ts;
};

class DeepPathStore
{
public:
    DeepPathStore();

    void storePath( const std::string &path, time_t ts, DeepPathNode::node_change_t *changed = NULL );
    std::list< std::pair< std::string, time_t > > getPaths();
    void clear();
private:
    void storePath( DeepPathNode &node,
                    const std::string &path,
                    time_t ts,
                    std::string::size_type start_pos,
                    DeepPathNode::node_change_t *changed );

    DeepPathNode m_root;
};

#endif
