// DBus-template.hh --- DBUS template
//
// Copyright (C) 2007, 2008 Rob Caelers <robc@krandor.nl>
// All rights reserved.
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// $Id: DBus.hh 1404 2008-01-07 20:48:30Z rcaelers $
//

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string>
#include <list>
#include <map>
#include <deque>

#include "DBus.hh"
#include "DBusBinding.hh"
#include "DBusException.hh"
#include "DBusWorkrave.hh"

using namespace std;



using namespace workrave;

#include "Core.hh"

class org_workrave_CoreInterface_Stub : public DBusBindingBase, public org_workrave_CoreInterface
{
private:
  typedef DBusMessage * (org_workrave_CoreInterface_Stub::*DBusMethod)(void *object, DBusMessage *message);

  virtual DBusMessage *call(int method_num, void *object, DBusMessage *message);
   
  virtual DBusIntrospect *get_method_introspect()
  {
    return method_introspect;
  }

  virtual DBusIntrospect *get_signal_introspect()
  {
    return signal_introspect;
  }

public:
  org_workrave_CoreInterface_Stub(DBus *dbus);
  ~org_workrave_CoreInterface_Stub();
  
  void MicrobreakChanged(const string &path,    std::string progress  );
  void RestbreakChanged(const string &path,    std::string progress  );
  void DailylimitChanged(const string &path,    std::string progress  );

  
private:
  DBusMessage *SetOperationMode(void *object, DBusMessage *message);
  DBusMessage *GetOperationMode(void *object, DBusMessage *message);
  DBusMessage *ReportActivity(void *object, DBusMessage *message);
  DBusMessage *IsTimerRunning(void *object, DBusMessage *message);
  DBusMessage *GetTimerIdle(void *object, DBusMessage *message);
  DBusMessage *GetTimerElapsed(void *object, DBusMessage *message);
  DBusMessage *GetTime(void *object, DBusMessage *message);
  DBusMessage *IsActive(void *object, DBusMessage *message);
  DBusMessage *PostponeBreak(void *object, DBusMessage *message);
  DBusMessage *SkipBreak(void *object, DBusMessage *message);

  void get_operation_mode(DBusMessageIter *reader, OperationMode *result);
  void put_operation_mode(DBusMessageIter *writer, const OperationMode *result);
  void get_break_id(DBusMessageIter *reader, BreakId *result);
  void put_break_id(DBusMessageIter *writer, const BreakId *result);



  
  static DBusMethod method_table[];
  static DBusIntrospect method_introspect[];
  static DBusIntrospect signal_introspect[];
};


org_workrave_CoreInterface *org_workrave_CoreInterface::instance(const DBus *dbus)
{
  org_workrave_CoreInterface_Stub *iface = NULL;
  DBusBindingBase *binding = dbus->find_binding("org.workrave.CoreInterface");
  
  if (binding != NULL)
    {
      iface = dynamic_cast<org_workrave_CoreInterface_Stub *>(binding);
    }

  return iface;
}

org_workrave_CoreInterface_Stub::org_workrave_CoreInterface_Stub(DBus *dbus)
  : DBusBindingBase(dbus)
{
}


org_workrave_CoreInterface_Stub::~org_workrave_CoreInterface_Stub()
{
}

DBusMessage *
org_workrave_CoreInterface_Stub::call(int method_num, void *object, DBusMessage *message)
{
  DBusMessage *ret = NULL;
  
  if (method_num >=0 && method_num < 10 )
    {
      DBusMethod m = method_table[method_num];
      if (m != NULL)
        {
          ret = (this->*m)(object, message);
        }
    }
  
  return ret;
}
 

void
org_workrave_CoreInterface_Stub::get_operation_mode(DBusMessageIter *reader, OperationMode *result)
{
  std::string value;
	int argtype = dbus_message_iter_get_arg_type(reader);

  if (argtype != DBUS_TYPE_STRING)
		throw DBusTypeException("Type mismatch. Excepted string");

  get_string(reader, &value);
  
  if ("normal" == value)
    {
      *result = OPERATION_MODE_NORMAL;
    }
  else if ("suspended" == value)
    {
      *result = OPERATION_MODE_SUSPENDED;
    }
  else if ("quiet" == value)
    {
      *result = OPERATION_MODE_QUIET;
    }
  else
    {
      throw DBusTypeException("Illegal enum value");
    }
}

void
org_workrave_CoreInterface_Stub::put_operation_mode(DBusMessageIter *writer, const OperationMode *result)
{
  string value;
  switch (*result)
    {
    case OPERATION_MODE_NORMAL:
      value = "normal";
      break;
    case OPERATION_MODE_SUSPENDED:
      value = "suspended";
      break;
    case OPERATION_MODE_QUIET:
      value = "quiet";
      break;
    default:
      throw DBusTypeException("Illegal enum value");
    }
        	
  put_string(writer, &value);
}


void
org_workrave_CoreInterface_Stub::get_break_id(DBusMessageIter *reader, BreakId *result)
{
  std::string value;
	int argtype = dbus_message_iter_get_arg_type(reader);

  if (argtype != DBUS_TYPE_STRING)
		throw DBusTypeException("Type mismatch. Excepted string");

  get_string(reader, &value);
  
  if ("microbreak" == value)
    {
      *result = BREAK_ID_MICRO_BREAK;
    }
  else if ("restbreak" == value)
    {
      *result = BREAK_ID_REST_BREAK;
    }
  else if ("dailylimit" == value)
    {
      *result = BREAK_ID_DAILY_LIMIT;
    }
  else
    {
      throw DBusTypeException("Illegal enum value");
    }
}

void
org_workrave_CoreInterface_Stub::put_break_id(DBusMessageIter *writer, const BreakId *result)
{
  string value;
  switch (*result)
    {
    case BREAK_ID_MICRO_BREAK:
      value = "microbreak";
      break;
    case BREAK_ID_REST_BREAK:
      value = "restbreak";
      break;
    case BREAK_ID_DAILY_LIMIT:
      value = "dailylimit";
      break;
    default:
      throw DBusTypeException("Illegal enum value");
    }
        	
  put_string(writer, &value);
}







DBusMessage *
org_workrave_CoreInterface_Stub::SetOperationMode(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      OperationMode mode;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_operation_mode(&reader, &mode);

      dbus_object->set_operation_mode(        mode      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::GetOperationMode(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      OperationMode mode;
      
      ok = dbus_message_iter_init(message, &reader);
        

      mode = dbus_object->get_operation_mode(       );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_operation_mode(&writer, &mode);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::ReportActivity(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      std::string who;
      bool act;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &who);
      get_bool(&reader, &act);

      dbus_object->report_external_activity(        who      , act      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::IsTimerRunning(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      BreakId timer_id;
      bool value;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_break_id(&reader, &timer_id);

      dbus_object->is_timer_running(        timer_id      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &value);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::GetTimerIdle(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      BreakId timer_id;
      gint32 value;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_break_id(&reader, &timer_id);

      dbus_object->get_timer_idle(        timer_id      , &value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_int32(&writer, &value);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::GetTimerElapsed(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      BreakId timer_id;
      gint32 value;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_break_id(&reader, &timer_id);

      dbus_object->get_timer_elapsed(        timer_id      , &value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_int32(&writer, &value);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::GetTime(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      gint32 value;
      
      ok = dbus_message_iter_init(message, &reader);
        

      value = dbus_object->get_time(       );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_int32(&writer, &value);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::IsActive(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      bool value;
      
      ok = dbus_message_iter_init(message, &reader);
        

      value = dbus_object->is_user_active(       );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &value);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::PostponeBreak(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      BreakId timer_id;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_break_id(&reader, &timer_id);

      dbus_object->postpone_break(        timer_id      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_CoreInterface_Stub::SkipBreak(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      Core *dbus_object = (Core *) object;
      
      BreakId timer_id;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_break_id(&reader, &timer_id);

      dbus_object->skip_break(        timer_id      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

void org_workrave_CoreInterface_Stub::MicrobreakChanged(const string &path,  std::string progress)
{
  DBusMessage *msg = NULL;
  DBusMessageIter writer;

  msg = dbus_message_new_signal(path.c_str(),
                                "org.workrave.CoreInterface",
                                "MicrobreakChanged");
  if (msg == NULL)
    {
      throw DBusSystemException("Unable to send signal");
    }

  dbus_message_iter_init_append(msg, &writer);

  try
    {
      put_string(&writer, &progress);
    }
  catch (DBusException &e)
    {
      dbus_message_unref(msg);
      throw;
    }

  send(msg);
}

void org_workrave_CoreInterface_Stub::RestbreakChanged(const string &path,  std::string progress)
{
  DBusMessage *msg = NULL;
  DBusMessageIter writer;

  msg = dbus_message_new_signal(path.c_str(),
                                "org.workrave.CoreInterface",
                                "RestbreakChanged");
  if (msg == NULL)
    {
      throw DBusSystemException("Unable to send signal");
    }

  dbus_message_iter_init_append(msg, &writer);

  try
    {
      put_string(&writer, &progress);
    }
  catch (DBusException &e)
    {
      dbus_message_unref(msg);
      throw;
    }

  send(msg);
}

void org_workrave_CoreInterface_Stub::DailylimitChanged(const string &path,  std::string progress)
{
  DBusMessage *msg = NULL;
  DBusMessageIter writer;

  msg = dbus_message_new_signal(path.c_str(),
                                "org.workrave.CoreInterface",
                                "DailylimitChanged");
  if (msg == NULL)
    {
      throw DBusSystemException("Unable to send signal");
    }

  dbus_message_iter_init_append(msg, &writer);

  try
    {
      put_string(&writer, &progress);
    }
  catch (DBusException &e)
    {
      dbus_message_unref(msg);
      throw;
    }

  send(msg);
}

  
org_workrave_CoreInterface_Stub::DBusMethod org_workrave_CoreInterface_Stub::method_table[] = {
  &org_workrave_CoreInterface_Stub::SetOperationMode,
  &org_workrave_CoreInterface_Stub::GetOperationMode,
  &org_workrave_CoreInterface_Stub::ReportActivity,
  &org_workrave_CoreInterface_Stub::IsTimerRunning,
  &org_workrave_CoreInterface_Stub::GetTimerIdle,
  &org_workrave_CoreInterface_Stub::GetTimerElapsed,
  &org_workrave_CoreInterface_Stub::GetTime,
  &org_workrave_CoreInterface_Stub::IsActive,
  &org_workrave_CoreInterface_Stub::PostponeBreak,
  &org_workrave_CoreInterface_Stub::SkipBreak,
};

DBusIntrospect org_workrave_CoreInterface_Stub::method_introspect[] = {
  { "SetOperationMode",
    "in\0s\0mode\0"
  },
  { "GetOperationMode",
    "out\0s\0mode\0"
  },
  { "ReportActivity",
    "in\0s\0who\0in\0b\0act\0"
  },
  { "IsTimerRunning",
    "in\0s\0timer_id\0out\0b\0value\0"
  },
  { "GetTimerIdle",
    "in\0s\0timer_id\0out\0i\0value\0"
  },
  { "GetTimerElapsed",
    "in\0s\0timer_id\0out\0i\0value\0"
  },
  { "GetTime",
    "out\0i\0value\0"
  },
  { "IsActive",
    "out\0b\0value\0"
  },
  { "PostponeBreak",
    "in\0s\0timer_id\0"
  },
  { "SkipBreak",
    "in\0s\0timer_id\0"
  },
  { NULL,
    NULL
  }
};
  
DBusIntrospect org_workrave_CoreInterface_Stub::signal_introspect[] = {
  { "MicrobreakChanged",
    "s\0progress\0"
  },
  { "RestbreakChanged",
    "s\0progress\0"
  },
  { "DailylimitChanged",
    "s\0progress\0"
  },
  { NULL,
    NULL
  }
};

 


using namespace workrave;

#include "IConfigurator.hh"

class org_workrave_ConfigInterface_Stub : public DBusBindingBase, public org_workrave_ConfigInterface
{
private:
  typedef DBusMessage * (org_workrave_ConfigInterface_Stub::*DBusMethod)(void *object, DBusMessage *message);

  virtual DBusMessage *call(int method_num, void *object, DBusMessage *message);
   
  virtual DBusIntrospect *get_method_introspect()
  {
    return method_introspect;
  }

  virtual DBusIntrospect *get_signal_introspect()
  {
    return signal_introspect;
  }

public:
  org_workrave_ConfigInterface_Stub(DBus *dbus);
  ~org_workrave_ConfigInterface_Stub();
  

  
private:
  DBusMessage *SetString(void *object, DBusMessage *message);
  DBusMessage *SetInt(void *object, DBusMessage *message);
  DBusMessage *SetBool(void *object, DBusMessage *message);
  DBusMessage *SetDouble(void *object, DBusMessage *message);
  DBusMessage *GetString(void *object, DBusMessage *message);
  DBusMessage *GetInt(void *object, DBusMessage *message);
  DBusMessage *GetBool(void *object, DBusMessage *message);
  DBusMessage *GetDouble(void *object, DBusMessage *message);




  
  static DBusMethod method_table[];
  static DBusIntrospect method_introspect[];
  static DBusIntrospect signal_introspect[];
};


org_workrave_ConfigInterface *org_workrave_ConfigInterface::instance(const DBus *dbus)
{
  org_workrave_ConfigInterface_Stub *iface = NULL;
  DBusBindingBase *binding = dbus->find_binding("org.workrave.ConfigInterface");
  
  if (binding != NULL)
    {
      iface = dynamic_cast<org_workrave_ConfigInterface_Stub *>(binding);
    }

  return iface;
}

org_workrave_ConfigInterface_Stub::org_workrave_ConfigInterface_Stub(DBus *dbus)
  : DBusBindingBase(dbus)
{
}


org_workrave_ConfigInterface_Stub::~org_workrave_ConfigInterface_Stub()
{
}

DBusMessage *
org_workrave_ConfigInterface_Stub::call(int method_num, void *object, DBusMessage *message)
{
  DBusMessage *ret = NULL;
  
  if (method_num >=0 && method_num < 8 )
    {
      DBusMethod m = method_table[method_num];
      if (m != NULL)
        {
          ret = (this->*m)(object, message);
        }
    }
  
  return ret;
}
 






DBusMessage *
org_workrave_ConfigInterface_Stub::SetString(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      std::string value;
      bool success;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);
      get_string(&reader, &value);

      success = dbus_object->set_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &success);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_ConfigInterface_Stub::SetInt(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      gint32 value;
      bool success;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);
      get_int32(&reader, &value);

      success = dbus_object->set_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &success);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_ConfigInterface_Stub::SetBool(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      bool value;
      bool success;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);
      get_bool(&reader, &value);

      success = dbus_object->set_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &success);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_ConfigInterface_Stub::SetDouble(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      double value;
      bool success;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);
      get_double(&reader, &value);

      success = dbus_object->set_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &success);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_ConfigInterface_Stub::GetString(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      bool found;
      std::string value;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);

      found = dbus_object->get_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &found);
      put_string(&writer, &value);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_ConfigInterface_Stub::GetInt(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      gint32 value;
      bool found;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);

      found = dbus_object->get_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_int32(&writer, &value);
      put_bool(&writer, &found);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_ConfigInterface_Stub::GetBool(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      bool value;
      bool found;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);

      found = dbus_object->get_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_bool(&writer, &value);
      put_bool(&writer, &found);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

DBusMessage *
org_workrave_ConfigInterface_Stub::GetDouble(void *object, DBusMessage *message)
{
  DBusMessage *reply = NULL;

  DBusMessageIter reader;
  DBusMessageIter writer;
  dbus_bool_t ok;

  try
    {
      IConfigurator *dbus_object = (IConfigurator *) object;
      
      std::string key;
      double value;
      bool found;
      
      ok = dbus_message_iter_init(message, &reader);
      if (!ok)
        {
          throw DBusSystemException("No parameters");
        }
        
      get_string(&reader, &key);

      found = dbus_object->get_value(        key      , value      );
                                                              
      reply = dbus_message_new_method_return(message);
      if (reply == NULL)
        {
          throw DBusSystemException("Internal error");
        }
                                                              
      dbus_message_iter_init_append(reply, &writer);
      
      put_double(&writer, &value);
      put_bool(&writer, &found);
  }
  catch (DBusException)
    {
      if (reply != NULL)
        {
          dbus_message_unref(reply);
        }

      throw;
    }
                                                          
  return reply;

}

  

  
org_workrave_ConfigInterface_Stub::DBusMethod org_workrave_ConfigInterface_Stub::method_table[] = {
  &org_workrave_ConfigInterface_Stub::SetString,
  &org_workrave_ConfigInterface_Stub::SetInt,
  &org_workrave_ConfigInterface_Stub::SetBool,
  &org_workrave_ConfigInterface_Stub::SetDouble,
  &org_workrave_ConfigInterface_Stub::GetString,
  &org_workrave_ConfigInterface_Stub::GetInt,
  &org_workrave_ConfigInterface_Stub::GetBool,
  &org_workrave_ConfigInterface_Stub::GetDouble,
};

DBusIntrospect org_workrave_ConfigInterface_Stub::method_introspect[] = {
  { "SetString",
    "in\0s\0key\0in\0s\0value\0out\0b\0success\0"
  },
  { "SetInt",
    "in\0s\0key\0in\0i\0value\0out\0b\0success\0"
  },
  { "SetBool",
    "in\0s\0key\0in\0b\0value\0out\0b\0success\0"
  },
  { "SetDouble",
    "in\0s\0key\0in\0d\0value\0out\0b\0success\0"
  },
  { "GetString",
    "in\0s\0key\0out\0b\0found\0out\0s\0value\0"
  },
  { "GetInt",
    "in\0s\0key\0out\0i\0value\0out\0b\0found\0"
  },
  { "GetBool",
    "in\0s\0key\0out\0b\0value\0out\0b\0found\0"
  },
  { "GetDouble",
    "in\0s\0key\0out\0d\0value\0out\0b\0found\0"
  },
  { NULL,
    NULL
  }
};
  
DBusIntrospect org_workrave_ConfigInterface_Stub::signal_introspect[] = {
  { NULL,
    NULL
  }
};

 

void init_DBusWorkrave(DBus *dbus)
{
  dbus->register_binding("org.workrave.CoreInterface", new org_workrave_CoreInterface_Stub(dbus));
  dbus->register_binding("org.workrave.ConfigInterface", new org_workrave_ConfigInterface_Stub(dbus));
}
