/* constants */
/* how tall the keyboard should be by default (can be overriden) */
#define KBD_PIXEL_HEIGHT 240

/* how tall the keyboard should be by default (can be overriden) */
#define KBD_PIXEL_LANDSCAPE_HEIGHT 120

/* spacing around each key */
#define KBD_KEY_BORDER 1

/* layout declarations */
enum layout_id {
	Full = 0,
	Special,
	Simple,
	SimpleGrid,
	Dialer,
	Cyrillic,
	Arabic,
	Persian,
	Greek,
	Georgian,
	Emoji,
	Nav,
	Landscape,
	ComposeA,
	ComposeE,
	ComposeY,
	ComposeU,
	ComposeI,
	ComposeO,
	ComposeW,
	ComposeR,
	ComposeT,
	ComposeP,
	ComposeS,
	ComposeD,
	ComposeF,
	ComposeG,
	ComposeH,
	ComposeJ,
	ComposeK,
	ComposeL,
	ComposeZ,
	ComposeX,
	ComposeC,
	ComposeV,
	ComposeB,
	ComposeN,
	ComposeM,
	ComposeMath,
	ComposePunctuation,
	ComposeBracket,
	ComposeCyrI,
	ComposeCyrJ,
	ComposeCyrE,
	ComposeCyrU,
	ComposeCyrL,
	ComposeCyrN,
	ComposeCyrTse,
	ComposeCyrChe,
	ComposeCyrG,
	ComposeCyrK,
	NumLayouts,
};

static struct key keys_full[], keys_special[], keys_simple[], keys_simplegrid[],
  keys_cyrillic[], keys_arabic[], keys_persian[], keys_georgian[], keys_greek[],
  keys_emoji[], keys_nav[], keys_landscape[], keys_compose_a[],
  keys_compose_e[], keys_compose_y[], keys_compose_u[], keys_compose_i[],
  keys_compose_o[], keys_compose_w[], keys_compose_r[], keys_compose_t[],
  keys_compose_p[], keys_compose_s[], keys_compose_d[], keys_compose_f[],
  keys_compose_g[], keys_compose_h[], keys_compose_j[], keys_compose_k[],
  keys_compose_l[], keys_compose_z[], keys_compose_x[], keys_compose_c[],
  keys_compose_v[], keys_compose_b[], keys_compose_n[], keys_compose_m[],
  keys_compose_math[], keys_compose_punctuation[], keys_compose_bracket[],
  keys_compose_cyr_i[], keys_compose_cyr_j[], keys_compose_cyr_e[],
  keys_compose_cyr_u[], keys_compose_cyr_l[], keys_compose_cyr_n[],
  keys_compose_cyr_tse[], keys_compose_cyr_che[], keys_compose_cyr_g[],
  keys_compose_cyr_k[], keys_dialer[];

static struct layout layouts[NumLayouts] = {
  [Full] = {keys_full, "latin", "full"}, // second parameter is the keymap name
                                         // third parameter is the layout name
  [Special] = {keys_special, "latin", "special"},
  [Simple] = {keys_simple, "latin", "simple"},
  [SimpleGrid] = {keys_simplegrid, "latin", "simplegrid"},
  [Cyrillic] = {keys_cyrillic, "cyrillic", "cyrillic"},
  [Dialer] = {keys_dialer, "latin", "dialer"},
  [Arabic] = {keys_arabic, "arabic", "arabic"},
  [Georgian] = {keys_georgian, "georgian", "georgian"},
  [Persian] = {keys_persian, "persian", "persian"},
  [Greek] = {keys_greek, "greek", "greek"},
  [Emoji] = {keys_emoji, "latin", "emoji"},
  [Nav] = {keys_nav, "latin", "nav"},
  [Landscape] = {keys_landscape, "latin", "landscape"},
  [ComposeA] = {keys_compose_a, "latin"},
  [ComposeE] = {keys_compose_e, "latin"},
  [ComposeY] = {keys_compose_y, "latin"},
  [ComposeU] = {keys_compose_u, "latin"},
  [ComposeI] = {keys_compose_i, "latin"},
  [ComposeO] = {keys_compose_o, "latin"},
  [ComposeW] = {keys_compose_w, "latin"},
  [ComposeR] = {keys_compose_r, "latin"},
  [ComposeT] = {keys_compose_t, "latin"},
  [ComposeP] = {keys_compose_p, "latin"},
  [ComposeS] = {keys_compose_s, "latin"},
  [ComposeD] = {keys_compose_d, "latin"},
  [ComposeF] = {keys_compose_f, "latin"},
  [ComposeG] = {keys_compose_g, "latin"},
  [ComposeH] = {keys_compose_h, "latin"},
  [ComposeJ] = {keys_compose_j, "latin"},
  [ComposeK] = {keys_compose_k, "latin"},
  [ComposeL] = {keys_compose_l, "latin"},
  [ComposeZ] = {keys_compose_z, "latin"},
  [ComposeX] = {keys_compose_x, "latin"},
  [ComposeC] = {keys_compose_c, "latin"},
  [ComposeV] = {keys_compose_v, "latin"},
  [ComposeB] = {keys_compose_b, "latin"},
  [ComposeN] = {keys_compose_n, "latin"},
  [ComposeM] = {keys_compose_m, "latin"},
  [ComposeMath] = {keys_compose_math, "latin"},
  [ComposePunctuation] = {keys_compose_punctuation, "latin"},
  [ComposeBracket] = {keys_compose_bracket, "latin"},
  [ComposeCyrI] = {keys_compose_cyr_i, "cyrillic"},
  [ComposeCyrJ] = {keys_compose_cyr_j, "cyrillic"},
  [ComposeCyrE] = {keys_compose_cyr_e, "cyrillic"},
  [ComposeCyrU] = {keys_compose_cyr_u, "cyrillic"},
  [ComposeCyrL] = {keys_compose_cyr_l, "cyrillic"},
  [ComposeCyrN] = {keys_compose_cyr_n, "cyrillic"},
  [ComposeCyrTse] = {keys_compose_cyr_tse, "cyrillic"},
  [ComposeCyrChe] = {keys_compose_cyr_che, "cyrillic"},
  [ComposeCyrG] = {keys_compose_cyr_g, "cyrillic"},
  [ComposeCyrK] = {keys_compose_cyr_k, "cyrillic"},
};

/* key layouts
 *
 * define keys like:
 *
 *  `{
 *     "label",
 *     "SHIFT_LABEL",
 *     1,
 *     [Code, Mod, Layout, EndRow, Last],
 *     [KEY_CODE, Modifier],
 *     [&layout]
 *  },`
 *
 * - label: normal label for key
 *
 * - shift_label: label for key in shifted (uppercase) layout
 *
 * - width: column width of key
 *
 * - type: what kind of action this key peforms (emit keycode, toggle modifier,
 *   switch layout, or end the layout)
 *
 * - code: key scancode or modifier name (see
 *   `/usr/include/linux/input-event-codes.h` for scancode names, and
 *   `keyboard.h` for modifiers)
 *
 * - layout: layout to switch to when key is pressed
 */
static struct key keys_full[] = {
  {"Esc", "Esc", 1.0, Code, KEY_ESC, .scheme = 1},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {"Tab", "Tab", 1.0, Code, KEY_TAB, .scheme = 1},
  {"↑", "↑", 1.0, Code, KEY_UP, .scheme = 1},
  {"↓", "↓", 1.0, Code, KEY_DOWN, .scheme = 1},
  {"←", "←", 1.0, Code, KEY_LEFT, .scheme = 1},
  {"→", "→", 1.0, Code, KEY_RIGHT, .scheme = 1},
  {"'", "\"", 1.0, Code, KEY_APOSTROPHE, .scheme = 1},
  {";", ":", 1.0, Code, KEY_SEMICOLON, .scheme = 1},
  {"/", ">", 1.0, Code, KEY_SLASH, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"1", "!", 1.0, Code, KEY_1},
  {"2", "@", 1.0, Code, KEY_2},
  {"3", "#", 1.0, Code, KEY_3},
  {"4", "$", 1.0, Code, KEY_4},
  {"5", "%", 1.0, Code, KEY_5},
  {"6", "^", 1.0, Code, KEY_6},
  {"7", "&", 1.0, Code, KEY_7},
  {"8", "*", 1.0, Code, KEY_8},
  {"9", "(", 1.0, Code, KEY_9},
  {"0", ")", 1.0, Code, KEY_0},
  {"-", "_", 1.0, Code, KEY_MINUS},
  {"=", "+", 1.0, Code, KEY_EQUAL},
  {"", "", 0.0, EndRow},

  {"", "", 0.5, Pad},
  {"q", "Q", 1.0, Code, KEY_Q, &layouts[Emoji]},
  {"w", "W", 1.0, Code, KEY_W, &layouts[ComposeW]},
  {"e", "E", 1.0, Code, KEY_E, &layouts[ComposeE]},
  {"r", "R", 1.0, Code, KEY_R, &layouts[ComposeR]},
  {"t", "T", 1.0, Code, KEY_T, &layouts[ComposeT]},
  {"y", "Y", 1.0, Code, KEY_Y, &layouts[ComposeY]},
  {"u", "U", 1.0, Code, KEY_U, &layouts[ComposeU]},
  {"i", "I", 1.0, Code, KEY_I, &layouts[ComposeI]},
  {"o", "O", 1.0, Code, KEY_O, &layouts[ComposeO]},
  {"p", "P", 1.0, Code, KEY_P, &layouts[ComposeP]},
  {"", "", 0.0, EndRow},

  {"Cmp", "Cmp", 1.0, Compose, .scheme = 1},
  {"a", "A", 1.0, Code, KEY_A, &layouts[ComposeA]},
  {"s", "S", 1.0, Code, KEY_S, &layouts[ComposeS]},
  {"d", "D", 1.0, Code, KEY_D, &layouts[ComposeD]},
  {"f", "F", 1.0, Code, KEY_F, &layouts[ComposeF]},
  {"g", "G", 1.0, Code, KEY_G, &layouts[ComposeG]},
  {"h", "H", 1.0, Code, KEY_H, &layouts[ComposeH]},
  {"j", "J", 1.0, Code, KEY_J, &layouts[ComposeJ]},
  {"k", "K", 1.0, Code, KEY_K, &layouts[ComposeK]},
  {"l", "L", 1.0, Code, KEY_L, &layouts[ComposeL]},
  {"", "", 0.5, Pad},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"z", "Z", 1.0, Code, KEY_Z, &layouts[ComposeZ]},
  {"x", "X", 1.0, Code, KEY_X, &layouts[ComposeX]},
  {"c", "C", 1.0, Code, KEY_C, &layouts[ComposeC]},
  {"v", "V", 1.0, Code, KEY_V, &layouts[ComposeV]},
  {"b", "B", 1.0, Code, KEY_B, &layouts[ComposeB]},
  {"n", "N", 1.0, Code, KEY_N, &layouts[ComposeN]},
  {"m", "M", 1.0, Code, KEY_M, &layouts[ComposeM]},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Sym", "Sym", 1.0, NextLayer, .scheme = 1},
  {"Alt", "Alt", 1.0, Mod, Alt, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_special[] = {
  {"Esc", "Esc", 1.0, Code, KEY_ESC, .scheme = 1},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {"Tab", "Tab", 1.0, Code, KEY_TAB, .scheme = 1},
  {"↑", "↑", 1.0, Code, KEY_UP, .scheme = 1},
  {"↓", "↓", 1.0, Code, KEY_DOWN, .scheme = 1},
  {"←", "←", 1.0, Code, KEY_LEFT, .scheme = 1},
  {"→", "→", 1.0, Code, KEY_RIGHT, .scheme = 1},
  {"⇈", "⇈", 1.0, Code, KEY_PAGEUP, .scheme = 1},
  {"⇊", "⇊", 1.0, Code, KEY_PAGEDOWN, .scheme = 1},
  {"⇤", "⇤", 1.0, Code, KEY_HOME, .scheme = 1},
  {"⇥", "⇥", 1.0, Code, KEY_END, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"1", "!", 1.0, Code, KEY_1},
  {"2", "@", 1.0, Code, KEY_2},
  {"3", "#", 1.0, Code, KEY_3},
  {"4", "$", 1.0, Code, KEY_4},
  {"5", "%", 1.0, Code, KEY_5},
  {"6", "^", 1.0, Code, KEY_6},
  {"7", "&", 1.0, Code, KEY_7},
  {"8", "*", 1.0, Code, KEY_8},
  {"9", "(", 1.0, Code, KEY_9},
  {"0", ")", 1.0, Code, KEY_0},
  {"", "", 0.0, EndRow},

  {"Caps", "Caps", 1.0, Mod, CapsLock, .scheme = 1},
  {"Sup", "Sup", 1.0, Mod, Super, .scheme = 1},
  {"`", "~", 1.0, Code, KEY_GRAVE},
  {"'", "\"", 1.0, Code, KEY_APOSTROPHE},
  {"-", "_", 1.0, Code, KEY_MINUS},
  {"=", "+", 1.0, Code, KEY_EQUAL},
  {"[", "{", 1.0, Code, KEY_LEFTBRACE},
  {"]", "}", 1.0, Code, KEY_RIGHTBRACE},
  {"\\", "|", 1.0, Code, KEY_BACKSLASH},
  {"Del", "Del", 1.0, Code, KEY_DELETE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 2.0, Mod, Shift, .scheme = 1},
  {";", ":", 1.0, Code, KEY_SEMICOLON},
  {"/", ">", 1.0, Code, KEY_SLASH},
  {"<", "«", 1.0, Code, KEY_COMMA, 0, AltGr},
  {">", "»", 1.0, Code, KEY_DOT, 0, AltGr},
  {"¡", "¿", 1.0, Code, KEY_MENU},
  {"🙂", "🙂", 1.0, Layout, 0, &layouts[Emoji]},
  {"AGr", "AGr", 1.0, Mod, AltGr, .scheme = 1},
  {"⌫", "⌫", 1.0, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"Alt", "Alt", 1.0, Mod, Alt, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_simple[] = {
  {"↹", "↹", 0.5, Code, KEY_TAB, .scheme = 1},
  {"q", "Q", 1.0, Code, KEY_Q, &layouts[Emoji]},
  {"w", "W", 1.0, Code, KEY_W, &layouts[ComposeW]},
  {"e", "E", 1.0, Code, KEY_E, &layouts[ComposeE]},
  {"r", "R", 1.0, Code, KEY_R, &layouts[ComposeR]},
  {"t", "T", 1.0, Code, KEY_T, &layouts[ComposeT]},
  {"y", "Y", 1.0, Code, KEY_Y, &layouts[ComposeY]},
  {"u", "U", 1.0, Code, KEY_U, &layouts[ComposeU]},
  {"i", "I", 1.0, Code, KEY_I, &layouts[ComposeI]},
  {"o", "O", 1.0, Code, KEY_O, &layouts[ComposeO]},
  {"p", "P", 1.0, Code, KEY_P, &layouts[ComposeP]},
  {"", "", 0.0, EndRow},

  {"Cmp", "Cmp", 1.0, Compose, .scheme = 1},
  {"a", "A", 1.0, Code, KEY_A, &layouts[ComposeA]},
  {"s", "S", 1.0, Code, KEY_S, &layouts[ComposeS]},
  {"d", "D", 1.0, Code, KEY_D, &layouts[ComposeD]},
  {"f", "F", 1.0, Code, KEY_F, &layouts[ComposeF]},
  {"g", "G", 1.0, Code, KEY_G, &layouts[ComposeG]},
  {"h", "H", 1.0, Code, KEY_H, &layouts[ComposeH]},
  {"j", "J", 1.0, Code, KEY_J, &layouts[ComposeJ]},
  {"k", "K", 1.0, Code, KEY_K, &layouts[ComposeK]},
  {"l", "L", 1.0, Code, KEY_L, &layouts[ComposeL]},
  {"'", "\"", 0.5, Code, KEY_APOSTROPHE, &layouts[ComposeBracket]},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"z", "Z", 1.0, Code, KEY_Z, &layouts[ComposeZ]},
  {"x", "X", 1.0, Code, KEY_X, &layouts[ComposeX]},
  {"c", "C", 1.0, Code, KEY_C, &layouts[ComposeC]},
  {"v", "V", 1.0, Code, KEY_V, &layouts[ComposeV]},
  {"b", "B", 1.0, Code, KEY_B, &layouts[ComposeB]},
  {"n", "N", 1.0, Code, KEY_N, &layouts[ComposeN]},
  {"m", "M", 1.0, Code, KEY_M, &layouts[ComposeM]},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {"-", "_", 1.0, Code, KEY_MINUS, &layouts[ComposeMath]},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT, &layouts[ComposePunctuation]},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_dialer[] = {
  {"Esc", "Esc", 1.0, Code, KEY_ESC},
  {"+","+", 1.0, Code, KEY_KPPLUS},
  {"⌫", "⌫", 1.0, Code, KEY_BACKSPACE},
  {"", "", 0.0, EndRow},
  {"1", "1", 1.0, Code, KEY_1},
  {"2", "2", 1.0, Code, KEY_2},
  {"3", "3", 1.0, Code, KEY_3},
  {"", "", 0.0, EndRow},
  {"4", "4", 1.0, Code, KEY_4},
  {"5", "5", 1.0, Code, KEY_5},
  {"6", "6", 1.0, Code, KEY_6},
  {"", "", 0.0, EndRow},
  {"7", "7", 1.0, Code, KEY_7},
  {"8", "8", 1.0, Code, KEY_8},
  {"9", "9", 1.0, Code, KEY_9},
  {"", "", 0.0, EndRow},
  {"*", "*", 1.0, Code, KEY_KPASTERISK},
  {"0", "0", 1.0, Code, KEY_0},
  {"#", "#", 1.0, Code, KEY_3, 0, Shift},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_simplegrid[] = {
  {"q", "Q", 1.0, Code, KEY_Q, &layouts[Emoji]},
  {"w", "W", 1.0, Code, KEY_W, &layouts[ComposeW]},
  {"e", "E", 1.0, Code, KEY_E, &layouts[ComposeE]},
  {"r", "R", 1.0, Code, KEY_R, &layouts[ComposeR]},
  {"t", "T", 1.0, Code, KEY_T, &layouts[ComposeT]},
  {"y", "Y", 1.0, Code, KEY_Y, &layouts[ComposeY]},
  {"u", "U", 1.0, Code, KEY_U, &layouts[ComposeU]},
  {"i", "I", 1.0, Code, KEY_I, &layouts[ComposeI]},
  {"o", "O", 1.0, Code, KEY_O, &layouts[ComposeO]},
  {"p", "P", 1.0, Code, KEY_P, &layouts[ComposeP]},
  {"", "", 0.0, EndRow},

  {"a", "A", 1.0, Code, KEY_A, &layouts[ComposeA]},
  {"s", "S", 1.0, Code, KEY_S, &layouts[ComposeS]},
  {"d", "D", 1.0, Code, KEY_D, &layouts[ComposeD]},
  {"f", "F", 1.0, Code, KEY_F, &layouts[ComposeF]},
  {"g", "G", 1.0, Code, KEY_G, &layouts[ComposeG]},
  {"h", "H", 1.0, Code, KEY_H, &layouts[ComposeH]},
  {"j", "J", 1.0, Code, KEY_J, &layouts[ComposeJ]},
  {"k", "K", 1.0, Code, KEY_K, &layouts[ComposeK]},
  {"l", "L", 1.0, Code, KEY_L, &layouts[ComposeL]},
  {"'", "\"", 1.0, Code, KEY_APOSTROPHE, &layouts[ComposeBracket]},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 1.0, Mod, Shift, .scheme = 1},
  {"z", "Z", 1.0, Code, KEY_Z, &layouts[ComposeZ]},
  {"x", "X", 1.0, Code, KEY_X, &layouts[ComposeX]},
  {"c", "C", 1.0, Code, KEY_C, &layouts[ComposeC]},
  {"v", "V", 1.0, Code, KEY_V, &layouts[ComposeV]},
  {"b", "B", 1.0, Code, KEY_B, &layouts[ComposeB]},
  {"n", "N", 1.0, Code, KEY_N, &layouts[ComposeN]},
  {"m", "M", 1.0, Code, KEY_M, &layouts[ComposeM]},
  {"Tab", "Tab", 1.0, Code, KEY_TAB, .scheme = 1},
  {"⌫", "⌫", 1.0, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {"-", "_", 1.0, Code, KEY_MINUS, &layouts[ComposeMath]},
  {"Cmp", "Cmp", 1.0, Compose, .scheme = 1},
  {"", "", 3.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT, &layouts[ComposePunctuation]},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_cyrillic[] = {
  {"1", "!", 1.0, Code, KEY_1},
  {"2", "@", 1.0, Code, KEY_2},
  {"3", "#", 1.0, Code, KEY_3},
  {"4", ";", 1.0, Code, KEY_4},
  {"5", "%", 1.0, Code, KEY_5},
  {"6", ":", 1.0, Code, KEY_6},
  {"7", "&", 1.0, Code, KEY_7},
  {"8", "*", 1.0, Code, KEY_8},
  {"9", "(", 1.0, Code, KEY_9},
  {"0", ")", 1.0, Code, KEY_0},
  {"ю", "Ю", 1.0, Code, KEY_DOT},
  {"э", "Э", 1.0, Code, KEY_APOSTROPHE, &layouts[ComposeCyrE]},
  {"ё", "Ё", 1.0, Code, KEY_GRAVE, &layouts[ComposeCyrE]},
  {"", "", 0.0, EndRow},

  {"й", "Й", 1.0, Code, KEY_Q, &layouts[ComposeCyrJ]},
  {"ц", "Ц", 1.0, Code, KEY_W, &layouts[ComposeCyrTse]},
  {"у", "У", 1.0, Code, KEY_E},
  {"к", "К", 1.0, Code, KEY_R, &layouts[ComposeCyrK]},
  {"е", "Е", 1.0, Code, KEY_T, &layouts[ComposeCyrE]},
  {"н", "Н", 1.0, Code, KEY_Y},
  {"г", "Г", 1.0, Code, KEY_U, &layouts[ComposeCyrG]},
  {"ш", "ш", 1.0, Code, KEY_I},
  {"щ", "щ", 1.0, Code, KEY_O},
  {"з", "з", 1.0, Code, KEY_P},
  {"х", "Х", 1.0, Code, KEY_LEFTBRACE},
  {"ъ", "Ъ", 0.5, Code, KEY_RIGHTBRACE},
  {"", "", 0.0, EndRow},

  {"ф", "Ф", 1.0, Code, KEY_A},
  {"ы", "Ы", 1.0, Code, KEY_S, &layouts[ComposeCyrI]},
  {"в", "В", 1.0, Code, KEY_D},
  {"а", "А", 1.0, Code, KEY_F},
  {"п", "П", 1.0, Code, KEY_G},
  {"р", "Р", 1.0, Code, KEY_H},
  {"о", "О", 1.0, Code, KEY_J},
  {"л", "Л", 1.0, Code, KEY_K, &layouts[ComposeCyrL]},
  {"д", "Д", 1.0, Code, KEY_L},
  {"ж", "Ж", 1.0, Code, KEY_SEMICOLON},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"я", "Я", 1.0, Code, KEY_Z},
  {"ч", "Ч", 1.0, Code, KEY_X, &layouts[ComposeCyrChe]},
  {"c", "С", 1.0, Code, KEY_C},
  {"м", "М", 1.0, Code, KEY_V},
  {"и", "И", 1.0, Code, KEY_B, &layouts[ComposeCyrI]},
  {"т", "Т", 1.0, Code, KEY_N},
  {"ь", "Ь", 1.0, Code, KEY_M},
  {"б", "Б", 1.0, Code, KEY_COMMA},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},

  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"Cmp", "Cmp", 1.0, Compose, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_arabic[] = {
  {"١", "!", 1.0, Code, KEY_1},
  {"٢", "@", 1.0, Code, KEY_2},
  {"٣", "#", 1.0, Code, KEY_3},
  {"٤", ";", 1.0, Code, KEY_4},
  {"٥", "%", 1.0, Code, KEY_5},
  {"٦", ":", 1.0, Code, KEY_6},
  {"٧", "&", 1.0, Code, KEY_7},
  {"٨", "*", 1.0, Code, KEY_8},
  {"٩", "(", 1.0, Code, KEY_9},
  {"٠", ")", 1.0, Code, KEY_0},
  {"", "", 0.0, EndRow},

  {"ض", "ـَ", 1.0, Code, KEY_Q},
  {"ص", "ـً", 1.0, Code, KEY_W},
  {"ث", "ـُ", 1.0, Code, KEY_E},
  {"ق", "ـٌ", 1.0, Code, KEY_R},
  {"ف", "ﻹ", 1.0, Code, KEY_T},
  {"غ", "إ", 1.0, Code, KEY_Y},
  {"ع", "`", 1.0, Code, KEY_U},
  {"ه", "", 1.0, Code, KEY_I},
  {"خ", ";", 1.0, Code, KEY_O},
  {"ح", "", 1.0, Code, KEY_P},
  {"ج", "<", 1.0, Code, KEY_LEFTBRACE},
  {"", "", 0.0, EndRow},

  {"ش", "ـِ", 1.0, Code, KEY_A},
  {"س", "ـٍ", 1.0, Code, KEY_S},
  {"ي", "]", 1.0, Code, KEY_D},
  {"ب", "[", 1.0, Code, KEY_F},
  {"ل", "ﻷ", 1.0, Code, KEY_G},
  {"ا", "أ", 1.0, Code, KEY_H},
  {"ت", "ـ", 1.0, Code, KEY_J},
  {"ن", "،", 1.0, Code, KEY_K},
  {"م", "/", 1.0, Code, KEY_L},
  {"ك", ":", 1.0, Code, KEY_SEMICOLON},
  {"ط", "\"", 1.0, Code, KEY_APOSTROPHE},
  {"", "", 0.0, EndRow},

  {"ذ", "~", 1.0, Code, KEY_Z},
  {"ء", "ـْ", 1.0, Code, KEY_X},
  {"ؤ", "}", 1.0, Code, KEY_C},
  {"ر", "{", 1.0, Code, KEY_V},
  {"ى", "ﻵ", 1.0, Code, KEY_B},
  {"ة", "'", 1.0, Code, KEY_N},
  {"و", ",", 1.0, Code, KEY_M},
  {"ز", "؟", 1.0, Code, KEY_COMMA},
  {"ظ", ".", 1.0, Code, KEY_DOT},
  {"د", ">", 1.0, Code, KEY_RIGHTBRACE},
  {"⌫", "⌫", 1.0, Code, KEY_BACKSPACE, .scheme = 1},

  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"⇧", "⇫", 1.0, Mod, Shift, .scheme = 1},
  {"Cmp", "Cmp", 1.0, Compose, .scheme = 1},
  {"", "", 5.0, Code, KEY_SPACE},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_georgian[] = {
  {"1", "!", 1.0, Code, KEY_1},
  {"2", "@", 1.0, Code, KEY_2},
  {"3", "#", 1.0, Code, KEY_3},
  {"4", ";", 1.0, Code, KEY_4},
  {"5", "%", 1.0, Code, KEY_5},
  {"6", ":", 1.0, Code, KEY_6},
  {"7", "&", 1.0, Code, KEY_7},
  {"8", "*", 1.0, Code, KEY_8},
  {"9", "(", 1.0, Code, KEY_9},
  {"0", ")", 1.0, Code, KEY_0},
  {"", "", 0.0, EndRow},

  {"ქ", "Q", 1.0, Code, KEY_Q},
  {"წ", "ჭ", 1.0, Code, KEY_W},
  {"ე", "E", 1.0, Code, KEY_E},
  {"რ", "ღ", 1.0, Code, KEY_R},
  {"ტ", "თ", 1.0, Code, KEY_T},
  {"ყ", "Y", 1.0, Code, KEY_Y},
  {"უ", "U", 1.0, Code, KEY_U},
  {"ი", "I", 1.0, Code, KEY_I},
  {"ო", "O", 1.0, Code, KEY_O},
  {"პ", "P", 1.0, Code, KEY_P},
  {"", "", 0.0, EndRow},

  {"ა", "A", 1.0, Code, KEY_A},
  {"ს", "შ", 1.0, Code, KEY_S},
  {"დ", "D", 1.0, Code, KEY_D},
  {"ფ", "F", 1.0, Code, KEY_F},
  {"გ", "G", 1.0, Code, KEY_G},
  {"ჰ", "H", 1.0, Code, KEY_H},
  {"ჯ", "ჟ", 1.0, Code, KEY_J},
  {"კ", "K", 1.0, Code, KEY_K},
  {"ლ", "L", 1.0, Code, KEY_L},
  {";", ":", 1.0, Code, KEY_SEMICOLON},
  {"", "", 0.0, EndRow},

  {"ზ", "ძ", 1.0, Code, KEY_Z},
  {"ხ", "X", 1.0, Code, KEY_X},
  {"ც", "ჩ", 1.0, Code, KEY_C},
  {"ვ", "V", 1.0, Code, KEY_V},
  {"ბ", "B", 1.0, Code, KEY_B},
  {"ნ", "N", 1.0, Code, KEY_N},
  {"მ", "M", 1.0, Code, KEY_M},
  {",", "<", 1.0, Code, KEY_COMMA},
  {".", ">", 1.0, Code, KEY_DOT},
  {"⌫", "⌫", 1.0, Code, KEY_BACKSPACE, .scheme = 1},

  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"⇧", "⇫", 1.0, Mod, Shift, .scheme = 1},
  {"Cmp", "Cmp", 1.0, Compose, .scheme = 1},
  {"", "", 5.0, Code, KEY_SPACE},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_persian[] = {
  {"۱", "|", 1.0, Code, KEY_1},
  {"۲", "٬", 1.0, Code, KEY_2},
  {"۳", "#", 1.0, Code, KEY_3},
  {"۴", "﷼", 1.0, Code, KEY_4},
  {"۵", "٪", 1.0, Code, KEY_5},
  {"۶", "×", 1.0, Code, KEY_6},
  {"۷", "/", 1.0, Code, KEY_7},
  {"۸", "*", 1.0, Code, KEY_8},
  {"۹", ")", 1.0, Code, KEY_9},
  {"۰", "(", 1.0, Code, KEY_0},
  {"-", "_", 1.0, Code, KEY_MINUS},
  {"=", "+", 1.0, Code, KEY_EQUAL},
  {"", "", 0.0, EndRow},

  {"ض", "ﹿ", 1.0, Code, KEY_Q},
  {"ص", "ـٌ", 1.0, Code, KEY_W},
  {"ث", "ـٍ", 1.0, Code, KEY_E},
  {"ق", "ـً", 1.0, Code, KEY_R},
  {"ف", "ـُ", 1.0, Code, KEY_T},
  {"غ", "ـِ", 1.0, Code, KEY_Y},
  {"ع", "ـَ", 1.0, Code, KEY_U},
  {"ه", "ﹽ", 1.0, Code, KEY_I},
  {"خ", "]", 1.0, Code, KEY_O},
  {"ح", "[", 1.0, Code, KEY_P},
  {"ج", "}", 1.0, Code, KEY_LEFTBRACE},
  {"چ", "{", 1.0, Code, KEY_RIGHTBRACE},
  {"", "", 0.0, EndRow},

  {"ش", "ؤ", 1.0, Code, KEY_A},
  {"س", "ئ", 1.0, Code, KEY_S},
  {"ی", "ي", 1.0, Code, KEY_D},
  {"ب", "إ", 1.0, Code, KEY_F},
  {"ل", "أ", 1.0, Code, KEY_G},
  {"ا", "آ", 1.0, Code, KEY_H},
  {"ت", "ة", 1.0, Code, KEY_J},
  {"ن", "»", 1.0, Code, KEY_K},
  {"م", "«", 1.0, Code, KEY_L},
  {"ک", ":", 1.0, Code, KEY_SEMICOLON},
  {"گ", "؛", 1.0, Code, KEY_APOSTROPHE},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"ظ", "ك", 1.0, Code, KEY_Z},
  {"ط", "ـٓ", 1.0, Code, KEY_X},
  {"ژ", ">", 1.0, Code, KEY_BACKSLASH},
  {"ز", "<", 1.0, Code, KEY_C},
  {"ر", "ـٰ", 1.0, Code, KEY_V},
  {"ذ", "⇑", 1.0, Code, KEY_B},
  {"د", "ـٔ", 1.0, Code, KEY_N},
  {"پ", "ء", 1.0, Code, KEY_M},
  {"و", "\\", 1.0, Code, KEY_COMMA},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.0, NextLayer, .scheme = 1},
  {"Cmp", "Cmp", 1.0, Compose, .scheme = 1},
  {".", "،", 1.0, Code, KEY_DOT},
  {"", "", 4.0, Code, KEY_SPACE},
  {"⥣", "↥", 1.0, Code, KEY_GRAVE},
  {"؟", "!", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_greek[] = {
  {"1", "!", 1.0, Code, KEY_1},
  {"2", "@", 1.0, Code, KEY_2},
  {"3", "#", 1.0, Code, KEY_3},
  {"4", "$", 1.0, Code, KEY_4},
  {"5", "%", 1.0, Code, KEY_5},
  {"6", "^", 1.0, Code, KEY_6},
  {"7", "&", 1.0, Code, KEY_7},
  {"8", "*", 1.0, Code, KEY_8},
  {"9", "(", 1.0, Code, KEY_9},
  {"0", ")", 1.0, Code, KEY_0},
  {"-", "_", 1.0, Code, KEY_MINUS},
  {"=", "+", 1.0, Code, KEY_EQUAL},
  {"", "", 0.0, EndRow},

  {";", ":", 1.0, Code, KEY_Q},
  {"ς", "Σ", 1.0, Code, KEY_W},
  {"ε", "Ε", 1.0, Code, KEY_E},
  {"ρ", "Ρ", 1.0, Code, KEY_R},
  {"τ", "Τ", 1.0, Code, KEY_T},
  {"υ", "Υ", 1.0, Code, KEY_Y},
  {"θ", "Θ", 1.0, Code, KEY_U},
  {"ι", "Ι", 1.0, Code, KEY_I},
  {"ο", "Ο", 1.0, Code, KEY_O},
  {"π", "Π", 1.0, Code, KEY_P},
  {"", "", 0.0, EndRow},

  {"α", "A", 1.0, Code, KEY_A},
  {"σ", "Σ", 1.0, Code, KEY_S},
  {"δ", "Δ", 1.0, Code, KEY_D},
  {"φ", "Φ", 1.0, Code, KEY_F},
  {"γ", "Γ", 1.0, Code, KEY_G},
  {"η", "Η", 1.0, Code, KEY_H},
  {"ξ", "Ξ", 1.0, Code, KEY_J},
  {"κ", "Κ", 1.0, Code, KEY_K},
  {"λ", "Λ", 1.0, Code, KEY_L},
  {"΄", "¨", 1.0, Code, KEY_SEMICOLON},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"ζ", "Ζ", 1.0, Code, KEY_Z},
  {"χ", "Χ", 1.0, Code, KEY_X},
  {"ψ", "Ψ", 1.0, Code, KEY_C},
  {"ω", "Ω", 1.0, Code, KEY_V},
  {"β", "Β", 1.0, Code, KEY_B},
  {"ν", "Ν", 1.0, Code, KEY_N},
  {"μ", "Μ", 1.0, Code, KEY_M},
  {"/", "\\", 1.0, Code, KEY_SLASH},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Abc", "Abc", 1.5, NextLayer, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 5, Code, KEY_SPACE},
  {".", "\"", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_compose_a[] = {
  {"à", "À", 1.0, Copy, 0x00E0, 0, 0x00C0},
  {"á", "Á", 1.0, Copy, 0x00E1, 0, 0x00C1},
  {"â", "Â", 1.0, Copy, 0x00E2, 0, 0x00C2},
  {"ã", "Ã", 1.0, Copy, 0x00E3, 0, 0x00C3},
  {"ä", "Ä", 1.0, Copy, 0x00E4, 0, 0x00C4},
  {"å", "Å", 1.0, Copy, 0x00E5, 0, 0x00C5},
  {"æ", "Æ", 1.0, Copy, 0x00E7, 0, 0x00C6},
  {"ā", "Ā", 1.0, Copy, 0x0101, 0, 0x0100},
  {"ă", "Ă", 1.0, Copy, 0x0103, 0, 0x0102},
  {"ą", "Ą", 1.0, Copy, 0x0105, 0, 0x0104},
  {"", "", 0.0, EndRow},
  {"α", "Α", 1.0, Copy, 0x03B1, 0, 0x0391},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer, .scheme = 1},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_e[] = {
  {"è", "È", 1.0, Copy, 0x00E8, 0, 0x00C8},
  {"é", "É", 1.0, Copy, 0x00E9, 0, 0x00C9},
  {"ê", "Ê", 1.0, Copy, 0x00EA, 0, 0x00CA},
  {"ë", "Ë", 1.0, Copy, 0x00EB, 0, 0x00CB},
  {"ē", "Ē", 1.0, Copy, 0x0113, 0, 0x0112},
  {"ĕ", "Ĕ", 1.0, Copy, 0x0115, 0, 0x0114},
  {"ė", "Ė", 1.0, Copy, 0x0117, 0, 0x0116},
  {"ę", "Ę", 1.0, Copy, 0x0119, 0, 0x0118},
  {"ě", "Ě", 1.0, Copy, 0x011B, 0, 0x011A},
  {"", "", 0.0, EndRow},
  {"ε", "Ε", 1.0, Copy, 0x03B5, 0, 0x0395},
  {"ǝ", "Ə", 1.0, Copy, 0x0259, 0, 0x018F},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER},
  {"", "", 0.0, Last},
};

static struct key keys_compose_y[] = {
  {"ý", "Ý", 1.0, Copy, 0x00FD, 0, 0x00DD},
  {"ÿ", "Ÿ", 1.0, Copy, 0x00FF, 0, 0x0178},
  {"ŷ", "Ŷ", 1.0, Copy, 0x0177, 0, 0x0176},
  {"", "", 7.0, Pad},
  {"", "", 0.0, EndRow},
  {"υ", "Υ", 1.0, Copy, 0x03C5, 0, 0x03A5},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_u[] = {
  {"ù", "Ù", 1.0, Copy, 0x00F9, 0, 0x00D9},
  {"ú", "Ú", 1.0, Copy, 0x00FA, 0, 0x00DA},
  {"û", "Û", 1.0, Copy, 0x00FB, 0, 0x00DB},
  {"ü", "Ü", 1.0, Copy, 0x00FC, 0, 0x00DC},
  {"ũ", "Ũ", 1.0, Copy, 0x0169, 0, 0x0168},
  {"ū", "Ū", 1.0, Copy, 0x016B, 0, 0x016A},
  {"ŭ", "Ŭ", 1.0, Copy, 0x016D, 0, 0x016C},
  {"ů", "Ů", 1.0, Copy, 0x016F, 0, 0x016E},
  {"ű", "Ű", 1.0, Copy, 0x0171, 0, 0x0170},
  {"ų", "Ų", 1.0, Copy, 0x0173, 0, 0x0172},
  {"", "", 0.0, EndRow},
  {"υ", "Υ", 1.0, Copy, 0x03C5, 0, 0x03A5},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_o[] = {
  {"ò", "Ò", 1.0, Copy, 0x00F2, 0, 0x00D2},
  {"ó", "Ó", 1.0, Copy, 0x00F3, 0, 0x00D3},
  {"ô", "Ô", 1.0, Copy, 0x00F4, 0, 0x00D4},
  {"õ", "Õ", 1.0, Copy, 0x00F5, 0, 0x00D5},
  {"ö", "Ö", 1.0, Copy, 0x00F6, 0, 0x00D6},
  {"ø", "Ø", 1.0, Copy, 0x00F8, 0, 0x00D8},
  {"ō", "Ō", 1.0, Copy, 0x014D, 0, 0x014C},
  {"ŏ", "Ŏ", 1.0, Copy, 0x014F, 0, 0x014E},
  {"ő", "Ő", 1.0, Copy, 0x0151, 0, 0x0150},
  {"œ", "Œ", 1.0, Copy, 0x0153, 0, 0x0152},
  {"", "", 0.0, EndRow},
  {"ο", "Ο", 1.0, Copy, 0x03BF, 0, 0x039F},
  {"ω", "Ο", 1.0, Copy, 0x03C9, 0, 0x03A9},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_i[] = {
  {"ì", "Ì", 1.0, Copy, 0x00EC, 0, 0x00CC},
  {"í", "Í", 1.0, Copy, 0x00ED, 0, 0x00CD},
  {"î", "Î", 1.0, Copy, 0x00EE, 0, 0x00CE},
  {"ï", "Ï", 1.0, Copy, 0x00EF, 0, 0x00CF},
  {"ĩ", "Ĩ", 1.0, Copy, 0x0129, 0, 0x0128},
  {"ī", "Ī", 1.0, Copy, 0x012B, 0, 0x012A},
  {"ĭ", "Ĭ", 1.0, Copy, 0x012D, 0, 0x012C},
  {"į", "Į", 1.0, Copy, 0x012F, 0, 0x012E},
  {"ı", "I", 1.0, Copy, 0x0131, 0, 0x0049},
  {"i", "İ", 1.0, Copy, 0x0069, 0, 0x0130},
  {"", "", 0.0, EndRow},
  {"ι", "Ι", 1.0, Copy, 0x03B9, 0, 0x0399},
  {"η", "Η", 1.0, Copy, 0x03B7, 0, 0x0397},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_emoji[] = {
  {"🙂", "🙏", 1.0, Copy, 0x1f642, 0, 0x1f64f}, // Emojis
  {"😀", "🙋", 1.0, Copy, 0x1f600, 0, 0x1f64b},
  {"😁", "✋", 1.0, Copy, 0x1f601, 0, 0x270B},
  {"😂", "🙇", 1.0, Copy, 0x1f602, 0, 0x1f647},
  {"😃", "👨", 1.0, Copy, 0x1f603, 0, 0x1f468},
  {"😄", "👩", 1.0, Copy, 0x1f604, 0, 0x1f469},
  {"😅", "👶", 1.0, Copy, 0x1f605, 0, 0x1f476},
  {"😆", "👻", 1.0, Copy, 0x1f606, 0, 0x1f47b},
  {"😇", "✨", 1.0, Copy, 0x1f607, 0, 0x2728},
  {"🤣", "💰", 1.0, Copy, 0x1f923, 0, 0x1f4b0},
  {"", "", 0.0, EndRow},
  {"😉", "🐕", 1.0, Copy, 0x1f609, 0, 0x1f415},
  {"😊", "🐈", 1.0, Copy, 0x1f60a, 0, 0x1f408},
  {"😋", "🐧", 1.0, Copy, 0x1f60b, 0, 0x1f427},
  {"😌", "🐇", 1.0, Copy, 0x1f60c, 0, 0x1f407},
  {"😍", "💔", 1.0, Copy, 0x1f60d, 0, 0x1f494},
  {"😘", "💕", 1.0, Copy, 0x1f618, 0, 0x1f495},
  {"😎", "❤", 1.0, Copy, 0x1f60e, 0, 0x2764},
  {"😐", "😏", 1.0, Copy, 0x1f610, 0, 0x1f60f},
  {"😒", "👀", 1.0, Copy, 0x1f612, 0, 0x1f440},
  {"😓", "💀", 1.0, Copy, 0x1f613, 0, 0x1f480},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.0, Mod, Shift, .scheme = 1},
  {"😛", "😜", 1.0, Copy, 0x1f61b, 0, 0x1f61c},
  {"😮", "😝", 1.0, Copy, 0x1f62e, 0, 0x1f61d},
  {"😟", "😞", 1.0, Copy, 0x1f61f, 0, 0x1f61e},
  {"😠", "🥺", 1.0, Copy, 0x1f620, 0, 0x1f97a},
  {"😢", "👿", 1.0, Copy, 0x1f622, 0, 0x1f47f},
  {"😭", "😯", 1.0, Copy, 0x1f62d, 0, 0x1f62f},
  {"😳", "😕", 1.0, Copy, 0x1f633, 0, 0x1f615},
  {"😴", "😵", 1.0, Copy, 0x1f634, 0, 0x1f635},
  {"⌫", "⌫", 1.0, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer, .scheme = 1},
  {"👆", "👊", 1.0, Copy, 0x1f446, 0, 0x1f44a},
  {"👇", "👌", 1.0, Copy, 0x1f447, 0, 0x1f44c},
  {"👈", "👏", 1.0, Copy, 0x1f448, 0, 0x1f44f},
  {"👉", "👐", 1.0, Copy, 0x1f449, 0, 0x1f450},
  {"👋", "🙌", 1.0, Copy, 0x1f44b, 0, 0x1f64c},
  {"👍", "✅", 1.0, Copy, 0x1f44d, 0, 0x2705},
  {"👎", "💪", 1.0, Copy, 0x1f44e, 0, 0x1f4aa},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_nav[] = {
  {"Esc", "Esc", 1.0, Code, KEY_ESC, .scheme = 1},
  {"⌫", "⌫", 1.0, Code, KEY_BACKSPACE, .scheme = 1},
  {"↑", "↑", 1.0, Code, KEY_UP, .scheme = 1},
  {"⇈", "⇈", 1.0, Code, KEY_PAGEUP, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"q", "q", 1.0, Code, KEY_Q, .scheme = 1},
  {"←", "←", 1.0, Code, KEY_LEFT, .scheme = 1},
  {"Enter", "Enter", 1.0, Code, KEY_ENTER, .scheme = 1},
  {"→", "→", 1.0, Code, KEY_RIGHT, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Sym", "Sym", 1.0, NextLayer, .scheme = 1},
  {"", "", 1.0, Code, KEY_SPACE, .scheme = 1},
  {"↓", "↓", 1.0, Code, KEY_DOWN, .scheme = 1},
  {"⇊", "⇊", 1.0, Code, KEY_PAGEDOWN, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_landscape[] = {
  {"Esc", "Esc", 1.0, Code, KEY_ESC, .scheme = 1},
  {"q", "Q", 1.0, Code, KEY_Q, &layouts[Emoji]},
  {"w", "W", 1.0, Code, KEY_W, &layouts[ComposeW]},
  {"e", "E", 1.0, Code, KEY_E, &layouts[ComposeE]},
  {"r", "R", 1.0, Code, KEY_R, &layouts[ComposeR]},
  {"t", "T", 1.0, Code, KEY_T, &layouts[ComposeT]},
  {"y", "Y", 1.0, Code, KEY_Y, &layouts[ComposeY]},
  {"u", "U", 1.0, Code, KEY_U, &layouts[ComposeU]},
  {"i", "I", 1.0, Code, KEY_I, &layouts[ComposeI]},
  {"o", "O", 1.0, Code, KEY_O, &layouts[ComposeO]},
  {"p", "P", 1.0, Code, KEY_P, &layouts[ComposeP]},
  {"(", "(", 0.75, Code, KEY_9, &layouts[ComposeBracket], Shift, .scheme = 1},
  {")", ")", 0.75, Code, KEY_0, &layouts[ComposeBracket], Shift, .scheme = 1},
  {"Tab", "Tab", 0.75, Code, KEY_TAB, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Cmp", "Cmp", 1.5, Compose, .scheme = 1},
  {"a", "A", 1.0, Code, KEY_A, &layouts[ComposeA]},
  {"s", "S", 1.0, Code, KEY_S, &layouts[ComposeS]},
  {"d", "D", 1.0, Code, KEY_D, &layouts[ComposeD]},
  {"f", "F", 1.0, Code, KEY_F, &layouts[ComposeF]},
  {"g", "G", 1.0, Code, KEY_G, &layouts[ComposeG]},
  {"h", "H", 1.0, Code, KEY_H, &layouts[ComposeH]},
  {"j", "J", 1.0, Code, KEY_J, &layouts[ComposeJ]},
  {"k", "K", 1.0, Code, KEY_K, &layouts[ComposeK]},
  {"l", "L", 1.0, Code, KEY_L, &layouts[ComposeL]},
  {";", ":", 1.0, Code, KEY_SEMICOLON, .scheme = 1},
  {"'", "\"", 1.0, Code, KEY_APOSTROPHE, &layouts[ComposeBracket], .scheme = 1},
  {"-", "_", 0.5, Code, KEY_MINUS, &layouts[ComposeBracket], .scheme = 1},
  {"", "", 0.0, EndRow},

  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"z", "Z", 1.0, Code, KEY_Z, &layouts[ComposeZ]},
  {"x", "X", 1.0, Code, KEY_X, &layouts[ComposeX]},
  {"c", "C", 1.0, Code, KEY_C, &layouts[ComposeC]},
  {"v", "V", 1.0, Code, KEY_V, &layouts[ComposeV]},
  {"b", "B", 1.0, Code, KEY_B, &layouts[ComposeB]},
  {"n", "N", 1.0, Code, KEY_N, &layouts[ComposeN]},
  {"m", "M", 1.0, Code, KEY_M, &layouts[ComposeM]},
  {"↑", "↑", 0.75, Code, KEY_UP, .scheme = 1},
  {"↓", "↓", 0.75, Code, KEY_DOWN, .scheme = 1},
  {"/", ">", 0.75, Code, KEY_SLASH, .scheme = 1},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},

  {"Sym", "Sym", 1.0, NextLayer, .scheme = 1},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {"Alt", "Alt", 0.5, Mod, Alt, .scheme = 1},
  {",", "'", 0.5, Code, KEY_COMMA, &layouts[ComposeMath]},
  {"", "", 3.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT, &layouts[ComposeMath]},
  {"Agr", "AGr", 1.0, Mod, AltGr, .scheme = 1},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},

  /* end of layout */
  {"", "", 0.0, Last},
};

static struct key keys_compose_w[] = {
  {"ŵ", "Ŵ", 1.0, Copy, 0x0175, 0, 0x0174},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_r[] = {
  {"ŕ", "Ŕ", 1.0, Copy, 0x0155, 0, 0x0154},
  {"ŗ", "Ŗ", 1.0, Copy, 0x0157, 0, 0x0156},
  {"ř", "Ř", 1.0, Copy, 0x0159, 0, 0x0158},
  {"", "", 7.0, Pad},
  {"", "", 0.0, EndRow},
  {"ρ", "Ρ", 1.0, Copy, 0x03C1, 0, 0x03A1},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_t[] = {
  {"ț", "Ț", 1.0, Copy, 0x021B, 0, 0x021A},
  {"ť", "Ť", 1.0, Copy, 0x0165, 0, 0x0164},
  {"ŧ", "Ŧ", 1.0, Copy, 0x0167, 0, 0x0166},
  {"þ", "Þ", 1.0, Copy, 0x00FE, 0, 0x00DE},
  {"", "", 6.0, Pad},
  {"", "", 0.0, EndRow},
  {"τ", "Τ", 1.0, Copy, 0x03C4, 0, 0x03A4},
  {"θ", "Θ", 1.0, Copy, 0x03B8, 0, 0x0398},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_p[] = {
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"π", "Π", 1.0, Copy, 0x03C0, 0, 0x03A0},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_s[] = {
  {"ś", "Ś", 1.0, Copy, 0x015B, 0, 0x015A},
  {"ŝ", "Ŝ", 1.0, Copy, 0x015D, 0, 0x015C},
  {"ş", "Ş", 1.0, Copy, 0x015F, 0, 0x015E},
  {"š", "Š", 1.0, Copy, 0x0161, 0, 0x0160},
  {"ß", "ẞ", 1.0, Copy, 0x00DF, 0, 0x1E9E},
  {"", "", 5.0, Pad},
  {"", "", 0.0, EndRow},
  {"σ", "Σ", 1.0, Copy, 0x03C3, 0, 0x03A3},
  {"ς", "Σ", 1.0, Copy, 0x03C2, 0, 0x03A3},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_d[] = {
  {"ð", "Ð", 1.0, Copy, 0x00F0, 0, 0x00D0},
  {"ď", "Ď", 1.0, Copy, 0x010F, 0, 0x010E},
  {"đ", "Đ", 1.0, Copy, 0x0111, 0, 0x0110},
  {"", "", 7.0, Pad},
  {"", "", 0.0, EndRow},
  {"δ", "Δ", 1.0, Copy, 0x03B4, 0, 0x0394},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_f[] = {
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"φ", "Φ", 1.0, Copy, 0x03C6, 0, 0x03A6},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_g[] = {
  {"ĝ", "Ĝ", 1.0, Copy, 0x011D, 0, 0x011C},
  {"ğ", "Ğ", 1.0, Copy, 0x011F, 0, 0x011E},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"γ", "Γ", 1.0, Copy, 0x03B3, 0, 0x0393},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_h[] = {
  {"ĥ", "Ĥ", 1.0, Copy, 0x0125, 0, 0x0124},
  {"ħ", "Ħ", 1.0, Copy, 0x0127, 0, 0x0126},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"η", "Η", 1.0, Copy, 0x03B7, 0, 0x0397},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_j[] = {
  {"ĵ", "Ĵ", 1.0, Copy, 0x0135, 0, 0x0134},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_k[] = {
  {"ķ", "Ķ", 1.0, Copy, 0x0137, 0, 0x0136},
  {"ǩ", "Ǩ", 1.0, Copy, 0x01E9, 0, 0x01E8},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"κ", "Κ", 1.0, Copy, 0x03BA, 0, 0x039A},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_l[] = {
  {"ľ", "Ľ", 1.0, Copy, 0x013E, 0, 0x013D},
  {"ŀ", "Ŀ", 1.0, Copy, 0x0140, 0, 0x013F},
  {"ł", "Ł", 1.0, Copy, 0x0142, 0, 0x0141},
  {"", "", 7.0, Pad},
  {"", "", 0.0, EndRow},
  {"λ", "Λ", 1.0, Copy, 0x03BB, 0, 0x039B},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_z[] = {
  {"ź", "Ź", 1.0, Copy, 0x017A, 0, 0x0179},
  {"ż", "Ż", 1.0, Copy, 0x017C, 0, 0x017B},
  {"ž", "Ž", 1.0, Copy, 0x017E, 0, 0x017D},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"ζ", "Ζ", 1.0, Copy, 0x03B6, 0, 0x0396},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_x[] = {
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"χ", "Χ", 1.0, Copy, 0x03C7, 0, 0x03A7},
  {"ξ", "Ξ", 1.0, Copy, 0x03BE, 0, 0x039E},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_c[] = {
  {"ç", "Ç", 1.0, Copy, 0x00E7, 0, 0x00C7},
  {"ć", "Ć", 1.0, Copy, 0x0107, 0, 0x0106},
  {"ĉ", "Ĉ", 1.0, Copy, 0x0109, 0, 0x0108},
  {"ċ", "Ċ", 1.0, Copy, 0x010B, 0, 0x010A},
  {"č", "Č", 1.0, Copy, 0x010D, 0, 0x010C},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"χ", "Χ", 1.0, Copy, 0x03C7, 0, 0x03A7},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_v[] = {
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_b[] = {
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"β", "Β", 1.0, Copy, 0x03B2, 0, 0x0392},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_n[] = {
  {"ñ", "Ñ", 1.0, Copy, 0x00F1, 0, 0x00D1},
  {"ń", "Ń", 1.0, Copy, 0x0144, 0, 0x0143},
  {"ņ", "Ņ", 1.0, Copy, 0x0146, 0, 0x0145},
  {"ň", "Ň", 1.0, Copy, 0x0148, 0, 0x0147},
  {"ŋ", "Ŋ", 1.0, Copy, 0x014B, 0, 0x014A},
  {"", "", 5.0, Pad},
  {"", "", 0.0, EndRow},
  {"ν", "Ν", 1.0, Copy, 0x03BD, 0, 0x039D},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer, .scheme = 1},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_m[] = {
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"μ", "Μ", 1.0, Copy, 0x03BC, 0, 0x039C},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_i[] = {
  {"і", "І", 1.0, Copy, 0x0456, 0, 0x0406},
  {"ї", "Ї", 1.0, Copy, 0x0457, 0, 0x0407},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_j[] = {
  {"ј", "Ј", 1.0, Copy, 0x0458, 0, 0x0408},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_e[] = {
  {"є", "Є", 1.0, Copy, 0x0454, 0, 0x0404},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_u[] = {
  {"ў", "Ў", 1.0, Copy, 0x045E, 0, 0x040E},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_l[] = {
  {"љ", "Љ", 1.0, Copy, 0x0459, 0, 0x0409},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_n[] = {
  {"њ", "Њ", 1.0, Copy, 0x045A, 0, 0x040A},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_che[] = {
  {"ћ", "Ћ", 1.0, Copy, 0x045B, 0, 0x040B},
  {"ђ", "Ђ", 1.0, Copy, 0x0452, 0, 0x0402},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_tse[] = {
  {"џ", "Џ", 1.0, Copy, 0x045F, 0, 0x040F},
  {"ѕ", "Ѕ", 1.0, Copy, 0x0455, 0, 0x0405},
  {"", "", 8.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_g[] = {
  {"ѓ", "Ѓ", 1.0, Copy, 0x0453, 0, 0x0403},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_cyr_k[] = {
  {"ќ", "Ќ", 1.0, Copy, 0x0453, 0, 0x040C},
  {"", "", 9.0, Pad},
  {"", "", 0.0, EndRow},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 1.5, Mod, Shift, .scheme = 1},
  {"", "", 7, Pad},
  {"⌫", "⌫", 1.5, Code, KEY_BACKSPACE, .scheme = 1},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_EQUAL},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_SLASH},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_math[] = {
  {"1", "!", 1.0, Code, KEY_1},
  {"2", "@", 1.0, Code, KEY_2},
  {"3", "#", 1.0, Code, KEY_3},
  {"4", "$", 1.0, Code, KEY_4},
  {"5", "%", 1.0, Code, KEY_5},
  {"6", "^", 1.0, Code, KEY_6},
  {"7", "&", 1.0, Code, KEY_7},
  {"8", "*", 1.0, Code, KEY_8},
  {"9", "(", 1.0, Code, KEY_9},
  {"0", ")", 1.0, Code, KEY_0},
  {"", "", 0.0, EndRow},
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 2, Mod, Shift, .scheme = 1},
  {"+", "+", 1, Code, KEY_EQUAL, 0, Shift},
  {"/", "/", 1, Code, KEY_SLASH},
  {"*", "*", 1, Code, KEY_8, 0, Shift},
  {"=", "=", 1, Code, KEY_EQUAL},
  {"_", "_", 1, Code, KEY_MINUS, 0, Shift},
  {"—", "—", 1, Copy, 0x2014, 0, 0x2014},
  {"", "", 2, Pad},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {"-", "-", 1, Code, KEY_MINUS},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT, &layouts[ComposePunctuation]},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_punctuation[] = {
  {"1", "!", 1.0, Code, KEY_1},
  {"2", "@", 1.0, Code, KEY_2},
  {"3", "#", 1.0, Code, KEY_3},
  {"4", "$", 1.0, Code, KEY_4},
  {"5", "%", 1.0, Code, KEY_5},
  {"6", "^", 1.0, Code, KEY_6},
  {"7", "&", 1.0, Code, KEY_7},
  {"8", "*", 1.0, Code, KEY_8},
  {"9", "(", 1.0, Code, KEY_9},
  {"0", ")", 1.0, Code, KEY_0},
  {"", "", 0.0, EndRow},
  {"", "", 10.0, Pad},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 2, Mod, Shift, .scheme = 1},
  {"…", "…", 1, Copy, 0x2026, 0, 0x2026},
  {":", ":", 1, Code, KEY_SEMICOLON, 0, Shift},
  {";", ";", 1, Code, KEY_SEMICOLON, 0},
  {"!", "!", 1, Code, KEY_1, 0, Shift},
  {"?", "?", 1, Code, KEY_DOT, 0, Shift},
  {",", ",", 1, Code, KEY_COMMA},
  {"·", "·", 1, Copy, 0x2027, 0, 0x2027},
  {"", "", 1, Pad},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA, &layouts[ComposeMath]},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT, &layouts[ComposePunctuation]},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};

static struct key keys_compose_bracket[] = {
  {"", "", 5, Pad},
  {"{", "{", 1, Code, KEY_LEFTBRACE, 0, Shift},
  {"}", "}", 1, Code, KEY_RIGHTBRACE, 0, Shift},
  {"[", "[", 1, Code, KEY_LEFTBRACE},
  {"]", "]", 1, Code, KEY_RIGHTBRACE},
  {"", "", 0, EndRow},
  {"", "", 5, Pad},
  {"<", "<", 1, Code, KEY_COMMA, 0, AltGr},
  {">", ">", 1, Code, KEY_SLASH, 0, Shift},
  {"`", "`", 1, Code, KEY_GRAVE},
  {"\"", "\"", 1, Code, KEY_APOSTROPHE, 0, Shift},
  {"'", "'", 1, Code, KEY_APOSTROPHE},
  {"", "", 0.0, EndRow},
  {"⇧", "⇫", 2, Mod, Shift, .scheme = 1},
  {"", "", 8, Pad},
  {"", "", 0.0, EndRow},
  {"Abc", "Abc", 1.0, BackLayer},
  {"Ctr", "Ctr", 1.0, Mod, Ctrl, .scheme = 1},
  {",", "'", 1.0, Code, KEY_COMMA, &layouts[ComposeMath]},
  {"", "", 4.0, Code, KEY_SPACE},
  {".", "?", 1.0, Code, KEY_DOT, &layouts[ComposePunctuation]},
  {"Enter", "Enter", 2.0, Code, KEY_ENTER, .scheme = 1},
  {"", "", 0.0, Last},
};
