#!/usr/bin/perl -w
#
# Package: Xdebconfigurator
# convert and merge different hardware-tables

# sax: Identity.map, (Keyboard.map, Probe.map) (source: sax/sysp/maps/)
# kudzu: /usr/share/kudzu/pcitable (installed) (make sure you also have package knoppix installed)
# hwdata: /usr/share/hwdata/ (installed) CardMonitorCombos, Cards, MonitorsDB, pci.ids, pcitable, usb.ids
# harddrake:
# libdetect(detect): /usr/share/detect/ (installed) pci.lst, isa.lst, pcmcia.lst, usb.lst
# discover: (discover-data) /usr/share/discover (installed) pci.lst, pcmcia.lst, usb.lst

# discover-data and libdetect have same format (possibly same files), using vendors as "topics"
# THE DIFFERENCE BETWEEN DISCOVER AND DETECT IS:
# discover-data has video Server:XFree86(ati) while detect has video Card:Ati Rage Pro II....
# So.. discover-data is the knoppix version of detect.. in a way..
# kudzu is simpler: 0x0001  0x1002  "bttv"  "ATI|TV Wonder"
# sax is well documented

# establish filenames
$sax_map = "/usr/share/xdebconfigurator/Identity.map";
$sax_out = "/usr/share/xdebconfigurator/sax.map";
$kudzu_map = "/usr/share/kudzu/pcitable";
$kudzu_out = "/usr/share/xdebconfigurator/kudzu.map";
$discover_map = "/usr/share/discover/pci.lst";
$discover_out = "/usr/share/xdebconfigurator/discover.map";


# establish infrastructure
# for an item to be unique in kudzu, it'll need vendid-devid-subvendid-subdevid as key!
# taking a closer look.. vendid-devid-subdevid should be enough for key..
# except.. sax has entries that are sort of duplicate.. vendid-devid-subvendid-subdevid is
# exactly the same but some of the other info differs. talk about biased.

%vendor_hash = (); # vendor name
%device_hash = (); # device name (0x4755->Rage 3D II+)
%vendor_id_hash = (); # vendor ID hex (0x4755->0x1002)
%server_mod_hash = (); #server module name (0x4755->ati)
# optional hashes (for sax2)
%ext_hash = (); # [+-]e1,[+-]e2,.. (not sure about this one)
%opt_hash = (); # o1,o2,.. (not sure about this one)
%raw_hash = (); # (not sure about this one)
%profile_hash = (); # file name of profile file (not sure about this one)
%script3d_hash = (); # script name of 3D environment script (not sure about this one)
%package3d_hash = (); # package list needed to enable 3D (not sure about this one)
%flag_hash = (); # flag to perform a user decision [ 3D, DEFAULT ] (not sure about this one)
%sub_vendor_hash = (); # Subvendor ID (not sure about this one)
%sub_device_hash = (); # Subdevice ID (not sure about this one)

%type_hash = (); # discover describes each entry by type (video, unknown, bridge, usb, scsi, ethernet, joystick, sound)

%arg_map = (); #hash to determine out-file syntax

# Establish hash of arguments to use
if ($#ARGV < 0)
	{
		%arg_map = (
			out_file => "sax",
		);	# default out_file-format is sax
	}
elsif ($#ARGV == 0 && $ARGV[0] =~ /help/i)
	{
			print "The Table Converter Help Screen\n";	
			print "usage: table_converter.pl [-out_file_format]\n";
			print "[-out_file_format] is one of:\n";
			print "  -discover => discover\n";
			print "  -hwinfo => hwinfo (not yet implemented)\n";
			print "  -kudzu => kudzu\n";
			print "  -libdetect => libdetect (use same as discover for now)\n";
			print "  -sax => sax\n";
			print "No argument => use default (sax)\n";
			print "\n";
			exit(0);
	}
elsif ($#ARGV == 0)
	{
		print "Outfile: $ARGV[0]\n";
		$arg_map{out_file} = $ARGV[0];
	}
# end if

sub from_sax()
{
# ex: NAME=ATIDEVICE=Rage 3D II+VID=0x1002DID=0x4755SERVER=atiEXT=OPT=RAW=PROFILE=SCRIPT3D=PACKAGE3D=FLAG=DEFAULT
# Note:
# -----
#  - each description needs a single line
#  - every line needs to specify all keywords even if there
#    is no setting for it (biased - Subdevice/Subvendor is not always specified)
#  - field seperator is the  sign

	print "Reading SaX Identity.map\n";
	open (FIL,"$sax_map") or die "Can't open $sax_map !\n";
	while ($line = <FIL>)   # <FIL> returning an empty string at file-end
  	{
  		if($line =~ /^NAME/)
   	{
   		chomp($line);
   		($vname,$dname,$vid,$did,$smod,$ext,$opt,$raw,$profile,$script3d,$package3d,$flag,$subvendor,$subdevice) = split(//,"$line");
			
			#bah! sax is not following their :own: standard (subvendor and subdevice is not always present)
			if(!$subvendor)
			{
				$subvendor = "";	
			}
			if(!$subdevice)
			{
				$subdevice = "";	
			}
   	
   		$did =~ s/.+=0x(\S+)/$1/;

   		$vname =~ s/.+=(.*)/$1/;
   		$dname =~ s/.+=(.*)/$1/;
   		$vid =~ s/.+=0x(\S+)/$1/;
   		$smod =~ s/.+=(.*)/$1/;
   		$ext =~ s/.+=(.*)/$1/;
   		$opt =~ s/.+=(.*)/$1/;
   		$raw =~ s/.+=(.*)/$1/;
   		$profile =~ s/.+=(.*)/$1/;
   		$script3d =~ s/.+=(.*)/$1/;
   		$package3d =~ s/.+=(.*)/$1/;
   		$flag =~ s/.+=(.*)/$1/;
   		$subvendor =~ s/.+=0x(\S+)/$1/;
   		$subdevice =~ s/.+=0x(\S+)/$1/;
   		
   		$key = "$vid:$did:$subvendor:$subdevice";
   		
   		if($arg_map{out_file} =~ /kudzu/)
			{
				# kudzu can take all the SaX entries, but need knoppix-syntax for driver module
				$smod =~ s/(\w+)/Server:XFree86\($1\)/i;
			}
			elsif($arg_map{out_file} =~ /discover/)
			{
				# discover needs same syntax as kudzu-knoppix, but also need the type_hash
				$smod =~ s/(\w+)/Server:XFree86\($1\)/i;
				$type_hash{$key} = "video"; # SaX only has video entries
			}
				
   		$vendor_hash{$key}= $vname;
			$device_hash{$key}= $dname;
			$vendor_id_hash{$key}= $vid; 
			$server_mod_hash{$key}= $smod; 
			$ext_hash{$key}= $ext; 
			$opt_hash{$key}= $opt; 
			$raw_hash{$key}= $raw; 
			$profile_hash{$key}= $profile;
			$script3d_hash{$key}= $script3d;
			$package3d_hash{$key}= $package3d; 
			$flag_hash{$key}= $flag;
			$sub_vendor_hash{$key}= $subvendor;
			$sub_device_hash{$key}= $subdevice;
		}
   }
   close(FIL);
   
   # the hashes should now be complete

}

sub to_sax()
{	
	print "Writing SaX $sax_out\n";
	open (OUT, ">$sax_out") || die ("couldn't open $sax_out");
   #loop through hashes
   foreach (sort keys %vendor_hash)
	{
		#print "$_\n";	
		($vid,$did,$subvendid,$subdid) = split(":",$_);
		if($sub_vendor_hash{$_} ne "" )
		{
			print OUT "NAME=$vendor_hash{$_}DEVICE=$device_hash{$_}VID=0x$vendor_id_hash{$_}DID=0x$didSERVER=$server_mod_hash{$_}EXT=$ext_hash{$_}OPT=$opt_hash{$_}RAW=$raw_hash{$_}PROFILE=$profile_hash{$_}SCRIPT3D=$script3d_hash{$_}PACKAGE3D=$package3d_hash{$_}FLAG=$flag_hash{$_}SUBVENDOR=0x$sub_vendor_hash{$_}SUBDEVICE=0x$sub_device_hash{$_}\n";
		}
		else
		{
			# skip printing SUB-sets
			print OUT "NAME=$vendor_hash{$_}DEVICE=$device_hash{$_}VID=0x$vendor_id_hash{$_}DID=0x$didSERVER=$server_mod_hash{$_}EXT=$ext_hash{$_}OPT=$opt_hash{$_}RAW=$raw_hash{$_}PROFILE=$profile_hash{$_}SCRIPT3D=$script3d_hash{$_}PACKAGE3D=$package3d_hash{$_}FLAG=$flag_hash{$_}\n";	
		}
		
	}
   close (OUT);
	
}


sub from_kudzu()
{
	#/usr/share/kudzu/pcitable (installed)
	#ex:  0x1002  0x4755  "Server:XFree86(ati)"   "Mach64 3D RAGE II 3D Rage II+ 215GTB [Mach64 GTB]"
	#ex2: 0x1011  0x0046  0x9005  0x1365  "aacraid"  "DEC|Dell PowerEdge RAID Control
	# pci-table, not just graphic cards	
	# ignore? and treat everything as same?
	
	# The format is ("%d\t%d\t%s\t"%s"\n", vendid, devid, moduleName, cardDescription)
	# or ("%d\t%d\t%d\t%d\t%s\t"%s"\n", vendid, devid, subvendid, subdevid, moduleName, cardDescription)

	#ouch! devid is not unique! the only unique thing is the combination: vendid-devid-subvendid-subdevid!
	
	print "Reading Kudzu pcitable\n";
	my $filename = $kudzu_map;
	
	open(PCIFILE, "<$filename") || die "Unable to read $filename";
   while (<PCIFILE>) 
   {
		chomp;

		# Remove comments
		s/\#.+$//;

		# Skip empty lines
		next if /^\s*$/;
		
		# Skip comment lines
		next if /^\#/;
		
		# NULL previous values so they don't carry same value to next device
		($subvendid,$subdevid,$vendordesc,$desc,$vendorid,$driverinfo,$devid)=("","","","","","","");

		($vendorid, $rest) = m/^0x(\S+)\s+(\S+.+)$/;
		#$vendorid = "\L$vendorid"; #unnecessary?

		# there are 2 formats we can have (as mentioned above)
		if($rest =~ m/^0x(\S+)\s+0x(\S+)\s+0x(\S+)\s+(\S+.+)$/)
		{
			($devid, $subvendid, $subdevid, $rest2) = ($1,$2,$3,$4);
		}
		elsif($rest =~ m/^0x(\S+)\s+(\S+.+)$/)
		{
			($devid, $rest2) = ($1,$2);
		}
		else
		{
			# something went wrong when getting IDs
			($devid, $rest2) = ("bad_id", $rest); 	
		}

		($driverinfo, $desc) = $rest2 =~ m/^\"(.*)\"\s+\"(.*)\"$/;

		if ($desc =~ /^(.+)\|(.+)$/) 
		{
	   	$vendordesc = $1;
	    	$desc = $2;
		}
		
		# make sure we have :something: for SUBs
		if(!$subvendid)
		{
			$subvendid = "";	
		}
		if(!$subdevid)
		{
			$subdevid = "";	
		}
		
		# we should now have all values
		$key = "$vendorid:$devid:$subvendid:$subdevid";
		
		#print "$vendorid $devid $subvendid $subdevid $driverinfo\n";
		# do a test here.. cause if we're writing a SaX map then we only want video-card entries.
		# entrys with Server:XFree86(foo), Server:foo or Card:foo as $server_mod_hash{entry}
		if($arg_map{out_file} =~ /sax/)
		{
			if($driverinfo =~ /Server:XFree86\((\w+)\)/i)
			{
					# this variant exists if package kudzu-common-knoppix is installed
					# and might be the only one which is of interrest to sax
					$driverinfo = $1;
					$videocard = 1;
			}
			elsif($driverinfo =~ /Server:(\w+)/)
			{
					# this one only reports a server, like SVGA
					# with knoppix it will report servers like XF86_S3 or XF86_SVGA etc too.
					$driverinfo = $1;
					#$videocard = 1;
			}
			elsif($driverinfo =~ /Card:(\w+)/)
			{
				# These looks like the ones we want for sax, but they report a cardname, not a driver!
				# Do we have to lookup the driver someplace else?
				# Most Card: entries disappear when we install knoppix though!
					$driverinfo = $1;
					#$videocard = 1;
			}
			if($videocard)
			{
				#add to hashes
				$vendor_hash{$key}= $vendordesc;
				$device_hash{$key}= $desc;
				$vendor_id_hash{$key}= $vendorid; 
				$server_mod_hash{$key}= $driverinfo; # this one looks different from format to format
				# the other hashes are empty for kudzu, but must exist for sax
				$ext_hash{$key}= ""; 
				$opt_hash{$key}= ""; 
				$raw_hash{$key}= ""; 
				$profile_hash{$key}= "";
				$script3d_hash{$key}= "";
				$package3d_hash{$key}= ""; 
				$flag_hash{$key}= "DEFAULT"; # can this be empty or must be in [ 3D, DEFAULT ] ???
				# these can exist in kudzu too
				$sub_vendor_hash{$key}= $subvendid;
				$sub_device_hash{$key}= $subdevid;
				$videocard = 0;
			}
		}
		else
		{
			if($arg_map{out_file} =~ /discover/)
			{
				# discover needs same syntax as kudzu-knoppix, but also need the type_hash
				if($driverinfo =~ /Server:/)
				{
					$type_hash{$key} = "video"; # Anything with an X-Server is a video-card
				}
				elsif($driverinfo =~ /Card:/)
				{
					$type_hash{$key} = "video"; # Card entries is video-cards
				}
				else
				{
					$type_hash{$key} = "unknown"; # I can't guess what type it is.. 
				}
			}
			$vendor_hash{$key}= $vendordesc;
			$device_hash{$key}= $desc;
			$vendor_id_hash{$key}= $vendorid; 
			$server_mod_hash{$key}= $driverinfo; # this one looks different from format to format
			# the other hashes are empty for kudzu, but must exist for sax
			$ext_hash{$key}= ""; 
			$opt_hash{$key}= ""; 
			$raw_hash{$key}= ""; 
			$profile_hash{$key}= "";
			$script3d_hash{$key}= "";
			$package3d_hash{$key}= ""; 
			$flag_hash{$key}= ""; # can this be empty or must be in [ 3D, DEFAULT ] ???
			# these can exist in kudzu too
			$sub_vendor_hash{$key}= $subvendid;
			$sub_device_hash{$key}= $subdevid;
		}
	}
   close(PCIFILE);

}

sub load_kudzu_pci()
{
	# mostly done by pere@hungry.com
	#my ($filename) = @_;
	my $filename = $kudzu_map;

   open(PCIFILE, "<$filename") || die "Unable to read $filename";
   while (<PCIFILE>) 
   {
		chomp;

		# Remove comments
		s/\#.+$//;

		# Skip empty lines
		next if /^\s*$/;

		# 0x1011  0x0046  0x9005  0x1365  "aacraid"  "DEC|Dell PowerEdge RAID Control

		my ($vendorid, $rest) = m/^0x(\S+)\s+(\S+.+)$/;
		$vendorid = "\L$vendorid";

		my @cardids;
		while ($rest =~ m/^0x(\S+)\s+(\S+.+)$/) 
		{
	    	push(@cardids, "\l$1");
	    	$rest = $2;
		}

		my ($driverinfo, $desc) = $rest =~ m/^\"(.*)\"\s+\"(.*)\"$/;

		my $vendordesc;
		if ($desc =~ /^(.+)\|(.+)$/) 
		{
	   	$vendordesc = $1;
	    	$desc = $2;
		}

		#print "D: $driverinfo\n" if $debug;

		if ( defined $vendordesc && "" ne $vendordesc ) 
		{
	    	#$pci{'vendor'}{$vendorid}{'desc'} = $vendordesc;	
		}
		else { $vendordesc = ""; } # need it defined for sax-hash

		my $cardid;
		#for $cardid (@cardids) 
		#{
	   # 	my $id = "$vendorid$cardid";
	   # 	$pci{'vendor'}{$vendorid}{'cards'}{$id} = $desc;
		#  	$pci{'card'}{$id}{'desc'}   = $desc;
	   # 	if ( ! exists $pci{'card'}{$id}{'driver'}
		# 		&& defined $driverinfo && "" ne $driverinfo ) 
		# 	{
		#		$pci{'card'}{$id}{'driver'} = "$driverinfo";
	   # 	}
		#}
		for $cardid (@cardids)
		{
			#print "$vendorid  $cardid  $driverinfo\n";
			$vendor_hash{$cardid}= $vendordesc;
			$device_hash{$cardid}= $desc;
			$vendor_id_hash{$cardid}= $vendorid; 
			$server_mod_hash{$cardid}= $driverinfo; 
			# the other hashes are empty for kudzu, but must exist for sax
			$ext_hash{$cardid}= ""; 
			$opt_hash{$cardid}= ""; 
			$raw_hash{$cardid}= ""; 
			$profile_hash{$cardid}= "";
			$script3d_hash{$cardid}= "";
			$package3d_hash{$cardid}= ""; 
			$flag_hash{$cardid}= ""; # can this be empty or must be in [ 3D, DEFAULT ] ???
			$sub_vendor_hash{$cardid}= "";
			$sub_device_hash{$cardid}= "";
		}
    }
    close(PCIFILE);
}


sub to_kudzu()
{
	print "Writing Kudzu $kudzu_out\n";
	open (OUT, ">$kudzu_out") || die ("couldn't open $kudzu_out");
   #loop through hashes
   
   foreach (sort keys %vendor_hash)
	{
		$key = $_;
		# NULL previous values
		($vid,$did,$subvendid,$subdid)=("","","","");
   	($vid,$did,$subvendid,$subdid) = split(":",$key);
		#2 formats: ("%d\t%d\t%s\t"%s"\n", vendid, devid, moduleName, cardDescription)
		#or:("%d\t%d\t%d\t%d\t%s\t"%s"\n", vendid, devid, subvendid, subdevid, moduleName, cardDescription)
		
		# do we have a vendor-description or just a device-description?
		if($vendor_hash{$key} eq "")
		{
			$descriptions = $device_hash{$key};
		}
		else
		{
			$descriptions = "$vendor_hash{$key}|$device_hash{$key}";
		}
		
		if($sub_vendor_hash{$key} ne "")
		{
			#print OUT "0x$vendor_id_hash{$key}  0x$did  0x$sub_vendor_hash{$key}  0x$sub_device_hash{$key}  \"$server_mod_hash{$key}\"  \"$vendor_hash{$key}|$device_hash{$key}\"\n";	
			print OUT "0x$vendor_id_hash{$key}\t0x$did\t0x$sub_vendor_hash{$key}\t0x$sub_device_hash{$key}\t\"$server_mod_hash{$key}\"\t\"$descriptions\"\n";	
		}
		else
		{
			print OUT "0x".$vendor_id_hash{$key}."\t0x".$did."\t\"".$server_mod_hash{$key}."\"\t\"".$descriptions."\"\n";
		}
		
	}
   close (OUT);
}



sub from_discover()
{
	# discover uses same syntax as libdetect.. ex:
	#102b Matrox Graphics, Inc.
   #     102b0010        video   Server:XFree86(mga)     Millennium [0010]
   #     102b0518        video   Server:XFree86(mga)     Millennium II 2085PX [Athena]
   #     102b0519        video   Server:XFree86(mga)     Millennium 2064W
	#     102b4536        unknown unknown VIA Framegrabber
   #     102b6573        unknown unknown Shark 10/100 Multiport SwitchNIC
	#102c Chips and Technologies
   #     102c00b8        video   Server:XFree86(chips)   CT64300 F64310
   #     102c00c0        video   Server:XFree86(chips)   CT69000 F69000 HiQVideo
   #     102c00d0        video   Server:XFree86(chips)   CT65545 F65545
   #     102c00d8        video   Server:XFree86(chips)   CT65545 F65545
   #     102c00f5        unknown unknown F68555
   #     102c0c30        video   Server:XFree86(chips)   CT69030
	#102d Wyse Technology Inc.
   #     102d50dc        unknown unknown 3328 Audio
	#102e Olivetti Advanced Technology
	
	#so we have..
	#$vendorid \t $vendordesc
	#	\t $vendorid$devid \t \t? $type? \t $driverinfo \t $desc	 

	print "Reading Discover pci.lst\n";
	my $filename = $discover_map;
	
	open(PCIFILE, "<$filename") || die "Unable to read $filename";
   while (<PCIFILE>) 
   {
		chomp;

		# Remove comments (allthough I don't think pci.lst has any comments)
		s/\#.+$//;

		# Skip empty lines (OK)
		next if /^\s*$/;
		
		# Skip comment lines (see above)
		next if /^\#/;

		# discover doesn't use sub-ids (far as I can tell)
		($vendorid,$desc,$driverinfo,$devid,$type)=("","","","","");
				
		if( ($vendorid_tmp, $vendordesc) = m/^(\S+)\s+(.+)$/ )
		{
			# we need to hold onto this vendor description
			$tmp_vendor_hash{$vendorid_tmp}= $vendordesc;
			if($arg_map{out_file} =~ /discover/)
			{
				# discover even wants vendors without devices
				$vendorid = $vendorid_tmp;
				($vendorid_tmp, $vendordesc) = ("","");
			}
			else
			{
				($vendorid_tmp, $vendordesc) = ("","");
				next; # this way, we don't catch vendors with no devices
			}
			
		}		
		elsif(($vendorid, $devid, $type, $driverinfo, $desc) = m/(\S{4})(\S{4})\s+(\S+)\s+(\S+)\s+(.+)/)
		{
			# we have a vendor, now get all their devices
			# ex:         102b0525        video   Server:XFree86(mga)     Millennium G400
			#if($vendorid ne $vendorid_temp)	
			#{
			#	print "Something might be wrong: Vendor: $vendorid ne $vendorid_temp for card $devid\n";
			#}
		}
		else
		{
			print "Debug: Didn't match!";
		}
		
		$subvendid = ""; # discover doesn't use these
		$subdevid = "";
		# we should now have all values
		$key = "$vendorid:$devid:$subvendid:$subdevid";
		
		#print "$vendorid $devid $subvendid $subdevid $driverinfo\n";
		# do a test here.. cause if we're writing a SaX map then we only want video-card entries.
		# entrys with Server:XFree86(foo), Server:foo as $server_mod_hash{entry}
		# and possibly type=video

		if($arg_map{out_file} =~ /sax/)
		{
			if($driverinfo =~ /Server:XFree86\((\w+)\)/i)
			{
					# this variant might be the only one which is of interrest to sax
					$driverinfo = $1;
					$videocard = 1;
			}
			elsif($driverinfo =~ /Server:(\w+)/)
			{
					# this one only reports a server, like SVGA
					# with knoppix it will report servers like XF86_S3 or XF86_SVGA etc too.
					$driverinfo = $1;
					#$videocard = 1;
			}
			
			if($videocard && $type =~ /video/)
			{
				#add to hashes
				$vendor_hash{$key}= $tmp_vendor_hash{$vendorid}; # vendor description stored in it's own hash
				$device_hash{$key}= $desc;
				$vendor_id_hash{$key}= $vendorid; 
				$server_mod_hash{$key}= $driverinfo; # this one looks different from format to format
				# the other hashes are empty for kudzu, but must exist for sax
				$ext_hash{$key}= ""; 
				$opt_hash{$key}= ""; 
				$raw_hash{$key}= ""; 
				$profile_hash{$key}= "";
				$script3d_hash{$key}= "";
				$package3d_hash{$key}= ""; 
				$flag_hash{$key}= "DEFAULT"; # can this be empty or must be in [ 3D, DEFAULT ] ???
				# these can exist in kudzu too
				$sub_vendor_hash{$key}= ""; # discover don't use subs
				$sub_device_hash{$key}= "";
				$videocard = 0;
			}
		}
		else
		{
			if($arg_map{out_file} =~ /discover/)
			{
				$type_hash{$key} = $type;
			}
			$vendor_hash{$key}= $tmp_vendor_hash{$vendorid};
			$device_hash{$key}= $desc;
			$vendor_id_hash{$key}= $vendorid; 
			$server_mod_hash{$key}= $driverinfo; # this one looks different from format to format
			# the other hashes are empty for kudzu/discover, but must exist for sax
			$ext_hash{$key}= ""; 
			$opt_hash{$key}= ""; 
			$raw_hash{$key}= ""; 
			$profile_hash{$key}= "";
			$script3d_hash{$key}= "";
			$package3d_hash{$key}= ""; 
			$flag_hash{$key}= ""; # can this be empty or must be in [ 3D, DEFAULT ] ???
			# these can exist in kudzu too (but not in discover)
			$sub_vendor_hash{$key}= ""; # discover don't use subs
			$sub_device_hash{$key}= "";
		}
		
	}
   close(PCIFILE);
	
}


sub to_discover()
{
	# vendor's without devices don't get written
	print "Writing Discover $discover_out\n";
	open (OUT, ">$discover_out") || die ("couldn't open $discover_out");
   #loop through hashes
   my $this_vendor = "unknown";
   foreach (sort keys %vendor_hash)
	{
		$key = $_;
		# NULL previous values
		($vid,$did,$subvendid,$subdid)=("","","","");
   	($vid,$did,$subvendid,$subdid) = split(":",$key);
		#format:
		#102b Matrox Graphics, Inc.
   	#     102b0010        video   Server:XFree86(mga)     Millennium [0010]
		
		#$vendorid \t $vendordesc
		#	\t $vendorid$devid \t \t? $type? \t $driverinfo \t $desc	 
		
		if($vid ne $this_vendor)
		{
			print OUT "$vid $vendor_hash{$key}\n"; # "title"
			$this_vendor = $vid;
		}
		if($vid eq $this_vendor && $did ne "")
		{
			print OUT "\t$vid$did\t$type_hash{$key}\t$server_mod_hash{$key}\t$device_hash{$key}\n";	
		}
		
	}
   close (OUT);
}


# output file/format is decided by $arg_map{out_file}

#from_sax();
#to_sax();
#load_kudzu_pci();
# to_kudzu();

# lastly we read whichever table is gonna be output to preserve that table "as is"
# and write out the new table

SWITCH: {
	if($arg_map{out_file} =~ /sax/){
		from_discover();
		from_kudzu();
		from_sax(); # read the old table
		to_sax(); # write out the new table
		last SWITCH; #end sax
	}
	if($arg_map{out_file} =~ /kudzu/){
		from_discover();
		from_sax();
		from_kudzu();
		to_kudzu();
		last SWITCH; #end kudzu
	}
	if($arg_map{out_file} =~ /discover/){
		from_sax();
		from_kudzu();
		from_discover();
		to_discover();
		last SWITCH; #end kudzu
	}
}
