/* -*-Mode: C;-*-
 * $Id: edsio.h,v 1.12 2001/12/08 03:17:21 jmacd Exp $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _EDSIO_H_
#define _EDSIO_H_

#ifdef __cplusplus
extern "C" {
#if 0
}
#endif
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <glib.h>

typedef struct _HandleFuncTable HandleFuncTable;
typedef struct _FileHandle      FileHandle;
typedef struct _MessageDigestCtx MessageDigestCtx;

typedef gint32                  SerialType;
typedef struct _FilePosition    FilePosition;
typedef struct _HookList        HookList;

#define EDSIO_LIBRARY_OFFSET_BITS 8
#define EDSIO_LIBRARY_OFFSET_MASK 0xff

#define ARRAY_SIZE(x) sizeof((x))/sizeof(*(x))

typedef struct _SerialSource    SerialSource;
typedef struct _SerialSink      SerialSink;

#include "edsio_edsio.h"

#define ST_Error         -1
#define ST_IncorrectType -2
#define ST_NotFound      -3

#define ALIGN_8(v) if (((v) % 8) != 0) { (v) += 8; (v) &= ~7; }

struct _SerialSource {
  /* Internal variables: don't touch. */
  guint32  alloc_total;
  guint32  alloc_pos;
  void    *alloc_buf;
  void    *alloc_buf_orig;

  /* These are setup by init.
   */
  SerialType (* source_type)           (SerialSource* source, gboolean set_allocation);
  gboolean   (* source_close)          (SerialSource* source);
  gboolean   (* source_read)           (SerialSource* source, guint8 *ptr, guint32 len);
  void       (* source_free)           (SerialSource* source);

  /* These may be NULL
   */
  void*  (* salloc_func)           (SerialSource* source,
				    guint32       len);
  void   (* sfree_func)            (SerialSource* source,
				    void*         ptr);

  /* Public functions, defaulted, but may be over-ridden
   * before calls to unserialize.
   */
  gboolean   (* next_bytes_known)    (SerialSource* source, guint8        *ptr, guint32  len);
  gboolean   (* next_bytes)          (SerialSource* source, const guint8 **ptr, guint32 *len);
  gboolean   (* next_uint)           (SerialSource* source, guint32       *ptr);
  gboolean   (* next_uint32)         (SerialSource* source, guint32       *ptr);
  gboolean   (* next_uint16)         (SerialSource* source, guint16       *ptr);
  gboolean   (* next_uint8)          (SerialSource* source, guint8        *ptr);
  gboolean   (* next_bool)           (SerialSource* source, gboolean      *ptr);
  gboolean   (* next_string)         (SerialSource* source, const char   **ptr);
};

struct _SerialSink {

  /* These are setup by init.
   */
  gboolean     (* sink_type)          (SerialSink* sink, SerialType type, guint mem_size, gboolean set_allocation);
  gboolean     (* sink_close)         (SerialSink* sink);
  gboolean     (* sink_write)         (SerialSink* sink, const guint8 *ptr, guint32 len);
  void         (* sink_free)          (SerialSink* sink);

  /* This may be null, called after each object is serialized. */
  gboolean     (* sink_quantum)       (SerialSink* sink);

  /* Public functions, defaulted, but may be over-ridden
   * before calls to serialize.
   */
  gboolean   (* next_bytes_known)   (SerialSink* sink, const guint8 *ptr, guint32 len);
  gboolean   (* next_bytes)         (SerialSink* sink, const guint8 *ptr, guint32 len);
  gboolean   (* next_uint)          (SerialSink* sink, guint32       ptr);
  gboolean   (* next_uint32)        (SerialSink* sink, guint32       ptr);
  gboolean   (* next_uint16)        (SerialSink* sink, guint16       ptr);
  gboolean   (* next_uint8)         (SerialSink* sink, guint8        ptr);
  gboolean   (* next_bool)          (SerialSink* sink, gboolean      ptr);
  gboolean   (* next_string)        (SerialSink* sink, const char   *ptr);
};

void           serializeio_initialize_type                (const char* name,
							   guint32     val,
							   gboolean  (*unserialize_func) (),
							   gboolean  (*serialize_func) (),
							   guint     (*count_func) (),
							   void      (*print_func) ());

const char*    serializeio_generic_type_to_string         (SerialType type);
void           serializeio_generic_print                  (SerialType type, void* object, guint indent_spaces);

gboolean       serializeio_serialize_generic              (SerialSink    *sink,
							   SerialType     object_type,
							   void          *object);

guint          serializeio_generic_count                  (SerialType     object_type,
							   void          *object);

gboolean       serializeio_unserialize_generic            (SerialSource  *source,
							   SerialType    *object_type,
							   void         **object);

gboolean       serializeio_unserialize_generic_noerr      (SerialSource  *source,
							   SerialType    *object_type,
							   void         **object);

gboolean       serializeio_unserialize_generic_acceptable (SerialSource*  source,
							   guint32        acceptable,
							   SerialType    *object_type,
							   void         **object);

gboolean       serializeio_serialize_generic_internal     (SerialSink    *sink,
							   SerialType     object_type,
							   void          *object,
							   gboolean       set_allocation);

gboolean       serializeio_unserialize_generic_internal   (SerialSource  *source,
							   SerialType    *object_type,
							   void         **object,
							   gboolean       set_allocation,
							   gboolean       error);


void           serializeio_sink_init                      (SerialSink* sink,
							   gboolean (* sink_type) (SerialSink* sink,
										   SerialType type,
										   guint mem_size,
										   gboolean set_allocation),
							   gboolean (* sink_close) (SerialSink* sink),
							   gboolean (* sink_write) (SerialSink* sink,
										    const guint8 *ptr,
										    guint32 len),
							   void     (* sink_free) (SerialSink* sink),
							   gboolean (* sink_quantum) (SerialSink* sink));

void           serializeio_source_init                    (SerialSource* source,
							   SerialType (* source_type) (SerialSource* source,
										       gboolean set_allocation),
							   gboolean   (* source_close) (SerialSource* source),
							   gboolean   (* source_read) (SerialSource* source,
										       guint8 *ptr,
										       guint32 len),
							   void       (* source_free) (SerialSource* source),
							   void*      (* salloc_func) (SerialSource* source,
										       guint32       len),
							   void       (* sfree_func) (SerialSource* source,
										      void*         ptr));

/* These two functions are internal, don't use. */
gboolean       serializeio_source_object_received         (SerialSource* source);
void*          serializeio_source_alloc                   (SerialSource* source,
							   guint32       len);

SerialSink*    serializeio_gzip_sink   (SerialSink* sink);
SerialSource*  serializeio_gzip_source (SerialSource* source);

SerialSink*    serializeio_checksum_sink   (SerialSink* sink);
SerialSource*  serializeio_checksum_source (SerialSource* source);

SerialSink*    serializeio_base64_sink   (SerialSink* sink);
SerialSource*  serializeio_base64_source (SerialSource* source);

/* Event delivery
 */

enum _EventLevel
{
  EL_Information   = 1<<0,
  EL_Warning       = 1<<5,
  EL_Error         = 1<<10,
  EL_InternalError = 1<<15,
  EL_FatalError    = 1<<20
};

typedef enum _EventLevel EventLevel;

enum _EventFlags
{
  EF_None    = 1<<0,
  EF_OpenSSL = 1<<1
};

typedef enum _EventFlags EventFlags;

const char* eventdelivery_int_to_string     (int x);
const char* eventdelivery_string_to_string  (const char* x);
const char* eventdelivery_source_to_string  (SerialSource* x);
const char* eventdelivery_sink_to_string    (SerialSink* x);

void serializeio_print_bytes (const guint8* buf, guint len);

/* Event delivery privates
 */

typedef struct _GenericEventDef GenericEventDef;
typedef struct _GenericEvent    GenericEvent;

struct _GenericEvent
{
  gint        code;
  const char* srcfile;
  guint       srcline;
};

GenericEventDef* eventdelivery_event_lookup (gint code);

void eventdelivery_event_deliver        (GenericEvent* e);

typedef gboolean (* ErrorDeliveryFunc) (GenericEvent* ev, GenericEventDef* def, const char* message);

void eventdelivery_event_watch_all (ErrorDeliveryFunc func);

void eventdelivery_initialize_event_def (gint        code,
					 gint        level,
					 gint        flags,
					 const char* name,
					 const char* oneline,
					 const char * (* field_to_string) (GenericEvent* ev, gint field));

const char* eventdelivery_ssl_errors_to_string (void);

struct _GenericEventDef
{
  gint        code;
  gint        level;
  gint        flags;
  const char *name;
  const char *oneline;

  const char * (* field_to_string) (GenericEvent* ev, gint field);
};

/* MD5.H - header file for MD5C.C */

/* Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991. All
   rights reserved.

   License to copy and use this software is granted provided that it
   is identified as the "RSA Data Security, Inc. MD5 Message-Digest
   Algorithm" in all material mentioning or referencing this software
   or this function.

   License is also granted to make and use derivative works provided
   that such works are identified as "derived from the RSA Data
   Security, Inc. MD5 Message-Digest Algorithm" in all material
   mentioning or referencing the derived work.

   RSA Data Security, Inc. makes no representations concerning either
   the merchantability of this software or the suitability of this
   software for any particular purpose. It is provided "as is"
   without express or implied warranty of any kind.

   These notices must be retained in any copies of any part of this
   documentation and/or software.
   */

/* MD5 context. */
typedef struct {
    guint32 state[4]; /* state (ABCD) */
    guint32 count[2]; /* number of bits, modulo 2^64 (lsb first) */
    guint8 buffer[64]; /* input buffer */
} EdsioMD5Ctx;

void edsio_md5_init   (EdsioMD5Ctx *);
void edsio_md5_update (EdsioMD5Ctx *, const guint8 *, guint);
void edsio_md5_final  (guint8*, EdsioMD5Ctx *);

gint  edsio_md5_equal (gconstpointer   v,
		       gconstpointer   v2);
guint edsio_md5_hash  (gconstpointer   v);

void      edsio_md5_to_string   (const guint8* md5, char buf[33]);
gboolean  edsio_md5_from_string (guint8* md5,       const char buf[33]);

/* NIST Secure Hash Algorithm */
/* heavily modified from Peter C. Gutmann's implementation */
/* then taken from from Uwe Hollerbach, */
/* and then modified a little by Josh MacDonald. */

/* This code is in the public domain */

typedef struct {
    guint32 digest[5];             /* message digest */
    guint32 count_lo, count_hi;    /* 64-bit bit count */
    guint32 data[16];              /* SHA data buffer */
    int local;                  /* unprocessed amount in data */
} EdsioSHACtx;

void edsio_sha_init   (EdsioSHACtx *);
void edsio_sha_update (EdsioSHACtx *, const guint8 *, guint);
void edsio_sha_final  (guint8 *, EdsioSHACtx *);

gint  edsio_sha_equal (gconstpointer   v,
		       gconstpointer   v2);
guint edsio_sha_hash  (gconstpointer   v);

typedef struct _MessageDigest MessageDigest;

struct _MessageDigest {
  guint ctx_size;
  guint cksum_len;
  guint id;
  const char* name;
  void (*init)();
  void (*update)();
  void (*final)();
};

const MessageDigest* edsio_message_digest_by_name (const char* name);
const MessageDigest* edsio_message_digest_by_id   (guint       id);

const MessageDigest* edsio_message_digest_md5     (void);
const MessageDigest* edsio_message_digest_sha     (void);

void      edsio_digest_to_string   (const MessageDigest* md, const guint8* md5, char *buf);
gboolean  edsio_digest_from_string (const MessageDigest* md, guint8* md5,       const char *str);

#define EDSIO_DIGEST_MAX 20
#define EDSIO_DIGEST_STRING_MAX (2*EDSIO_DIGEST_MAX+1)

/* Misc stuff.
 */

/* These raise an error if errmsg is non-null.  The errmsg should
 * be something like "Invalid port number".  See edsio.ser for the
 * format.
 */
gboolean strtosi_checked (const char* str, gint32* i, const char* errmsg);
gboolean strtoss_checked (const char* str, gint16* s, const char* errmsg);

gboolean strtoui_checked (const char* str, guint32* i, const char* errmsg);
gboolean strtous_checked (const char* str, guint16* i, const char* errmsg);

const char* edsio_intern_string (const char* str);

GByteArray*    edsio_base64_encode_region           (const guint8* data, guint data_len);
GByteArray*    edsio_base64_decode_region           (const guint8* data, guint data_len);
gboolean       edsio_base64_encode_region_into      (const guint8* data, guint data_len, guint8* out, guint *out_len);
gboolean       edsio_base64_decode_region_into      (const guint8* data, guint data_len, guint8* out, guint *out_len);

gchar*   edsio_time_to_iso8601   (SerialGenericTime* time);
gchar*   edsio_time_t_to_iso8601 (GTime time);
gboolean edsio_time_of_day       (SerialGenericTime* time);

enum _SimpleBufferFlags {
  SBF_None     = 0,
  SBF_Compress = 1 << 0,
  SBF_Checksum = 1 << 1,
  SBF_Base64   = 1 << 2
};

typedef enum _SimpleBufferFlags SimpleBufferFlags;

SerialSource*  edsio_simple_source                  (const guint8* data, guint len, guint flags);
SerialSink*    edsio_simple_sink                    (gpointer data,
						     guint    flags,
						     gboolean free_result,
						     void (* success) (gpointer data, GByteArray* result),
						     GByteArray **result);

gboolean edsio_library_check                        (guint32 number, gboolean err);
void     edsio_library_register                     (guint32 number, const char*name);


/**********************************************************************/
/*			  Handle Operations                           */
/**********************************************************************/

/* This serves as a virtual table for I/O to the FileHandle */

struct _FilePosition {
  gssize  page;
  gsize   offset;
  gssize  page_size;
};

struct _HookList {
  void      *data;
  gboolean (*func) (FileHandle* fh, void* data);
  HookList  *next;
};

struct _FileHandle {
  const HandleFuncTable* table;

  gint32       fh_open_flags;
  gint16       fh_open_mode;

  guint        fh_closed : 1;
  guint        fh_has_len : 1;
  guint        fh_read_mapped : 1;
  guint        fh_at_eof : 1;

  guint8      *fh_cur_page;  /* current page, if mapped (pgno = curpos.page), else NULL */
  guint8      *fh_read_page; /* read page, if not seekable, for implementing map_page */

  FilePosition fh_cur_pos;  /* Current read/write position */
  FilePosition fh_file_len; /* EOF in current state */

  HookList         *fh_close_hooks;

  MessageDigestCtx *fh_digests;
};

enum _HandleOpenFlags {
  HV_Read                    = 1<<0,
  HV_Replace                 = 1<<1,
  HV_NoSeek                  = 1<<2
};

typedef enum _HandleOpenFlags HandleOpenFlags;

struct _HandleFuncTable
{
  gboolean          (* fh_close)   (FileHandle* fh);
  void              (* fh_abort)   (FileHandle* fh);
  const gchar*      (* fh_name)    (FileHandle *fh); /* user must free */
  void              (* fh_free)    (FileHandle* fh);

  /* Seekable, buffered files */
  gssize            (* fh_map)     (FileHandle *fh, guint pgno, const guint8** mem);
  gboolean          (* fh_unmap)   (FileHandle *fh, guint pgno, const guint8** mem);

  /* Unseekable, unbuffered files */
  gssize            (* fh_read)    (FileHandle *fh, guint8* buf, gsize nbyet);
  gboolean          (* fh_write)   (FileHandle *fh, const guint8 *buf, gsize nbyte);
  gboolean          (* fh_eof)     (FileHandle *fh);

  /* Sync */
  gboolean          (* fh_sync)    (FileHandle* fh);
};


gssize  file_position_rem_on_page_no (const FilePosition *len, guint pgno);
gssize  file_position_to_abs         (const FilePosition *pos);
void    file_position_from_abs       (guint page_size, guint abs, FilePosition *pos);
void    file_position_incr           (FilePosition *pos, gint incr_by);

#define HANDLE_SEEK_SET 9
#define HANDLE_SEEK_CUR 10
#define HANDLE_SEEK_END 11

gssize   handle_length     (FileHandle *fh);
gsize    handle_position   (FileHandle *fh);
gssize   handle_pages      (FileHandle *fh);
gssize   handle_pagesize   (FileHandle *fh);
gboolean handle_eof        (FileHandle *fh);
gssize   handle_read       (FileHandle *fh, guint8 *buf, gsize nbyte);
gboolean handle_write      (FileHandle *fh, const guint8 *buf, gsize nbyte);
gint     handle_getc       (FileHandle *fh);
gboolean handle_putc       (FileHandle *fh, gchar c);
gboolean handle_getui      (FileHandle *fh, guint32* i);
gboolean handle_putui      (FileHandle *fh, guint32 i);
gboolean handle_printf     (FileHandle *fh, const char* fmt, ...);
gboolean handle_puts       (FileHandle *fh, const char* linebuf);
gssize   handle_gets       (FileHandle *fh, char* buf, gsize nbyte);
gssize   handle_seek       (FileHandle *fh, gssize offset,     gint whence);
gboolean handle_close      (FileHandle *fh);
gboolean handle_sync       (FileHandle *fh);
void     handle_abort      (FileHandle *fh);
void     handle_free       (FileHandle *fh);
gpointer handle_unique     (FileHandle* fh);
gssize   handle_map_page   (FileHandle *fh, guint pgno, const guint8** mem);
gboolean handle_unmap_page (FileHandle *fh, guint pgno, const guint8** mem /* may be NULL */);
void     handle_close_hook (FileHandle *fh, void* data, gboolean (*) (FileHandle* fh, void* data));

const guint8*  handle_digest         (FileHandle *fh, const MessageDigest *md);
gboolean       handle_digest_compute (FileHandle *fh, const MessageDigest *md);

gboolean handle_copy       (FileHandle *from,
			    FileHandle *to,
			    guint       off,
			    guint       len);

gboolean handle_copy_len   (FileHandle *from,
			    FileHandle *to,
			    guint       len);

gboolean handle_drain      (FileHandle *from,
			    FileHandle *to);

SerialSource*  handle_source (FileHandle *fh);

SerialSink*    handle_sink   (FileHandle *fh,
			      gpointer data1,
			      gpointer data2,
			      gpointer data3,
			      gboolean (* cont_onclose) (gpointer data1, gpointer data2, gpointer data3));

const char* eventdelivery_handle_to_string  (FileHandle* x);

gboolean edsio_library_init (void);

FileHandle* handle_read_mem  (const guint8* buf, guint len);
FileHandle* handle_write_mem (void);
void        handle_write_mem_result (FileHandle* fh, const guint8** buf, guint* len);

gboolean handle_cmp (FileHandle *f1, FileHandle *f2, guint *pos, gboolean *diff);

#ifdef __cplusplus
}
#endif

#endif /* _EDSIO_H_ */
