/* -*-Mode: C++;-*-
 * $Id: curs.cc 1.16 Fri, 29 Jun 2001 16:59:43 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"
#include <errno.h>
#include <unistd.h>

static int
dbfs_rmw (DBFS_RMWFLAG flag)
{
    g_assert (flag == DBFS_RMW || flag == DBFS_NORMW);

    if (flag & DBFS_RMW) {
	return DB_RMW;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			      STATIC DB
//////////////////////////////////////////////////////////////////////

STATIC_DB::STATIC_DB (BASIC_DBENV &dbfs,
		      const char  *name,
		      DBTYPE       type,
		      int          rec_size)
    : _name     (name),
      _type     (type),
      _rec_size (rec_size)
{
    _next = dbfs._static_list;
            dbfs._static_list = this;
}

int
DBREF::common_open (Db           *dbp,
		    const char   *name,
		    DBTYPE        type,
		    int           rec_size,
		    int           oflag)
{
    int ret;
    int dbflag = DB_THREAD;

    if (oflag & DBFS_CREATE) {

	dbflag |= DB_CREATE;

	if ((rec_size >= 0) && (ret = dbp->set_re_len (rec_size))) {
	    DB_ERROR (ret) ("db_set_re_len: %s", name);
	    return ret;
	}
    }

    if ((ret = dbp->open (name, NULL, type, dbflag, 0666))) {
	DB_ERROR (ret) ("db_open: %s", name);
	return ret;
    }

    return 0;
}

int
STATIC_DB::open (BASIC_DBENV &dbfs,
		 int          oflag)
{
    int        ret;
    SHDB_DESC *sdbd = NULL;

    if ((ret = dbfs._shared_dbs.static_alloc (sdbd))) {
	DBFS_ERROR (ret) ("Not enough Shared-DB descriptors for static set");
	return ret;
    }

    sdbd->refs = 0;
    sdbd->key  = XFID (0);
    sdbd->dbp  = new Db (dbfs._env, 0);

    if ((ret = common_open (sdbd->dbp, _name, _type, _rec_size, oflag))) {
	PROP_ERROR (ret) ("common_open: %s", _name);
	return ret;
    }

    copy (sdbd);

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			      DYNAMIC DB
//////////////////////////////////////////////////////////////////////

int
DYNAMIC_DB::open (DBFS       &dbfs,
		  const XFID &fid,
		  DBTYPE      type,
		  int         rec_size,
		  int         oflag)
{
    int ret;

    _fid      = fid;
    _type     = type;
    _rec_size = rec_size;

    g_assert (_fid.key () != 0);

    SHDB_DESC *sdbd;
    MONITOR    sdbd_mon;

    if ((ret = dbfs._shared_dbs.find_shared (_fid, sdbd, sdbd_mon))) {
	PROP_ERROR (ret) ("shared_dbs_find");
	return ret;
    }

    if (sdbd->dbp == NULL) {

	string name;

	dbfs.relative_fname (_fid, name);

	sdbd->dbp = new Db (dbfs._env, 0);

	if ((ret = common_open (sdbd->dbp, name.c_str (), _type, _rec_size, oflag))) {
	    // @@ Shared-dset error handling
	    PROP_ERROR (ret) ("common_open");
	    return ret;
	}
    }

    copy (sdbd);

    return 0;
}

//////////////////////////////////////////////////////////////////////
//				DBREF
//////////////////////////////////////////////////////////////////////

int
DBREF::cursor (STXN   &stxn,
	       DBCREF &dbcref) const
{
    Dbc *dbc = NULL;
    int  ret;

    if ((ret = dbp ()->cursor (stxn.dbtxn (), & dbc, 0))) {
	DB_ERROR (ret) ("cursor");
	return ret;
    }

    dbcref.set_open (*this, stxn, dbc);

    return 0;
}

int
DBREF::append (STXN         &stxn,
	       XSEQNO       &seqno_out,
	       const COMDBT &data_in) const
{
    int            ret;
    RECDBT<XSEQNO> seqno_dbt (seqno_out);

    if ((ret = dbp ()->put (stxn.dbtxn (), seqno_dbt (), data_in (), DB_APPEND))) {
	DB_ERROR (ret) ("db_append");
	return ret;
    }

    return 0;
}

int
DBREF::get_must_exist (STXN         &stxn,
		       const COMDBT &key,
		       COMDBT       &data,
		       DBFS_RMWFLAG  rmwflag) const
{
    int    ret;
    DBCREF dbc;

    if ((ret = cursor (stxn, dbc)) ||
	(ret = dbc.get_must_exist (key, data, rmwflag)) ||
	(ret = dbc.close ())) {
	return ret;
    }

    return 0;
}

int
DBREF::get_or_notfound (STXN         &stxn,
			const COMDBT &key,
			COMDBT       &data,
			DBFS_RMWFLAG  rmwflag) const
{
    int    ret;
    DBCREF dbc;

    if ((ret = cursor (stxn, dbc)) ||
	(ret = dbc.get_or_notfound (key, data, rmwflag)) ||
	(ret = dbc.close ())) {
	return ret;
    }

    return 0;
}

int
DBREF::put_overwrite (STXN         &stxn,
		      const COMDBT &key,
		      const COMDBT &data) const
{
    int    ret;
    DBCREF dbc;

    if ((ret = cursor (stxn, dbc)) ||
	(ret = dbc.put_overwrite (key, data)) ||
	(ret = dbc.close ())) {
	return ret;
    }

    return 0;
}

int
DBREF::put_no_overwrite (STXN         &stxn,
			 const COMDBT &key,
			 const COMDBT &data) const
{
    int    ret;
    DBCREF dbc;

    if ((ret = cursor (stxn, dbc)) ||
	(ret = dbc.put_no_overwrite (key, data)) ||
	(ret = dbc.close ())) {
	return ret;
    }

    return 0;
}

int
DBREF::delete_must_exist (STXN         &stxn,
			  const COMDBT &key) const
{
    int    ret;
    DBCREF dbc;

    if ((ret = cursor (stxn, dbc)) ||
	(ret = dbc.delete_must_exist (key)) ||
	(ret = dbc.close ())) {
	return ret;
    }

    return 0;
}

int
DBREF::delete_or_notfound (STXN         &stxn,
			   const COMDBT &key) const
{
    int    ret;
    DBCREF dbc;

    if ((ret = cursor (stxn, dbc)) ||
	(ret = dbc.delete_or_notfound (key)) ||
	(ret = dbc.close ())) {
	return ret;
    }

    return 0;
}

int
DBREF::consume (STXN         &stxn,
		XSEQNO       &seqno,
		COMDBT       &data) const
{
    int    ret;
    DBCREF dbc;

    if ((ret = cursor (stxn, dbc)) ||
	(ret = dbc.consume (seqno, data)) ||
	(ret = dbc.close ())) {
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//				DBCREF COMPOUND
//////////////////////////////////////////////////////////////////////

DBCREF::~DBCREF ()
{
    if (_dbc) {

	int ret;

	if ((ret = DBCREF::close ())) {
	    // @@ Should remember?
	    DBFS_ERROR (ret) ("cursor auto close");
	}
    }
}

int
DBCREF::close ()
{
    int ret = 0;

    if (_dbc) {

	if ((ret = _dbc->close ())) {
	    DB_ERROR (ret) ("dbc_close");
	}

	_dbc = NULL;

	_dbr.close ();
    }

    return ret;
}

int
DBCREF::get_must_exist (const COMDBT &key,
			COMDBT       &data,
			DBFS_RMWFLAG  rmwflag) const
{
    int ret;

    if ((ret = get_or_notfound (key, data, rmwflag)) && (ret != DBFS_NOTFOUND)) {
	PROP_ERROR (ret) ("get_or_notfound");
	return ret;
    }

    if (ret == DBFS_NOTFOUND) {
	DBFS_ERROR (ret) ("get_must_exist");
	return DBFS_LINK_INCONSISTENT;
    }

    return ret;
}

int
DBCREF::put_no_overwrite (const COMDBT &key,
			  const COMDBT &data) const
{
    int     ret;
    NULLDBT null;

    if ((ret = get_or_notfound (key, null, DBFS_RMW)) && (ret != DBFS_NOTFOUND)) {
	PROP_ERROR (ret) ("get_or_notfound");
	return ret;
    }

    if (ret != DBFS_NOTFOUND) {
	ret = DBFS_LINK_INCONSISTENT;
	DBFS_ERROR ("put_no_overwrite: record exists");
	return ret;
    }

    if ((ret = put_overwrite (key, data))) {
	PROP_ERROR (ret) ("put_overwrite");
	return ret;
    }

    return 0;
}

int
DBCREF::delete_must_exist (const COMDBT &key) const
{
    int     ret;
    NULLDBT null;

    if ((ret = get_must_exist (key, null, DBFS_RMW))) {
	PROP_ERROR (ret) ("get_must_exist");
	return ret;
    }

    if ((ret = delete_current ())) {
	PROP_ERROR (ret) ("delete_current");
	return ret;
    }

    return 0;
}

int
DBCREF::delete_or_notfound (const COMDBT &key) const
{
    int     ret;
    NULLDBT null;

    if ((ret = get_or_notfound (key, null, DBFS_RMW))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("get_or_notfound");
	}
	return ret;
    }

    if ((ret = delete_current ())) {
	PROP_ERROR (ret) ("delete_current");
	return ret;
    }

    return 0;
}

int
DBCREF::set_range (COMDBT &key,
		   COMDBT &data) const
{
    return move_pos (key, data, DB_SET_RANGE, DBFS_NORMW);
}

//////////////////////////////////////////////////////////////////////
//			      DBCREF BASIC
//////////////////////////////////////////////////////////////////////

//  int
//  DBCREF::get_current (const COMDBT &data) const
//  {
//      int ret;
//      NULLDBT null;

//      DEBUG_CURSOR (this) ("get_current");

//      if ((ret = _dbc->get (null (), data (), DB_CURRENT))) {

//  	if (ret == DB_NOTFOUND) {
//  	    return DBFS_NOTFOUND;
//  	}

//  	DB_ERROR (ret) ("dbc_get");
//  	return ret;
//      }

//      return 0;
//  }

int
DBCREF::put_current (const COMDBT &data) const
{
    int ret;
    // Note: This is a dangerous option!  Positioning commands are
    // ignored when DB_NOTFOUND or DB_KEYEMPTY is returned, so will
    // overwrite the last successful get, even if you consider
    // NOTFOUND "success".  (could there be a flag to turn this on?)

    DEBUG_CURSOR (this) ("put_current");

    if ((ret = _dbc->put (NULL, data (), DB_CURRENT))) {

	if (ret == DB_NOTFOUND) {
	    DBFS_ERROR ("Hash-DB cursor cannot delete then put");
	    return DBFS_NOTFOUND;
	}

	DB_ERROR (ret) ("dbc_put");
	return ret;
    }

    return 0;
}

int
DBCREF::delete_current (void) const
{
    int ret;

    DEBUG_CURSOR (this) ("delete_current");

    if ((ret = _dbc->del (0))) {

	if (ret == DB_KEYEMPTY) {
	    DBFS_ERROR (ret) ("delete_current");
	    return DBFS_NOTFOUND;
	}

	DB_ERROR ("dbc_del");
	return ret;
    }

    return 0;
}

int
DBCREF::get_or_notfound (const COMDBT &key,
			 COMDBT       &data,
			 DBFS_RMWFLAG  rmwflag) const
{
    int ret;
    int dbflag = DB_SET;
    int times = 0;

    dbflag |= dbfs_rmw (rmwflag);

    DEBUG_CURSOR (this) ("get_or_notfound: ") (key);

  again:

    times += 1;

    g_assert (times <= 2);

    if ((ret = _dbc->get (key (), data (), dbflag))) {

	if (ret == ENOMEM) {

	    key.const_post ();
	    data.enlarge ();
	    DEBUG_ENLARGE (this) ("get enlarge %d: key", times) (key) (" data ") (data);
	    goto again;
	}

	if ((ret == DB_KEYEMPTY) || (ret == DB_NOTFOUND)) {
	    return DBFS_NOTFOUND;
	}

	DB_ERROR (ret) ("dbc_set");
	return ret;
    }

    key.const_post ();
    data.post ();

    return 0;
}

int
DBCREF::put_overwrite (const COMDBT &key,
		       const COMDBT &data) const
{
    int ret;

    DEBUG_CURSOR (this) ("put_overwrite: ") (key);

    if (dbtype () == DB_QUEUE) {
	NULLDBT null;

	if ((ret = _dbc->get (key (), null (), DB_SET))) {
	    DB_ERROR (ret) ("dbc_get");
	    return ret;
	}

	if ((ret = _dbc->put (NULL, data (), DB_CURRENT))) {
	    DB_ERROR (ret) ("dbc_put");
	    return ret;
	}

    } else {

	// This doesn't work for QUEUE or RECNO, not sure why
	if ((ret = _dbc->put (key (), data (), DB_KEYLAST))) {
	    DB_ERROR (ret) ("dbc_put");
	    return ret;
	}
    }

    return 0;
}

int
DBCREF::move_pos (COMDBT       &key,
		  COMDBT       &data,
		  int          moveflag,
		  DBFS_RMWFLAG rmwflag) const
{
    int ret   = 0;
    int times = 0;

    moveflag |= dbfs_rmw (rmwflag);

  again:

    times += 1;

    g_assert (times <= 3);

    if ((ret = _dbc->get (key (), data (), moveflag))) {

	if (ret == ENOMEM) {

	    // The only difference from above is the different
	    // treatment of the const key argument.  This is because
	    // DB_SET operations do not modify key, while cursor moves
	    // do.
	    key.enlarge ();
	    data.enlarge ();
	    DEBUG_ENLARGE (this) ("move_pos enlarge %d: key", times) (key) (" data ") (data);
	    goto again;
	}

	if (ret == DB_KEYEMPTY) {
	    // Should we skip, or does the cursor do that?
	    g_assert_not_reached ();
	}

	if (ret == DB_NOTFOUND) {
	    return DBFS_NOTFOUND;
	}

	DB_ERROR (ret) ("dbc_move");
	return ret;
    }

    DEBUG_CURSOR (this) ("move_pos: ") (key);

    key.post ();
    data.post ();

    return 0;
}

int
DBCREF::consume (XSEQNO &seqno, COMDBT &data) const
{
    int            ret;
    RECDBT<XSEQNO> key_dbt (seqno);

    if ((ret = _dbc->get (key_dbt (), data (), DB_CONSUME))) {

	if (ret == DB_NOTFOUND) {
	    return DBFS_NOTFOUND;
	}

	DB_ERROR (ret) ("dbc_consume");
	return ret;
    }

    DEBUG_CURSOR (this) ("consume: %u", seqno.key ());

    data.post ();

    return 0;
}
