/* -*-Mode: C++;-*-
 * $Id: curs.h 1.14 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _CURS_H_
#define _CURS_H_

//////////////////////////////////////////////////////////////////////
//			      COMMON DBT
//////////////////////////////////////////////////////////////////////

class COMDBT
{
public:

    COMDBT ()                        { set_dbt (NULL, 0); }
    COMDBT (void       *p, size_t l) { set_dbt (p, l); }
    COMDBT (const void *p, size_t l) { set_dbt (p, l); }

    virtual ~COMDBT () { }

    // Error to enlarge by default
    virtual void enlarge () { const_post (); }
    virtual void post    () { const_post (); }

    void    const_post () const { g_assert (_base_size == get_size ()); }
    size_t  get_size   () const { return _dbt.get_size (); }
    Dbt*    operator() () const { _base_size = get_size (); return & _dbt; }

protected:

    void    set_dbt   (const void* p, size_t l) const
    {
	memset (& _dbt, 0, sizeof (_dbt));

	_base_size = l;

	_dbt.set_data  ((void*) p); // Bogus const-cast
	_dbt.set_size  (l);
	_dbt.set_ulen  (l);
	_dbt.set_flags (DB_DBT_USERMEM);
    }

    void    reset_dbt   (const void* p, size_t l) const
    {
	_dbt.set_data  ((void*) p); // Bogus const-cast
	_dbt.set_size  (l);
	_dbt.set_ulen  (l);
    }

    void    reset_ulen  (const void* p, size_t l) const
    {
	_dbt.set_data  ((void*) p); // Bogus const-cast
	_dbt.set_ulen  (l);
    }

    void    set_flags  (int flags) const { _dbt.set_flags (flags); }

private:

    mutable Dbt    _dbt;
    mutable size_t _base_size;
};

//////////////////////////////////////////////////////////////////////
//			       MISC DBT
//////////////////////////////////////////////////////////////////////

template <class T>
class RECDBT
    : public COMDBT
{
public:

    RECDBT (T       &v) : COMDBT (& v, sizeof (v)) { }
    RECDBT (const T &v) : COMDBT (& v, sizeof (v)) { }
};

class NULLDBT
    : public COMDBT
{
public:

    NULLDBT ()
    {
	set_flags (DB_DBT_USERMEM | DB_DBT_PARTIAL);
    }
};

class BYTEDBT
    : public COMDBT
{
public:

    BYTEDBT (const guint8 *p, size_t l)
	: COMDBT (p, l)
    {
    }
    BYTEDBT (guint8 *p, size_t l)
	: COMDBT (p, l)
    {
    }
};

//////////////////////////////////////////////////////////////////////
//				DBREF
//////////////////////////////////////////////////////////////////////

class DBREF
    : protected DESC_REF<SHDB_DESC>
{
    friend class DBCREF;
    friend class DBFS;

public:

    DBREF () { }

    DBREF (const DBREF& dbr)
    {
	copy (dbr._sdesc);
    }

    DBREF& operator= (const DBREF& dbr)
    {
	decr ();

	copy (dbr._sdesc);

	return *this;
    }

    void   close () { decr (); }

    DBTYPE dbtype () const
    {
	int ret;
	DBTYPE type;
	if ((ret = dbp ()->get_type (& type)) != 0) {
	    /* @@@ lie */
	    return DB_HASH;
	}
	return type;
    }

    int  cursor               (STXN         &stxn,
			       DBCREF       &dbc) const;

    int  append               (STXN         &stxn,
			       XSEQNO       &seqno_out,
			       const COMDBT &data_in) const;

    int  get_must_exist       (STXN         &stxn,
			       const COMDBT &key,
			       COMDBT       &data,
			       DBFS_RMWFLAG  rmwflag) const;

    int  get_or_notfound      (STXN         &stxn,
			       const COMDBT &key,
			       COMDBT       &data,
			       DBFS_RMWFLAG  rmwflag) const;

    int  put_overwrite        (STXN         &stxn,
			       const COMDBT &key,
			       const COMDBT &data) const;

    int  put_no_overwrite     (STXN         &stxn,
			       const COMDBT &key,
			       const COMDBT &data) const;

    int  delete_must_exist    (STXN         &stxn,
			       const COMDBT &key) const;

    int  delete_or_notfound   (STXN         &stxn,
			       const COMDBT &key) const;

    int  consume              (STXN         &stxn,
			       XSEQNO       &seqno,
			       COMDBT       &data) const;

    template <class D>
    int append_T              (STXN         &stxn,
			       XSEQNO       &seqno_out,
			       const D      &data_in) const;

    template<class K, class D>
    int  get_must_exist_T     (STXN         &stxn,
			       const K      &key,
			       D            &data,
			       DBFS_RMWFLAG  rmwflag) const;

    template<class K, class D>
    int  get_or_notfound_T    (STXN         &stxn,
			       const K      &key,
			       D            &data,
			       DBFS_RMWFLAG  rmwflag) const;

    template<class K, class D>
    int  put_overwrite_T      (STXN         &stxn,
			       const K      &key,
			       const D      &data) const;

    template<class K, class D>
    int  put_no_overwrite_T   (STXN         &stxn,
			       const K      &key,
			       const D      &data) const;

    template<class K>
    int  delete_must_exist_T  (STXN         &stxn,
			       const K      &key) const;

    template<class K>
    int  delete_or_notfound_T (STXN         &stxn,
			       const K      &key) const;

    template <class D>
    int  consume_T            (STXN         &stxn,
			       XSEQNO       &seqno,
			       D            &data) const;

protected:

    Db* dbp () const
    {
	g_assert (_sdesc);
	g_assert (_sdesc->dbp);

	_sdesc->use ();

	return _sdesc->dbp;
    }

    static int common_open (Db           *dbp,
			    const char   *name,
			    DBTYPE        type,
			    int           rec_size,
			    int           oflag);

    friend ostream& operator<< (ostream &os, const DBREF& dbr);
};

inline ostream&
operator<< (ostream &os, const DBREF& dbr)
{
    os << dbr._sdesc->key;
    return os;
}

//////////////////////////////////////////////////////////////////////
//				DBCREF
//////////////////////////////////////////////////////////////////////

class DBCREF
{
    friend class DBREF;

private:

    DBCREF            (const DBCREF &nocopy);
    DBCREF& operator= (const DBCREF &noassign);

    void set_open (const DBREF& dbr, STXN &txn, Dbc *dbc)
    {
	g_assert (_dbc == NULL);

	_txn = & txn;
	_dbr = dbr;
	_dbc = dbc;
    }

public:

    DBCREF ()
	: _txn (NULL),
	  _dbc (NULL)
    {
    }

    virtual ~DBCREF ();

    virtual int close ();

    DBTYPE dbtype () const { return _dbr.dbtype (); }

    int  move_pos            (COMDBT       &key,
			      COMDBT       &data,
			      int           move_flag,
			      DBFS_RMWFLAG  flag) const;

    int  set_range            (COMDBT       &key,
			       COMDBT       &data) const;

    int  get_must_exist      (const COMDBT &key,
			      COMDBT       &data,
			      DBFS_RMWFLAG  rmwflag) const;
    int  get_or_notfound     (const COMDBT &key,
			      COMDBT       &data,
			      DBFS_RMWFLAG  rmwflag) const;
    int  get_current         (const COMDBT &data) const;

    int  put_overwrite       (const COMDBT &key,
			      const COMDBT &data) const;
    int  put_no_overwrite    (const COMDBT &key,
			      const COMDBT &data) const;
    int  put_current         (const COMDBT &data) const;

    int  delete_must_exist   (const COMDBT &key) const;
    int  delete_or_notfound  (const COMDBT &key) const;
    int  delete_current      (void) const;

    int  consume             (XSEQNO       &seqno,
			      COMDBT       &data) const;

    template<class K, class D>
    int  get_must_exist_T    (const K      &key,
			      D            &data,
			      DBFS_RMWFLAG  rmwflag) const;

    template<class K, class D>
    int  get_or_notfound_T   (const K      &key,
			      D            &data,
			      DBFS_RMWFLAG rmwflag) const;

    template<class K, class D>
    int  put_overwrite_T     (const K &key,
			      const D &data) const;

    template<class K, class D>
    int  put_no_overwrite_T  (const K &key,
			      const D &data) const;

    template<class D>
    int  put_current_T       (const D &data) const;

    template<class D>
    int  get_current_T       (const D &data) const;

    template<class K>
    int  delete_must_exist_T (const K &key) const;

    template<class K>
    int  delete_or_notfound_T (const K &key) const;

    template <class D>
    int  consume_T           (XSEQNO  &seqno,
			      D       &data) const;

private:

    friend class MESSAGE_CLOSURE;

    STXN  *_txn;
    DBREF  _dbr;
    Dbc   *_dbc;
};

//////////////////////////////////////////////////////////////////////
//			      STATIC DBREF
//////////////////////////////////////////////////////////////////////

class STATIC_DB
    : public DBREF
{
public:

    STATIC_DB (BASIC_DBENV  &dbfs,
	       const char   *name,
	       DBTYPE        type,
	       int           rec_size = -1);

    int        open  (BASIC_DBENV  &dbfs,
		      int           oflag);

    const char *_name;
    DBTYPE      _type;
    int         _rec_size;
    STATIC_DB  *_next;
};

//////////////////////////////////////////////////////////////////////
//			      DYNAMIC DBREF
//////////////////////////////////////////////////////////////////////

class DYNAMIC_DB
    : public DBREF
{
public:

    int        open  (DBFS       &dbfs,
		      const XFID &fid,
		      DBTYPE      type,
		      int         rec_size,
		      int         oflag);

private:

    XFID        _fid;
    DBTYPE      _type;
    int         _rec_size;
};

//////////////////////////////////////////////////////////////////////
//			    DBCREF INLINES
//////////////////////////////////////////////////////////////////////

template<class K, class D>
inline int
DBCREF::get_must_exist_T (const K &key,
			  D       &data,
			  DBFS_RMWFLAG rmwflag) const
{
    RECDBT<D> data_dbt (data);
    return get_must_exist (RECDBT<K> (key), data_dbt, rmwflag);
}

template<class K, class D>
inline int
DBCREF::get_or_notfound_T (const K      &key,
			   D            &data,
			   DBFS_RMWFLAG  rmwflag) const
{
    RECDBT<D> data_dbt (data);
    return get_or_notfound (RECDBT<K> (key), data_dbt, rmwflag);
}

template<class K, class D>
inline int
DBCREF::put_overwrite_T (const K &key,
			 const D &data) const
{
    return put_overwrite (RECDBT<K> (key), RECDBT<D> (data));
}

template<class K, class D>
inline int
DBCREF::put_no_overwrite_T (const K &key,
			    const D &data) const
{
    return put_no_overwrite (RECDBT<K> (key), RECDBT<D> (data));
}

template<class D>
inline int
DBCREF::put_current_T (const D &data) const
{
    return put_current (RECDBT<D> (data));
}

template<class D>
inline int
DBCREF::get_current_T (const D &data) const
{
    return get_current (RECDBT<D> (data));
}

template<class K>
inline int
DBCREF::delete_must_exist_T (const K &key) const
{
    return delete_must_exist (RECDBT<K> (key));
}

template<class K>
inline int
DBCREF::delete_or_notfound_T (const K &key) const
{
    return delete_or_notfound (RECDBT<K> (key));
}

template <class D>
inline int
DBCREF::consume_T (XSEQNO &seqno, D &data) const
{
    RECDBT<D> data_dbt (data);
    return consume (seqno, data_dbt);
}

//////////////////////////////////////////////////////////////////////
//			    DBCREF INLINES
//////////////////////////////////////////////////////////////////////

template <class D>
inline int
DBREF::append_T (STXN         &stxn,
		 XSEQNO       &seqno_out,
		 const D      &data_in) const
{
    return append (stxn, seqno_out, RECDBT<D> (data_in));
}

template<class K, class D>
inline int
DBREF::get_must_exist_T (STXN         &stxn,
			 const K      &key,
			 D            &data,
			 DBFS_RMWFLAG  rmwflag) const
{
    RECDBT<D> data_dbt (data);
    return get_must_exist (stxn, RECDBT<K> (key), data_dbt, rmwflag);
}

template<class K, class D>
inline int
DBREF::get_or_notfound_T (STXN         &stxn,
			  const K      &key,
			  D            &data,
			  DBFS_RMWFLAG  rmwflag) const
{
    RECDBT<D> data_dbt (data);
    return get_or_notfound (stxn, RECDBT<K> (key), data_dbt, rmwflag);
}

template<class K, class D>
inline int
DBREF::put_overwrite_T (STXN         &stxn,
			const K      &key,
			const D      &data) const
{
    return put_overwrite (stxn, RECDBT<K> (key), RECDBT<D> (data));
}

template<class K, class D>
inline int
DBREF::put_no_overwrite_T (STXN         &stxn,
			   const K      &key,
			   const D      &data) const
{
    return put_no_overwrite (stxn, RECDBT<K> (key), RECDBT<D> (data));
}

template<class K>
inline int
DBREF::delete_must_exist_T (STXN         &stxn,
			    const K      &key) const
{
    return delete_must_exist (stxn, RECDBT<K> (key));
}

template <class D>
inline int
DBREF::consume_T (STXN         &stxn,
		  XSEQNO       &seqno,
		  D            &data) const
{
    RECDBT<D> data_dbt (data);
    return consume (stxn, seqno, data_dbt);
}

#endif /* _CURS_H_ */
