/* -*-Mode: C++;-*-
 * $Id: delete.cc 1.14 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"

#include <unistd.h>
#include <errno.h>

//////////////////////////////////////////////////////////////////////
//			  DELETE OPERATIONS
//////////////////////////////////////////////////////////////////////

int
NODEC::pre_overwrite_when (int flags, bool del_now) const
{
    int ret;

    if ((ret = dbfs_check_ow_flags (flags))) {
	PROP_ERROR (ret) ("dbfs_check_ow_flags");
	return ret;
    }

    if (type () == FT_NotPresent) {
	return 0;
    }

    if (flags & DBFS_OVERWRITE) {

	if (del_now && (ret = del_minor ())) {
	    PROP_ERROR (ret) ("del_minor");
	    return ret;
	}

	return 0;
    }

    return DBFS_EXISTS;
}

int
MAJORC::del_major () const
{
    g_assert (nlinks () == DBFS_ZERO_REFS);
    g_assert (number () != DBFS_ROOT_DIRECTORY);

    // Note: Happens during TXN::pre_commit

    return 0;
}

int
NODEC::del_minor () const
{
    int ret;

    if (type () == FT_NotPresent) {
	return DBFS_NEXISTS;
    }

    if ((ret = pre_delete ())) {
	PROP_ERROR (ret) ("pre_delete");
	return ret;
    }

    return 0;
}

int
NODEC::pre_delete_cont () const
{
    int  ret;
    DIRC curs;

    if ((ret = dir_open (curs))) {
	PROP_ERROR (ret) (this) ("dir_open");
	return ret;
    }

    while (curs.next ()) {

	MAJORC major;

	if ((ret = curs.get_node (major))) {
	    PROP_ERROR (ret) (this) ("get_node");
	    return ret;
	}

	if ((ret = major.del_link_to (cont_id ().file_id ()))) {
	    PROP_ERROR (ret) (major) ("del_link_to");
	    return ret;
	}
    }

    if ((ret = curs.close ())) {
	PROP_ERROR (ret) (this) ("curs_close");
	return ret;
    }

    return 0;
}

int
NODEC::pre_delete () const
{
    int   ret;
    DXN   dxn (txn (), ret);
    DBFS &dbfs = txn ().dbfs ();

    switch (type ()) {
    case FT_DirHash:
    case FT_DirBtree:
    case FT_DirSeq: {

	if ((ret = pre_delete_cont ())) {
	    PROP_ERROR (ret) (this) ("pre_delete_cont");
	    return ret;
	}

	break;
    }

    case FT_Reflink: {

	MAJORC decr;

	if ((ret = read_reflink (decr, DBFS_LINK_RMW))) {
	    PROP_ERROR (ret) (this) ("read_reflink");
	    return ret;
	}

	decr.incr_refcount (-1);

	break;
    }

    case FT_ShortSeg: {

	if ((ret = dxn.del_short (*this))) {
	    PROP_ERROR (ret) (this) ("del_short");
	    return ret;
	}

	break;
    }

    case FT_Arealink:
    case FT_ViewSeg:
    case FT_LongSeg: {
	break;
    }

    default:
	g_assert_not_reached ();
    }

    // Return FID to freelist
    switch (type ()) {
    case FT_DirHash:
    case FT_DirBtree:
    case FT_DirSeq:
    case FT_LongSeg: {

	XFID free_fid = cont_id ().file_id ();

	if ((ret = dxn.deallocate_fid (free_fid, xtype ()))) {
	    PROP_ERROR (ret) ("deallocate_fid");
	    return ret;
	}

	// Delete/close FID
	if (type () == FT_LongSeg) {

	    if ((ret = dbfs._shared_fds.remove (free_fid))) {
		PROP_ERROR (ret) ("shared_fds_remove");
		return ret;
	    }

	} else {

	    if ((ret = dbfs._shared_dbs.remove (free_fid))) {
		PROP_ERROR (ret) ("shared_dbs_remove");
		return ret;
	    }
	}

	break;
    }

    default:

	break;
    }

    MINOR_DESC &desc = const_cast<MINOR_DESC&> (ref ());

    desc.set_deleted ();

    memset (& desc.rec, 0, sizeof (desc.rec));

    desc.rec.type = XTYPE_NOT_PRESENT;

    return 0;
}
