/* -*-Mode: C++;-*-
 * $Id: unsafe.cc 1.3 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"

#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/mman.h>

inline XPGNO PAGE_COUNT  (XSIZ size)   { return XPGNO (((size.key()-1)/DBFS_PAGE_SIZE)+1); }
inline XPGNO PAGE_OF     (XSIZ offset) { return XPGNO (offset.key ()/DBFS_PAGE_SIZE); }
inline guint PAGE_OFFSET (XSIZ offset) { return offset.key ()%DBFS_PAGE_SIZE; }

/**********************************************************************/
/*			  Close mechanism                             */
/**********************************************************************/

void
SHDB_DESC::reset (const XFID& fid)
{
    PARENT::reset (fid);

    g_assert (dbp == NULL);
}

void
SHFD_DESC::reset (const XFID& fid)
{
    PARENT::reset (fid);

    g_assert (fd == -1);
}

void
SHPG_DESC::reset (const XFIDPGNO& xpgno)
{
    PARENT::reset (xpgno);

    g_assert (mem == NULL);
    g_assert (onpage == 0);
}

void
SHDB_DESC::close_after (CLOSE_AFTER &ca, BASIC_DBENV &dbfs)
{
    ca.dbp = dbp;
    dbp = NULL;
}

void
SHPG_DESC::close_after (CLOSE_AFTER &ca, BASIC_DBENV &dbfs)
{
    ca.mem    = mem;
    ca.onpage = onpage;
    mem    = NULL;
    onpage = 0;

    SHFD_DESC *shfd = shfd_desc;

    shfd_desc = NULL;

    if (shfd) {
	// @@@ Does this need synchronization?
	shfd->pgs.erase (this);
    }
}

void
SHFD_DESC::close_after (CLOSE_AFTER &ca, BASIC_DBENV &dbfs)
{
    ca.fd = fd;
    fd = -1;

    for (SHPG_LIST::iterator piter = pgs.begin ();
	 piter != pgs.end ();
	 ++ piter) {

	SHPG_DESC *shpg = & (* piter);

	g_assert (shpg->shfd_desc == this);

	// Prevent recursion: SHPG_DESC will remove itself from
	// pgs otherwise
	shpg->shfd_desc = NULL;

	dbfs._shared_pgs.remove (shpg->key);
    }

    pgs.clear ();
}

int
SHFD_DESC::truncate (BASIC_DBENV &dbfs, const XSIZ& size)
{
    SHPG_LIST::iterator piter = pgs.begin ();

    while (piter != pgs.end ()) {
	SHPG_DESC *shpg = & (* piter);

	g_assert (shpg->shfd_desc == this);

	XPGNO new_pages (PAGE_COUNT (size));

	if (shpg->key.pgno >= new_pages) {
	    // Prevent recursion: SHPG_DESC will remove itself from
	    // pgs otherwise
	    shpg->shfd_desc = NULL;

	    g_assert (shpg->refs == 0);

	    dbfs._shared_pgs.remove (shpg->key);

	    pgs.erase (piter);
	} else {
	    ++piter;
	}
    }

    return 0;
}

int
CLOSE_AFTER::close ()
{
    int ret;

    if (fd >= 0) {

	if ((ret = ::close (fd))) {
	    ret = errno;
	    SYS_ERROR (ret) ("fd_close");
	    return ret;
	}
    }

    if (dbp != NULL) {

	if ((ret = dbp->close (0))) {
	    DB_ERROR (ret) ("db_close");
	    return ret;
	}
    }

    if (mem != NULL) {

	if ((ret = munmap ((caddr_t) mem, onpage) < 0)) {
	    ret = errno;
	    SYS_ERROR (ret) ("munmap");
	    return ret;
	}
    }

    return 0;
}

/**********************************************************************/
/*			  Unsafe handles                              */
/**********************************************************************/

class UNSAFEH
{
public:

    UNSAFEH (const NODEC &node)
	:_node (node)
    {
	g_assert (_node.xtype () == XTYPE_LONGSEG);
    }

    int sfd_get (SHFD_DESC*&sfd, MONITOR&sfd_mon, bool should_exist)
    {
	DBFS      &dbfs = _node.txn ().dbfs ();
	XFID       fid = _node.cont_id ().file_id ();
	int ret;

	if ((ret = dbfs._shared_fds.find_shared (fid, sfd, sfd_mon))) {
	    PROP_ERROR (ret) ("shared_fds_find");
	    return ret;
	}

	g_assert (should_exist ? (sfd->fd >= 0) : (sfd->fd < 0));

	return 0;
    }

    int create_fd ()
    {
	SHFD_DESC *sfd;
	MONITOR    sfd_mon;
	string     name;
	int ret;

	if ((ret = sfd_get (sfd, sfd_mon, false))) {
	    PROP_ERROR (ret) ("sfd_get");
	    return ret;
	}

	DBFS &dbfs = _node.txn ().dbfs ();
	XFID  fid  = _node.cont_id ().file_id ();

	dbfs.absolute_fname (fid, name);

	if ((sfd->fd = open (name.c_str (), O_RDWR | O_CREAT | O_TRUNC, 0666)) < 0) {
	    // @@ Shared-dset error handling: should this be dxn.abort ()
	    ret = errno;
	    SYS_ERROR (ret) ("unsafe_create_fd");
	    return ret;
	}

	return 0;
    }

    int truncate_fd (const XSIZ &size)
    {
	int        ret;
	SHFD_DESC *sfd;
	MONITOR    sfd_mon;
	DBFS      &dbfs = _node.txn ().dbfs ();

	if ((ret = sfd_get (sfd, sfd_mon, true))) {
	    PROP_ERROR (ret) ("sfd_get");
	    return ret;
	}

	if ((ret = sfd->truncate (dbfs, size))) {
	    PROP_ERROR (ret) ("sfd_truncate");
	    return ret;
	}

	if ((ret = ftruncate (sfd->fd, size.key ()))) {
	    ret = errno;
	    SYS_ERROR (ret) ("ftruncate");
	    return ret;
	}

	return 0;
    }

    int copy_fd (guint8 *buf,    // base of data to copy
		 XSIZ    nbytes, // size of the data to copy (EOF checked by caller--
		                 // SHFD_DESC has no length, which is odd)
		 XSIZ    offset, // offset for the copy
		 bool    iswrite)
    {
	int        ret;
	SHFD_DESC *sfd;
	MONITOR    sfd_mon;

	if ((ret = sfd_get (sfd, sfd_mon, true))) {
	    PROP_ERROR (ret) ("sfd_get");
	    return ret;
	}

	while (nbytes > 0) {

	    // Scatter/gather data into buffers
	    XPGNO   pgno     = PAGE_OF     (offset);
	    guint   page_off = PAGE_OFFSET (offset);
	    guint   page_rem = DBFS_PAGE_SIZE - page_off;
	    guint   page_cnt = min (page_rem, nbytes.key ());
	    guint8 *page;

	    if ((ret = mmap_pg (sfd, pgno, &page))) {
		PROP_ERROR (ret) ("mmap_pg");
		return ret;
	    }

	    if (iswrite) {
		// Write -- copy into page
		memcpy (page+page_off, buf, page_cnt);
	    } else {
		// Read -- copy into user buf
		memcpy (buf, page+page_off, page_cnt);
	    }

	    nbytes -= page_cnt;
	    offset += page_cnt;
	    buf    += page_cnt;
	}

	return 0;
    }

    int write_fd (const guint8 *buf,
		  const XSIZ   &nbytes,
		  const XSIZ   &offset)
    {
	return copy_fd ((guint8*) buf, nbytes, offset, true);
    }

    int read_fd (guint8     *buf,
		 const XSIZ &nbytes,
		 const XSIZ &offset)
    {
	return copy_fd (buf, nbytes, offset, false);
    }

    int mmap_pg (SHFD_DESC* sfd, const XPGNO &pgno, guint8 **mem)
    {
	int ret;
	DBFS      &dbfs = _node.txn ().dbfs ();
	SHPG_DESC *spg;
	MONITOR    spg_mon;
	XFIDPGNO   key (sfd->key, pgno);

	if ((ret = dbfs._shared_pgs.find_shared (key, spg, spg_mon))) {
	    PROP_ERROR (ret) ("find_shared_page");
	    return ret;
	}

	if (spg->mem == NULL) {

	    if ((spg->mem = (guint8*) mmap (NULL, DBFS_PAGE_SIZE, PROT_READ | PROT_WRITE,
					    MAP_PRIVATE, sfd->fd,
					    (pgno * DBFS_PAGE_SIZE).key ())) == MAP_FAILED) {
		ret = errno;
		SYS_ERROR (ret) ("mmap");
		return ret;
	    }

	    spg->onpage    = DBFS_PAGE_SIZE;
	    spg->shfd_desc = sfd;

	    sfd->pgs.push_front (spg);
	}

	spg->use ();

	(*mem) = spg->mem;

	return 0;
    }

private:

    NODEC _node;
};

/**********************************************************************/
/*			     Unsafe longs                             */
/**********************************************************************/

int
NODEC::unsafe_create (int flags) const
{
    int        ret;
    DXN        dxn (txn (), ret);
    XFID       fid;

    if ((ret = dxn.allocate_fid (fid))) {
	PROP_ERROR (ret) ("allocate_fid");
	return ret;
    }

    if ((ret = dxn.create_long (*this, fid, XSIZ (0), flags))) {
	PROP_ERROR (ret) (this) ("create_long");
	return ret;
    }

    UNSAFEH unsafeh (*this);

    if ((ret = unsafeh.create_fd ())) {
	PROP_ERROR (ret) (this) ("ucreate_fd");
	return ret;
    }

    return 0;
}

int
NODEC::unsafe_truncate (const XSIZ &size) const
{
    int ret;
    UNSAFEH unsafeh (*this);

    if (size != length ()) {
	MINOR_DESC &desc = const_cast<MINOR_DESC&> (ref ());

	desc.rec.length = size;

	desc.set_dirty ();
    }

    if ((ret = unsafeh.truncate_fd (size))) {
	PROP_ERROR (ret) ("utruncate_fd");
	return ret;
    }

    return 0;
}

int
NODEC::unsafe_write (const guint8 *buf,
		     const XSIZ   &nbytes,
		     const XSIZ   &offset) const
{
    int ret;
    UNSAFEH unsafeh (*this);
    XSIZ total_len = offset + nbytes;

    // Update length
    if (total_len > length ()) {
	MINOR_DESC &desc = const_cast<MINOR_DESC&> (ref ());

	desc.rec.length = total_len;

	desc.set_dirty ();
    }

    if ((ret = unsafeh.write_fd (buf, offset, nbytes))) {
	PROP_ERROR (ret) ("uwrite_fd");
	return ret;
    }

    return 0;
}

int
NODEC::unsafe_read (guint8       *buf,
		    const XSIZ   &xnbytes,
		    const XSIZ   &offset,
		    XSIZ         &nbytes_out) const
{
    int     ret;
    UNSAFEH unsafeh (*this);
    XSIZ    len = length ();

    if (offset >= len) {
	return 0;
    }

    if ((offset + xnbytes) > len) {
	nbytes_out = len - offset;
    } else {
	nbytes_out = xnbytes;
    }

    if ((ret = unsafeh.read_fd (buf, nbytes_out, offset))) {
	PROP_ERROR (ret) ("uread_fd");
	return ret;
    }

    return 0;
}

int
NODEC::unsafe_sync (void) const
{
    // @@@ Sync unimplemented
    return 0;
}
