/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

#include "mod_dav_xdelta.h"
#include "xdfs_cpp.h"

// Notes: mod_dav.c:751: No default response to GET for a collection

#define DEBUG_DAV  MK_DEBUG_MESSAGE (DAV)

MESSAGE_TYPE DEBUG_DAV_MESSAGE ("dav",  MCLASS_DEBUG);

static char *__dav_xdelta_argv[] = { "<mod_dav_xdelta>" };

class DAV_SERVER
{
public:
    DAV_SERVER ()
	: _fspath (NULL),
	  _dbfs   (1, __dav_xdelta_argv) { }

    const char *_fspath;
    DBFS        _dbfs;
};

struct dav_db {
    apr_pool_t *pool;
};

struct dav_resource_private
{
    const char *_pathname;
};

struct dav_xdelta_walker_ctx
{
    const dav_walk_params *params;

    // http://plan9.bell-labs.com/sys/man/5/INDEX.html
    dav_walk_resource    wres;

    dav_resource         res1;
    dav_resource_private info1;
    dav_buffer           path1;
    dav_buffer           uri1;
};

static dav_error* dav_xdelta_walker (dav_xdelta_walker_ctx *xctx, int depth);

// Note: enum below corresponds to the this array
static const char * const dav_xdelta_namespace_uris[] =
{
    "DAV:",
    "http://apache.org/dav/props/",

    NULL	/* sentinel */
};
enum {
    DAV_XDELTA_URI_DAV, /* the DAV: namespace URI */
};

static const dav_liveprop_spec dav_xdelta_props[] =
{
    {
        DAV_XDELTA_URI_DAV,
        "creationdate",
        DAV_PROPID_creationdate,
        0
    },
    {
        DAV_XDELTA_URI_DAV,
        "getcontentlength",
        DAV_PROPID_getcontentlength,
        0
    },
    {
        DAV_XDELTA_URI_DAV,
        "getetag",
        DAV_PROPID_getetag,
        0
    },
    {
        DAV_XDELTA_URI_DAV,
        "getlastmodified",
        DAV_PROPID_getlastmodified,
        0
    },

    { 0 }	/* sentinel */
};

/* define the dav_stream structure for our use */
struct dav_stream {
    apr_pool_t *p;
    apr_file_t *f;
    const char *pathname;	/* we may need to remove it at close time */
};

#if 0
/* Note: picked up from ap_gm_timestr_822() */
/* NOTE: buf must be at least DAV_TIMEBUF_SIZE chars in size */
static void dav_format_time_xdelta (int style, apr_time_t sec, char *buf)
{
    apr_exploded_time_t tms;

    /* ### what to do if fails? */
    (void) apr_explode_gmt(&tms, sec);

    if (style == DAV_STYLE_ISO8601) {
	/* ### should we use "-00:00" instead of "Z" ?? */

	/* 20 chars plus null term */
	sprintf(buf, "%.4d-%.2d-%.2dT%.2d:%.2d:%.2dZ",
               tms.tm_year + 1900, tms.tm_mon + 1, tms.tm_mday,
               tms.tm_hour, tms.tm_min, tms.tm_sec);
        return;
    }

    /* RFC 822 date format; as strftime '%a, %d %b %Y %T GMT' */

    /* 29 chars plus null term */
    sprintf(buf,
	    "%s, %.2d %s %d %.2d:%.2d:%.2d GMT",
           apr_day_snames[tms.tm_wday],
           tms.tm_mday, apr_month_snames[tms.tm_mon],
           tms.tm_year + 1900,
           tms.tm_hour, tms.tm_min, tms.tm_sec);
}
#endif

//////////////////////////////////////////////////////////////////////
// REPOSITORY HOOK FUNCTIONS
//////////////////////////////////////////////////////////////////////

void *dav_xdelta_create_server_config (apr_pool_t *p, server_rec *s)
{
    // @@ Not using apr_pool for allocation
    return new DAV_SERVER;
}

static dav_error*
dav_xdelta_server (const request_rec *r, DAV_SERVER **srvp)
{
    DAV_SERVER *srv = (DAV_SERVER*) ap_get_module_config(r->server->module_config, & dav_xdelta);
    int ret;

    if (srv == NULL || srv->_fspath == NULL) {
	return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			      "No Xdelta server found for this request, check your XdeltaFSPath directive");
    }

    (*srvp) = srv;

    if (! srv->_dbfs.is_open () && (ret = srv->_dbfs.open (srv->_fspath, NULL, DBFS_RECOVER_KILLPROC))) {
	return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, ret,
			      "Error opening Xdelta file system");
    }

    return NULL;
}

const char*
dav_xdelta_cmd_fspath (cmd_parms *cmd,
		       void *config,
		       const char *arg1)
{
    DAV_SERVER *conf = (DAV_SERVER*) ap_get_module_config (cmd->server->module_config, & dav_xdelta);

    arg1 = ap_os_canonical_filename (cmd->pool, arg1);

    conf->_fspath = arg1;

    return NULL;
}

static dav_error*
dav_xdelta_lookup (TXN                  &txn,
		   dav_resource_private *ctx,
		   MAJORC               &major)
{
    MAJORC tmp = txn.root ();

    // @@@ Do the lookup
    major = tmp;
    return NULL;
}

static dav_error*
dav_xdelta_begin (DAV_SERVER  *srv,
		  TXN         &txn,
		  request_rec *r)
{
    int ret;

    if ((ret = txn.begin (srv->_dbfs, DBFS_TXN_SYNC))) {
	return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, ret, "Error in txn_begin");
    }

    return NULL;
}

static dav_error*
dav_xdelta_commit (TXN        &txn,
		  request_rec *r)
{
    int ret;

    if ((ret = txn.commit ())) {
	return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, ret, "Error in txn_commit");
    }

    return NULL;
}

static dav_error*
dav_xdelta_cursor (MAJORC      &major,
		   DIRC        &dirc,
		   request_rec *r)
{
    int ret;

    if ((ret = major.dir_open (dirc))) {
	return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, ret, "Error in dir_open");
    }
    return NULL;
}

static dav_error*
dav_xdelta_get_resource(request_rec *r,
			const char *root_dir,
			const char *label,
			int use_checked_in,
			dav_resource **result_resource)
{
    dav_error  *err;
    DAV_SERVER *srv;
    TXN         txn;
    MAJORC      major;

    if ((err = dav_xdelta_server (r, & srv))) {
	return err;
    }

    DEBUG_DAV ("get_resource: root=%s req_path=%s uri=%s label=%s use_checked_in=%d",
	       root_dir, r->path_info, r->uri, label, use_checked_in);

    // Notes: I don't know how to handle root_dir, dav/fs removes
    // trailing slashes from r->uri, don't understand what
    // r->path_info means: looks fishy
    g_assert (strcmp (root_dir, "/") == 0);

    dav_resource_private *ctx = (dav_resource_private*) apr_pcalloc (r->pool, sizeof (*ctx));
    dav_resource         *res = (dav_resource*) apr_pcalloc (r->pool, sizeof (*res));

    ctx->_pathname = r->uri;

    res->type  = DAV_RESOURCE_TYPE_REGULAR;
    res->info  = ctx;
    res->hooks = & dav_hooks_repository_xdelta;
    res->pool  = r->pool;
    res->uri   = r->uri;

    if ((err = dav_xdelta_begin (srv, txn, r)) ||
	(err = dav_xdelta_lookup (txn, ctx, major))) {
	return err;
    }

    // Note: dav/fs does some checkinf of path_info here, related to locknulls
    res->exists     = major.is_present ();
    res->collection = major.is_container ();

    if ((err = dav_xdelta_commit (txn, r))) {
	return err;
    }

    (*result_resource) = res;

    return NULL;
}

static void
dav_xdelta_checkslash (apr_pool_t *pool, dav_buffer &buf)
{

    if (buf.cur_len > 0 && buf.buf[buf.cur_len-1] != '/') {

	dav_check_bufsize (pool, & buf, DAV_BUFFER_PAD);

	buf.buf[buf.cur_len++] = '/';
	buf.buf[buf.cur_len++] = '/';
    }
}

static dav_error*
dav_xdelta_walk (const dav_walk_params *params,
		 int                    depth,
		 dav_response         **response)
{
    dav_xdelta_walker_ctx xctx = { 0 };

    xctx.params        = params;
    xctx.wres.walk_ctx = params->walk_ctx;
    xctx.wres.pool     = params->pool;

    xctx.res1          = *params->root;
    xctx.res1.pool     = params->pool;
    xctx.res1.info     = & xctx.info1;

    /* Copy the CTX pathname into buf from root->info */
    dav_buffer_init (params->pool, & xctx.uri1,  params->root->uri);
    /* The current resource's URI is stored in the uri1 buffer */
    dav_buffer_init (params->pool, & xctx.path1, params->root->info->_pathname);

    xctx.info1           = *params->root->info;
    xctx.info1._pathname = xctx.path1.buf;

    xctx.res1.uri        = xctx.uri1.buf;

    /* dav/fs ensures the URI has a trailing "/" */
    if (xctx.res1.collection) {
	dav_xdelta_checkslash (params->pool, xctx.uri1);
    }

    /* Point the callback's resource at our structure */
    xctx.wres.resource = & xctx.res1;

    DEBUG_DAV ("dav_xdelta_walk: depth=%d", depth);

    /* always return the error, and any/all multistatus responses */
    dav_error *err = dav_xdelta_walker (&xctx, depth);
    *response = xctx.wres.response;
    return err;
}

static dav_error*
dav_xdelta_walker (dav_xdelta_walker_ctx *xctx, int depth)
{
    const dav_walk_params *params = xctx->params;
    apr_pool_t *pool = params->pool;
    dav_error  *err  = NULL;
    bool isdir = xctx->res1.collection;
    DIRC dirc;

    DEBUG_DAV ("dav_xdelta_walker: %s depth=%d", xctx->res1.uri, depth);

    /* ensure the context is prepared properly, then call the func */
    if ((err = (*params->func) (&xctx->wres,
				isdir
				? DAV_CALLTYPE_COLLECTION
				: DAV_CALLTYPE_MEMBER))) {
	return err;
    }

    if (depth == 0 || !isdir) {
	return NULL;
    }

    dav_xdelta_checkslash (pool, xctx->path1);

    /* for this first pass of files, all resources exist */
    xctx->res1.exists = 1;

    /* a file is the default; we'll adjust if we hit a directory */
    xctx->res1.collection = 0;

    if ((err = dav_xdelta_cursor (MAJOR, dirc))) {
	return err;
    }

#if 0
    /* open and scan the directory */
    if ((apr_dir_open(&dirp, xctx->path1.buf, pool)) != APR_SUCCESS) {
	/* ### need a better error */
	return dav_new_error(pool, HTTP_NOT_FOUND, 0, NULL);
    }
    while ((apr_dir_read(&dirent, APR_FINFO_DIRENT, dirp)) == APR_SUCCESS) {
	apr_size_t len;

	len = strlen(dirent.name);

	/* append this file onto the path buffer (copy null term) */
	dav_buffer_place_mem(pool, &xctx->path1, dirent.name, len + 1, 0);

        /* ### Optimize me, dirent can give us what we need! */
        if (apr_lstat(&xctx->info1.finfo, xctx->path1.buf,
                      APR_FINFO_NORM, pool) != APR_SUCCESS) {
	    /* woah! where'd it go? */
	    /* ### should have a better error here */
	    err = dav_new_error(pool, HTTP_NOT_FOUND, 0, NULL);
	    break;
	}

	/* copy the file to the URI, too. NOTE: we will pad an extra byte
	   for the trailing slash later. */
	dav_buffer_place_mem(pool, &xctx->uri_buf, dirent.name, len + 1, 1);

	/* set up the (internal) pathnames for the two resources */
	xctx->info1.pathname = xctx->path1.buf;
	xctx->info2.pathname = xctx->path2.buf;

	/* set up the URI for the current resource */
	xctx->res1.uri = xctx->uri_buf.buf;

	/* ### for now, only process regular files (e.g. skip symlinks) */
	if (xctx->info1.finfo.filetype == APR_REG) {
	    /* call the function for the specified dir + file */
	    if ((err = (*params->func)(&xctx->wres,
                                       DAV_CALLTYPE_MEMBER)) != NULL) {
		/* ### maybe add a higher-level description? */
		break;
	    }
	}
	else if (xctx->info1.finfo.filetype == APR_DIR) {
	    apr_size_t save_path_len = xctx->path1.cur_len;
	    apr_size_t save_uri_len = xctx->uri_buf.cur_len;
	    apr_size_t save_path2_len = xctx->path2.cur_len;

	    /* adjust length to incorporate the subdir name */
	    xctx->path1.cur_len += len;
	    xctx->path2.cur_len += len;

	    /* adjust URI length to incorporate subdir and a slash */
	    xctx->uri_buf.cur_len += len + 1;
	    xctx->uri_buf.buf[xctx->uri_buf.cur_len - 1] = '/';
	    xctx->uri_buf.buf[xctx->uri_buf.cur_len] = '\0';

	    /* switch over to a collection */
	    xctx->res1.collection = 1;
	    xctx->res2.collection = 1;

	    /* recurse on the subdir */
	    /* ### don't always want to quit on error from single child */
	    if ((err = dav_fs_walker(xctx, depth - 1)) != NULL) {
		/* ### maybe add a higher-level description? */
		break;
	    }

	    /* put the various information back */
	    xctx->path1.cur_len = save_path_len;
	    xctx->path2.cur_len = save_path2_len;
	    xctx->uri_buf.cur_len = save_uri_len;

	    xctx->res1.collection = 0;
	    xctx->res2.collection = 0;

	    /* assert: res1.exists == 1 */
	}
    }

    /* ### check the return value of this? */
    apr_dir_close(dirp);

    if (err != NULL)
	return err;

    if (params->walk_type & DAV_WALKTYPE_LOCKNULL) {
	apr_size_t offset = 0;

	/* null terminate the directory name */
	xctx->path1.buf[xctx->path1.cur_len - 1] = '\0';

	/* Include any lock null resources found in this collection */
	xctx->res1.collection = 1;
	if ((err = dav_fs_get_locknull_members(&xctx->res1,
                                               &xctx->locknull_buf)) != NULL) {
            /* ### maybe add a higher-level description? */
            return err;
	}

	/* put a slash back on the end of the directory */
	xctx->path1.buf[xctx->path1.cur_len - 1] = '/';

	/* these are all non-existant (files) */
	xctx->res1.exists = 0;
	xctx->res1.collection = 0;
	memset(&xctx->info1.finfo, 0, sizeof(xctx->info1.finfo));

	while (offset < xctx->locknull_buf.cur_len) {
	    apr_size_t len = strlen(xctx->locknull_buf.buf + offset);
	    dav_lock *locks = NULL;

	    /*
	    ** Append the locknull file to the paths and the URI. Note that
	    ** we don't have to pad the URI for a slash since a locknull
	    ** resource is not a collection.
	    */
	    dav_buffer_place_mem(pool, &xctx->path1,
				 xctx->locknull_buf.buf + offset, len + 1, 0);
	    dav_buffer_place_mem(pool, &xctx->uri_buf,
				 xctx->locknull_buf.buf + offset, len + 1, 0);
	    if (xctx->path2.buf != NULL) {
		dav_buffer_place_mem(pool, &xctx->path2,
				     xctx->locknull_buf.buf + offset,
                                     len + 1, 0);
	    }

	    /* set up the (internal) pathnames for the two resources */
	    xctx->info1.pathname = xctx->path1.buf;
	    xctx->info2.pathname = xctx->path2.buf;

	    /* set up the URI for the current resource */
	    xctx->res1.uri = xctx->uri_buf.buf;

	    /*
	    ** To prevent a PROPFIND showing an expired locknull
	    ** resource, query the lock database to force removal
	    ** of both the lock entry and .locknull, if necessary..
	    ** Sure, the query in PROPFIND would do this.. after
	    ** the locknull resource was already included in the
	    ** return.
	    **
	    ** NOTE: we assume the caller has opened the lock database
	    **       if they have provided DAV_WALKTYPE_LOCKNULL.
	    */
	    /* ### we should also look into opening it read-only and
	       ### eliding timed-out items from the walk, yet leaving
	       ### them in the locknull database until somebody opens
	       ### the thing writable.
	       */
	    /* ### probably ought to use has_locks. note the problem
	       ### mentioned above, though... we would traverse this as
	       ### a locknull, but then a PROPFIND would load the lock
	       ### info, causing a timeout and the locks would not be
	       ### reported. Therefore, a null resource would be returned
	       ### in the PROPFIND.
	       ###
	       ### alternative: just load unresolved locks. any direct
	       ### locks will be timed out (correct). any indirect will
	       ### not (correct; consider if a parent timed out -- the
	       ### timeout routines do not walk and remove indirects;
	       ### even the resolve func would probably fail when it
	       ### tried to find a timed-out direct lock).
	    */
	    if ((err = dav_lock_query(params->lockdb, &xctx->res1,
                                      &locks)) != NULL) {
		/* ### maybe add a higher-level description? */
		return err;
	    }

	    /* call the function for the specified dir + file */
	    if (locks != NULL &&
		(err = (*params->func)(&xctx->wres,
                                       DAV_CALLTYPE_LOCKNULL)) != NULL) {
		/* ### maybe add a higher-level description? */
		return err;
	    }

	    offset += len + 1;
	}

	/* reset the exists flag */
	xctx->res1.exists = 1;
    }

    if (params->walk_type & DAV_WALKTYPE_POSTFIX) {
	/* replace the dirs' trailing slashes with null terms */
	xctx->path1.buf[--xctx->path1.cur_len] = '\0';
	xctx->uri_buf.buf[--xctx->uri_buf.cur_len] = '\0';
	if (xctx->path2.buf != NULL) {
	    xctx->path2.buf[--xctx->path2.cur_len] = '\0';
	}

	/* this is a collection which exists */
	xctx->res1.collection = 1;

	return (*params->func)(&xctx->wres, DAV_CALLTYPE_POSTFIX);
    }
#endif

    return NULL;
}

static dav_error * dav_xdelta_get_parent_resource(const dav_resource *resource,
						  dav_resource **result_parent)
{
    DEBUG_DAV ("get_parent_resource");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static int dav_xdelta_is_same_resource (const dav_resource *res1,
					const dav_resource *res2)
{
    DEBUG_DAV ("is_same_resource");
    return 0;
}

static int dav_xdelta_is_parent_resource (const dav_resource *res1,
					  const dav_resource *res2)
{
    DEBUG_DAV ("is_parent_resource");
    return 0;
}

static dav_error * dav_xdelta_open_stream(const dav_resource *resource,
				      dav_stream_mode mode,
				      dav_stream **stream)
{
    DEBUG_DAV ("open_stream");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_close_stream(dav_stream *stream, int commit)
{
    DEBUG_DAV ("close_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_read_stream(dav_stream *stream,
				      void *buf, apr_size_t *bufsize)
{
    DEBUG_DAV ("read_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_write_stream(dav_stream *stream,
				       const void *buf, apr_size_t bufsize)
{
    DEBUG_DAV ("write_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_seek_stream(dav_stream *stream, apr_off_t abs_pos)
{
    DEBUG_DAV ("seek_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_set_headers(request_rec *r,
				      const dav_resource *resource)
{
    DEBUG_DAV ("set_headers");
    return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static const char * dav_xdelta_get_pathname(const dav_resource *resource,
					    void **free_handle_p)
{
    DEBUG_DAV ("get_pathname: %s", resource->info->_pathname);
    return resource->info->_pathname;
}

static void dav_xdelta_free_file(void *free_handle)
{
    /* nothing to free ... */
}

static dav_error * dav_xdelta_create_collection(dav_resource *resource)
{
    DEBUG_DAV ("create_collection");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_copy_resource(const dav_resource *src,
					    dav_resource *dst,
					    int depth,
					    dav_response **response)
{
    DEBUG_DAV ("copy_resource");
    return dav_new_error (src->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_move_resource(dav_resource *src,
					    dav_resource *dst,
					    dav_response **response)
{
    DEBUG_DAV ("move_resource");
    return dav_new_error (src->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_remove_resource(dav_resource *resource,
                                          dav_response **response)
{
    DEBUG_DAV ("remove_resource");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static const char *dav_xdelta_getetag(const dav_resource *resource)
{
    DEBUG_DAV ("get_etag");
    return "etag";
}

static dav_prop_insert
dav_xdelta_insert_prop(const dav_resource *resource,
		       int propid, dav_prop_insert what,
		       ap_text_header *phdr)
{
    DEBUG_DAV ("insert_prop");
    return what;
}

static int dav_xdelta_is_writable(const dav_resource *resource, int propid)
{
    DEBUG_DAV ("is_writeable");
    return 1;
}

static dav_error *dav_xdelta_patch_validate (const dav_resource *resource,
					     const ap_xml_elem *elem,
					     int operation,
					     void **context,
					     int *defer_to_dead)
{
    DEBUG_DAV ("patch_validate");
    // SVN: Don't need these until support for writable props
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error *dav_xdelta_patch_exec(const dav_resource *resource,
					const ap_xml_elem *elem,
					int operation,
					void *context,
					dav_liveprop_rollback **rollback_ctx)
{
    DEBUG_DAV ("patch_exec");
    // SVN: Don't need these until support for writable props
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static void dav_xdelta_patch_commit(const dav_resource *resource,
				    int operation,
				    void *context,
				    dav_liveprop_rollback *rollback_ctx)
{
    DEBUG_DAV ("patch_commit");
    // SVN: Don't need these until support for writable props
}

static dav_error *dav_xdelta_patch_rollback(const dav_resource *resource,
					int operation,
					void *context,
					dav_liveprop_rollback *rollback_ctx)
{
    DEBUG_DAV ("patch_rollback");
    // SVN: Don't need these until support for writable props
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

void dav_xdelta_gather_propsets(apr_array_header_t *uris)
{
    DEBUG_DAV ("gather_propsets");
}

int dav_xdelta_find_liveprop(const dav_resource *resource,
			     const char *ns_uri, const char *name,
			     const dav_hooks_liveprop **hooks)
{
    /* don't try to find any liveprops if this isn't "our" resource */
    if (resource->hooks != &dav_hooks_repository_xdelta) {
        return 0;
    }

    DEBUG_DAV ("find_liveprop: %s:%s", ns_uri, name);

    return dav_do_find_liveprop(ns_uri, name, &dav_xdelta_liveprop_group, hooks);
}

void dav_xdelta_insert_all_liveprops(request_rec *r, const dav_resource *resource,
				     dav_prop_insert what, ap_text_header *phdr)
{
    /* don't insert any liveprops if this isn't "our" resource */
    if (resource->hooks != &dav_hooks_repository_xdelta) {
        return;
    }

    DEBUG_DAV ("insert_all_liveprops");

    if (!resource->exists) {
	/* a lock-null resource */
	/*
	** ### technically, we should insert empty properties. dunno offhand
	** ### what part of the spec said this, but it was essentially thus:
	** ### "the properties should be defined, but may have no value".
	*/
	return;
    }

    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_creationdate,
				  what, phdr);
    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_getcontentlength,
				  what, phdr);
    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_getlastmodified,
				  what, phdr);
    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_getetag,
				  what, phdr);

    /* ### we know the others aren't defined as liveprops */
}

//////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////

static dav_error * dav_xdelta_propdb_open(apr_pool_t * p, const dav_resource *resource,
                                int ro, dav_db **pdb)
{
    DEBUG_DAV ("propdb_open");
    return dav_new_error (p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static void dav_xdelta_propdb_close(dav_db *db)
{
    DEBUG_DAV ("propdb_close");
}

static dav_error * dav_xdelta_propdb_fetch(dav_db *db, dav_datum key, dav_datum *pvalue)
{
    DEBUG_DAV ("propdb_fetch");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_propdb_store(dav_db *db, dav_datum key, dav_datum value)
{
    DEBUG_DAV ("propdb_store");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_propdb_delete(dav_db *db, dav_datum key)
{
    DEBUG_DAV ("propdb_delete");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static int dav_xdelta_propdb_exists(dav_db *db, dav_datum key)
{
    DEBUG_DAV ("propdb_exists");
    return 0;
}

static dav_error * dav_xdelta_propdb_firstkey(dav_db *db, dav_datum *pkey)
{
    DEBUG_DAV ("propdb_firstkey");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_propdb_nextkey(dav_db *db, dav_datum *pkey)
{
    DEBUG_DAV ("propdb_nextkey");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static void dav_xdelta_propdb_freedatum(dav_db *db, dav_datum data)
{
}

//////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////

const dav_hooks_db dav_hooks_db_xdelta =
{
    dav_xdelta_propdb_open,
    dav_xdelta_propdb_close,
    dav_xdelta_propdb_fetch,
    dav_xdelta_propdb_store,
    dav_xdelta_propdb_delete,
    dav_xdelta_propdb_exists,
    dav_xdelta_propdb_firstkey,
    dav_xdelta_propdb_nextkey,
    dav_xdelta_propdb_freedatum,
};

const dav_hooks_repository dav_hooks_repository_xdelta =
{
    1, // Require GET handling
    dav_xdelta_get_resource,
    dav_xdelta_get_parent_resource,
    dav_xdelta_is_same_resource,
    dav_xdelta_is_parent_resource,
    dav_xdelta_open_stream,
    dav_xdelta_close_stream,
    dav_xdelta_read_stream,
    dav_xdelta_write_stream,
    dav_xdelta_seek_stream,
    dav_xdelta_set_headers,
    dav_xdelta_get_pathname,
    dav_xdelta_free_file,
    dav_xdelta_create_collection,
    dav_xdelta_copy_resource,
    dav_xdelta_move_resource,
    dav_xdelta_remove_resource,
    dav_xdelta_walk,
    dav_xdelta_getetag,
};

const dav_hooks_liveprop dav_hooks_liveprop_xdelta =
{
    dav_xdelta_insert_prop,
    dav_xdelta_is_writable,
    dav_xdelta_namespace_uris,
    dav_xdelta_patch_validate,
    dav_xdelta_patch_exec,
    dav_xdelta_patch_commit,
    dav_xdelta_patch_rollback
};

const dav_liveprop_group dav_xdelta_liveprop_group =
{
    dav_xdelta_props,
    dav_xdelta_namespace_uris,
    & dav_hooks_liveprop_xdelta
};

const dav_provider dav_xdelta_provider =
{
    & dav_hooks_repository_xdelta,
    & dav_hooks_db_xdelta,
    NULL,               /* locks */
    NULL,               /* vsn */
    NULL                /* binding */
};
