/*
  3Com ADSL Modem USB extract utility
  Copyright (C) 2001 Josep Comas

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  Author     : Josep Comas <jcomas@gna.es>
  Creation   : 17/12/2001

  Description: This program extracts EZ-USB firmware from 3Com ADSL Modem USB Windows driver (3CF002LD.SYS)
*/

#include <stdio.h>
#include <stdlib.h>

//#define DEBUG  /* uncomment line to print debug info */

/* show printable char */
void print_char(unsigned char c)
{
  if (c >= ' ' && c < 0x7f)
    printf("%c", c);
  else
    printf(".");
}

/* show buffer */
void dump(unsigned char *buf, int lenbuf, int lenline)
{
  int i, j;  /* counters */

  for (i = 0; i < lenbuf; i+= lenline)
  {
    for (j = i; j < lenbuf && j < i + lenline; j++)
      printf("%02x ", buf[j]);
    for (; j < i + lenline; j++)
      printf("   ");
    for (j = i; j < lenbuf && j < i + lenline; j++)
      print_char(buf[j]);
    printf("\n");
  }
}

/* search buf_key bytes into buf */
unsigned char *search_bytes(unsigned char * buf, long len, unsigned char * buf_key, long len_key)
{
  long i, pos;  /* positions */
  int found;  /* found key buffer? */

  i = 0;
  pos = 0;
  found = 0;
  while (!found && i < len)
  {
    if (*(buf+i) == *(buf_key+pos))
    {
      pos++;
      if (pos == len_key)
        found = 1;
    }
    else
    {
      if (pos > 0)
      {
        i = i - pos;
        pos = 0;
      }
    }
    i++;
  }
  if (found)
    return buf+i-pos;
  else
    return NULL;
}

int main(int argc, char *argv[])
{
  const char * filename = NULL;  /* filename */
  const char * filesoft = "ez-usb.bin";  /* EZ-USB firmware filename */
  long len;  /* length */
  FILE * soft;  /* file handle */
  unsigned char *pbuf, *pbuf2, *pbuf3;  /* pointers to buffer */

  /* start and end bytes to search -> firmware */
  unsigned char startbytes[] = {0x04, 0x00, 0x20, 0x13, 0x00, 0xc1, 0x03, 0xc1, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  unsigned char endbytes[] = {0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  /* show program information */
  printf("3Com ADSL Modem USB extract utility (it extracts EZ-USB firmware)\n");
  printf("Josep Comas <jcomas@gna.es>. 17/12/2001\n\n");

  /* check parameters */
  if (argc <= 1)
  {
    printf("Usage: %s ez_usb_firmware_windows_driver_filename\n", argv[0]);
    printf("\nExample:\n");
    printf("   Extract EZ-USB firmware from Windows driver: %s 3cf002ld.sys\n", argv[0]);
    return -1;
  }

  /* read file into memory */
  filename = argv[1];
  soft = fopen(filename, "rb");
  if (soft == NULL)
  {
    printf("Error: I can't open file %s\n", filename);
    return -1;
  }
  fseek(soft, 0, SEEK_END);
  len = ftell(soft);
  if (len <= 10)
  {
    printf("Error getting size from file %s\n", filename);
    return -1;
  }
  fseek(soft, 0, SEEK_SET);
  pbuf = malloc(len);
  if (pbuf == NULL)
  {
    printf("Error allocating buffer\n");
    return -1;
  }
  if (fread(pbuf, 1, len, soft) <= 0)
  {
    printf("Error reading file %s\n", filename);
    return -1;
  }
  fclose(soft);

  /* search firmware */
  printf("Searching firmware...\n");
  pbuf2 = search_bytes(pbuf, len, startbytes, sizeof(startbytes));
  if (pbuf2 == NULL)
  {
    printf("Error: firmware not found (initial bytes)\n");
    return -1;
  }
  pbuf3 = search_bytes(pbuf2, len-(pbuf2-pbuf), endbytes, sizeof(endbytes));
  if (pbuf3 == NULL)
  {
    printf("Error: firmware not found (end bytes)\n");
    return -1;
  }
  len = pbuf3-pbuf2+sizeof(endbytes);
#ifdef DEBUG
  printf("Bytes to write:\n");
  dump(pbuf2, len, 22);
#endif
  printf("Firmware found!\n");

  /* write firmware to file */
  soft = fopen(filesoft, "rb");
  if (soft != NULL)
  {
    printf("Error: file %s exists\n", filesoft);
    return -1;
  }
  soft = fopen(filesoft, "wb");
  if (soft == NULL)
  {
    printf("Error: file %s can't be created\n", filesoft);
    return -1;
  }
  if (fwrite(pbuf2, 1, len, soft) != len)
  {
    printf("Error writting file %s\n", filesoft);
    return -1;
  }
  fclose(soft);
  printf("Firmware extracted successfully!\n");

  free(pbuf);
  return 0;
}


