/*
  3Com ADSL Modem USB init process
  Copyright (C) 2002 Josep Comas

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  Author     : Josep Comas <jcomas@gna.es>
  Creation   : 14/9/2002

  Description: This program inits 3Com ADSL Modem USB.

  Log:

  5/6/2003 Josep Comas
  Updated to work with libusb-0.1.7

  18/6/2003 Josep Comas
  Changed resetep instead of clear_halt endpoint

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <usb.h>
#include "usbi.h"
#include <libintl.h>
#include <locale.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

/* uncomment following line to print debug info or define in compiler parameters: */
/* #define DEBUG 1 */
/* uncomment following line to print debug transfers or define in compiler parameters: */
/* #define DEBUG_TRANSFER 1 */

#ifdef DEBUG
#define PDEBUG(arg...)  printf(arg)
#else
#define PDEBUG(arg...)
#endif

/* modem usb identifications */
#define ID_VENDOR    0x0506  /* Vendor = 3Com */
#define ID_PRODUCT1  0xF002  /* Product = ADSL modem without EZ-USB firmware loaded */
#define ID_PRODUCT1b 0xF100  /* Product = ADSL modem without EZ-USB firmware loaded */
#define ID_PRODUCT2  0xF003  /* Product = ADSL modem with EZ-USB firmware loaded */

/* EZ-USB codes */
#define CPU_CONTROL_REGISTER 0x7F92  /* 0x7F92 = 1 -> stop 8051 processor, 0 -> start 8051 processor */
#define FIRMWARE_LOAD 0xA0           /* Special code to download firmware */
#define VENDOR_REQUEST_OUT 0x40      /* Vendor specific requests, OUT direction */
#define VENDOR_REQUEST_IN 0xC0       /* Vendor specific requets, IN direction */

/* usb input and ouput address against modem */
#define USB_IN_INFO  0x81    /* IN endpoint address, receive info */
#define USB_OUT_INFO 0x01    /* OUT endpoint address, ask info */
#define USB_IN_DATA  0x82    /* IN endpoint address, receive ATM cells */
#define USB_OUT_DATA 0x02    /* OUT endpoint address, send ATM cells
 (for each ATM cell you must send: 53 bytes cell + 11 bytes set to 0x00 = 64 bytes) */

/* timeouts & retries */
#define DATA_TIMEOUT 5000    /* Timeout (ms) for in/out data packets */ 
#define CTRL_TIMEOUT 1000    /* Timeout (ms) for in/out control packets */
#define TIMEOUT_ADD  1000    /* We get timeout then add additional time */
#define SEND_BULK_RETRIES 4  /* Max retries when you send a bulk packet */
#define READ_BULK_RETRIES 4  /* Max retries when you wait a bulk packet */
#define CTRL_MSG_RETRIES  4  /* Max retries when you transfer a control message */
#define MAX_WAIT_LINE_UP 90  /* Seconds to wait until ADSL line is up */

/* usb errors */
#define EPIPE      32        /* When we receive a STALL */
#define ETIMEDOUT 110        /* When we receive a NAK */

/* buffers sizes */
#define MAX_INPUT_BUF 128    /* Maximum bytes that we can read */
#define MAX_INIT_FIRM 2250   /* Maximum size of Alcatel init firmware */
#define MAX_BLOCK_CODE 8000  /* Maximum bytes that we can send for each block of code (Alcatel firmware) */

/* leds */
#define LED_USB_ON   0x08    /* bit 4 */
#define LED_ADSL_ON  0x10    /* bit 5 */
#define LED_DATA_ON  0x20    /* bit 6 */

/* led values that I found:
0x26 = 00100110
0x06 = 00000110 USB led off  (bit 4)
0x0e = 00001110 USB led on   (bit 4)
0x0b = 00001011 ADSL led off (bit 5)
0x1b = 00011011 ADSL led on  (bit 5)
0x2b = 00101011 DATA led on  (bit 6)
0xac = 10101100 Maybe when modem is connected
0xab = 10101011 Send disconnect command?
*/

/* identified codes (bulk transfers in info endpoint):
 To know if DSP (Alcatel processor) is booted send: 0x1c + 0xbf (you receive 0x1c + 0xbf + status, status = 0 not booted, status = 1 booted)
 To receive leds state send: 0x78 + 0x26  You receive: 0x78 + 0x26 + leds state
 To change leds state send: 0x78 + 0x26 + leds state
 If a USB packet begins with 0x78 is a register of EZ-USB, look at datasheet chip 
 If a USB packet begins in range 0x79 + 0x00 until 0x79 + 7f we are programming GPIF interface of EZ-USB
 If a USB packet begins with 0x7f is a register of EZ-USB, look at datasheet chip 
 If a USB packet begins with 0x80 or 0x81 is a CTRLE protocol sequence
 If a USB packet begins with 0xa1 is firmware for Alcatel chip
*/

/* addresses */
#define DSP_Booted   0x1cbf
#define LED_REGISTER 0x7826
#define ADSL_OpenCommand 0x1cb3
#define Firmware_Major_Version 0x1cba
#define Firmware_Minor_Version 0x1cbb

/* eeprom */
#define I2CS  0x7fa5
#define I2DAT 0x7fa6
#define I2CS_START 0x80
#define I2CS_STOP  0x40
#define I2CS_LASTREAD 0x20
#define I2CS_DONE  0x01
#define I2CS_ACK   0x02
#define I2CS_WRITE_ADDRESS 0xa0
#define I2CS_READ_ADDRESS 0xa1


/******************/
/* CTRLE protocol */
/******************/

#define TX_COMMAND 0x8000
#define RX_COMMAND 0x8001
#define TX_COM_AV  0x8002
#define RX_COM_AV  0x8003
#define TX_COM_BUSY 0x1
#define RX_COM_BUSY 0x1
#define MODEM_STATUS 0x8100

/* status of modem (MODEM_STATUS -> 0x8100) */
#define MODEM_DOWN  0x00
#define MODEM_WAIT  0x08
#define MODEM_INIT  0x10
#define MODEM_UP    0x20

/* responses from modem a receive command (RX_COMMAND -> 0x8001) */
#define ADSL_Operational_Mode_Ack  0x12
#define ADSL_Connection_Opened     0x24
#define ADSL_Connection_Closed     0x25
#define ADSL_Connection_Lost       0x41
#define Modem_Passed_Init_Routine  0x80


/* Commands accepted by CTRLE */

#define ADSL_Open            0x03
#define ADSL_Close           0x04

/* open mode */
#define ADSL_Open_ANSI       0x05
#define ADSL_Open_GDMT       0x06
#define ADSL_Open_GLITE      0x07
#define ADSL_Open_MULTIMODE  0x08

#define MODEM_JUMP_ADDRESS   0x14

/* end commands */

/* Alcatel firmware version */
#define SW_Version_LSB          0x81fe
#define SW_Version_MSB          0x81ff

/* channel down bitrate */
#define R_ChanDataIntNear_LSB   0x81a0
#define R_ChanDataIntNear_MSB   0x81a1
#define R_ChanDataFastNear_LSB  0x81a2
#define R_ChanDataFastNear_MSB  0x81a3

/* channel up bitrate */
#define R_ChanDataIntFar_LSB    0x81a4
#define R_ChanDataIntFar_MSB    0x81a5
#define R_ChanDataFastFar_LSB   0x81a6
#define R_ChanDataFastFar_MSB   0x81a7

/* end CTRLE protocol */


/* constants to blink led (speed in us) */
#define FAST_BLINK 100000
#define SLOW_BLINK 600000

/* files and columns sizes */
#define NFIL_INIT_EZUSB 26
#define NCOL_INIT_EZUSB 3
#define NFIL_MODEM_JUMP 5
#define NCOL_MODEM_JUMP 3
#define NFIL_INIT_GPIF 143
#define NCOL_INIT_GPIF 3
#define NFIL_CONNECT1 5
#define NCOL_CONNECT1 3
#define NFIL_CONNECT2 4
#define NCOL_CONNECT2 3

/* translation files */
#define TF_CODE "3cload"

/* initialize, EZ-USB chip */
char modem_init_ezusb[NFIL_INIT_EZUSB][NCOL_INIT_EZUSB] = {
  /* init ports */
  {0x7f, 0x98, 0x01},
  {0x7f, 0x9e, 0x01},
  {0x7f, 0x96, 0x00},
  {0x7f, 0x9c, 0x2e},
  {0x7f, 0x93, 0x0c},
  {0x7f, 0x96, 0x25},
  /* init FIFOs */
  {0x7f, 0x96, 0x27},
  {0x78, 0x5e, 0x07},
  {0x78, 0x57, 0x00},
  {0x78, 0x4a, 0x07},
  {0x78, 0x4b, 0x00},
  {0x78, 0x1c, 0x00},
  {0x78, 0x1d, 0x00},
  {0x78, 0x1e, 0x00},
  {0x78, 0x16, 0xb5},
  {0x78, 0x02, 0xb4},
  {0x78, 0x03, 0xb4},
  {0x78, 0x1a, 0x3f},
  {0x78, 0x19, 0x00},
  {0x78, 0x0c, 0x3f},
  {0x78, 0x0b, 0x00},
  {0x78, 0x4a, 0x06},
  {0x78, 0x3c, 0x07},
  {0x78, 0x3b, 0x00},
  {0x78, 0x39, 0x00},
  {0x7f, 0x98, 0x01}
};

/* this indicates that jump to 0 address in Alcatel code, CTRLE protocol */
char modem_jump[NFIL_MODEM_JUMP][NCOL_MODEM_JUMP] = {
  {0x80, 0x08, 0x00},  /* BLOCK_BASE_ADDRESS_1 = 0x00 */
  {0x80, 0x09, 0x00},  /* BLOCK_BASE_ADDRESS_2 = 0x00 */
  {0x80, 0x0a, 0x00},  /* BLOCK_BASE_ADDRESS_3 = 0x00 */
  {0x80, 0x0b, 0x00},  /* BLOCK_BASE_ADDRESS_4 = 0x00 */
  {0x80, 0x00, 0x14}   /* MODEM_JUMP_ADDRESS (TX_COMMAND (0x8000) + CTRLE COMMAND (0x14 = JUMP)) */
};

/* programming GPIF interface, EZ-USB chip */
char modem_init_gpif[NFIL_INIT_GPIF][NCOL_INIT_GPIF] = {
  {0x78, 0x26, 0x0b},
  {0x78, 0x27, 0x00},
  {0x78, 0x15, 0x01},
  {0x78, 0x16, 0x01},
  {0x78, 0x02, 0x35},
  {0x78, 0x38, 0x40},
  {0x78, 0x1e, 0x00},
  {0x78, 0x39, 0x00},
  {0x79, 0x00, 0x01},
  {0x79, 0x01, 0x11},
  {0x79, 0x02, 0x01},
  {0x79, 0x03, 0x16},
  {0x79, 0x04, 0x1f},
  {0x79, 0x05, 0x27},
  {0x79, 0x06, 0x02},
  {0x79, 0x07, 0xe0},
  {0x79, 0x08, 0x01},
  {0x79, 0x09, 0x01},
  {0x79, 0x0a, 0x02},
  {0x79, 0x0b, 0x01},
  {0x79, 0x0c, 0x01},
  {0x79, 0x0d, 0x03},
  {0x79, 0x0e, 0x00},
  {0x79, 0x0f, 0x70},
  {0x79, 0x10, 0x23},
  {0x79, 0x11, 0x22},
  {0x79, 0x12, 0x22},
  {0x79, 0x13, 0x22},
  {0x79, 0x14, 0x22},
  {0x79, 0x15, 0x22},
  {0x79, 0x16, 0x23},
  {0x79, 0x17, 0x9c},
  {0x79, 0x18, 0x12},
  {0x79, 0x19, 0x11},
  {0x79, 0x1a, 0x00},
  {0x79, 0x1b, 0x00},
  {0x79, 0x1c, 0x00},
  {0x79, 0x1d, 0x00},
  {0x79, 0x1e, 0x00},
  {0x79, 0x1f, 0x98},
  {0x79, 0x20, 0x01},
  {0x79, 0x21, 0x02},
  {0x79, 0x22, 0x33},
  {0x79, 0x23, 0x12},
  {0x79, 0x24, 0x36},
  {0x79, 0x25, 0x00},
  {0x79, 0x26, 0x01},
  {0x79, 0x27, 0xe0},
  {0x79, 0x28, 0x01},
  {0x79, 0x29, 0x02},
  {0x79, 0x2a, 0x07},
  {0x79, 0x2b, 0x03},
  {0x79, 0x2c, 0x01},
  {0x79, 0x2d, 0x00},
  {0x79, 0x2e, 0x06},
  {0x79, 0x2f, 0xf1},
  {0x79, 0x30, 0x23},
  {0x79, 0x31, 0x25},
  {0x79, 0x32, 0x21},
  {0x79, 0x33, 0x21},
  {0x79, 0x34, 0x23},
  {0x79, 0x35, 0x00},
  {0x79, 0x36, 0x21},
  {0x79, 0x37, 0x71},
  {0x79, 0x38, 0x12},
  {0x79, 0x39, 0x00},
  {0x79, 0x3a, 0x36},
  {0x79, 0x3b, 0x36},
  {0x79, 0x3c, 0x36},
  {0x79, 0x3d, 0x00},
  {0x79, 0x3e, 0x00},
  {0x79, 0x3f, 0xe1},
  {0x79, 0x40, 0x01},
  {0x79, 0x41, 0x02},
  {0x79, 0x42, 0x03},
  {0x79, 0x43, 0x04},
  {0x79, 0x44, 0x05},
  {0x79, 0x45, 0x06},
  {0x79, 0x46, 0x07},
  {0x79, 0x47, 0x11},
  {0x79, 0x48, 0x00},
  {0x79, 0x49, 0x00},
  {0x79, 0x4a, 0x00},
  {0x79, 0x4b, 0x00},
  {0x79, 0x4c, 0x00},
  {0x79, 0x4d, 0x00},
  {0x79, 0x4e, 0x00},
  {0x79, 0x4f, 0x7c},
  {0x79, 0x50, 0x3f},
  {0x79, 0x51, 0x20},
  {0x79, 0x52, 0x3f},
  {0x79, 0x53, 0x20},
  {0x79, 0x54, 0x3f},
  {0x79, 0x55, 0x20},
  {0x79, 0x56, 0x3f},
  {0x79, 0x57, 0x64},
  {0x79, 0x58, 0x00},
  {0x79, 0x59, 0x00},
  {0x79, 0x5a, 0x00},
  {0x79, 0x5b, 0x00},
  {0x79, 0x5c, 0x00},
  {0x79, 0x5d, 0x00},
  {0x79, 0x5e, 0x00},
  {0x79, 0x5f, 0xe0},
  {0x79, 0x60, 0x01},
  {0x79, 0x61, 0x02},
  {0x79, 0x62, 0x03},
  {0x79, 0x63, 0x04},
  {0x79, 0x64, 0x05},
  {0x79, 0x65, 0x06},
  {0x79, 0x66, 0x07},
  {0x79, 0x67, 0xfa},
  {0x79, 0x68, 0x00},
  {0x79, 0x69, 0x00},
  {0x79, 0x6a, 0x00},
  {0x79, 0x6b, 0x00},
  {0x79, 0x6c, 0x00},
  {0x79, 0x6d, 0x00},
  {0x79, 0x6e, 0x00},
  {0x79, 0x6f, 0x50},
  {0x79, 0x70, 0x3f},
  {0x79, 0x71, 0x20},
  {0x79, 0x72, 0x3f},
  {0x79, 0x73, 0x20},
  {0x79, 0x74, 0x3f},
  {0x79, 0x75, 0x20},
  {0x79, 0x76, 0x3f},
  {0x79, 0x77, 0xf4},
  {0x79, 0x78, 0x00},
  {0x79, 0x79, 0x00},
  {0x79, 0x7a, 0x00},
  {0x79, 0x7b, 0x00},
  {0x79, 0x7c, 0x00},
  {0x79, 0x7d, 0x00},
  {0x79, 0x7e, 0x00},
  {0x79, 0x7f, 0xb0},
  {0x78, 0x39, 0x00},
  {0x81, 0xfd, 0x0d},  /* set AFE value, R_Function_Code = 0x0d (adjust Alcatel DSP for our configuration) */
  {0x80, 0x4a, 0x01},
  {0x80, 0x4b, 0x00},
  {0x80, 0x4c, 0x0f},
  {0x7f, 0xae, 0x1f},  /* Enable USB interrupts (view EZ-USB datasheet) */
  {0x7f, 0xab, 0x02}   /* Clear SOF IRQ (view EZ-USB datasheet) */
};

/* ask connect again */
char modem_connect1[NFIL_CONNECT1][NCOL_CONNECT1] = {
  {0x1c, 0xbd, 0x00},  /* ADDRESS_ADSL_LineUp (maybe it indicates to ez-usb that line is up or set up) */
  {0x81, 0xfd, 0x0d},  /* set AFE value, R_Function_Code = 0x0d (adjust Alcatel DSP for our configuration) */
  {0x80, 0x4a, 0x01},
  {0x80, 0x4b, 0x00},
  {0x80, 0x4c, 0x0f}   
};
char modem_connect2[NFIL_CONNECT2][NCOL_CONNECT2] = {
  {0x1c, 0xbd, 0x00},  /* ADDRESS_ADSL_LineUP (maybe it indicates to ez-usb that line is up or set up) */
  {0x1c, 0xbc, 0x00},  /* ADDRESS_Modem_State = startup */
  {0x7f, 0xae, 0x1f},  /* Enable USB interrupts (view EZ-USB datasheet) */
  {0x7f, 0xab, 0x02}   /* Clear SOF IRQ (view EZ-USB datasheet) */
};

/* info about ADSL modem */
struct adsl_info {
   unsigned char rx_cmd;  /* received command */
   unsigned char adsl_status;  /* status of ADSL line (up, down, ...) */
   char sw_ver_lsb;  /* low version of Alcatel firmware */
   char sw_ver_msb;  /* high version of Alcatel firmware */
   unsigned int downstream_bit_rate;  /* max downstream speed of channel, ADSL line */
   unsigned int upstream_bit_rate;    /* max upstream speed of channel, ADSL line */
};

/* check if a file exists */
int file_exists(const char *filename)
{
  struct stat info_file;

  return stat(filename, &info_file) == 0;
}

/* show printable char */
void print_char(unsigned char c)
{
  if (c >= ' ' && c < 0x7f)
    printf("%c", c);
  else
    printf(".");
}

/* show buffer */
void dump(unsigned char *buf, int lenbuf, int lenline)
{
  int i, j;  /* counters */

  for (i = 0; i < lenbuf; i+= lenline)
  {
    for (j = i; j < lenbuf && j < i + lenline; j++)
      printf("%02x ", buf[j]);
    for (; j < i + lenline; j++)
      printf("   ");
    for (j = i; j < lenbuf && j < i + lenline; j++)
      print_char(buf[j]);
    printf("\n");
  }
}

/* transfer a control message to USB bus */
int transfer_ctrl_msg(usb_dev_handle *adsl_handle, int requesttype, int request, int value, int index, char *buf, int size)
{
  int j;  /* counter */
  int n;  /* bytes transfed or error code */
  int tmout = CTRL_TIMEOUT;  /* timeout value */

  n = 0;
  for (j = 0; j < CTRL_MSG_RETRIES; j++) {
    n = usb_control_msg(adsl_handle, requesttype, request, value, index,  buf, size, tmout);
    if (n >= 0) {
#if DEBUG_TRANSFER
      printf(gettext("%d bytes transferred:\n"), n);
      dump(buf, n, 16);
#endif
      break;
    }
    else {
      printf(gettext("Error: usb_control_msg: %s\n"), usb_strerror());
      if (n == -EPIPE) {
        usb_clear_halt(adsl_handle, 0x00);
        usb_clear_halt(adsl_handle, 0x80);
      }
      else if (n == -ETIMEDOUT) {
	tmout += TIMEOUT_ADD;
      }
    }
  }
  if (n < 0) {
    printf(gettext("Error: usb_control_msg failed after %d retries\n"), CTRL_MSG_RETRIES);
    return -1;
  }
  return n;
}

/* clear endpoints that we use */
void clear_endpoints(usb_dev_handle *adsl_handle, int step) {
  usb_clear_halt(adsl_handle, 0x00);
  usb_clear_halt(adsl_handle, 0x80);
  if (step >= 2) {
    usb_resetep(adsl_handle, USB_IN_INFO);
    usb_resetep(adsl_handle, USB_OUT_INFO);
    usb_resetep(adsl_handle, USB_IN_DATA);
    usb_resetep(adsl_handle, USB_OUT_DATA);
/*
    usb_clear_halt(adsl_handle, USB_IN_INFO);
    usb_clear_halt(adsl_handle, USB_OUT_INFO);
    usb_clear_halt(adsl_handle, USB_IN_DATA);
    usb_clear_halt(adsl_handle, USB_OUT_DATA);
*/
  }
}

/* set 8051 processor state (pos = 1 -> stop, 0 -> start) */
int set_ez_usb(usb_dev_handle *adsl_handle, char pos)
{
  int r;  /* return code */

  if (pos != 0 && pos != 1)
  {
    printf(gettext("Error setting EZ-USB CPU, incorrect value: %x\n"), pos);
    return -1;
  }
  if (pos != 0)
  {
    r = transfer_ctrl_msg(adsl_handle, VENDOR_REQUEST_OUT, FIRMWARE_LOAD, CPU_CONTROL_REGISTER, 0, &pos, 1);
    if (r != 1)
    {
      printf(gettext("Error setting EZ-USB CPU in state %x\n"), pos);
      return -1;
    }
  }
  else
    r = usb_control_msg(adsl_handle, VENDOR_REQUEST_OUT, FIRMWARE_LOAD, CPU_CONTROL_REGISTER, 0, &pos, 1, CTRL_TIMEOUT);
    /* Ignore error code when setting cpu on, it's renum process */
  return 0;
}

/* stop 8051 processor */
int reset_ez_usb(usb_dev_handle *adsl_handle)
{
  return set_ez_usb(adsl_handle, 1);
}

/* start 8051 processor */
int unreset_ez_usb(usb_dev_handle *adsl_handle)
{
  return set_ez_usb(adsl_handle, 0);
}

/* show EZ-USB internal memory program/data (0x0000-0x1B3F) */
int view_ez_usb(usb_dev_handle *adsl_handle)
{
  int r;  /* return code */
  unsigned char buf[16];  /* buffer */
  unsigned int i;  /* counter */

  for (i = 0; i <= 0x1B3F; i += 16)
  {
    r = transfer_ctrl_msg(adsl_handle, VENDOR_REQUEST_IN, FIRMWARE_LOAD, i, 0, buf, 16);
    if (r != 16)
    {
      printf(gettext("Error reading from address 0x%04x\n"), i);
      return -1;
    }
    printf(gettext("Address 0x%04x: "), i);
    dump(buf, 16, 16);
  }
  return 0;
}

/* load EZ-USB firmware */
int load_ez_usb(usb_dev_handle *adsl_handle, const char *filename)
{
  FILE *soft;  /* file handle */
  unsigned char buf[10000];  /* buffer */
  int checkOK;       /* boolean */
  int r;             /* result code */
  unsigned int i;    /* counter */
  unsigned int len;  /* length */
  unsigned lines;    /* lines of code to send */
  unsigned total;    /* bytes uploaded */
  unsigned target;   /* target address of code */


  /* clear endpoints */
  clear_endpoints(adsl_handle, 1);

  printf(gettext("Loading and sending %s...\n"), filename); 
  soft = fopen(filename, "rb");
  if (soft == NULL)
  {
    printf(gettext("Error: I can't open file %s\n"), filename);
    return -1;
  }
  len = fread(buf, 1, sizeof(buf), soft);
  if (len <= 0)
  {
    printf(gettext("Error: No bytes to read from file %s\n"), filename);
    return -1;
  }
  fclose(soft);
  if (len % 22 != 0) {
    printf(gettext("Error: Incorrect size of file %s\n"), filename);
    return -1;
  }
  i = 0; 
  checkOK = 1;
  lines = (len / 22) - 1;
  while (i < lines && checkOK)
  {
#if DEBUG
    printf(gettext("Verifying line %03d: "), i + 1);
    dump(buf + (i * 22), 22, 22);
#endif
    if (buf[(i * 22)] == 0 || buf[(i * 22)] > 16)
      checkOK = 0;
    else
      if (buf[(i * 22) + 1] != 0)
        checkOK = 0;
      else
      {
        target = (buf[(i * 22) + 3] << 8) + buf[(i * 22) + 2];
#if DEBUG
        printf(gettext("Target address: 0x%04x\n"), target);
        printf(gettext("Bytes to send: 0x%02x\n"), buf[i*22]);
#endif
        if (target > 0x1B3F)
          checkOK = 0;
        else
          if (buf[(i* 22) + 4] != 0)
            checkOK = 0;
          else
            i++;
      }
  }
  if (checkOK)
    if (buf[(i * 22)] != 0)
      checkOK = 0;
  if (!checkOK) {
    printf(gettext("Error: one or more lines are incorrect in file %s\n"), filename);
    return -1;
  }
  PDEBUG(gettext("Lines to send: %d\n"), lines);
#ifdef DEBUG
  printf(gettext("\nEZ-USB internal memory before firmware load:\n"));
  view_ez_usb(adsl_handle);
#endif
  if (reset_ez_usb(adsl_handle))
    return -1;
  PDEBUG(gettext("\nTransfering firmware...\n"));
  total = 0;
  for(i = 0; i < lines; i++)
  {
    target = (buf[(i * 22) + 3] << 8) + buf[(i * 22) + 2];
    r = transfer_ctrl_msg(adsl_handle, VENDOR_REQUEST_OUT, FIRMWARE_LOAD, target, 0, buf + (i * 22) + 5, buf[i * 22]);
    if (r != buf[i * 22])
    {
      printf(gettext("Error writing at address 0x%04x\n"), target);
      return -1;
    }
    total += buf[i*22];
  }
#ifdef DEBUG
  printf(gettext("\nEZ-USB internal memory after firmware load:\n"));
  view_ez_usb(adsl_handle);
#endif
  if (unreset_ez_usb(adsl_handle))
    return -1;
  printf(gettext("%d bytes uploaded to EZ-USB internal memory\n"), total);
  return 0;
}

/* send one or more packets to USB bus by bulk transfer */
int send_bulk(usb_dev_handle *adsl_handle, int ep, char *buf, int nfil, int ncol)
{
  int i, j;  /* counters */
  int n;  /* bytes sent or error code */
  int tmout = DATA_TIMEOUT;  /* timeout value */

  n = 0;
  for (i = 0; i < nfil; i++)
  {
    for (j = 0; j < SEND_BULK_RETRIES; j++) {
      n = usb_bulk_write(adsl_handle, ep, buf+(i*ncol), ncol, tmout);
      if (n >= 0) {
#if DEBUG_TRANSFER
        printf(gettext("%d bytes uploaded:\n"), n);
        dump(buf+(i*ncol), ncol, 16);
#endif
        break;
      }
      else {
        printf(gettext("Error: usb_bulk_write: %s\n"), usb_strerror());
	if (n == -EPIPE) {
	  usb_clear_halt(adsl_handle, ep);
	}
	else if (n == -ETIMEDOUT) {
	  tmout += TIMEOUT_ADD;
	}
      }
    }
    if (n < 0) {
      printf(gettext("Error: usb_bulk_write failed after %d retries\n"), SEND_BULK_RETRIES);
      return -1;
    }
  }
  return 0;
}

/* receive a packet from USB bus by bulk transfer */
int read_bulk(usb_dev_handle *adsl_handle, int ep, char *buf, int size)
{
  int n;  /* bytes readed or error code */
  int i;  /* counter */
  int tmout = DATA_TIMEOUT;  /* timeout value */

  memset(buf, 0, sizeof(buf));
  n = 0;
  for (i = 0; i < READ_BULK_RETRIES; i++) {
    n = usb_bulk_read(adsl_handle, ep, buf, size, tmout);
    if (n >= 0) {
#if DEBUG_TRANSFER
      printf(gettext("%d bytes downloaded:\n"), n);
      dump(buf, n, 16);
#endif
      break;
    }
    else {
      printf(gettext("Error: usb_bulk_read: %s\n"), usb_strerror());
      if (n == -EPIPE) {
        usb_clear_halt(adsl_handle, ep);
      }
      else if (n == -ETIMEDOUT) {
	tmout += TIMEOUT_ADD;
      }
    }
  }
  if (n < 0) {
    printf(gettext("Error: usb_bulk_read failed after %d retries\n"), READ_BULK_RETRIES);
    return -1;
  }
  return 0;
}

/* write 8051 status byte */
int write_status_byte(usb_dev_handle *adsl_handle, unsigned int address, char value)
{
  char bufout[3];

  bufout[0] = (char) (address >> 8);
  bufout[1] = (char) address;
  bufout[2] = value;
  if (send_bulk(adsl_handle, USB_OUT_INFO, bufout, 1, 3))
    return -1;
  return 0;
}

/* read 8051 status byte */
int read_status_byte(usb_dev_handle *adsl_handle, unsigned int address, char *value)
{
  char bufout[2];
  char bufin[3];

  bufout[0] = (char) (address >> 8);
  bufout[1] = (char) address;
  if (send_bulk(adsl_handle, USB_OUT_INFO, bufout, 1, 2))
    return -1;
  if (read_bulk(adsl_handle, USB_IN_INFO, (char *) &bufin, sizeof(bufin)))
    return -1;
  if (bufin[0] != bufout[0] || bufin[1] != bufout[1]) {
    printf(gettext("Error: bytes inexpected\n"));
    return -1;
  }
  *value = bufin[2];
  return 0;
}

/* read a word from 8051 */
int read_status_word(usb_dev_handle *adsl_handle, unsigned int addressMSB, unsigned int addressLSB, short *value) {
  unsigned char bufMSB, bufLSB;

  if (read_status_byte(adsl_handle, addressMSB, &bufMSB))
    return -1;
  if (read_status_byte(adsl_handle, addressLSB, &bufLSB))
    return -1;
  if (value == NULL)
    return -1;
  else {
    if ((bufMSB == 0xff) || (bufLSB == 0xff))
      *value = 0xffff;
    else
      *value = ((bufMSB << 8) | bufLSB) & 0xffff;
  }

  return 0;
}

/* change state of leds */
int set_led_state(usb_dev_handle *adsl_handle, char value)
{
  char current;  /* current state of leds */

  if (read_status_byte(adsl_handle, LED_REGISTER, &current))
    return -1;
  current &= ~(LED_USB_ON | LED_ADSL_ON | LED_DATA_ON);
  current |= value;
  if (write_status_byte(adsl_handle, LED_REGISTER, current))
    return -1;

  return 0;
}

/* invert a led */
int invert_led(usb_dev_handle *adsl_handle, char mask)
{
  char current;  /* current state of leds */
  
  if (read_status_byte(adsl_handle, LED_REGISTER, &current))
    return -1;
  if ((current & mask) == 0)
  {
    if (write_status_byte(adsl_handle, LED_REGISTER, current | mask))
      return -1;
  }
  else
  {
    if (write_status_byte(adsl_handle, LED_REGISTER, current & ~mask))
      return -1;
  }

  return 0;
}


/* wait 8051 data bit on */
int wait_data_bit_on(usb_dev_handle *adsl_handle, int address, char mask, int maxtime)
{
  char value;

  while (maxtime > 0) 
  {
    if (read_status_byte(adsl_handle, address, &value))
      return -1;
    if ((value & mask) == mask)
      return 0;
    else
    {
      usleep(50);
      maxtime -= 1;
    } 
  }
  return -1;
}

/* wait 8051 data bit off */
int wait_data_bit_off(usb_dev_handle *adsl_handle, int address, char mask, int maxtime)
{
  char value;

  while (maxtime > 0)
  {
    if (read_status_byte(adsl_handle, address, &value))
      return -1;
    if ((value & mask) == 0)
      return 0;
    else
    { 
      usleep(50);
      maxtime -= 50;
    }
  }
  return -1;
}

/* prepare code block to send */
unsigned long fill_buffer(unsigned char *buf, unsigned char *buf2, long len, unsigned long place)
{
  long i;    /* counter */
  unsigned long target;  /* target address */

  for(i = 0; i < MAX_BLOCK_CODE; i+=64)
  {
    buf[i] = 0xa1;
    target = (i / 64) * 18;
    if (target+18 <= len)
    {
      buf[i+1] = 0x08;
      buf[i+2] = 0x80;
      buf[i+3] = place + target;
      buf[i+4] = (place + target) >> 8;
      buf[i+5] = (place + target) >> 16;
      buf[i+6] = (place + target) >> 24;
      buf[i+7] = 0x12;
      memcpy(buf+i+9, buf2+target, 18);
    }
    else
      if (target < len)
      {
        buf[i+1] = 0x08;
        buf[i+2] = 0x80;
        buf[i+3] = place + target;
        buf[i+4] = (place + target) >> 8;
        buf[i+5] = (place + target) >> 16;
        buf[i+6] = (place + target) >> 24;
        buf[i+7] = len-target;
        memcpy(buf+i+9, buf2+target, len-target);
      }
  }
  return (place + target + 18);
}

/* read alcatel firmware version */
int read_alc_firm_ver(usb_dev_handle *adsl_handle, struct adsl_info *adapter)
{
  unsigned char status;  /* response code */

  if (read_status_byte(adsl_handle, SW_Version_LSB, &status))
    return -1;
  adapter->sw_ver_lsb = status;
  if (read_status_byte(adsl_handle, SW_Version_MSB, &status))
    return -1;
  adapter->sw_ver_msb = status;
  return 0;
}

/* wait response from eeprom */
int wait_eeprom(usb_dev_handle *adsl_handle, unsigned char command)
{
  int i = 0;  /* counter */
  unsigned char status;  /* response code */

  do
  {
    i++;
    if (read_status_byte(adsl_handle, I2CS, &status))
      return -1;
  }
  while ((!(status & command)) && (i < 4));
  if (i != 4)
    return 0;
  else
    return -1;
}

/* read eeprom info */
int read_eeprom(usb_dev_handle *adsl_handle)
{
  long i;     /* counter */
  unsigned char status;  /* response code */

  /* send commands to receive info */
  if (write_status_byte(adsl_handle, I2CS, I2CS_START))
    return -1;
  if (write_status_byte(adsl_handle, I2DAT, I2CS_WRITE_ADDRESS))
    return -1;
  if (wait_eeprom(adsl_handle, I2CS_ACK))
    return -1;
  if (write_status_byte(adsl_handle, I2DAT, 0x10))
    return -1;
  if (wait_eeprom(adsl_handle, I2CS_ACK))
    return -1;
  if (write_status_byte(adsl_handle, I2CS, I2CS_STOP))
    return -1;
  if (write_status_byte(adsl_handle, I2CS, I2CS_START))
    return -1;
  if (write_status_byte(adsl_handle, I2DAT, I2CS_READ_ADDRESS))
    return -1;
  if (wait_eeprom(adsl_handle, I2CS_ACK))
    return -1;

  /* skip first byte, is garbage */
  if (read_status_byte(adsl_handle, I2DAT, &status))
    return -1;

  /* read mac address */
  printf(gettext("MAC address = "));
  for (i = 16; i <= 21; i++)
  {
    if (read_status_byte(adsl_handle, I2DAT, &status))
      return -1;
    printf("%02x", status);
    if (i != 21)
      printf(".");
  }
  printf("\n");
 
  /* read until arrive to serial number */
  for (i = 22; i <= 31; i++)
  {
    if (read_status_byte(adsl_handle, I2DAT, &status))
      return -1;
  }

  /* read serial number */
  printf(gettext("Serial number = "));
  for (i = 32; i <= 43; i++)
  {
    if (read_status_byte(adsl_handle, I2DAT, &status))
      return -1;
    print_char(status);
  }
  printf("\n");

  /* read until arrive to hardware revision */
  for (i = 44; i <= 47; i++)
  {
    if (read_status_byte(adsl_handle, I2DAT, &status))
      return -1;
  }
   
  /* read hardware revision */
  printf(gettext("Hardware version = "));
  for (i = 48; i <= 48; i++) 
  {
    if (read_status_byte(adsl_handle, I2DAT, &status))
      return -1;
    if (status != 0xff)
      printf("%d", (int) status);
    else
      printf("-");
  }
  printf(":");

  /* read until arrive to date code */
  for (i = 49; i <= 63; i++)
  {
    if (read_status_byte(adsl_handle, I2DAT, &status))
      return -1;
  }
 
  /* read date code */
  for (i = 64; i <= 67; i++) 
  {
    if (read_status_byte(adsl_handle, I2DAT, &status))
      return -1;
    if (status != 0xff)
      printf("%02x", status);
    else
      printf("--");
  }
  printf(":");

  /* read firmware major version */
  if (read_status_byte(adsl_handle, Firmware_Major_Version, &status))
    return -1;
  printf("%d", (int) status);
  printf(".");
 
  /* read firmware minor version */
  if (read_status_byte(adsl_handle, Firmware_Minor_Version, &status))
    return -1;
  printf("%d", (int) status);
  printf("\n");

  /* send end commands */
  if (write_status_byte(adsl_handle, I2CS, I2CS_LASTREAD))
    return -1;
  if (wait_eeprom(adsl_handle, I2CS_ACK))
    return -1;
  if (read_status_byte(adsl_handle, I2CS, &status))
    return -1;
  if (wait_eeprom(adsl_handle, I2CS_DONE))
    return -1;
  if (read_status_byte(adsl_handle, I2CS, &status))
    return -1;

  return 0;
}

/* read up and down channel bitrate */
int read_bitrate(usb_dev_handle *adsl_handle, struct adsl_info *adapter)
{
  unsigned short statusWord;

  adapter->upstream_bit_rate = 0;
  adapter->downstream_bit_rate = 0;

  if (read_status_word(adsl_handle, R_ChanDataIntNear_MSB, R_ChanDataIntNear_LSB, &statusWord))
    return -1;
  if ((statusWord != 0) && (statusWord != 0xffff)) {
    adapter->downstream_bit_rate = statusWord * 1024;
  }

  if (read_status_word(adsl_handle, R_ChanDataFastNear_MSB, R_ChanDataFastNear_LSB, &statusWord))
    return -1;
  if ((statusWord != 0) && (statusWord != 0xffff)) {
    adapter->downstream_bit_rate = statusWord * 1024;
  }

  if (read_status_word(adsl_handle, R_ChanDataIntFar_MSB, R_ChanDataIntFar_LSB, &statusWord))
    return -1;
  if ((statusWord != 0) && (statusWord != 0xffff)) {
    adapter->upstream_bit_rate = statusWord * 1024;
  }

  if (read_status_word(adsl_handle, R_ChanDataFastFar_MSB, R_ChanDataFastFar_LSB, &statusWord))
    return -1;
  if ((statusWord != 0) && (statusWord != 0xffff)) {
    adapter->upstream_bit_rate = statusWord * 1024;
  }

  return 0;
}

/* show ADSL line status */
int show_adsl_status(usb_dev_handle *adsl_handle, struct adsl_info *adapter)
{
  if (adapter->adsl_status == MODEM_UP && (adapter->rx_cmd == ADSL_Operational_Mode_Ack || adapter->rx_cmd == ADSL_Connection_Opened)) {
    /* set ADSL led on */
    if (set_led_state(adsl_handle, LED_USB_ON | LED_ADSL_ON))
      return -1;
    if (read_bitrate(adsl_handle, adapter))
      return -1;
    printf(gettext("ADSL line is up (Downstream %u bits/s, Upstream %u bits/s)\n"), adapter->downstream_bit_rate, adapter->upstream_bit_rate);
  }
  else {
    /* set ADSL led off */
    if (set_led_state(adsl_handle, LED_USB_ON))
      return -1;
    printf(gettext("ADSL line is down\n"));
  }
  /*
  switch (adapter->adsl_status) {
    case MODEM_DOWN: printf("ADSL line is down\n"); break;
    case MODEM_WAIT: printf("ADSL line is blocked?\n"); break;
    case MODEM_INIT: printf("ADSL line is synchronising\n"); break;
    case MODEM_UP: printf("ADSL line is up (Downstream %u bits/s, Upstream %u bits/s)\n", adapter->downstream_bit_rate, adapter->upstream_bit_rate); break;
    default: printf("ADSL line status is unknown\n"); break;
  }
  */
  return 0;
}

/* read info provided by Alcatel processor */
int query_adsl(usb_dev_handle *adsl_handle, struct adsl_info *adapter, struct timeval *lastblink)
{
  int diff_blink;
  struct timeval tv;
  struct timezone tz;

  adapter->rx_cmd = 0xff; adapter->adsl_status = 0xff;

  if (read_status_byte(adsl_handle, RX_COMMAND, &adapter->rx_cmd))
    return -1;
  if (read_status_byte(adsl_handle, MODEM_STATUS, &adapter->adsl_status))
    return -1;

  if (adapter->adsl_status == MODEM_UP && adapter->rx_cmd == ADSL_Connection_Opened) {
    if (write_status_byte(adsl_handle, TX_COMMAND, 0x31))  /* update ops mode */
      return -1;
    if (set_led_state(adsl_handle, LED_USB_ON | LED_ADSL_ON))
      return -1;
    if (read_bitrate(adsl_handle, adapter))
      return -1;
  }
  else {
    if (adapter->adsl_status == MODEM_INIT || adapter->adsl_status == MODEM_WAIT) {
      if (adapter->adsl_status == MODEM_INIT)
        diff_blink = FAST_BLINK;
      else
        diff_blink = SLOW_BLINK;
      gettimeofday(&tv, &tz);
      if (((tv.tv_sec - lastblink->tv_sec) > 1) || (abs(tv.tv_usec - lastblink->tv_usec) > diff_blink)) {
	lastblink->tv_sec = tv.tv_sec;
        lastblink->tv_usec = tv.tv_usec;
        if (invert_led(adsl_handle, LED_ADSL_ON))
          return -1;
      }
    }
  }

  return 0;
}

/* load Alcatel firmware */
int load_alcatel(usb_dev_handle *adsl_handle, const char *filename1, const char *filename2, int open_mode)
{
  unsigned char buf[MAX_BLOCK_CODE];  /* block of code that we send */
  unsigned char buf2[MAX_INIT_FIRM];  /* buffer to put code */
  unsigned char bufin[MAX_INPUT_BUF]; /* input buffer, read bulk transfers */
  FILE * soft;  /* file handle */
  long i;       /* counter */
  long len;     /* length */
  unsigned long place;   /* initial target address */
#if DEBUG
  int n;      /* counter */
#endif
  struct adsl_info adapter;  /* info about ADSL modem */
  time_t first, last, before;  /* to wait */
  struct timeval tv;  /* for blink */
  struct timezone tz;  /* for blink */


  /* clear endpoints */
  clear_endpoints(adsl_handle, 2);

  /* check DSP status, is booted (firmware downloaded)? */
  if (read_status_byte(adsl_handle, DSP_Booted, &bufin[0]))
    return -1;
  if (bufin[0] == 1) {
    printf(gettext("Alcatel firmware has already been downloaded. No need to do it again!\n"));
    return 0;
  }

  /* turn off all leds and activate USB led */
  if (set_led_state(adsl_handle, LED_USB_ON))
    return -1;

  /* initialize */
  if (send_bulk(adsl_handle, USB_OUT_INFO, (char *) &modem_init_ezusb, NFIL_INIT_EZUSB, NCOL_INIT_EZUSB))
    return -1;

  /* activate USB led */
  if (set_led_state(adsl_handle, LED_USB_ON))
    return -1;

  /* send init Alcatel firmware */
  printf(gettext("Loading and sending %s...\n"), filename1); 
  memset(buf, 0, sizeof(buf));
  soft = fopen(filename1, "rb");
  if (soft == NULL)
  {
    printf(gettext("Error: I can't open file %s\n"), filename1);
    return -1;
  }
  fseek(soft, 0, SEEK_END);
  len = ftell(soft);
  PDEBUG(gettext("Length of file %s = %ld bytes\n"), filename1, len);
  if (len > MAX_INIT_FIRM)
  {
    printf(gettext("Error: file %s exceeds buffer size\n"), filename1);
    return -1;
  }
  fseek(soft, 0, SEEK_SET);
  len = fread(buf2, 1, sizeof(buf2), soft);
  PDEBUG(gettext("%ld bytes readed from file %s\n"), len, filename1);
  if (len <= 0)
  {
    printf(gettext("Error: No bytes to read from file %s\n"), filename1);
    return -1;
  }
  fclose(soft);
  /* check initial bytes */
  PDEBUG(gettext("Initial bytes from file %s:\n"), filename1);
#if DEBUG
  dump(buf2, 5, 5);
#endif
  if (buf2[0] != 0x04 || buf2[1] != 0xa0 || buf2[2] != 0x00 || buf2[3] != 0x47 || buf2[4] != 0x02) {
    printf(gettext("Error: Maybe file %s isn't Alcatel init firmware, contact with author of this program\n"), filename1);
    return -1;
  }
  fill_buffer(buf, buf2, len, 0);
  if (send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, sizeof(buf)))
    return -1;

  /* checking initialization */
  if (send_bulk(adsl_handle, USB_OUT_INFO, (char *) &modem_jump, NFIL_MODEM_JUMP, NCOL_MODEM_JUMP))
    return -1;
  if (wait_data_bit_off(adsl_handle, TX_COM_AV, TX_COM_BUSY, 1000))
    return -1;
  if (wait_data_bit_on(adsl_handle, RX_COM_AV, RX_COM_BUSY, 1000))
    return -1;
  if (read_status_byte(adsl_handle, RX_COMMAND, &bufin[0]))
    return -1;
  if (bufin[0] != Modem_Passed_Init_Routine) {
    printf(gettext("Error: init Alcatel routine not passed!\n"));
    return -1;
  }
  printf(gettext("Init Alcatel routine passed\n"));

  /* send main Alcatel firmware */
  printf(gettext("Loading and sending %s...\n"), filename2); 
  soft = fopen(filename2, "rb");
  if (soft == NULL)
  {
    printf(gettext("Error: I can't open file %s\n"), filename2);
    return -1;
  }
  fseek(soft, 0L, SEEK_END);
  len = ftell(soft);
  PDEBUG(gettext("Length of file %s = %ld bytes\n"), filename2, len);
  fseek(soft, 0L, SEEK_SET);
  len = fread(buf2, 1, 5, soft);
  if (len <= 0)
  {
    printf(gettext("Error: No bytes to read from file %s\n"), filename2);
    return -1;
  }
  if (len != 5)
  {
    printf(gettext("Error: I can't read initial 5 bytes from file %s\n"), filename2);
    return -1;
  }
  /* check initial bytes */
  PDEBUG(gettext("Initial bytes from file %s:\n"), filename2);
#if DEBUG
  dump(buf2, 5, 5);
#endif
  if (buf2[0] != 0x11 || buf2[1] != 0xa0 || buf2[2] != 0x00 || buf2[3] != 0x47 || buf2[4] != 0x0f) {
    printf(gettext("Error: Maybe file %s isn't Alcatel main firmware, contact with author of this program\n"), filename2);
    return -1;
  }
  fseek(soft, 0L, SEEK_SET);
  place = 0;
  i = 0; 
  while ((len = fread(buf2, 1, sizeof(buf2), soft)) > 0)
  {
    PDEBUG(gettext("%ld bytes readed from file %s\n"), len, filename2);
    memset(buf, 0, sizeof(buf));
    place = fill_buffer(buf, buf2, len, place);
    if (send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, sizeof(buf)))
      return -1;
    if ((i % 4) == 0)
      invert_led(adsl_handle, LED_USB_ON);
    i++;
  }
  fclose(soft);
  if (set_led_state(adsl_handle, LED_USB_ON))
    return -1;
  if (send_bulk(adsl_handle, USB_OUT_INFO, (char *) &modem_jump, NFIL_MODEM_JUMP, NCOL_MODEM_JUMP))
    return -1;
  if (wait_data_bit_off(adsl_handle, TX_COM_AV, TX_COM_BUSY, 1000))
    return -1;
  if (send_bulk(adsl_handle, USB_OUT_INFO, (char *) &modem_init_gpif, NFIL_INIT_GPIF, NCOL_INIT_GPIF))
    return -1;
  if (write_status_byte(adsl_handle, ADSL_OpenCommand, open_mode))
    return -1;
  if (write_status_byte(adsl_handle, TX_COMMAND, open_mode))
    return -1;

  /* indicate that DSP is booted */
  if (write_status_byte(adsl_handle, DSP_Booted, 1))
    return -1;

  /* turn on USB led */
  if (set_led_state(adsl_handle, LED_USB_ON))
    return -1;

  memset(&adapter, 0, sizeof(adapter));

  /* show firmware version */
  if (read_alc_firm_ver(adsl_handle, &adapter))
    return -1;
  printf(gettext("Alcatel firmware version = %d.%d.%d\n"), (int) adapter.sw_ver_msb >> 4, (int) adapter.sw_ver_msb &0xf, (int) adapter.sw_ver_lsb);

  /* show eeprom info */
  if (read_eeprom(adsl_handle))
    return -1;

  /* waiting until line is up (a maximum time) */
  printf(gettext("Waiting ADSL line is up (until %d seconds)...\n"), MAX_WAIT_LINE_UP);
  time(&first); before = first;
  gettimeofday(&tv, &tz);
  do {
    if (query_adsl(adsl_handle, &adapter, &tv))
      return -1;
    PDEBUG(gettext("Command = %02x\n"), adapter.rx_cmd);
    PDEBUG(gettext("Status = %02x\n"), adapter.adsl_status);
    if (difftime(time(&last), before) > 1) {
#ifndef DEBUG
      printf(".");
      fflush(stdout);
#endif
      before = last;
    }
  }
  while (!(adapter.adsl_status == MODEM_UP && adapter.rx_cmd == ADSL_Connection_Opened) && (difftime(last, first) < MAX_WAIT_LINE_UP));
  printf("\n");

  /* show info about ADSL */
  if (show_adsl_status(adsl_handle, &adapter))
    return -1;

  /* clear endpoints */
  clear_endpoints(adsl_handle, 2);

#if DEBUG
  if (adapter.adsl_status == MODEM_UP) {
    /* waiting until we receive first ATM cells */
    printf(gettext("Waiting to receive first ATM cells...\n"));
    while (1) {
      n = usb_bulk_read(adsl_handle, USB_IN_DATA, bufin, sizeof(bufin), DATA_TIMEOUT);
      if (n > 0)  {
        printf(gettext("ATM cells received:\n"));
        dump(bufin, n, 16);
        return 0;
      }
    }
  } 
#endif

  return 0;
}

/* print time */
void print_time(void)
{
  time_t tp;  /* struct time */
  char strtime[80];  /* string with time */

  time(&tp);
  strcpy(strtime, "");
  strncat(strtime, asctime(localtime((const time_t *) &tp)), strlen(asctime(localtime((const time_t *) &tp)))-1);
  printf("[ %s ] ", strtime);
}

int connect_again(usb_dev_handle *adsl_handle, int open_mode)
{
  /* put ADSL led off */
  if (set_led_state(adsl_handle, LED_USB_ON))
    return -1;

  /* try again to connect */
  if (send_bulk(adsl_handle, USB_OUT_INFO, (char *) &modem_connect1, NFIL_CONNECT1, NCOL_CONNECT1))
    return -1;
  if (write_status_byte(adsl_handle, TX_COMMAND, open_mode))
    return -1;
  if (send_bulk(adsl_handle, USB_OUT_INFO, (char *) &modem_connect2, NFIL_CONNECT2, NCOL_CONNECT2))
    return -1;

  return 0;
}

/* monitorize ADSL line */
int monitorize(usb_dev_handle *adsl_handle, int open_mode)
{
  long j;  /* counter */
  unsigned char lastcmd;      /* last received command */
  unsigned char laststatus;   /* last ADSL line status */
  unsigned char bufin[MAX_INPUT_BUF];  /* buffer to receive data */
  int k;   /* slow or fast blink */
  struct adsl_info adapter;   /* info about ADSL modem */
  struct timeval tv;  /* for blink */
  struct timezone tz;  /* for blink */

  lastcmd = 0xff; laststatus = MODEM_UP;


  /* clear endpoints */
  clear_endpoints(adsl_handle, 3);

  /* check if Alcatel firmware is loaded */
  if (read_status_byte(adsl_handle, DSP_Booted, &bufin[0]))
    return -1;
  if (bufin[0] != 1) {
    printf(gettext("Error: Alcatel firmware not downloaded!\n"));
    return 0;
  } 


  /* monitorize line, activate/desactivate ADSL led */
  k = SLOW_BLINK; j = 0;
  gettimeofday(&tv, &tz);
  if (query_adsl(adsl_handle, &adapter, &tv))
    return -1;
  PDEBUG(gettext("Command = %02x\n"), adapter.rx_cmd);
  PDEBUG(gettext("Status = %02x\n"), adapter.adsl_status);
  if (!((adapter.adsl_status == MODEM_DOWN  && adapter.rx_cmd == ADSL_Connection_Closed) ||
      (adapter.adsl_status == MODEM_UP && adapter.rx_cmd == ADSL_Connection_Opened) ||
      (adapter.adsl_status == MODEM_UP && adapter.rx_cmd == ADSL_Connection_Lost)))
  {
    print_time();
    if (show_adsl_status(adsl_handle, &adapter))
      return -1;
  }
  while (1) {
    if (lastcmd != adapter.rx_cmd || laststatus != adapter.adsl_status) {
      switch (adapter.adsl_status) {
        case MODEM_DOWN:
          if (laststatus == MODEM_UP && adapter.rx_cmd == ADSL_Connection_Closed) {
            print_time();
            printf(gettext("Connection closed\n"));
            if (connect_again(adsl_handle, open_mode))
              return -1;
          }
          break;
        case MODEM_UP:
          if (adapter.rx_cmd == ADSL_Connection_Opened) {
            print_time();
            if (show_adsl_status(adsl_handle, &adapter))
              return -1;
            /* put ADSL led on */
            if (set_led_state(adsl_handle, LED_USB_ON | LED_ADSL_ON))
              return -1;
          }
          else if (laststatus == MODEM_UP && adapter.rx_cmd == ADSL_Connection_Lost) {
            print_time();
            printf(gettext("Connection lost\n"));
            if (connect_again(adsl_handle, open_mode))
              return -1;
          }
          break;
      }
      lastcmd = adapter.rx_cmd;
      laststatus = adapter.adsl_status;
    }
    if (query_adsl(adsl_handle, &adapter, &tv))
      return -1;
  }

  return 0;
}


int main(int argc, char *argv[])
{
  /* filenames */
  const char * filename1, * filename2 = NULL;

  /* bus structures variables */
  struct usb_bus * bus;
  struct usb_device * dev;
  struct usb_device * adsl_dev = NULL;
  usb_dev_handle * adsl_handle;

  /* boolean value */
  int goon;

  /* result code */
  int r; 

  /* open mode */
  int open_mode = ADSL_Open_ANSI;

  /* init locale */
  setlocale(LC_ALL, "");
  //if (file_exists("./locale")) 
  //  bindtextdomain(TF_CODE, "./locale");  /* set directory for a domain (source code messages) */
  //else 
    bindtextdomain(TF_CODE, "/usr/share/locale");  /* set directory for a domain (source code messages) */
  textdomain(TF_CODE);  /* set domain */

  /* show program information */
  printf(gettext("3Com ADSL Modem USB microcode upload program\n"));
  printf("Josep Comas <jcomas@gna.es>. 18/6/2003\n\n");

  /* check parameters */
  if ((argc == 2 && strcmp(argv[1], "3")) || argc < 2)
  {
    printf(gettext("Usage: %s step parameters\n"), argv[0]);
    printf(gettext("\nSteps and parameters:\n"));
    printf(gettext("  1 ez_usb_firmware_filename\n"));
    printf(gettext("  2 alcatel_init_firmware_filename alcatel_main_firmware_filename [open_mode]\n"));
    printf(gettext("  3 [open_mode]\n"));
    printf(gettext("\nOpen mode (optional):\n"));
    printf(gettext("  5 = ANSI T1.413 (default)\n"));
    printf(gettext("  7 = ITU-T G.992.2 (G.LITE)\n"));
    printf(gettext("  6 = ITU-T G.992.1 (G.DMT)\n"));
    printf(gettext("  8 = multimode\n"));
    printf(gettext("  3 = other\n"));
    printf(gettext("\nExamples:\n"));
    printf(gettext("   Load EZ-USB firmware: %s 1 ez-usb.bin\n"), argv[0]);
    printf(gettext("   Load Alcatel firmware: %s 2 3cinit.bin 3cmain.bin\n"), argv[0]);
    printf(gettext("   Monitorize ADSL modem: %s 3\n"), argv[0]);
    return -1;
  }
  if (strcmp(argv[1], "1")  && strcmp(argv[1], "2") && strcmp(argv[1], "3"))
  {
    printf(gettext("Error: invalid step: %s\n"), argv[1]);
    return -1;
  }
  filename1 = argv[2];
  if (argv[1][0] == '2')
  {
    if (argc <= 3)
    {
      printf(gettext("Error: step 2 needs two filenames\n"));
      return -1;
    }
    else
      if (argc > 4)
        open_mode = atoi(argv[4]);
    filename2 = argv[3];
  }
  else 
    if (argv[1][0] == '3')
    { 
      if (argc > 2)
	open_mode = atoi(argv[2]);
    }

  if (open_mode != ADSL_Open && open_mode != ADSL_Open_ANSI && open_mode != ADSL_Open_GDMT && open_mode != ADSL_Open_GLITE && open_mode != ADSL_Open_MULTIMODE)
  {
    printf(gettext("Error: invalid open mode\n"));
    return -1;    
  }

  /* init USB bus and find devices */
  usb_init();
  if (usb_find_busses() < 0)
  {
    printf(gettext("Error: I can't find busses\n"));
    return -1;
  }
  if (usb_find_devices() < 0)
  {
    printf(gettext("Error: I can't find devices\n"));
    return -1;
  }

  /* search first 3Com ADSL modem */
  bus = usb_busses;
  goon = 1;
  while (bus && goon)
  {
    dev = bus->devices;
    while (dev && goon)
    {
      if (argv[1][0] == '1')
      {
        if (dev->descriptor.idVendor == ID_VENDOR &&
           (dev->descriptor.idProduct == ID_PRODUCT1 ||
            dev->descriptor.idProduct == ID_PRODUCT1b ||
            dev->descriptor.idProduct == ID_PRODUCT2))
        {
          goon = 0;
          adsl_dev = dev;
        }
      }
      else
      {
        if (dev->descriptor.idVendor == ID_VENDOR)
        {
          if (dev->descriptor.idProduct == ID_PRODUCT1 || dev->descriptor.idProduct == ID_PRODUCT1b)
          {
            printf(gettext("Error: you need load EZ-USB firmware before Alcatel firmware\n"));
            return -1;
          }
          if (dev->descriptor.idProduct == ID_PRODUCT2)
          {
            goon = 0;
            adsl_dev = dev;
          }
        }
      }
      if (goon)
        dev = dev->next;
    }
    bus = bus->next;
  }
  if (adsl_dev == NULL)
  {
    printf(gettext("Error: I didn't find ADSL modem\n"));
    return -1;
  }
  printf(gettext("I found ADSL modem with VendorID = %04x & ProductID = %04x\n"),
         adsl_dev->descriptor.idVendor, adsl_dev->descriptor.idProduct);

  /* connect to ADSL modem */
  adsl_handle = usb_open(adsl_dev);
  if (adsl_handle == NULL)
  {
    printf(gettext("Error: Couldn't get device handle for ADSL modem\n"));
    return -1;
  }
  /* set configuration (not necessary, we have only one configuration) */
  /*
  if (usb_set_configuration(adsl_handle, 1) < 0)
  {
    printf("Error: usb_set_configuration: %s\n", usb_strerror());
    return -1;
  }
  */
  /* check if other program is using interface 0 */
  if (usb_claim_interface(adsl_handle, 0) < 0)
  {
    printf("Error: usb_claim_interface: %s\n", usb_strerror());
    return -1;
  }
  /* if ez-usb firmware is not loaded then set alternate setting to 1 */
  if (adsl_dev->descriptor.idProduct == ID_PRODUCT1 || adsl_dev->descriptor.idProduct == ID_PRODUCT1b)
  {
    if (usb_set_altinterface(adsl_handle, 1) < 0)
    {
      printf("Error: usb_set_configuration: %s\n", usb_strerror());
      return -1;
    }
  }
#ifdef DEBUG
  printf("Interface = %d\n", adsl_handle->interface);
  if (adsl_dev->descriptor.idProduct == ID_PRODUCT1 || adsl_dev->descriptor.idProduct == ID_PRODUCT1b)
    printf("Alternate setting = %d\n", adsl_handle->altsetting);
#endif

  /* call to option */
  if (argv[1][0] == '1')  /* we permit reload ez-usb firmware if it is loaded already */
    r = load_ez_usb(adsl_handle, argv[2]); 
  else if (argv[1][0] == '2') 
    r = load_alcatel(adsl_handle, argv[2], argv[3], open_mode);
  else
    r = monitorize(adsl_handle, open_mode);

  PDEBUG(gettext("Releasing interface...\n"));
  usb_release_interface(adsl_handle, 0);
  PDEBUG(gettext("Releasing device...\n"));
  usb_close(adsl_handle);

  return r;
   
}

