#!/usr/bin/perl -w
#
#  Test that every perl script accepts and processes each of the options
# documented in its POD.
#
#  Cute test :)
#
# Steve
# --
# $Id: getopt.t,v 1.3 2006/09/10 21:04:20 steve Exp $


use strict;
use File::Find;
use Test::More qw( no_plan );


#
#  Test each file
#
foreach my $file ( sort( glob "./bin/*-*" ) )
{
    testFile( $file );
}


#
#  Check that the given file implements all the option processing it
# is supposed to.
#
#
sub testFile
{
    my ($file ) = (@_);
    is( -e $file, 1, "File exists: $file" );
    is( -x $file, 1, "File is executable" );

    #
    #  Run the file with "--help" and capture the output.
    #
    my $output = `$file --help`;

    #
    #  Parse out the options we accept
    #
    my @documented = ();

    foreach my $line ( split( /\n/, $output ) )
    {
        if ( $line =~ /[ \t]*--([a-z-]+)/ )
        {
            push @documented, $1;
        }
    }

    #
    #  Test we discovered some documented options.
    #
    ok( $#documented > 1, "We found some options documented." );



    #
    #  Now read the input file so that we can see if these advertised
    # options are actually used.
    #
    open( IN, "<", $file ) or die "Failed to open file for reading $file - $!";
    my @LINES = <IN>;
    close( IN );

    #
    #  Options accepted
    #
    my %accepted;

    #
    #  Do minimal parsing to find the options we process with
    # Getopt::Long;
    #
    my $complete = join( "\n", @LINES );
    if ( $complete =~ /GetOptions\(([^\)]+)\)/mi )
    {
        #
        #  Multi-line text which should have all the options we've
        # invoked GetOptions with.
        #
        my $opt = $1;

        #
        #  Process each one.
        #
        foreach my $o ( split( /\n/, $opt ) )
        {
            #
            #  Strip trailing comments.
            #
            if ( $o =~ /([^#]+)#/ )
            {
                $o = $1;
            }

            #
            #  Remove "" from around it.
            #
            if ( $o =~ /"([^"]+)"/ )
            {
                $o = $1;
            }

            #
            #  Discard anything after "=", or " "
            #
            if ( $o =~ /(.*)[ \t=]+(.*)/ )
            {
                $o = $1;
            }

            #
            #  Now avoid blank lines.
            #
            next if ( $o =~ /^[ \t]*$/ );


            #
            #  Phew.  Now we're done.
            #
            #  This option '$o' is something we call GetOptions with.
            #
            $accepted{$o} = 1;
        }
    }

    #
    #  Now we want to make sure that each documented option is
    # present in the list of options we pass to getopt.
    #
    foreach my $argument ( @documented )
    {
        is( $accepted{$argument}, 1, "Option '--$argument' accepted: $file" );
    }
}
