#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2007 Jd & Hap Hazard
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#


# All classes in thse files are required Interfaces for managing
# machines/host/nodes with virtualization technology.
# Currently this is implemented for Xen. (Later we can separate this
# and fromalize the interface. )

# TODO:
# - Remoting integration, remote file enhancements
# - Exception handling

# TBD: Later (not required immediately
# - Add DiskEntry to Dom too, to reprsent mounted disk
# - Add NetworkEntry to DomConfig as well as Dom


import sys,os,re,types

from XenServerProxy import VMOperationException
from utils import search_tree, XMConfig, PyConfig
from NodeProxy import Node
from constants import *


   # TODO : currently using the internal representation returned by
   #        xen API. will keep it like this till we really need our own

class XenDomain:
    """
    This represent Doms. It encapsulates information about
    running virtual machine : state as well as resource stats
    """
    RUNNING  = 0
    BLOCKED  = 1
    PAUSED   = 2
    SHUTDOWN = 3
    CRASHED  = 4
    NOT_STARTED    = -1
    UNKNOWN  = -2
    
    def __init__(self, node,context):
        """
        initialize using dom_info.
        a. can be used to create a new dom
        b. can be used to instantiate dom representation of a runnig dom.
        """
        self.id = -1
        self._config   = None
        self._dom_info = None
        self._stats    = None
        self.node = node
        
        if type(context) == types.StringType:
            filename = context
            self._config = DomConfig(node,filename)
            self.name = self._config.name
            self.is_resident = False
        
        else:
            # assume dom_info
            dom_info = context
            self._dom_info = dom_info
            self.name = search_tree(self._dom_info, 'name')
            self.id = search_tree(self._dom_info, 'domid')
            self.is_resident = True
            

        self.state = self._state()

    @classmethod
    def get_state(cls,state_str):
        if state_str == None or len(state_str) < 5:
            return XenDomain.UNKNOWN
        
        if state_str[0] == "r":
            return XenDomain.RUNNING
        elif state_str[1] == "b":
                return XenDomain.BLOCKED
        elif state_str[2] == 'p':
            return XenDomain.PAUSED
        elif state_str[3] == "s":
            return XenDomain.SHUTDOWN
        elif state_str[4] == "c":
            return XenDomain.CRASHED
        return XenDomain.UNKNOWN
        

    def _state(self):
        """
        Translate the string in to state enum
        returns one of the following
        XenDomain.RUNNING, XenDomain.BLOCKED, XenDomain.PAUSED,
        XenDomain.SHUTDOWN, XenDomain.CRASHED, XenDomain.UNKNOWN.
        XenDomain.NOT_STARTED 
        """
        if self._dom_info == None:
            return XenDomain.NOT_STARTED
        
        state_str = search_tree(self._dom_info, 'state')
        return self.get_state(state_str)
    def isDom0(self):
        """Test wether the dom is dom0"""

        return self.is_resident and self.id == 0

    def isDomU(self):
        """Test wether the dom is a guest dom"""

        return self.is_resident and self.id > 0

    
    def __getitem__(self, param):
        if param == "name":
            return self.name
        else:
            return search_tree(self._dom_info, param)

    # refresh the dom
    def refresh(self):
        if self.name is None:
            if self._config is not None:
                name = self._conifig.name
            else:
                raise Exception("Invalid domain state, no Name found")

        # try to get the dom information from xen
        try:
            self._dom_info= self.node.node_proxy.xend.domain(self.name)
            self.name = search_tree(self._dom_info, 'name')
            self.id   = search_tree(self._dom_info, 'domid')
            self.is_resident = True
        except VMOperationException, ex:
            self._dom_info = None
            self.is_resident = False

        self.state = self._state()
            
    # operation on doms.
    def _save(self, filename):
        """Save a snapshot of the dom"""

        self.node.node_proxy.xend.domain.save(self.id, filename)

    def _execShutdownCmd(self, cmd):
        """Execute the requested shutdown command on the dom"""
        self.node.node_proxy.xend.domain.shutdown(self.id, cmd)
        self._dom_info = None
        self.is_resident = False

    def _reboot(self):
        """Reboot the dom"""

        self._execShutdownCmd("reboot")

    def _shutdown(self):
        """Shutdown the dom"""

        self._execShutdownCmd("halt")

    def _destroy(self):
        """Destroy the dom, hence removing it from resident running doms
        without any attempt at shutting anything down on the guest OS"""

        self.node.node_proxy.xend.domain.destroy(self.id)
        self._dom_info = None
        self.is_resident = False

    def _pause(self):
        """Pause the dom"""

        self.node.node_proxy.xend.domain.pause(self.id)
        
    def _resume(self):
        """Resume the paused dom"""

        self.node.node_proxy.xend.domain.unpause(self.id)

    ## start
    def _start(self):
        self._start_dom()


    ## Changing properties of running dom
    def setMem(self, value):
        """Change the running memory setting"""
        
        self.node.node_proxy.xend.domain.setMemoryTarget(self.id, value)
        
    def setVCPUs(self, value):
        """Change the running number of CPUs setting"""

        self.node.node_proxy.xend.domain.setVCpuCount(self.id, value)


    ## config related intefaces
    def get_config(self):
        return self._config

    def set_config(self, config):
        self._config = config
    
    def _start_dom(self):
        if self._config is None:
            raise Exception("Configuration not set. Can not start domain")
        cmd = "xm create " + self._config.filename
        (output, exit_code) = self.node.node_proxy.exec_cmd(cmd,self.node.exec_path)
        if not exit_code:
            self.refresh()            
        else:
            raise Exception("Domain could not be started: " + output)
        
    ## get stats
    def get_snapshot(self):
        if self._stats == None:
            self._stats = DomStats(self)
        return self._stats.get_snapshot()
        






# For now, we can implement the Xen specific stuff. Later we should
# drive it through metadata. (what attributes supported in file and
# running doms,nested attribute support) [May be do it now?]
#
class DomConfig(PyConfig):
    """
    represnts startup config object (information in the conf file)
    """

    imports = ["import sys",
               "import os",
               "import os.path",
               "from xen.util.ip import *",
               ]

    # Disk entry class
    class DiskEntry:
        """This class provides a structured representation of the disk
        string. The following attributes are available:

         type:     {file|lvm|part}
         filename: the path to the VBD or device
         device:   the device to be used in the guest system
         mode:     {r|w}
         """

        def __init__(self, input):
            if type(input) is str:
                m = re.match("^(phy|file):(\S+),([a-z|0-9]+(:cdrom)?|ioemu:[a-z|0-9]+),(r|w)$", input)
                if m:
                    self.filename = m.group(2)
                    self.device = m.group(3)
                    self.mode = m.group(5)
                    
                    if m.group(1) == "file":
                        self.type = "file"
                    else:
                        device = os.path.realpath(m.group(2))
                        if device.startswith("/dev/mapper"):
                            self.type = "lvm"
                        else:
                            self.type = "part"
            elif type(input) in (list, tuple):
                self.type, self.filename, self.device, self.mode = input

        def __repr__(self):
            if self.type == "file":
                dtype = "file"
            else:
                dtype = "phy"

            return "%s:%s,%s,%s" % (dtype, self.filename, self.device,
                                    self.mode)


    ### Module variable.
    signature = "# Automtically generated by XenMan\n"

    # DomConfig follows
    # ### Damm... node needs to be passed... can we do better?
    def __init__(self, node, filename = None):
        """
        read stuff from file and populate the config
        when filename is None, creates an empty config
        """
        self.node = node
        self.xenman_generated = False
    
        PyConfig.__init__(self,
                          node,
                          filename,
                          DomConfig.signature)


        # TBD : implement isfile in node proxy and use it.
        #if filename is None or not os.path.isfile(filename): return
        
        if filename is None : return

        if len(self.lines) > 0:
            if self.lines[0].find(self.signature) >= 0:
                    self.xenman_generated = True
            
        for k in ("xm_help", "xm_file"):
            if self.options.has_key(k):
                del self.options[k]
                
        if self["name"] is None:
            raise Exception("No dom name specified")

        #pick up the name from the file
        self.name = self["name"]

    # custom read_conf
    def read_conf(self, init_glob=None, init_locs=None):
        # Leverage the fact that conf files are python syntax.
        # save on parsing code
        
        # Create global and local dicts for the file.
        # Initialize locals to the vars.
        globs = {}
        locs = {}

        cmd = '\n'.join(self.imports + 
                        [ "from xen.xm.help import Vars",
                          "xm_file = '%s'" % self.filename,
                          "xm_help = %d" % 0,
                          "xm_vars = Vars(xm_file, xm_help, locals())"
                          ])

        # Use exec to do the standard imports and
        # define variables we are passing to the script.
        exec cmd in globs, locs
        return PyConfig.read_conf(self, globs, locs)
    
    def __setitem__(self, name, item):
        self.options[name] = item
        if name == "name":
            self.name = item

    # try to keep them in sync
    def set_name(self, name):
        self.name = name
        self["name"] = name
    
    
    # get the configured disks
    def getDisks(self):
        """
        This method returns a more structured version of the config file's
        disk entry
        """
        reslist = []
        for str in self["disk"]:
            reslist.append(self.DiskEntry(str))
        return reslist


    # Assumes that the file read contains single line values
    # now preserves the comments and order
    def write(self):
        """Writes the settings out to the filename specified during
        initialization"""
        self.name = self["name"]
        PyConfig.write(self)
        self.xenman_generated = True


    def is_xenman_generated(self):
        return self.xenman_generated

    ### Get raw file content    
    def get_contents(self):
        f = self.node.node_proxy.open(filename)
        lines = f.readlines()
        f.close()
        contents = "".join(lines)
        return contents

    ### Write raw contents
    def write_contents(self, contents):
        """Writes the settings out to the filename specified during
        initialization"""
        outfile = self.node.node_proxy.open(self.filename, 'w')
        outfile.write(contents)
        outfile.close()



    # get representation required for underlying create API
    # may be __repr__ ?
    def internal_rep(self):
        """ create config required by xen create_dom api"""
        # the problem is that this needs to be complete, otherwise some
        # existing conf files that are super set of xenman created files
        # would not work
        # May be we should just go the xm create route for now.


    # xenman specific validation.
    def validate(self):
        """Attempts to validate that the settings are not going to lead to
        any errors when the dom is started, and returns a list of the
        errors as strings"""

        result = []

        if not self["name"]:
            result.append("Missing domain name.")

        if not self["disk"]:
            result.append("Missing disk specification.")

        if not self["bootloader"]:
            # if a bootloader is not specified,
            # check for valid kernel and ramdisk
            for parm in ("kernel", "ramdisk"):
                if not self[parm] or not self.node.node_proxy.file_exists(self[parm]):
                    result.append("Invalid file name for %s." % parm)
                    

        for parm, nodeval in (("memory", "total_memory"),
                               ("vcpus", "nr_cpus")):
            if self[parm] and int(self[parm]) > int(ManagedNode.getNodeVal(nodeval)):
                result.append("More %s requested than total %s." % (parm,
                                                                    parm))

        return result



    

class DomStats:
    """
    represents statatistics/measurements for a dom. (CPU, I/O etc)
    This is abstracted out so we can cut over to some other source for
    runtime statastics/measurements
    """

    def __init__(self, dom):
        """
        constructor, dom for which the stats are to be obtained.
        """
        self.dom = dom
        self.stat = {}

    def get_snapshot(self):
        self.dom.refresh()
        for stat in ("memory", "cpu_time"):
            self.stat[stat] = dom[stat]
        return self.stat

    
### module initialization

    







    
    

    
    
