#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2007 Jd & Hap Hazard
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#


# All classes in thse files are required Interfaces for managing
# machines/host/nodes with virtualization technology.
# Currently this is implemented for Xen. (Later we can separate this
# and fromalize the interface. )

# TODO:
# - Remoting integration, remote file enhancements
# - Exception handling

# TBD: Later (not required immediately
# - Add DiskEntry to Dom too, to reprsent mounted disk
# - Add NetworkEntry to DomConfig as well as Dom


import sys,os,re,types

from ManagedNode import ManagedNode
from NodeProxy import Node
from utils import XMConfig, constants, search_tree, Poller
from XenDomain import *


class XenNode(ManagedNode):
    """
    Interface that represents a node being managed.It defines useful APIs
    for clients to be able to do Management for a virtualized Node
    """                

    def __init__(self,
                 hostname = None,
                 username=Node.DEFAULT_USER,
                 password=None,
                 isRemote=False,
                 protocol = "ssh_tunnel",
                 tcp_port = Node.DEFAULT_TCP_PORT,
                 helper = None):

        ManagedNode.__init__(self, hostname,
                             username, password,
                             isRemote, protocol,
                             tcp_port, helper)
        self._node_info = None
        self.dom_list = DomListHelper(self)
        self._dom0     = None
        self._managed_domfiles = None
        self.xenauto = '/etc/xen/auto'

        self.metrics_helper = MetricsHelper(self)
        self.metrics_poller = None
        self.POLLING_INTERVAL = 5.0 # default no.of seconds between each metric fetch
        self.MAX_POLLS = 4 # default number of times metrics poller iterates
        

    def _init_node_info(self):
        if self._node_info is None:
            self._node_info = self.node_proxy.xend.node.info()
        return self._node_info
    
    def _init_managed_domfiles(self):
        if self._managed_domfiles is None:
            mdf = self.config.get(XMConfig.APP_DATA,
                                  constants.prop_domfiles,
                                  )            
            if mdf is None:
                self._managed_domfiles = []
            else:
                self._managed_domfiles = eval(mdf)        
        return self._managed_domfiles
    
            
    # implement lazy initialization. 
    def __getattr__(self, name):
        if name == 'node_info':
            return self._init_node_info()
        if name == 'managed_domfiles':
            return self._init_managed_domfiles()
        else:
            return ManagedNode.__getattr__(self,name)
        
        
    
    # encapsulate the host and dom0 information here.        
    def __getitem__(self, param):
        val = search_tree(self.node_info, param)
        if  val == None:
            # try the dom0
            return self._dom0[param]
        else:
            return val


    def _set_dom0(self,dom):
        self._dom0 = dom



    # public methods
    def get_dom_names(self):
        """
        return lists containing names of doms running on this node.
        exceptions :
        """
        return self.dom_list.iterkeys()


    def get_doms(self):
        """
        returns list containing information about nodes running on
        this node.
        returns list of Dom objects
        exceptions :
        NOTE: in most cases, get_dom_names() is a better option.
        """
        return self.dom_list

    def get_dom(self, name):
        return self.dom_list[name]

    
    def isDom0(self, name):
        if self.get_dom(name):
            return self.get_dom(name).isDom0()
        else:
            return False
    
    def isDomU(self, name):
        if self.get_dom(name):
            return self.get_dom(name).isDomU()
        else:
            return False
    
    def get_state(self, name):
        if self.get_dom(name):
            return self.get_dom(name)._state()
        else:
            return None

    def isResident(self, name):
        if self.get_dom(name):
            return self.get_dom(name).is_resident
        else:
            return False
    

    def create_dom(self, config):
       """
       create a new dom given a particular config.
       exceptions:
       """
       if config.filename is None:
           raise Exception("filename must be set in the config.")
       config.write()
       dom_name = self.add_dom_config(config.filename)
       self.start_dom(dom_name)
       self.refresh()
       return dom_name

    def create_dom_from_file(self, filename):
        dom_name = self.add_dom_config(filename)
        self.start_dom(dom_name)
        self.refresh()
        return dom_name

    def refresh(self):
        self.dom_list.refresh()
    
    # Manage external files.
    def add_dom_config(self, filename):
        return self.dom_list.add_dom_config(filename)

    def remove_dom_config(self, filename):
        return self.dom_list.remove_dom_config(filename)

    # Metrics
    def get_metrics(self, refresh=False):
        if refresh:
            self.metrics = self.metrics_helper.getFrame()
        else:
            if self.metrics is None:
                # first time get_metrics has been called
                self.get_metrics(refresh=True)

            if self.metrics_poller is None or not self.metrics_poller.isAlive():
                # kick off ascynchronous metrics polling
                # ... MAX_POLLS polls at POLLING_INTERVAL second intervals
                self.metrics_poller = Poller(self.POLLING_INTERVAL,self.get_metrics,
                                             args=[True],max_polls=self.MAX_POLLS)
                self.metrics_poller.start()
                
        return self.metrics


    # dom operations
    
    def start_dom(self, name):
       """
       start the given dom 
       exceptions:
       """
       # deligate to the dom itself
       self.get_dom(name)._start()
       

    def pause_dom(self, name):
       """
       pause a running dom
       exceptions:
       """
       self.get_dom(name)._pause()
       

    def resume_dom(self, name):
        """
        pause a running dom
        exceptions:
        """
        self.get_dom(name)._resume()
       

    def shutdown_dom(self, name):
       """
       shutdown a running dom. 
       """
       self.get_dom(name)._shutdown()
       

    def destroy_dom(self, name):
       """
       destroy dom
       """
       self.get_dom(name)._destroy()
       self.dom_list.refresh()
       
    def reboot_dom(self, name):
       """
       reboot dom
       """
       self.get_dom(name)._reboot()
       
    def restore_dom(self, filename):
        """
        restore from snapshot file
        """
        self.node_proxy.xend.domain.restore(filename)
        self.dom_list.refresh()

    def get_console(self, name):
       """
       get the console for the dom
       API migght need revision...
       """
       pass

    def get_terminal(self, dom, username, password):
       """
       return tty terminal (telnet, ssh session ?)
       """
       pass

    def get_vnc(self,dom):
       """
       get VNC session for this dom. VNC would popup username/password.
       """
       pass

    
       

class DomListHelper:
    """
    Class represent list of dom being tracked by this managed
    node. 
    """
    # for implementing domlist

    def __init__(self, node):
        self._dom_dict = None
        self.node = node

    def _init_dom_list(self):
        """ take the dominfo from the API and return list of doms
        """
        if self._dom_dict is None:
            self.refresh()
        return self._dom_dict

    def __getattr__(self, name):
        if name == 'dom_dict': 
            return self._init_dom_list()

    # TODO : add synchronization ?
    def refresh(self):
        current_dict = {}
        # get the current running doms
        dom_list_info = self.node.node_proxy.xend.domains(1)

        for dom_info in dom_list_info:
            dom = XenDomain(self.node,dom_info[1:])
            if dom.id == 0:
                self.node._set_dom0(dom)
            #else
            #ALLOW Dom 0 to be in the list too
            #This would allow twaking mem and cpus
            current_dict[dom.name] = dom

        # now lets get /etc/xen/auto files and associate with
        # running doms, also add the doms for managed files.

        for filename in self.node.node_proxy.listdir(self.node.xenauto) + \
                self.node.managed_domfiles:

            try: 
                if filename.find('/') >= 0:
                    if self.node.node_proxy.file_exists(filename):
                        new_dom = XenDomain(self.node, filename)                    
                    else:
                        continue
                else:
                    filename = "%s/%s" % (self.node.xenauto, filename)
                    if self.node.node_proxy.file_exists(filename):
                        new_dom = XenDomain(self.node, filename)
                    else:
                        continue
            except (Exception, StandardError), e:
                print "Domain File %s is invalid. Reason: %s. Removing from list"\
                      % (filename, str(e))
                self.remove_dom_config(filename)
                continue

            dom_name = new_dom.name
            if current_dict.has_key(dom_name):
                current_dict[dom_name].set_config(new_dom.get_config())
            else:
                current_dict[dom_name] = new_dom                   

        self._dom_dict = current_dict

    def __getitem__(self, item):
        if not item: return None
        if type(item) is int:
            for name, dom in self.dom_dict.iteritems():
                if dom.is_resident and dom.id == item:
                    return dom
        else:
            if self.dom_dict.has_key(item):
                return self.dom_dict[item]
            else:
                return None


    def __iter__(self):
        return self.dom_dict.itervalues()


    def iterkeys(self):
        return self.dom_dict.keys()


    # start tracking file
    def add_dom_config(self, filename):
        if filename in self.node.managed_domfiles:
            return DomConfig(self.node,filename)['name']
        else:
            new_dom = XenDomain(self.node, filename)
            self.node.managed_domfiles.append(filename)
            self.node.config.set(XMConfig.APP_DATA,constants.prop_domfiles,
                                          repr(self.node.managed_domfiles),
                                          )
            self.dom_dict[new_dom.name] = new_dom

            return new_dom.name

    def remove_dom_config(self, filename):

        if filename in self.node.managed_domfiles:
            self.node.managed_domfiles.remove(filename)
            self.node.config.set(XMConfig.APP_DATA,constants.prop_domfiles,
                                          repr(self.node.managed_domfiles),
                                          )
            # check if running, shutdown, remove, etc.
            for d in self.dom_dict.itervalues():
                if d.get_config() is not None and \
                       d.get_config().filename == filename:
                    del self.dom_dict[d.name]
                    return True

        return False




class MetricsHelper:
    """A Helper to fetch and format runtime metrics from a Xen Host"""

    FRAME_CMD = 'xentop -b -i 2 -d 1'
    
    def __init__(self, node):
        self.node = node

    def getFrame(self):
        """returns a dictionary containing metrics for all running domains
        in a frame"""
        (retbuf, retcode) = self.node.node_proxy.exec_cmd(self.FRAME_CMD,
                                                          self.node.exec_path)
        if retcode: return None

        # hack to eliminate unwanted vbd entries.
        cleansed_retbuf = re.sub('vbd.*\n','',retbuf)

        frame = {} # the output metric frame (dict of dict's)
        
        #extract the xen version
        m = re.search('xentop.*(Xen.*?)\n',cleansed_retbuf,re.S)
        if not m: return None
        frame['XEN_VER'] = m.group(1)

        #initialise aggregation counters
        frame['VM_TOTAL_CPU(%)'] = 0.0
        frame['VM_TOTAL_MEM(%)'] = 0.0

        # split the returned buffer into individual frame buffers ...
        frames = re.split('xentop.*\n',cleansed_retbuf,re.S)
        #... and use the last frame buffer for creating the metric frame
        fbuffer = frames[-1:][0]

        # extract host cpu and mem configuration
        m = re.search('Mem:(.*) total.*CPUs:(.*)',fbuffer)
        if not m: return None
        frame['SERVER_CPUs'] = m.group(2).strip()
        frame['SERVER_MEM'] = m.group(1).strip()

        # extract overall runtime domain stats
        m = re.search('(\d+) domains.*(\d+) paused.*(\d+) crashed',fbuffer)
        if not m: return None
        frame['RUNNING_VMs'] = m.group(1).strip()
        frame['PAUSED_VMs']  = m.group(2).strip()
        frame['CRASHED_VMs'] = m.group(3).strip()
        
        
        # parse the metric frame buffer for per domain stats
        lines = fbuffer.split('\n')[:-1]
        mbuffer = ''
        # strip unused entries at the top of the buffer
        # and extract the metric (sub)buffer
        for l in lines:
            if l.strip().startswith('NAME'):
                mbuffer = lines[lines.index(l):]
                break
        # construct the metric frame as a dict of dictionaries
        # containing metric-name, metric-value pairs
        cleanup_exp = re.compile('[a-zA-Z]+\s[a-zA-Z]+ | n/a')
        for d in mbuffer[1:]:
            cleansed = re.sub(cleanup_exp,'None',d)
            d_frame = dict(zip(mbuffer[0].split(),cleansed.split()))
            frame[d_frame["NAME"]] = d_frame
            # compute running aggregates
            if d_frame['NAME'] != 'Domain-0':
                frame['VM_TOTAL_CPU(%)'] += float(d_frame['CPU(%)'])
                frame['VM_TOTAL_MEM(%)'] += float(d_frame['MEM(%)'])


        return frame
        
    
        

        
            



# Test code
if __name__ == "__main__":
    test_domu = "test"
    host = "localhost"
    dom_file = '/etc/xen/test'
    dom_2b_started = 'test'
    dom_2b_shutdown = 'test'

    username = 'root'
    passwd = ''
    
    # basic connectivity
    remote = False
    if not remote:
        host = "localhost"
    else:
        host = '192.168.123.155'
        test_domu = "test"
        dom_file = '/etc/xen/test'
        dom_2b_started = 'test'
        dom_2b_shutdown = 'test'
        
    managed_node = XenNode(hostname=host,
                           username = username,
                           password = passwd,
                           isRemote=remote)

    ## create/destroy dom
    dom_config = DomConfig(managed_node, dom_file)
    dom_config["memory"] = 256
    dom_config["vif"] = ['bridge=xenbr1']
    m = { 'VM_NAME':'foo', 'IMAGE_NAME':'anaconda' } 
    dom_config.instantiate_config(m)
    print dom_config.default_computed_options, dom_config.get_computed_options()
    dom_config.save("/tmp/test_config")

    sys.exit(0)
    
    dom_config = DomConfig(managed_node,dom_file)
    dom_name = managed_node.create_dom(dom_config)
    print 'resident?', managed_node.isResident(dom_name)
    managed_node.destroy_dom(dom_name)
    print 'Doms b4 removal: ',managed_node.get_dom_names()
    managed_node.remove_dom_config(dom_file)
    print 'Doms post removal: ',managed_node.get_dom_names()

    dom_name = managed_node.create_dom_from_file(dom_file)
    print 'resident?', managed_node.isResident(dom_name)
    managed_node.destroy_dom(dom_name)
    print 'Doms b4 removal: ',managed_node.get_dom_names()
    managed_node.remove_dom_config(dom_file)
    print 'Doms post removal: ',managed_node.get_dom_names()
    
    ## start / stop dom and check its running state    

    managed_node.add_dom_config(dom_file)
    for name in  managed_node.get_dom_names():
        print name
        
    ## start / stop dom and check its running state
    managed_node.start_dom(dom_2b_started)
    print 'resident?', managed_node.isResident(dom_2b_started)
    print 'memory: ',managed_node.get_dom(dom_2b_started)["memory"] 
    print 'destroying ... '
    managed_node.destroy_dom(dom_2b_started)
    print "resident?" ,managed_node.isResident(dom_2b_shutdown)
    
    
    sys.exit(0)
    #################################################
    
    doms  = managed_node.get_doms()  # test dom information.
    for dom in doms:
        print "##### some info from dom object for dom  ###"
        for key in ("name", "memory", "kernel"):
            print key, "=" , dom[key]

        if dom.is_resident:
            print "device" ,"=", dom["device"]  #priniting first device only, BUG!!
            print "image" , "=", dom["image"]
            print "state", "=", dom.state
        else:
            print "disk"," ="
            for disk in dom.get_config().getDisks():
                print disk
            print "network", "=", dom.get_config()["vif"]


    managed_node.remove_dom_config(dom_file)
    for name in  managed_node.get_dom_names():
        print name


    # get doms by name or id
    print "Access domain by name as well as id"
    dom_by_name = doms[test_domu]
    dom_by_id = doms[doms[test_domu].id]

    print dom_by_name.name, dom_by_id.name


    # get the stats
    print "### get measurement snapshot ###"
    stats = dom_by_name.get_snapshot()
    for stat in stats:
        print stat,"=",stats[stat]


    # empty dom config, create a new file
    print "### new empty config and creating a file."
    newcfg = DomConfig(managed_node)
    newcfg["name"] = "Txx"
    newcfg["memory"] = 299
    newcfg.set_filename("/tmp/Txx")
    newcfg.write()

    f = managed_node.node_proxy.open("/tmp/Txx")
    x = f.read(1024)
    print x
    f.close()

    print "### read config from /etc/xen/auto and write them to /tmp"
    ## Dom Config
    for f in managed_node.node_proxy.listdir("/etc/xen/auto"):
        fin = "/etc/xen/auto/"+f
        print fin
        d = DomConfig(managed_node, fin)
        d.save("/tmp/" + f)
    

    print "### get first file in /etc/xen/auto and dump its info"
    ## access this through dom
    d = None
    for f in managed_node.node_proxy.listdir("/etc/xen/auto"):
        fin = "/etc/xen/auto/"+f
        d = XenDomain(managed_node, fin)
        break    # pick up first file
    

    if d != None:
        print "#### dumping config ####"
        cfg =  d.get_config()
        cfg.dump()
        print "disk config"
        disks = cfg.getDisks()
        for disk in disks:
            print disk
    
        print "### modified memory to 300, dumping again ### "
        cfg['memory'] = 300
        cfg.dump()
        
    else:
        print "No Dom in /etc/auto"
    


    ## test the nodeinfo
    print "########## Host information ##########"
    for key in ("system", "host","release", "version", "machine", "nr_cpus",
                "nr_nodes","sockets_per_node",
                "cores_per_socket", "threads_per_core",
                "cpu_mhz","total_memory","free_memory","xen_caps",
                "platform_params","xen_changeset"):
        
        print key,"=",managed_node[key]


    print "########## getting dom0 information from managed node ##########"
    # Note dom0 information does not contain few things available in
    # other domus, example, uptime,device, network..!? 
    for key in ("cpu_time", "state","uptime", "online_vcpus"):
        print key,"=",managed_node[key]
    

    

    sys.exit(0)
    
    ### test create
    cfg = DomConfig(managed_node, "/etc/xen/T99")
    cfg["name"] = "T99"
    cfg["memory"] = 256
    cfg["vif"] = []
    cfg["vcpu"] = 1
    cfg["disk"] = ['file:/domu_disks/T99',xvda,w]

    d = managed_node.create_dom(cfg)
    print d["memory"], d.is_resident

    
    

    
    
