#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdarg.h>

#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>

#include "popupmenu.h"
#include "xmms_plugin.h"
#include "playerctrl.h"

/* file chooser dialog  */

enum {
	RESPONSE_OPEN,	/// Open button
	RESPONSE_ADD	/// Add button
};

static void file_chooser_response_cb(GtkDialog *dialog, gint code, gpointer data) {
	GSList *list, *li;
	gchar *uri;
	plugin_data *pd = (plugin_data *)data;

	if (code == RESPONSE_OPEN || code == RESPONSE_ADD) {
		list = gtk_file_chooser_get_uris(GTK_FILE_CHOOSER(dialog));
		for (li = list; li != NULL; li = g_slist_next(li)) {
			uri = (gchar *)li->data;
			pd->pctrl->player_playlist_add_url_string(pd->xmms_session, uri);
			g_free(uri);
		}
		g_slist_free(list);
	}

	if (code == RESPONSE_ADD)
		return;

	gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * @param name displayed label
 * @param ... mimetype or pattern (gchar *)
 */
static void add_file_filter(GtkFileChooser *chooser, gchar *name, ...) {
	GtkFileFilter *filter;
	va_list ap;
	gchar *mask;

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, name);

	va_start(ap, name);
	while ((mask = va_arg(ap, gchar *)) != NULL) {
		if (g_strrstr(mask, "/") != NULL)
			gtk_file_filter_add_mime_type(filter, mask);
		else
			gtk_file_filter_add_pattern(filter, mask);
	}
	va_end(ap);
	
	gtk_file_chooser_add_filter(chooser, filter);
}

/* callback funcs */

/**
 * show add files dialog
 */
static void show_file_chooser(GtkWidget *widget, gpointer data) {
	static GtkWidget * dialog = NULL;
	GtkFileFilter *filter;
	plugin_data *pd = (plugin_data *) data;

	if (! pd->pctrl->player_is_running(pd->xmms_session))
		return;

	if (GTK_IS_WINDOW(dialog)) {
		gtk_window_present(GTK_WINDOW(dialog));
		return;
	}

	dialog = gtk_file_chooser_dialog_new(_("Add files"), NULL,
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_OPEN, RESPONSE_OPEN,
			GTK_STOCK_ADD, RESPONSE_ADD,
			GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
			NULL);
	gtk_window_set_skip_taskbar_hint(GTK_WINDOW(dialog), TRUE);
	gtk_window_set_skip_pager_hint(GTK_WINDOW(dialog), TRUE);
	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), TRUE);
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), g_get_home_dir());
	
	add_file_filter(GTK_FILE_CHOOSER(dialog), _("Audio files"), "audio/*", NULL);
	add_file_filter(GTK_FILE_CHOOSER(dialog), _("Playlist files"), "*.m3u", "*.pls", NULL);
	add_file_filter(GTK_FILE_CHOOSER(dialog), _("All files"), "*", NULL);

	g_signal_connect(G_OBJECT(dialog), "response",
			G_CALLBACK(file_chooser_response_cb), pd);
	gtk_widget_show(dialog);
}


static void show_open_file(GtkWidget *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_eject(pd->xmms_session);
}


/**
 * open xmms preference window
 */
static void show_xmms_preferences(GtkWidget *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_show_prefs_box(pd->xmms_session);
}

/**
 * crear current playlist
 */
static void clear_playlist(GtkWidget *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_playlist_clear(pd->xmms_session);
}

/**
 * startup xmms
 */
static void exec_xmms(GtkWidget *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	
	exec_command(player_control_get_command(pd->player));
}

/**
 * shutdown xmms
 */
static void quit_xmms(GtkWidget *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_quit(pd->xmms_session);
}

/**
 * toggle repeat state
 */
static void toggle_repeat(GtkCheckMenuItem *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_toggle_repeat(pd->xmms_session);
}

/**
 * toggle shuffle state
 */
static void toggle_shuffle(GtkCheckMenuItem *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_toggle_shuffle(pd->xmms_session);
}

/**
 * toggle main window visibility
 */
static void toggle_main_win(GtkCheckMenuItem *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_main_win_toggle(pd->xmms_session,
			! pd->pctrl->player_is_main_win(pd->xmms_session));
}

/**
 * toggle playlist window visibility
 */
static void toggle_pl_win(GtkCheckMenuItem *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_pl_win_toggle(pd->xmms_session,
			! pd->pctrl->player_is_pl_win(pd->xmms_session));
}

/**
 * toggle equalizer window visibility
 */
static void toggle_eq_win(GtkCheckMenuItem *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_eq_win_toggle(pd->xmms_session,
			! pd->pctrl->player_is_eq_win(pd->xmms_session));
}

/**
 * show "Jump to File" dialog
 */
static void show_jtf_box(GtkWidget *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_show_jtf_box(pd->xmms_session);
}

/**
 * selection-done callback
 */
static void destroy_popupmenu(GtkWidget *widget, gpointer data) {
	gtk_widget_destroy(widget);
}


/* menu creation */

/**
 * create a image item
 * @param label label with mnemonic char
 * @param stock_id GtkStockItem ID
 * @param data must be plugin_data
 * @see http://developer.gnome.org/doc/API/2.0/gtk/gtk-Stock-Items.html
 */
static void
add_image_item_from_stock(GtkWidget *menu, gchar *label, const gchar *stock_id, GCallback cb, gpointer data) {
	GtkWidget *mi, *img;
	mi = gtk_image_menu_item_new_with_mnemonic(label);
	img = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
	g_signal_connect(G_OBJECT(mi), "activate", cb, data);
	gtk_widget_show(mi);
}

/**
 * create a check item
 * @param label label with mnemonic char
 * @param data must be plugin_data
 */
static void
add_check_item(GtkWidget *menu, gchar *label, gboolean active, GCallback cb, gpointer data) {
	GtkWidget *mi = gtk_check_menu_item_new_with_mnemonic(label);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi), active);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
	g_signal_connect(G_OBJECT(mi), "toggled", cb, data);
	gtk_widget_show(mi);
}

/**
 * create popupmenu
 * this menu is automatically destroyed when selection is done.
 * @return popupmenu widget
 */
GtkWidget * create_popupmenu(plugin_data *pd) {
	GtkWidget *menu, *submenu, *mi, *img;

	menu = gtk_menu_new();
	gtk_widget_show(menu);
	g_signal_connect(G_OBJECT(menu), "selection-done",
			G_CALLBACK(destroy_popupmenu), NULL);

	// open files button
	add_image_item_from_stock(menu, _("_Open files"), GTK_STOCK_OPEN,
			G_CALLBACK(show_open_file), pd);

	// add files button
	add_image_item_from_stock(menu, _("_Add files"), GTK_STOCK_ADD,
			G_CALLBACK(show_file_chooser), pd);

	// jtf button (audacious 1.1)
	if (pd->pctrl->player_show_jtf_box)
		add_image_item_from_stock(menu, _("_Jump to File"), GTK_STOCK_JUMP_TO,
				G_CALLBACK(show_jtf_box), pd);

	// clear pl button
	add_image_item_from_stock(menu, _("_Clear Playlist"), GTK_STOCK_CLEAR,
			G_CALLBACK(clear_playlist), pd);

	// repeate button
	add_check_item(menu, _("_Repeat"), pd->pctrl->player_is_repeat(pd->xmms_session),
			G_CALLBACK(toggle_repeat), pd);

	// shuffle button
	add_check_item(menu, _("_Shuffle"), pd->pctrl->player_is_shuffle(pd->xmms_session),
			G_CALLBACK(toggle_shuffle), pd);

	// view menu
	submenu = gtk_menu_new();
	gtk_widget_show(submenu);

	// main window button
	add_check_item(submenu, _("_Main Window"), pd->pctrl->player_is_main_win(pd->xmms_session),
			G_CALLBACK(toggle_main_win), pd);
	
	// playlist button
	add_check_item(submenu, _("_Playlist"), pd->pctrl->player_is_pl_win(pd->xmms_session),
			G_CALLBACK(toggle_pl_win), pd);

	// equalizer button
	add_check_item(submenu, _("_Equalizer"), pd->pctrl->player_is_eq_win(pd->xmms_session),
			G_CALLBACK(toggle_eq_win), pd);

	mi = gtk_menu_item_new_with_mnemonic(_("_View"));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
	gtk_widget_show(mi);

	// separator
	mi = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
	gtk_widget_show(mi);

	// preference button
	add_image_item_from_stock(menu, _("_Preferences"), GTK_STOCK_PREFERENCES,
			G_CALLBACK(show_xmms_preferences), pd);

	if (pd->pctrl->player_is_running(pd->xmms_session))
		add_image_item_from_stock(menu, _("Power _Off"), GTK_STOCK_QUIT,
				G_CALLBACK(quit_xmms), pd);
	else
		add_image_item_from_stock(menu, _("Power _On"), GTK_STOCK_EXECUTE,
				G_CALLBACK(exec_xmms), pd);

	return menu;
}
