/* $Id: desktop-menu.c,v 1.11 2005/01/04 22:14:46 bmeurer Exp $ */
/*-
 * Copyright (c) 2004 os-cillation
 * All rights reserved.
 *
 * Written by Benedikt Meurer <bm@os-cillation.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#include <gmodule.h>

#include <libxfce4util/libxfce4util.h>

#include "desktop-menu.h"
#include "desktop-menu-parser.h"
#include "xdg-desktop-support.h"
#include "xdg-menu.h"
#include "xfce-menu.h"



static void   desktop_menu_rebuild (XfceDesktopMenu *desktop_menu);
static gchar *desktop_menu_get_menu_file (void);



struct _XfceDesktopMenu
{
  XdgMenu             *system_tree;

  gchar               *file_path; /* == NULL -> using default */
  time_t               file_mtime;

  GtkWidget           *menu;
};


G_MODULE_EXPORT XfceDesktopMenu*
xfce_desktop_menu_new_impl (const gchar *menu_file,
                            gboolean     deferred)
{
  XfceDesktopMenu *desktop_menu;
  GtkSettings     *settings;
  GParamSpec      *pspec;
  GError          *error = NULL;
  gchar           *system_file;

  /* install xfdesktop-menu-show-icons property */
  settings = gtk_settings_get_default ();
  if (g_object_class_find_property (G_OBJECT_GET_CLASS (settings), "xfdesktop-menu-show-icons") == NULL)
    {
      pspec = g_param_spec_boolean ("xfdesktop-menu-show-icons",
                                    "Show icons in the desktop menu",
                                    "Show icons in the desktop menu",
                                    TRUE, G_PARAM_READWRITE);
      gtk_settings_install_property (pspec);
    }

  /* try to locate xfce4.menu */
  system_file = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, "menus/xfce4.menu");
  if (G_UNLIKELY (system_file == NULL))
    {
      g_warning ("Unable to locate xfce4.menu");
      return NULL;
    }

  /* try to load xfce4.menu */
  desktop_menu = g_new0 (XfceDesktopMenu, 1);
  desktop_menu->system_tree = xdg_menu_load (system_file, &error);
  if (G_UNLIKELY (desktop_menu->system_tree == NULL))
    {
      g_warning ("Unable to parse %s: %s", system_file, error->message);
      g_free (desktop_menu);
      g_free (system_file);
      g_error_free (error);
      return NULL;
    }
  g_free (system_file);

  if (menu_file != NULL)
    desktop_menu->file_path = g_strdup (menu_file);
  else
    desktop_menu->file_path = NULL;

  desktop_menu_rebuild (desktop_menu);

  return desktop_menu;
}


static void
desktop_menu_rebuild (XfceDesktopMenu *desktop_menu)
{
  XdgDesktopCache *cache;
  struct stat      sb;
  GError          *error = NULL;
  gchar           *filename;

  if (desktop_menu->file_path != NULL)
    filename = g_strdup (desktop_menu->file_path);
  else
    filename = desktop_menu_get_menu_file ();

  if (stat (filename, &sb) == 0)
    desktop_menu->file_mtime = sb.st_mtime;

  if (G_LIKELY (desktop_menu->menu != NULL))
    {
      gtk_widget_destroy (desktop_menu->menu);
      desktop_menu->menu = NULL;
    }

  cache = xdg_desktop_cache_new ();

  desktop_menu->menu = xfce_menu_new_from_parsed (desktop_menu->system_tree, cache);

  if (!desktop_menu_parse (filename,
                           desktop_menu->system_tree,
                           cache,
                           desktop_menu->menu,
                           FALSE,
                           &error))
    {
      g_warning ("Unable to parse %s: %s",
                 filename, error->message);
      g_error_free (error);
    }

  g_object_unref (G_OBJECT (cache));

  g_free (filename);
}


G_MODULE_EXPORT GtkWidget*
xfce_desktop_menu_get_widget_impl (XfceDesktopMenu *desktop_menu)
{
  g_return_val_if_fail (desktop_menu != NULL, NULL);
  return desktop_menu->menu;
}



G_MODULE_EXPORT const gchar*
xfce_desktop_menu_get_menu_file_impl (XfceDesktopMenu *desktop_menu)
{
  g_return_val_if_fail (desktop_menu != NULL, NULL);
  if (desktop_menu->file_path != NULL)
    return desktop_menu->file_path;
  else
    return desktop_menu_get_menu_file ();
}



G_MODULE_EXPORT gboolean
xfce_desktop_menu_need_update_impl (XfceDesktopMenu *desktop_menu)
{
  struct stat sb;
  gboolean    result = FALSE;
  gchar      *filename;

  if (desktop_menu->file_path != NULL)
    filename = g_strdup (desktop_menu->file_path);
  else
    filename = desktop_menu_get_menu_file ();

  if (stat (filename, &sb) == 0 && sb.st_mtime != desktop_menu->file_mtime)
    result = TRUE;

  g_free (filename);

  return result;
}



G_MODULE_EXPORT void
xfce_desktop_menu_start_autoregen_impl (XfceDesktopMenu *desktop_menu,
                                        guint            delay)
{
}



G_MODULE_EXPORT void
xfce_desktop_menu_stop_autoregen_impl (XfceDesktopMenu *desktop_menu)
{
}



G_MODULE_EXPORT void
xfce_desktop_menu_force_regen_impl (XfceDesktopMenu *desktop_menu)
{
  desktop_menu_rebuild (desktop_menu);
}



G_MODULE_EXPORT void
xfce_desktop_menu_set_show_icons_impl (XfceDesktopMenu *desktop_menu,
                                       gboolean         show_icons)
{
  GtkSettings *settings;

  settings = gtk_settings_get_default ();
  g_object_set (G_OBJECT (settings),
                "xfdesktop-menu-show-icons", show_icons,
                NULL);
}



G_MODULE_EXPORT void
xfce_desktop_menu_destroy_impl (XfceDesktopMenu *desktop_menu)
{
}



static gchar*
desktop_menu_get_menu_file (void)
{
  const gchar *userhome;
	XfceKiosk   *kiosk;
	gboolean     user_menu;
  gchar      **all_dirs;
  gchar       *menu_file;
	gchar        filename[PATH_MAX];
  gchar        searchpath[PATH_MAX * 3 + 2];
	gint         i;

	kiosk = xfce_kiosk_new ("xfdesktop");
	user_menu = xfce_kiosk_query (kiosk, "UserMenu");
	xfce_kiosk_free (kiosk);
	
	if (!user_menu)
    {
		  userhome = xfce_get_homedir ();
		  all_dirs = xfce_resource_lookup_all (XFCE_RESOURCE_CONFIG, "xfce4/desktop/");
		
      for (i = 0; all_dirs[i]; i++)
        {
          if (strstr (all_dirs[i], userhome) != all_dirs[i])
            {
              g_snprintf (searchpath, PATH_MAX * 3 + 2, "%s%%F.%%L:%s%%F.%%l:%s%%F",
                          all_dirs[i], all_dirs[i], all_dirs[i]);
              if (xfce_get_path_localized (filename, PATH_MAX, searchpath,
                                           "menu.xml", G_FILE_TEST_IS_REGULAR))
                {
                  g_strfreev (all_dirs);
                  return g_strdup (filename);
                }
            }			
    		}
		
      g_strfreev(all_dirs);
    }
  else
    {
		  menu_file = xfce_resource_save_location (XFCE_RESOURCE_CONFIG,
                                               "xfce4/desktop/menu.xml", FALSE);
		  if (menu_file != NULL && g_file_test (menu_file, G_FILE_TEST_IS_REGULAR))
			  return menu_file;
			g_free (menu_file);
		
      all_dirs = xfce_resource_lookup_all (XFCE_RESOURCE_CONFIG,
                                           "xfce4/desktop/");

      for (i = 0; all_dirs[i]; i++)
        {
          g_snprintf (searchpath, PATH_MAX * 3 + 2, "%s%%F.%%L:%s%%F.%%l:%s%%F",
                      all_dirs[i], all_dirs[i], all_dirs[i]);
          if (xfce_get_path_localized (filename, PATH_MAX, searchpath,
                                       "menu.xml", G_FILE_TEST_IS_REGULAR))
            {
              g_strfreev (all_dirs);
              return g_strdup (filename);
            }		
        } 
      
      g_strfreev(all_dirs);
	  }

  g_warning ("%s: Could not locate a menu definition file", PACKAGE);

  return NULL;
}


static void
xdg_migrate_config (const gchar *filename)
{
	gchar *old_file;
  gchar *new_file;
  gchar  new_loc[PATH_MAX];
				gchar *contents = NULL;
				gsize len = 0;
FILE *fp;

	g_snprintf (new_loc, PATH_MAX, "xfce4/desktop/%s", filename);
	
	new_file = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, new_loc, FALSE);

	/* if the new file _does_ exist, assume we've already migrated */
	if (!g_file_test(new_file, G_FILE_TEST_IS_REGULAR))
    {
		  old_file = xfce_get_userfile (filename, NULL);
		  if (g_file_test (old_file, G_FILE_TEST_IS_REGULAR))
        {
    			/* we have to run it again to make sure the directory exists */
		    	g_free (new_file);
    			new_file = xfce_resource_save_location (XFCE_RESOURCE_CONFIG,
                                                  new_loc, TRUE);
			
    			/* try atomic move first, if not, resort to read->write->delete */
		    	if (link (old_file, new_file) == 0)
            {
      				unlink (old_file);
            }
		    	else
            {
				      if (g_file_get_contents (old_file, &contents, &len, NULL))
                {
                  fp = fopen(new_file, "w");
                  if (fp != NULL)
                    {
                      if (fwrite(contents, len, 1, fp) == len)
                        {
                          fclose (fp);
                          unlink (old_file);
                        }
                      else
                        {
                          fclose (fp);
                          g_critical ("XfceDesktopMenu: Unable to migrate %s to new location (error writing to file)", filename);
                        }
                    }
                  else
                    {
                    g_critical ("XfceDesktopMenu: Unable to migrate %s to new location (error opening target file for writing)", filename);
                    }
                }
              else
                {
                  g_critical ("XfceDesktopMenu: Unable to migrate %s to new location (error reading old file)", filename);
                }
            }
        }
      
      g_free (old_file);
    }
	
  g_free (new_file);	
}


G_MODULE_EXPORT gchar*
g_module_check_init (GModule *module)
{
  /* keep the module in memory at any rate! */
  g_module_make_resident (module);

  /* move menu.xml to new XDG location */
  xdg_migrate_config ("menu.xml");
  return NULL;
}
