/* $Id: xfce-menu.c,v 1.6 2004/08/08 15:31:06 bmeurer Exp $ */
/*-
 * Copyright (c) 2004 os-cillation
 * All rights reserved.
 *
 * Written by Benedikt Meurer <bm@os-cillation.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <libxfce4util/libxfce4util.h>

#include "xfce-menu.h"
#include "xfce-menu-entry-item.h"



static void     xfce_menu_class_init    (XfceMenuClass    *klass);
static void     xfce_menu_finalize      (GObject          *object);
static gboolean xfce_menu_entry_match   (XfceMenu         *menu,
                                         XdgDesktopEntry  *entry,
                                         gboolean          only_allocated);
static void     xfce_menu_entry_unlink  (XfceMenu         *menu,
                                         XdgDesktopEntry  *entry);
static void     xfce_menu_entry_added   (XdgDesktopDir    *appdir,
                                         XdgDesktopEntry  *entry,
                                         XfceMenu         *menu);
static void     xfce_menu_entry_removed (XdgDesktopDir    *appdir, 
                                         XdgDesktopEntry  *entry,
                                         XfceMenu         *menu);
static void     xfce_menu_insert        (GtkMenuShell     *menu_shell,
                                         GtkWidget        *child,
                                         gint              position);



static const gchar *directory_keywords[] =
{
  "Name", "Icon"
};



GType
xfce_menu_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo type_info = {
        sizeof (XfceMenuClass),
        NULL,
        NULL,
        (GClassInitFunc) xfce_menu_class_init,
        NULL,
        NULL,
        sizeof (XfceMenu),
        0,
        NULL,
        NULL,
      };

      type = g_type_register_static (GTK_TYPE_MENU,
                                     "XfceMenu",
                                     &type_info,
                                     0);
    }

  return type;
}



GtkWidget*
xfce_menu_new (const gchar *name,
               const gchar *icon)
{
  XfceMenu *menu;

  menu       = g_object_new (XFCE_TYPE_MENU, NULL);
  menu->name = g_strdup (name);
  menu->icon = g_strdup (icon);

  return (GtkWidget *) menu;
}



GtkWidget*
xfce_menu_new_from_parsed (const XdgMenu   *parsed_menu,
                           XdgDesktopCache *cache)
{
  XfceDesktopEntry  *dentry;
  XdgDesktopDir     *appdir;
  XfceMenu          *menu;
  gchar             *directory;
  GList             *lp;

  menu                   = g_object_new (XFCE_TYPE_MENU, NULL);
  menu->include          = xdg_pattern_copy (parsed_menu->include);
  menu->exclude          = xdg_pattern_copy (parsed_menu->exclude);
  menu->only_unallocated = parsed_menu->only_unallocated;

  /* read information from .directory file if any */
  directory = xdg_menu_get_directory (parsed_menu);
  if (G_LIKELY (directory != NULL))
    {
      dentry = xfce_desktop_entry_new (directory,
                                       directory_keywords,
                                       G_N_ELEMENTS (directory_keywords));

      if (G_LIKELY (dentry != NULL))
        {
          xfce_desktop_entry_get_string (dentry, "Name", TRUE, &menu->name);
          xfce_desktop_entry_get_string (dentry, "Icon", TRUE, &menu->icon);
          g_object_unref (dentry);
        }
      g_free (directory);
    }

  if (menu->name == NULL)
    menu->name = g_strdup (parsed_menu->name);

  /* setup AppDir listeners */
  for (lp = parsed_menu->appdirs; lp != NULL; lp = lp->next)
    {
      appdir = xdg_desktop_cache_append_dir (cache,
                                             XDG_APP_DIR (lp->data)->path,
                                             XDG_APP_DIR (lp->data)->prefix);

      menu->appdirs = g_list_append (menu->appdirs, appdir);

      g_signal_connect (G_OBJECT (appdir), "added",
                        G_CALLBACK (xfce_menu_entry_added), menu);
      g_signal_connect (G_OBJECT (appdir), "removed",
                        G_CALLBACK (xfce_menu_entry_removed), menu);
    }

  return (GtkWidget *) menu;
}



void
xfce_menu_append_submenu (XfceMenu  *menu,
                          GtkWidget *submenu)
{
  GtkWidget *item;
  
  g_return_if_fail (XFCE_IS_MENU (submenu));

  item = xfce_menu_item_new (XFCE_MENU (submenu)->name,
                             XFCE_MENU (submenu)->icon,
                             NULL);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), submenu);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  XFCE_MENU (submenu)->hook = item;
}



static void
xfce_menu_class_init (XfceMenuClass *klass)
{
  G_OBJECT_CLASS (klass)->finalize      = xfce_menu_finalize;
  GTK_MENU_SHELL_CLASS (klass)->insert  = xfce_menu_insert;
}



static void
xfce_menu_finalize (GObject *object)
{
  GObjectClass *parent_class;

  parent_class = g_type_class_peek_parent (XFCE_MENU_GET_CLASS (object));
  parent_class->finalize (object);
}



static gboolean
xfce_menu_entry_match (XfceMenu        *menu,
                       XdgDesktopEntry *entry,
                       gboolean         only_allocated)
{
  GtkWidget   *submenu;
  GtkWidget   *item;
  GList       *lp;
  gint         n;

  /* check if one of our children matches */
  for (lp = GTK_MENU_SHELL (menu)->children; lp != NULL; lp = lp->next)
    {
      submenu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (lp->data));
      if (!XFCE_IS_MENU (submenu))
        continue;

      if (XFCE_MENU (submenu)->appdirs == NULL
          && xfce_menu_entry_match (XFCE_MENU (submenu), entry, only_allocated))
        {
          return TRUE;
        }
    }

  if (only_allocated)
    {
      if (G_UNLIKELY (menu->only_unallocated))
        return FALSE;
      
      /* check include */
      if (menu->include == NULL || !xdg_pattern_match (menu->include, entry))
        return FALSE;

      /* check exclude */
      if (menu->exclude != NULL && xdg_pattern_match (menu->exclude, entry))
        return FALSE;
    }
  else
    {
      if (!menu->only_unallocated)
        return FALSE;
    }

  /* ok, we got a match */
  for (n = 0, lp = GTK_MENU_SHELL (menu)->children; lp; lp = lp->next, ++n)
    if (gtk_menu_item_get_submenu (GTK_MENU_ITEM (lp->data)) == NULL)
      break;
  for (; lp != NULL; lp = lp->next, ++n)
    if (strcmp (entry->name, xfce_menu_item_get_text (XFCE_MENU_ITEM (lp->data))) < 0)
      break;

  item = xfce_menu_entry_item_new (entry);
  gtk_menu_shell_insert (GTK_MENU_SHELL (menu), item, n);
  gtk_widget_show (item);

  return TRUE;
}



static void
xfce_menu_entry_unlink (XfceMenu        *menu,
                        XdgDesktopEntry *entry)
{
  GtkWidget *submenu;
  GList     *children;
  GList     *lp;

  children = gtk_container_get_children (GTK_CONTAINER (menu));

  for (lp = children; lp != NULL; lp = lp->next)
    {
      if (XFCE_IS_MENU_ENTRY_ITEM (lp->data)
          && XFCE_MENU_ENTRY_ITEM (lp->data)->entry == entry)
        {
          gtk_widget_destroy (GTK_WIDGET (lp->data));
        }
      else
        {
          submenu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (lp->data));
          if (XFCE_IS_MENU (submenu))
            xfce_menu_entry_unlink (XFCE_MENU (submenu), entry);
        }
    }

  g_list_free (children);
}



static void
xfce_menu_entry_added (XdgDesktopDir    *appdir, 
                       XdgDesktopEntry  *entry,
                       XfceMenu         *menu)
{
  if (!xfce_menu_entry_match (menu, entry, TRUE))
    {
      /* no match, so its "unallocated" */
      xfce_menu_entry_match (menu, entry, FALSE);
    }
}



static void
xfce_menu_entry_removed (XdgDesktopDir    *appdir, 
                         XdgDesktopEntry  *entry,
                         XfceMenu         *menu)
{
  xfce_menu_entry_unlink (menu, entry);
}



static void
xfce_menu_insert (GtkMenuShell *menu_shell,
                  GtkWidget    *child,
                  gint          position)
{
  GtkMenuShellClass *klass;
  XfceMenu          *menu;

  klass = g_type_class_peek_parent (XFCE_MENU_GET_CLASS (menu_shell));
  klass->insert (menu_shell, child, position);

  /* make sure this menu item is accessible */
  if (XFCE_IS_MENU_APP_ITEM (child))
    {
      menu = XFCE_MENU (menu_shell);
      while (menu != NULL && menu->hook != NULL)
        {
          gtk_widget_show (menu->hook);
          menu = XFCE_MENU (gtk_widget_get_parent (menu->hook));
        }
    }
}



