// Directory list. Taken from the FOX library and slightly modified.

#include "config.h"
#include "i18n.h"

#include <fox/fx.h>
#include <fox/FXPNGIcon.h>

#include "icons.h"
#include "DirList.h"
#include "InputDialog.h"
#include "File.h"
#include "FileDict.h"
#include "MessageBox.h"

// Root directory string
#ifndef ROOTDIR
#define ROOTDIR "/"
#endif

// Maximum length of a file name
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

// Interval between refreshes
#define REFRESHINTERVAL     1000
#define REFRESHINTERVALLONG 15000

// File systems not supporting mod-time, refresh every nth time
#define REFRESHFREQUENCY    32

// Time interval before expanding a folder
#define EXPANDINTERVAL		500

extern FXbool ask_before_copy;

FXPNGIcon  *locked_folder;          // Locked folder icon
TreeItem *prevSelItem;

#if defined(linux)
extern FXStringDict* devices;
#endif

// Object implementation
FXIMPLEMENT(DirItem,FXTreeItem,NULL,0)



// Map
FXDEFMAP(DirList) DirListMap[]={
                                   FXMAPFUNC(SEL_DRAGGED,0,DirList::onDragged),
                                   FXMAPFUNC(SEL_TIMEOUT,DirList::ID_REFRESH,DirList::onRefresh),
                                   FXMAPFUNC(SEL_TIMEOUT,DirList::ID_EXPANDTIMER,DirList::onExpandTimer),
                                   FXMAPFUNC(SEL_DND_ENTER,0,DirList::onDNDEnter),
                                   FXMAPFUNC(SEL_DND_LEAVE,0,DirList::onDNDLeave),
                                   FXMAPFUNC(SEL_DND_DROP,0,DirList::onDNDDrop),
                                   FXMAPFUNC(SEL_DND_MOTION,0,DirList::onDNDMotion),
                                   FXMAPFUNC(SEL_DND_REQUEST,0,DirList::onDNDRequest),
                                   FXMAPFUNC(SEL_BEGINDRAG,0,DirList::onBeginDrag),
                                   FXMAPFUNC(SEL_ENDDRAG,0,DirList::onEndDrag),
                                   FXMAPFUNC(SEL_OPENED,0,DirList::onOpened),
                                   FXMAPFUNC(SEL_CLOSED,0,DirList::onClosed),
                                   FXMAPFUNC(SEL_EXPANDED,0,DirList::onExpanded),
                                   FXMAPFUNC(SEL_COLLAPSED,0,DirList::onCollapsed),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SHOW_HIDDEN,DirList::onUpdShowHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_HIDE_HIDDEN,DirList::onUpdHideHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_TOGGLE_HIDDEN,DirList::onUpdToggleHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SHOW_FILES,DirList::onUpdShowFiles),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_HIDE_FILES,DirList::onUpdHideFiles),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_TOGGLE_FILES,DirList::onUpdToggleFiles),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SET_PATTERN,DirList::onUpdSetPattern),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SORT_REVERSE,DirList::onUpdSortReverse),
                                   FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,DirList::onCmdSetValue),
                                   FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,DirList::onCmdSetStringValue),
                                   FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE,DirList::onCmdGetStringValue),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SHOW_HIDDEN,DirList::onCmdShowHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_HIDE_HIDDEN,DirList::onCmdHideHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_TOGGLE_HIDDEN,DirList::onCmdToggleHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SHOW_FILES,DirList::onCmdShowFiles),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_HIDE_FILES,DirList::onCmdHideFiles),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_TOGGLE_FILES,DirList::onCmdToggleFiles),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SET_PATTERN,DirList::onCmdSetPattern),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SORT_REVERSE,DirList::onCmdSortReverse),
                               };


// Object implementation
FXIMPLEMENT(DirList,FXTreeList,DirListMap,ARRAYNUMBER(DirListMap))


// For serialization
DirList::DirList()
{
    flags|=FLAG_ENABLED|FLAG_DROPTARGET;
    refresh=NULL;
    expandtimer=NULL;
    counter=0;
    associations=NULL;
    dropaction=DRAG_MOVE;
}


// Directory List Widget
DirList::DirList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXTreeList(p,0,tgt,sel,opts,x,y,w,h),pattern("*")
{
    flags|=FLAG_ENABLED|FLAG_DROPTARGET;
    associations=NULL;
    open_folder=new FXPNGIcon(getApp(),minifolderopen);
    closed_folder=new FXPNGIcon(getApp(),minifolderclosed);
    mini_doc=new FXPNGIcon(getApp(),minidoc);
    mini_app=new FXPNGIcon(getApp(),miniapp);
    locked_folder=new FXPNGIcon(getApp(),minifolderlocked);
    harddiskicon=new FXPNGIcon(getApp(),harddisk);
    linkicon=new FXPNGIcon(getApp(),minilink);
    matchmode=FILEMATCH_FILE_NAME|FILEMATCH_NOESCAPE;
    if(!(options&DIRLIST_NO_OWN_ASSOC))
        associations=new FileDict(getApp());
    sortfunc=(FXTreeListSortFunc)cmpFName;
    refresh=NULL;
    expandtimer=NULL;
    dropaction=DRAG_MOVE;
    counter=0;
    prevSelItem = NULL;
}


// Create the directory list
void DirList::create()
{
    FXTreeList::create();
    if(!deleteType)
        deleteType=getApp()->registerDragType(deleteTypeName);
    if(!urilistType)
        urilistType=getApp()->registerDragType(urilistTypeName);
    if(!refresh)
        refresh=getApp()->addTimeout(REFRESHINTERVAL,this,ID_REFRESH);
    dropEnable();
    open_folder->create();
    closed_folder->create();
    mini_doc->create();
    mini_app->create();
    locked_folder->create();
    harddiskicon->create();
    linkicon->create();
    scanRootDir(FALSE);
}


// Detach disconnects the icons
void DirList::detach()
{
    FXTreeList::detach();
    if(refresh)
        refresh=getApp()->removeTimeout(refresh);
    if(expandtimer)
        expandtimer=getApp()->removeTimeout(expandtimer);
    open_folder->detach();
    closed_folder->detach();
    mini_doc->detach();
    mini_app->detach();
    locked_folder->detach();
    deleteType=0;
    urilistType=0;
}


// Destroy zaps the icons
void DirList::destroy()
{
    FXTreeList::destroy();
    if(refresh)
        refresh=getApp()->removeTimeout(refresh);
    if(expandtimer)
        expandtimer=getApp()->removeTimeout(expandtimer);
    open_folder->destroy();
    closed_folder->destroy();
    mini_doc->destroy();
    mini_app->destroy();
    locked_folder->destroy();
}

// Expand folder tree when hovering long over a folder
long DirList::onExpandTimer(FXObject* sender,FXSelector sel,void* ptr)
{
    FXint xx,yy;
    FXuint state;
    DirItem *item;

    expandtimer=NULL;
    getCursorPosition(xx,yy,state);
    item=(DirItem*)getItemAt(xx,yy);

    FXchar path[MAXPATHLEN];
    FXbool changed;
    if(!(item->state&DirItem::FOLDER))
        return 0;

    // Expand tree item
    expandTree(item,TRUE);

    // Get path to item
    getpath((TreeItem*)item,path);

    // List stuff in item directory
    changed=listSubDir(item,path);

    // Sort items
    if(changed)
        sortChildItems(item);

    // Set open timer
    expandtimer=getApp()->addTimeout(EXPANDINTERVAL,this,ID_EXPANDTIMER);

    return 1;

}

// Create item
TreeItem* DirList::createItem(const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr)
{
    return (TreeItem*) new DirItem(text,oi,ci,ptr);
}


// Compare file names
FXint DirList::cmpFName(const TreeItem* a,const TreeItem* b)
{
    return compare(a->label,b->label);
}


// Reversed compare file names
FXint DirList::cmpRName(const TreeItem* a,const TreeItem* b)
{
    return -DirList::cmpFName(a,b);
}

// FIXME This needs to be redone using FXFile

// Helper function
FXchar *DirList::getpath(const TreeItem* item,FXchar* pathname) const
{
    register FXchar *ptr=pathname;
    register const FXchar *p;
    register FXuint ss=0;
    const FXchar *stack[100];
    FXASSERT(pathname);
    while(item)
    {
        stack[ss++]=item->label.text();
        item=(TreeItem*)item->parent;
    }
    FXASSERT(ss<100);
    if(ss)
    {
        p=stack[--ss];
        FXASSERT(p);
        while(*p)
            *ptr++=*p++;
        if(ss)
        {
            while(1)
            {
                p=stack[--ss];
                FXASSERT(p);
                while(*p)
                    *ptr++=*p++;
                if(!ss)
                    break;
                *ptr++=PATHSEP;
            }
        }
    }
    *ptr='\0';
    FXASSERT((ptr-pathname)<MAXPATHLEN);
    return pathname;
}

// FIXME This needs to be redone using FXFile
// FIXME also use new findItem() API's!!

// Helper function:- this is one ugly mother
TreeItem* DirList::getitem(FXchar* pathname)
{
    FXchar buffer[MAXPATHLEN];
    register const FXchar *ptr=pathname;
    register FXchar *p;
    FXbool changed;
    TreeItem *item,*it;
    recalc();
    update();
    if(!firstitem)
        scanRootDir(FALSE);
    if(!firstitem)
        return NULL;
    ptr=(FXchar*)strchr(pathname,PATHSEP);
    if(!ptr)
        return (TreeItem*)firstitem;
    ptr=ptr+1;
    p=(FXchar*)strchr(ptr,PATHSEP);
    if(p)
        *p=0;
    item=(TreeItem*)firstitem;
    while(*ptr)
    {
        for(it=(TreeItem*)(item->first); it; it=(TreeItem*)(it->next))
        {   // First try with what we've got
            if(compare(ptr,it->label)==0)
                goto x;
        }
        getpath(item,buffer);
        changed=listSubDir((DirItem*)item,buffer);      // Didn't find it, try relist
        if(!changed)
            goto ret;                         // No change, its not there!
        if(!item->first)                                  // Now have knowledge of subitems
            item->state&=~DirItem::HASITEMS;
        else
            item->state|=DirItem::HASITEMS;
        //////// Missing update here somewhere...
        sortChildItems(item);                             // Sort items
        for(it=(TreeItem*)(item->first); it; it=(TreeItem*)(it->next))
        {             // Try find it again with new list
            if(compare(ptr,it->label)==0)
                goto x;
        }
        goto ret;                                      // No luck!
x:
        item=it;
        if(!p)
            break;
        ptr=p+1;
        if(*ptr==0)
            break;
        p=(FXchar*)strchr(ptr,PATHSEP);
        if(p)
            *p=0;
    }

// Recompute item position before leaving
ret:
	recompute();
    return item;
}


// Handle drag-and-drop enter
long DirList::onDNDEnter(FXObject* sender,FXSelector sel,void* ptr)
{
    FXTreeList::onDNDEnter(sender,sel,ptr);
    return 1;
}


// Handle drag-and-drop leave
long DirList::onDNDLeave(FXObject* sender,FXSelector sel,void* ptr)
{
    // Cancel open up timer
    if(expandtimer)
        expandtimer=getApp()->removeTimeout(expandtimer);

    stopAutoScroll();
    FXTreeList::onDNDLeave(sender,sel,ptr);
    if(prevSelItem)
    {
        if(!isItemCurrent(prevSelItem))
            closeItem(prevSelItem);
        prevSelItem = NULL;
    }
    return 1;
}


// Handle drag-and-drop motion
long DirList::onDNDMotion(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    TreeItem *item;

    // Cancel open up timer
    if(expandtimer)
        expandtimer=getApp()->removeTimeout(expandtimer);

    // Start autoscrolling
    if(startAutoScroll(event->win_x,event->win_y,FALSE))
        return 1;

    // Give base class a shot
    if(FXTreeList::onDNDMotion(sender,sel,ptr))
        return 1;

    // Dropping list of filenames
    if(offeredDNDType(FROM_DRAGNDROP,urilistType))
    {

        // Locate drop place
        item=(TreeItem*)getItemAt(event->win_x,event->win_y);

        // We can drop in a directory
        if(item && isItemDirectory(item))
        {
            // Get drop directory
            dropdirectory=getItemPathname(item);

            // What is being done (move,copy,link)
            dropaction=inquireDNDAction();

            // Set open up timer
            expandtimer=getApp()->addTimeout(EXPANDINTERVAL,this,ID_EXPANDTIMER);

            // See if this is writable
            if(FXFile::isWritable(dropdirectory))
            {
                acceptDrop(DRAG_ACCEPT);
                FXint x,y;
                FXuint state;
                getCursorPosition(x,y,state);
                TreeItem* item=(TreeItem*)getItemAt(x,y);

                if(prevSelItem && prevSelItem != item)
                {
                    if(!isItemCurrent(prevSelItem))
                        closeItem(prevSelItem);
                    prevSelItem = NULL;
                }
                if(item && prevSelItem != item)
                {
                    openItem(item);
                    prevSelItem = item;
                }
            }
        }
        return 1;
    }
    return 0;
}


// Handle drag-and-drop drop
long DirList::onDNDDrop(FXObject* sender,FXSelector sel,void* ptr)
{
    FXuchar *data;
    FXuint len;
    FXbool showdialog=TRUE;
    File *f=NULL;

    if(prevSelItem)
    {
        if(!isItemCurrent(prevSelItem))
            closeItem(prevSelItem);
        prevSelItem = NULL;
    }

    // Cancel open up timer
    if(expandtimer)
        expandtimer=getApp()->removeTimeout(expandtimer);

    // Stop scrolling
    stopAutoScroll();

    // Perhaps target wants to deal with it
    if(FXTreeList::onDNDDrop(sender,sel,ptr))
        return 1;

    // Get uri-list of files being dropped
    if(getDNDData(FROM_DRAGNDROP,urilistType,data,len))
    {
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
        FXchar *p,*q;
        p=q=(FXchar*)data;
        FXString buf=p;
        int num=buf.count('\n')+1; // number of selected items

        // File object
        if (dropaction==DRAG_COPY)
            f=new File(getApp(),_("File copy"),COPY);
        else if (dropaction==DRAG_MOVE)
            f=new File(getApp(),_("File move"),MOVE);
        else
            return 0;

        while(*p)
        {
            while(*q && *q!='\r')
                q++;
            FXString url(p,q-p);
            FXString source(FXURL::fileFromURL(url));
            FXString target(dropdirectory+PATHSEPSTRING+FXFile::name(source));
            FXString sourcedir=FXFile::directory(source);
            FXString targetdir=FXFile::directory(target);

            // File operation dialog, if needed
            if(ask_before_copy & showdialog)
            {
                FXIcon *icon=NULL;
                FXString title,message;
                if (dropaction==DRAG_COPY)
                {
                    title=_("Copy ");
                    icon = new FXPNGIcon(getApp(),copy_big);
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Copy ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if (dropaction==DRAG_MOVE)
                {
                    title=_("Move ");
                    icon = new FXPNGIcon(getApp(),move_big);
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Move ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }

                InputDialog* dialog = new InputDialog(this,targetdir,message,title,_("To:"),icon);
                dialog->CursorEnd();
                int rc=1;
                rc=dialog->execute();
                target=dialog->getText();
                target=::filePath(target);
                if (num>1)
                    showdialog=FALSE;
                delete dialog;
                delete icon;
                if (!rc)
                    return 0;
            }

            // Move or copy the source file
            if(dropaction==DRAG_MOVE)
            {
                // Move file
                f->create();
                f->move(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Move file operation cancelled!"));
                }
            }
            else if(dropaction==DRAG_COPY)
            {
                // Copy file
                f->create();
                f->copy(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Copy file operation cancelled!"));
                }
            }
            if(*q=='\r')
                q+=2;
            p=q;
        }
        delete f;
        FXFREE(&data);
        return 1;
    }
    return 0;
}


// Somebody wants our dragged data
long DirList::onDNDRequest(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXuchar *data;
    FXuint len;

    // Perhaps the target wants to supply its own data
    if(FXTreeList::onDNDRequest(sender,sel,ptr))
        return 1;

    // Return list of filenames as a uri-list
    if(event->target==urilistType)
    {
        if(!dragfiles.empty())
        {
            len=dragfiles.length();
            FXMEMDUP(&data,FXuchar,dragfiles.text(),len);
            setDNDData(FROM_DRAGNDROP,event->target,data,len);
        }
        return 1;
    }

    // Delete selected files
    if(event->target==deleteType)
        return 1;

    return 0;
}


// Start a drag operation
long DirList::onBeginDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    register TreeItem *item;
    if(FXTreeList::onBeginDrag(sender,sel,ptr))
        return 1;
    if(beginDrag(&urilistType,1))
    {
        dragfiles=FXString::null;
        item=(TreeItem*)firstitem;
        while(item)
        {
            if(item->isSelected())
            {
                if(!dragfiles.empty())
                    dragfiles+="\r\n";
                dragfiles+=FXURL::fileToURL(getItemPathname(item));
                //dragfiles+="\r\n";
            }
            if(item->first)
                item=(TreeItem*)item->first;
            else
            {
                while(!item->next && item->parent)
                    item=(TreeItem*)item->parent;
                item=(TreeItem*)item->next;
            }
        }
        return 1;
    }
    return 0;
}


// End drag operation
long DirList::onEndDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    if(FXTreeList::onEndDrag(sender,sel,ptr))
        return 1;
    endDrag((didAccept()!=DRAG_REJECT));
    setDragCursor(getDefaultCursor());
    return 1;
}


// Dragged stuff around
long DirList::onDragged(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    FXDragAction action;
    if(FXTreeList::onDragged(sender,sel,ptr))
        return 1;
    action=DRAG_MOVE;
    if(event->state&CONTROLMASK)
        action=DRAG_COPY;
    if(event->state&SHIFTMASK)
        action=DRAG_MOVE;
    handleDrag(event->root_x,event->root_y,action);
    if(didAccept()!=DRAG_REJECT)
    {
        if(action==DRAG_MOVE)
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDMOVE_CURSOR));
        else
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDCOPY_CURSOR));
    }
    else
        setDragCursor(getApp()->getDefaultCursor(DEF_DNDSTOP_CURSOR));
    return 1;
}


// Open up the path down to the given string
long DirList::onCmdSetValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr)
        setCurrentFile((const FXchar*)ptr);
    return 1;
}


// Open up the path down to the given string
long DirList::onCmdSetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr==NULL)
        fxerror("%s::onCmdSetStringValue: NULL pointer.\n",getClassName());
    setCurrentFile(*((FXString*)ptr));
    return 1;
}


// Obtain value of the current item
long DirList::onCmdGetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr==NULL)
        fxerror("%s::onCmdGetStringValue: NULL pointer.\n",getClassName());
    *((FXString*)ptr)=getCurrentFile();
    return 1;
}


// Toggle hidden files
long DirList::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(!showHiddenFiles());
    return 1;
}


// Update toggle hidden files widget
long DirList::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Show hidden files
long DirList::onCmdShowHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(TRUE);
    return 1;
}


// Update show hidden files widget
long DirList::onUpdShowHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Hide hidden files
long DirList::onCmdHideHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(FALSE);
    return 1;
}


// Update hide hidden files widget
long DirList::onUpdHideHidden(FXObject* sender,FXSelector,void*)
{
    if(!showHiddenFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Toggle files display
long DirList::onCmdToggleFiles(FXObject*,FXSelector,void*)
{
    showFiles(!showFiles());
    return 1;
}


// Update toggle files widget
long DirList::onUpdToggleFiles(FXObject* sender,FXSelector,void*)
{
    if(showFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Show files
long DirList::onCmdShowFiles(FXObject*,FXSelector,void*)
{
    showFiles(TRUE);
    return 1;
}


// Update show files widget
long DirList::onUpdShowFiles(FXObject* sender,FXSelector,void*)
{
    if(showFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Hide files
long DirList::onCmdHideFiles(FXObject*,FXSelector,void*)
{
    showFiles(FALSE);
    return 1;
}


// Update hide files widget
long DirList::onUpdHideFiles(FXObject* sender,FXSelector,void*)
{
    if(!showFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Change pattern
long DirList::onCmdSetPattern(FXObject*,FXSelector,void* ptr)
{
    if(!ptr)
        return 0;
    setPattern((const char*)ptr);
    return 1;
}


// Update pattern
long DirList::onUpdSetPattern(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)pattern.text());
    return 1;
}


// Reverse sort order
long DirList::onCmdSortReverse(FXObject*,FXSelector,void*)
{
    if(sortfunc==(FXTreeListSortFunc)cmpFName)
        sortfunc=(FXTreeListSortFunc)cmpRName;
    else if(sortfunc==(FXTreeListSortFunc)cmpRName)
        sortfunc=(FXTreeListSortFunc)cmpFName;
    scanRootDir(TRUE);
    return 1;
}


// Update sender
long DirList::onUpdSortReverse(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=(sortfunc==(FXTreeListSortFunc)cmpRName)?ID_CHECK:ID_UNCHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// FIXME look more carefully at this four handlers below...

// Close directory
long DirList::onClosed(FXObject*,FXSelector,void* ptr)
{
    DirItem *item=(DirItem*)ptr;
    if(item->state&DirItem::FOLDER)
        return target && target->handle(this,MKUINT(message,SEL_CLOSED),ptr);

    return 1;
}


// Open directory
long DirList::onOpened(FXObject*,FXSelector,void* ptr)
{
    DirItem *item=(DirItem*)ptr;
    if(item->state&DirItem::FOLDER)
        return target && target->handle(this,MKUINT(message,SEL_OPENED),ptr);
    return 1;
}


// Item opened
long DirList::onExpanded(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem *item=(DirItem*)ptr;
    FXchar path[MAXPATHLEN];
    FXbool changed;
    if(!(item->state&DirItem::FOLDER))
        return 0;

    // Expand tree item
    expandTree(item,TRUE);

    // Get path to item
    getpath((TreeItem*)item,path);

    // List stuff in item directory
    changed=listSubDir(item,path);

    // Now we know for sure whether we really have subitems or not
    if(!item->first)
        item->state&=~DirItem::HASITEMS;
    else
        item->state|=DirItem::HASITEMS;

    // Sort items
    if(changed)
        sortChildItems(item);
    return 1;
}


// Item closed
long DirList::onCollapsed(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem *item=(DirItem*)ptr;
    if(!(item->state&DirItem::FOLDER))
        return 0;

    // Collapse tree item
    collapseTree(item,TRUE);

    return 1;
}


// Refresh
long DirList::onRefresh(FXObject*,FXSelector,void*)
{
    FXbool changed;

    // Not necessary???
    // Only update if user is not interacting with the file list
    //if(flags&FLAG_UPDATE)
    //{
    changed=scanRootDir(FALSE);
    if(changed)
        recalc();
    counter=(counter+1)%REFRESHFREQUENCY;
    //}

    // Reset timer again
    refresh=getApp()->addTimeout(REFRESHINTERVAL,this,ID_REFRESH);

    return 0;
}


// List root directories
FXbool DirList::listRoots()
{
    if(firstitem)
        return FALSE;
    addItemLast(NULL,createItem(ROOTDIR,harddiskicon,harddiskicon,NULL));
    return TRUE;
}


// Scan root directory for changes
FXbool DirList::scanRootDir(FXbool relist)
{
    FXbool changed=FALSE;
    FXchar pathname[MAXPATHLEN];
    DirItem *item;
    long filetime;
    struct stat info;

    // Root directory name
    pathname[0]=PATHSEP;
    pathname[1]='\0';

    // Create root item if we don't have one yet
    if(!firstitem)
    {
        item=(DirItem*)createItem(pathname,harddiskicon,harddiskicon,NULL);
        item->parent=NULL;
        item->next=NULL;
        item->prev=NULL;
        item->inext=NULL;
        item->iprev=NULL;
        item->list=NULL;
        item->date=0;
        item->state=DirItem::FOLDER|DirItem::HASITEMS;
        item->first=NULL;
        item->last=NULL;
        firstitem=lastitem=item;
        changed=TRUE;
    }

    item=(DirItem*)firstitem;

    // Item is directory; check regardless of children
    if((item->state&DirItem::FOLDER) && (item->state&DirItem::EXPANDED))
    {

        // Refresh subitems of root if necessary
        if(stat(pathname,&info)==0)
        {
            filetime=FXMAX(info.st_mtime,info.st_ctime);
            if(relist || (item->date!=filetime) || ((filetime==0) && (counter==0)))
            {
                if(listSubDir(item,pathname))
                {
                    sortChildItems(item);
                    changed=TRUE;
                }
            }
            item->date=filetime;
        }

        // Then check subdirectories
        changed|=scanSubDir(item,pathname,relist);
    }
    return changed;
}


// Scan sub directories
FXbool DirList::scanSubDir(DirItem *par,FXchar *pathname,FXbool relist)
{
    FXbool changed=FALSE;
    FXchar *pathtail,*pathend;
    DirItem *item;
    long filetime;
    struct stat info;

    if(par->first)
    {

        // Build path prefix
        pathend=pathtail=pathname+strlen(pathname);
        if(!ISPATHSEP(*(pathtail-1)))
            *pathtail++=PATHSEP;

        // Loop over contents
        for(item=(DirItem*)par->first; item; item=(DirItem*)item->next)
        {

            // Item is directory; check regardless of children
            if((item->state&DirItem::FOLDER) && (item->state&DirItem::EXPANDED))
            {

                // Sub item path
                strcpy(pathtail,item->label.text());

                // Refresh subitems of item if necessary
                if(stat(pathname,&info)==0)
                {
                    filetime=FXMAX(info.st_mtime,info.st_ctime);
                    if(relist || (item->date!=filetime) || ((filetime==0) && (counter==0)))
                    {
                        if(listSubDir(item,pathname))
                        {
                            sortChildItems(item);
                            changed=TRUE;
                        }
                    }
                    item->date=filetime;
                }

                // Then check subdirectories
                changed|=scanSubDir(item,pathname,relist);
            }
        }
        *pathend='\0';
    }
    return changed;
}


// List subdirectories
FXbool DirList::listSubDir(DirItem *par,FXchar *pathname)
{
    DirItem *after,*before,*newlist,*item,*it;
    FXIcon *openicon,*closedicon;
    FXchar *pathtail,*pathend,*name;
    FXbool changed=FALSE;
    FileAssoc *fileassoc;
    long filetime;
    struct dirent *dp;
    struct stat info;
    DIR *dirp;
    int islink;

    // Find end of original pathname
    pathend=pathtail=pathname+strlen(pathname);

    // Build new insert-order list
    after=NULL;
    before=par->list;
    newlist=NULL;
    if(!par->first)
        before=NULL;

    // Get directory stream pointer
    dirp=opendir(pathname);

    // Managed to open directory
    if(dirp)
    {

        // Insert a '/' if needed
        if(!ISPATHSEP(*(pathtail-1)))
            *pathtail++=PATHSEP;

        // Process directory entries
        while((dp=readdir(dirp))!=NULL)
        {
            name=dp->d_name;

            // A dot special file?
            if(name[0]=='.' && (name[1]==0 || (name[1]=='.' && name[2]==0)))
                continue;

            // Hidden file or directory normally not shown
            if(name[0]=='.' && !(options&DIRLIST_SHOWHIDDEN))
                continue;

            // Build full pathname
            strcpy(pathtail,name);

            // Get File info
            if(lstat(pathname,&info)!=0)
                continue;

            // If its a link, get the file info
            islink=S_ISLNK(info.st_mode);
            if(islink && stat(pathname,&info)!=0)
                continue;

            // If it is not a directory, and not showing files and matching pattern skip it
            if(!S_ISDIR(info.st_mode) && !((options&DIRLIST_SHOWFILES) && fxfilematch(pattern.text(),name,matchmode)))
                continue;

            // File change/mod time
            filetime=info.st_mtime;
            if(filetime<0)
                filetime=0;

            // Find it
            for(item=before; item; item=item->inext)
            {
                if(compare(item->label,name)==0)
                {
                    while(before!=item)
                    {
                        it=before;
                        before=before->inext;

                        if(it->prev)
                            ((TreeItem*)it->prev)->next=it->next;
                        else
                            ((TreeItem*)it->parent)->first=it->next;

                        if(it->next)
                            ((TreeItem*)it->next)->prev=it->prev;
                        else
                            ((TreeItem*)it->parent)->last=it->prev;
                        removeItems(it->first,it->last);
                        if(currentitem==it)
                            currentitem=NULL;
                        if(anchoritem==it)
                            anchoritem=NULL;
                        changed=TRUE;
                        delete it;
                    }
                    before=item->inext;
                    if(before)
                        before->iprev=NULL;
                    item->inext=NULL;
                    item->iprev=NULL;
                    goto fnd;
                }
            }

            // Not found; prepend before list
            item=(DirItem*)createItem(name,open_folder,closed_folder,NULL);
            item->prev=par->last;
            item->next=NULL;
            item->parent=par;
            item->first=NULL;
            item->last=NULL;
            item->label=name;
            item->iprev=NULL;
            item->inext=NULL;
            item->list=NULL;
            item->state=DirItem::HASITEMS;
            item->date=0;
            if(item->prev)
                ((TreeItem*)item->prev)->next=item;
            else
                par->first=item;
            par->last=item;
            changed=TRUE;

            // Next gets hung after this one
fnd:
            item->iprev=after;
            if(after)
                after->inext=item;
            else
                newlist=item;

            // Fill item with updated information, if data has changed

            // Anything about the file changed?
            if((item->date!=filetime) || (filetime==0))
            {

                // Item flags
                if(info.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH))
                    item->state|=DirItem::EXECUTABLE;
                else
                    item->state&=~DirItem::EXECUTABLE;
                if(S_ISDIR(info.st_mode))
                    item->state|=DirItem::FOLDER;
                else
                    item->state&=~(DirItem::FOLDER|DirItem::HASITEMS);
                if(S_ISLNK(info.st_mode))
                    item->state|=DirItem::SYMLINK;
                else
                    item->state&=~DirItem::SYMLINK;
                if(S_ISCHR(info.st_mode))
                    item->state|=DirItem::CHARDEV;
                else
                    item->state&=~DirItem::CHARDEV;
                if(S_ISBLK(info.st_mode))
                    item->state|=DirItem::BLOCKDEV;
                else
                    item->state&=~DirItem::BLOCKDEV;
                if(S_ISFIFO(info.st_mode))
                    item->state|=DirItem::FIFO;
                else
                    item->state&=~DirItem::FIFO;
                if(S_ISSOCK(info.st_mode))
                    item->state|=DirItem::SOCK;
                else
                    item->state&=~DirItem::SOCK;


                // We can drag items
                item->state|=DirItem::DRAGGABLE;

                // Assume no associations
                fileassoc=NULL;

                // Determine icons and type
                if(item->state&DirItem::FOLDER)
                {
                    openicon=open_folder;
                    closedicon=closed_folder;
                    if(associations)
                        fileassoc=associations->findDirBinding(pathname);
                }
                else if(item->state&DirItem::EXECUTABLE)
                {
                    openicon=mini_app;
                    closedicon=mini_app;
                    if(associations)
                        fileassoc=associations->findExecBinding(pathname);
                }
                else
                {
                    openicon=mini_doc;
                    closedicon=mini_doc;
                    if(associations)
                        fileassoc=associations->findFileBinding(pathname);
                }

                // If association is found, use it
                if(fileassoc)
                {
                    if(fileassoc->miniicon)
                        closedicon=fileassoc->miniicon;
                    if(fileassoc->miniiconopen)
                        openicon=fileassoc->miniiconopen;
                }

                // Update item information
                item->openIcon=openicon;
                item->closedIcon=closedicon;
                item->size=(unsigned long)info.st_size;
                item->assoc=fileassoc;
                item->date=filetime;

#if defined(linux)
                // Devices have a specific icon
                if(devices->find(pathname))
                {
                    item->closedIcon=harddiskicon;
                    item->openIcon=harddiskicon;
                }
#endif

                // Symbolic links have a specific icon
                if(islink)
                {
                    item->closedIcon=linkicon;
                    item->openIcon=linkicon;
                }

                // Create item
                if(id())
                    item->create();

                // Info has changed
                changed=TRUE;
            }

            // Next one goes after item
            after=item;
        }

        // Close it
        closedir(dirp);
    }

    // Wipe items remaining in list:- they have disappeared!!
    while(before)
    {
        it=before;
        before=before->inext;
        if(it->prev)
            ((TreeItem*)it->prev)->next=it->next;
        else
            ((TreeItem*)it->parent)->first=it->next;
        if(it->next)
            ((TreeItem*)it->next)->prev=it->prev;
        else
            ((TreeItem*)it->parent)->last=it->prev;
        removeItems(it->first,it->last);
        if(currentitem==it)
            currentitem=NULL;
        if(anchoritem==it)
            anchoritem=NULL;
        changed=TRUE;
        delete it;
    }

    // Remember new list
    par->list=newlist;

    // Restore original path
    *pathend='\0';
    return changed;
}


// Is directory
FXbool DirList::isItemDirectory(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::isItemDirectory: item is NULL.\n",getClassName());
    return (item->state&DirItem::FOLDER)!=0;
}


// Is file
FXbool DirList::isItemFile(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::isItemFile: item is NULL.\n",getClassName());
    return (item->state&(DirItem::FOLDER|DirItem::CHARDEV|DirItem::BLOCKDEV|DirItem::FIFO|DirItem::SOCK))==0;
}


// Is executable
FXbool DirList::isItemExecutable(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::isItemExecutable: item is NULL.\n",getClassName());
    return (item->state&DirItem::EXECUTABLE)!=0;
}


// Obtain full pathname down from root
FXString DirList::getItemPathname(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::getItemPathname: item is NULL.\n",getClassName());
    if(!item->getParent())
        return item->getText();
    if(!item->getParent()->getParent())
        return item->getParent()->getText()+item->getText();
    return getItemPathname((TreeItem*)item->getParent())+PATHSEPSTRING+item->getText();
}


// Obtain item's file name only
FXString DirList::getItemFilename(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::getItemFilename: item is NULL.\n",getClassName());
    return item->label;
}


// Open all intermediate directories down toward given one
void DirList::setDirectory(const FXString& pathname)
{    // FIXME notify argument?
    TreeItem *item;
    if(!pathname.empty())
    {
        FXString path=FXFile::absolute(pathname);
        while(!FXFile::isTopDirectory(path) && !isDirectory(path))
            path=FXFile::upLevel(path);
        item=getitem((FXchar*)path.text());   // FIXME
        makeItemVisible(item);
        setCurrentItem(item);
    }
}


// Return directory part of path to current item
FXString DirList::getDirectory() const
{
    const TreeItem* item=(TreeItem*)currentitem;
    while(item)
    {
        if(item->state&DirItem::FOLDER)
            return getItemPathname(item);
        item=(TreeItem*)item->parent;
    }
    return "";
}


// Set current (dir/file) name path
void DirList::setCurrentFile(const FXString& pathname)
{    // FIXME notify argument?
    TreeItem *item;
    if(!pathname.empty())
    {
        FXString path=FXFile::absolute(pathname);
        while(!FXFile::isTopDirectory(path) && !exists(dequote(path)))
            path=FXFile::upLevel(path);
        item=getitem((FXchar*)path.text());     // FIXME
        makeItemVisible(item);
        setCurrentItem(item);
    }
}


// Get current (dir/file) name path
FXString DirList::getCurrentFile() const
{
    if(currentitem==NULL)
        return FXString::null;
    return getItemPathname((TreeItem*)currentitem);
}



// Get list style
FXbool DirList::showFiles() const
{
    return (options&DIRLIST_SHOWFILES)!=0;
}


// Change list style
void DirList::showFiles(FXbool showing)
{
    FXuint opts=options;
    if(showing)
        opts|=DIRLIST_SHOWFILES;
    else
        opts&=~DIRLIST_SHOWFILES;
    if(options!=opts)
    {
        options=opts;
        scanRootDir(TRUE);
    }
}


// Return TRUE if showing hidden files
FXbool DirList::showHiddenFiles() const
{
    return (options&DIRLIST_SHOWHIDDEN)!=0;
}


// Change show hidden files mode
void DirList::showHiddenFiles(FXbool showing)
{
    FXuint opts=options;
    if(showing)
        opts|=DIRLIST_SHOWHIDDEN;
    else
        opts&=~DIRLIST_SHOWHIDDEN;
    if(opts!=options)
    {
        options=opts;
        scanRootDir(TRUE);
    }
}


// Set associations
void DirList::setAssociations(FileDict* assoc)
{
    associations=assoc;
}


// Set the pattern to filter
void DirList::setPattern(const FXString& ptrn)
{
    if(ptrn.empty())
        return;
    if(pattern!=ptrn)
    {
        pattern=ptrn;
        scanRootDir(TRUE);
    }
}


// Change file match mode
void DirList::setMatchMode(FXuint mode)
{
    if(matchmode!=mode)
    {
        matchmode=mode;
        scanRootDir(TRUE);
    }
}


// Save data
void DirList::save(FXStream& store) const
{
    FXTreeList::save(store);
    store << associations;
    store << pattern;
    store << matchmode;
    store << closed_folder;
    store << open_folder;
    store << mini_doc;
    store << mini_app;
}


// Load data
void DirList::load(FXStream& store)
{
    FXTreeList::load(store);
    store >> associations;
    store >> pattern;
    store >> matchmode;
    store >> closed_folder;
    store >> open_folder;
    store >> mini_doc;
    store >> mini_app;
}


// Cleanup
DirList::~DirList()
{
    clearItems();
    if(refresh)
        getApp()->removeTimeout(refresh);
    if(expandtimer)
        getApp()->removeTimeout(expandtimer);
    if(!(options&DIRLIST_NO_OWN_ASSOC))
        delete associations;
    delete closed_folder;
    delete open_folder;
    delete mini_doc;
    delete mini_app;
    delete harddiskicon;
    delete linkicon;
    associations=(FileDict*)-1;
    closed_folder=(FXPNGIcon*)-1;
    open_folder=(FXPNGIcon*)-1;
    mini_doc=(FXPNGIcon*)-1;
    mini_app=(FXPNGIcon*)-1;
    harddiskicon=(FXPNGIcon*)-1;
    linkicon=(FXPNGIcon*)-1;
    refresh=(FXTimer*)-1;
    expandtimer=(FXTimer*)-1;
}


