// File management functions taken from the Fox library
// and modified to add a progress dialog 
// Some helper functions are also added to get large files support 

#include "config.h"
#include "i18n.h"

#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <unistd.h>

#include <fox/fx.h>
#include <fox/fxkeys.h>

#include "icons.h"
#include "File.h"
#include "OverwriteBox.h"
#include "MessageBox.h"
#include "CommandWindow.h"


// Maximum length of a file name
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

// Delay before the progress bar should be shown (ms)
#define SHOW_PBAR_DELAY 1000

// Progress dialog width
#define PDIALOG_WIDTH 150

extern FXbool confirm_overwrite;

// Write the file size in human readable form (bytes or Kbytes or Mbytes)
void Size(FXchar* size,FXchar *hsize)
{
    int flag=0;
    FXchar suf[50];
	
    unsigned long lsize = atol(size);
    float fsize=0.0;

	strcpy(suf,_("bytes")); 
	if(lsize>1000000)
    {
        fsize=lsize/1048576.0;
        strcpy(suf,_("MB"));
        flag=1;
    }
    else if(lsize>1000)
    {
        fsize=lsize/1024.0;
        strcpy(suf,_("KB"));
        flag=1;
    }
    if(flag)
    {
        if(fsize==(int)fsize)
            sprintf(hsize,"%.0f %s",fsize,suf);
        else
            sprintf(hsize,"%.1f %s",fsize,suf);
    }
    else
        sprintf(hsize,"%lu %s",lsize,suf);
}


// Remove terminating '/' on a path string to simplify a file or directory path
// Thus '/bla/bla////' becomes '/bla/bla'
// Special case : '/' stays to '/'
// And then return the absolute path, based on the current directory path
FXString filePath(const FXString path)
{
	FXString in=path, out=path;
	while (1)
	{
		if (in[in.length()-1]=='/' & in.length()!=1)
		{
			out=in.trunc(in.length()-1);
			in=out;
		}
		else
			break;
	}
	FXString dir=FXFile::getCurrentDirectory();

	// If absolute path
	if(ISPATHSEP(out[0]))
		return (out);
	else
		return (dir+PATHSEPSTRING+out);
}

// Remove terminating '/' on a path string to simplify a file or directory path
// Thus '/bla/bla////' becomes '/bla/bla'
// Special case : '/' stays to '/'
// And then return the absolute path, based on the specified directory path
FXString filePath(const FXString path, const FXString dir)
{
	FXString in=path, out=path;
	while (1)
	{
		if (in[in.length()-1]=='/' && in.length()!=1)
		{
			out=in.trunc(in.length()-1);
			in=out;
		}
		else
			break;
	}
	// If absolute path
	if(ISPATHSEP(out[0]))
		return (out);
	else
		return (dir+PATHSEPSTRING+out);
}

// Enquote filename to make it safe for shell
// (the function provided by Fox seems to be buggy??)
FXString quote(const FXString& file,FXbool forcequotes)
{
	FXString result;
	register FXint i,c;
	for(i=0; (c=file[i])!='\0'; i++)
	{
    	switch(c)
		{
      	case '"':              // Quote needs to be escaped
        	result+="\\\"";
        break;
      	case '\\':              // Backspace needs to be escaped, of course
        	result+="\\\\";
        break;
      	case '#':
      	case '~':
        	if(i) goto noquote;   // Only quote if at begin of filename
      	case '!':               // Special in csh
      	case '\'':
      	case '$':               // Variable substitution
      	case '&':
      	case '(':
      	case ')':
      	case ';':
      	case '<':               // Redirections, pipe
      	case '>':
      	case '|':
      	case '`':               // Command substitution
      	case '^':               // Special in sh
      	case '*':               // Wildcard characters
      	case '?':
      	case '[':
      	case ']':
      	case '\t':              // White space
      	case '\n':
      	case ' ':
        	forcequotes=TRUE;
      	default:                // Normal characters just added
			noquote:result+=c;
        break;
		}
	}
	if(forcequotes) return "\""+result+"\"";
	return result;
}

// Decode filename to get original again
FXString dequote(const FXString& file)
{
	FXString result;
	register FXint i,c;
	i=0;
	while((c=file[i])!='\0' && isspace((FXuchar)c)) i++;
  	if(file[i]=='\"')
	{
    	i++;
    	while((c=file[i])!='\0' && c!='\"')
		{
      		if(c=='\\' && file[i+1]!='\0') c=file[++i];
      		result+=c;
      		i++;
      	}
    }
	else
	{
    	while((c=file[i])!='\0' /*&& !isspace((FXuchar)c)*/)
		{
      		if(c=='\\' && file[i+1]!='\0') c=file[++i];
      		result+=c;
      		i++;
      	}
    }
  	return result;
}

// Test if a directory is empty
// Return -1 if not a directory, 1 if empty and 0 if not empty
FXint isEmptyDir(const FXString directory)
{
	FXint ret=-1;
	DIR* dir;
	struct dirent *entry;
	int n=0;
	
	if ((dir=opendir(directory.text()))!=NULL)
	{
		while (n<3)	{entry=readdir(dir); n++;}
		if (entry==NULL)
			ret=1;
		else
			ret=0;
	}
	closedir(dir);
	return ret;
}
		
// Check if file exists
FXbool exists(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (stat(file.text(),&status)==0);
}

// Check if file represents a directory
FXbool isDirectory(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (stat(file.text(),&status)==0) && S_ISDIR(status.st_mode);
}

// Check if file represents a file
FXbool isFile(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (stat(file.text(),&status)==0) && S_ISREG(status.st_mode);
}

// Check if file represents a link
FXbool isLink(const FXString& file)
{
  	struct stat status;
  	return !file.empty() && (lstat(file.text(),&status)==0) && S_ISLNK(status.st_mode);
}

// Get file info
FXbool info(const FXString& file,struct stat& inf)
{
  	return !file.empty() && (stat(file.text(),&inf)==0);
}

// Read symbolic link
FXString readlink(const FXString& file)
{
	FXchar lnk[MAXPATHLEN+1];
	FXint len=::readlink(file.text(),lnk,MAXPATHLEN);
	if(0<=len)
  		return FXString(lnk,len);
  	else
  		return FXString::null;
}



// Timer record (redefined here to use it in checkTimeout)
struct FXTimer
{
	FXTimer       *next;              // Next timeout in list
	FXObject      *target;            // Receiver object
	FXSelector     message;           // Message sent to receiver
	struct timeval due;               // When timer is due
};

// Message Map
FXDEFMAP(File) FileMap[]={
                             FXMAPFUNC(SEL_COMMAND,File::ID_CANCEL_BUTTON,File::onCmdCancel),
							 FXMAPFUNC(SEL_TIMEOUT,File::ID_TIMEOUT,File::onTimeout),
                         };

// Object implementation
FXIMPLEMENT(File,FXDialogBox,FileMap,ARRAYNUMBER(FileMap))

// Construct object
File::File(FXApp* a, FXString title, const unsigned int operation):FXDialogBox(a,title,DECOR_ALL)
{
    // Progress window
	FXPacker *buttons=new FXPacker(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,10,10,PDIALOG_WIDTH,PDIALOG_WIDTH,5,5);
    new FXHorizontalSeparator(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X|SEPARATOR_GROOVE);
    FXVerticalFrame *contents=new FXVerticalFrame(this,LAYOUT_SIDE_TOP|FRAME_NONE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	allowPdialog=TRUE;

    // Cancel Button
    cancelButton=new FXButton(buttons,_("&Cancel"),NULL,this,File::ID_CANCEL_BUTTON,FRAME_RAISED|FRAME_THICK|LAYOUT_CENTER_X,0,0,0,0,20,20);
    cancelButton->setFocus();
    cancelButton->addHotKey(KEY_Escape);
	isCancelled=FALSE;

	// Timer
	timer=NULL;
	
	// Progress bar
	pbar=NULL;

	// Progress dialog depends on the file operation
	switch(operation)	
	{
		case COPY:
	    	// Labels and progress bar
			uplabel=new FXLabel(contents,_("Source :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Target :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
			pbar=new FXProgressBar(contents,NULL,0,LAYOUT_FILL_X|FRAME_SUNKEN|FRAME_THICK|PROGRESSBAR_PERCENTAGE,0,0,0,0,PDIALOG_WIDTH);
		
			// Timer on
			timer=getApp()->addTimeout(SHOW_PBAR_DELAY,this,File::ID_TIMEOUT);
			break;

    	case MOVE:
	    	// Labels
			uplabel=new FXLabel(contents,_("Source :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Target :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
		
			// Timer on
			timer=getApp()->addTimeout(SHOW_PBAR_DELAY,this,File::ID_TIMEOUT);
			break;

    	case DELETE:
	    	// Labels
			uplabel=new FXLabel(contents,_("Delete :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("From :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);

			// Timer on
			timer=getApp()->addTimeout(SHOW_PBAR_DELAY,this,File::ID_TIMEOUT);
			break;

    	case CHMOD:
	    	// Labels
			uplabel=new FXLabel(contents,_("Changing permissions..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("File :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);

			// Timer on
			timer=getApp()->addTimeout(SHOW_PBAR_DELAY,this,File::ID_TIMEOUT);
			break;

    	case CHOWN:
	    	// Labels
			uplabel=new FXLabel(contents,_("Changing owner..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("File :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);

			// Timer on
			timer=getApp()->addTimeout(SHOW_PBAR_DELAY,this,File::ID_TIMEOUT);
			break;

#if defined(linux)
    	case MOUNT:
	    	// Labels
			uplabel=new FXLabel(contents,_("Mount file system..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Mount the folder :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
			break;

    	case UNMOUNT:
	    	// Labels
			uplabel=new FXLabel(contents,_("Unmount file system..."),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
    		downlabel=new FXLabel(contents,_("Unmount the folder :"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_X);
			break;
#endif

		default: // Other : RENAME, SYMLINK, ARCHIVE, EXTRACT, RPM_INSTALL, RPM_UNINSTALL
			// Progress dialog not used
			uplabel=NULL;
    		downlabel=NULL;
	}
	
	// Initialize the overwrite flags
	if (confirm_overwrite)
	{
		overwrite=FALSE;
		overwrite_all=FALSE;
		skip_all=FALSE;
	}
	else
	{
		overwrite=TRUE;
		overwrite_all=TRUE;
		skip_all=FALSE;
	}		    			
}

// Destructor
File::~File()
{
	getApp()->removeTimeout(timer);
	timer=NULL;
    delete pbar;
}



// Create and initialize
void File::create()
{
	FXDialogBox::create();
}


// Force check of timeout for progress dialog (to avoid latency problems)
int File::checkTimeout(FXTimer *timer)
{
	timeval now;
	
	if (timer)
	{
		gettimeofday(&now,NULL);
		if(now.tv_sec>timer->due.tv_sec || (now.tv_sec==timer->due.tv_sec && now.tv_usec>=timer->due.tv_usec))
		{
			getApp()->removeTimeout(timer);
			if (allowPdialog)
				show(PLACEMENT_OWNER);
			getApp()->forceRefresh();
			getApp()->flush();
			return 1;
		}
	}
	return 0;
}


// Read bytes
long File::fullread(int fd,unsigned char *ptr,long len)
{
    long nread;
#ifdef EINTR
    do
        nread=read(fd,ptr,len);
    while(nread<0 && errno==EINTR);
#else
    nread=read(fd,ptr,len);
#endif
    return nread;
}


// Write bytes
long File::fullwrite(int fd,const unsigned char *ptr,long len)
{
    long nwritten,ntotalwritten=0;
    while(len>0)
    {
        nwritten=write(fd,ptr,len);
        if(nwritten<0)
        {
#ifdef EINTR
            if(errno==EINTR)
                continue;
#endif
            return -1;
        }
        ntotalwritten+=nwritten;
        ptr+=nwritten;
        len-=nwritten;
    }
    return ntotalwritten;
}


// Copy ordinary file
FXbool File::copyfile(const FXString& source, const FXString& target)
{
    FXString destfile;
	unsigned char buffer[4096];
    struct stat status;
    long nread,nwritten;
	long size, nread_all=0;
	double pct=0;
    int src,dst;
    FXbool ok=FALSE;

    if((src=open(source.text(),O_RDONLY))>=0)
    {
        if(stat(source.text(),&status)==0)
        {
			// If destination is a directory
			if(isDirectory(target))
				destfile=target+PATHSEPSTRING+FXFile::name(source);
			else
				destfile=target;
				
         	// Copy file block by block
			size=status.st_size; 
			if((dst=open(destfile.text(),O_WRONLY|O_CREAT|O_TRUNC,status.st_mode))>=0)
            {
                while(1)
                {
                    nread=File::fullread(src,buffer,sizeof(buffer));
                    if(nread<0)
					{
						MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno));									
                        goto err;
					}
                    if(nread==0)
                        break;
											
					// Force timeout checking for progress dialog
					if(checkTimeout(timer))
						timer=NULL;

					// Set percentage value for progress dialog
					nread_all+=nread;
					pct=100*(double)nread_all/(double)size;	
					if (pbar)
						pbar->setProgress((int)pct);

					// Give cancel button an opportunity to be clicked
					if(cancelButton)
						getApp()->runWhileEvents(cancelButton);

					// Set labels for progress dialog
					FXString label=_("Source : ")+source;
					uplabel->setText(label);
					label=_("Target : ")+target;
					downlabel->setText(label);
					getApp()->repaint();

					// If cancel button was clicked, close files and return
					if (isCancelled)
					{
						close(dst);
						close(src);
						return FALSE;
					}
                    nwritten=File::fullwrite(dst,buffer,nread);
                    if(nwritten<0)
					{
						MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno) );			
                        goto err;
					}
                }
                ok=TRUE;
err:
                close(dst);
            }
        }
        close(src);
    }
    return ok;
}


// Copy directory
FXbool File::copydir(const FXString& source,const FXString& target,struct stat& parentstatus,inodelist* inodes)
{
    DIR *dirp;
    struct dirent *dp;
    struct stat status;
    inodelist *in,inode;
    FXString destfile,oldchild,newchild;

	// If destination is an existing directory
	if(isDirectory(target))
		destfile=target+PATHSEPSTRING+FXFile::name(source);
	else
		destfile=target;

    // See if visited this inode already
    for(in=inodes; in; in=in->next)
    {
        if(in->st_ino==parentstatus.st_ino)
            return TRUE;
    }

    // Try make directory, if none exists yet
    if(mkdir(destfile.text(),parentstatus.st_mode|S_IWUSR)!=0 && errno!=EEXIST)
        return FALSE;

    // Can we stat it
    if(lstat(destfile.text(),&status)!=0 || !S_ISDIR(status.st_mode))
        return FALSE;

    // Try open directory to copy
    dirp=opendir(source.text());
    if(!dirp)
        return FALSE;

    // Add this to the list
    inode.st_ino=status.st_ino;
    inode.next=inodes;

    // Copy stuff
    while((dp=readdir(dirp))!=NULL)
    {
        if(dp->d_name[0]!='.' || (dp->d_name[1]!='\0' && (dp->d_name[1]!='.' || dp->d_name[2]!='\0')))
        {
            oldchild=source;
            if(!ISPATHSEP(oldchild[oldchild.length()-1]))
                oldchild.append(PATHSEP);
            oldchild.append(dp->d_name);
            newchild=destfile;
            if(!ISPATHSEP(newchild[newchild.length()-1]))
                newchild.append(PATHSEP);
            newchild.append(dp->d_name);
            if(!copyrec(oldchild,newchild,&inode))
            {
                closedir(dirp);
                return FALSE;
            }
        }
    }

    // Close directory
    closedir(dirp);

    // Success
    return TRUE;
}


// Recursive copy
FXbool File::copyrec(const FXString& source,const FXString& target,inodelist* inodes)
{
    struct stat status1,status2;

    // Old file or directory does not exist
    if(lstat(source.text(),&status1)!=0)
        return FALSE;

    // If target is not a directory, remove it if allowed
    if(lstat(target.text(),&status2)==0)
    {
        if(!S_ISDIR(status2.st_mode))
        {
            if(!(overwrite|overwrite_all))
                return FALSE;
            if(::unlink(target.text())!=0)
                return FALSE;
        }
    }

    // Source is directory: copy recursively
    if(S_ISDIR(status1.st_mode))
        return File::copydir(source,target,status1,inodes);

    // Source is regular file: copy block by block
    if(S_ISREG(status1.st_mode))
        return File::copyfile(source,target);

    // Source is fifo: make a new one
    if(S_ISFIFO(status1.st_mode))
        return ::mkfifo(target.text(),status1.st_mode);

    // Source is device: make a new one
    if(S_ISBLK(status1.st_mode) || S_ISCHR(status1.st_mode) || S_ISSOCK(status1.st_mode))
        return ::mknod(target.text(),status1.st_mode,status1.st_rdev)==0;

    // Source is symbolic link: make a new one
    if(S_ISLNK(status1.st_mode))
    {
        FXString lnkfile=readlink(source);
        return ::symlink(lnkfile.text(),target.text())==0;
    }

    // This shouldn't happen
    return FALSE;
}



// Copy file
FXbool File::copy(const FXString& source,const FXString& target)
{
	FXString targetfile;

	// Source doesn't exist
	if(!exists(dequote(source)))
		return FALSE;

	// Source and target are identical
	if (target==source)
		return FALSE;

	// Target is an existing directory
	if (isDirectory(target))
		targetfile=target+PATHSEPSTRING+FXFile::name(source);
	else
		targetfile=target;

	// Source and target file are identical
	if (targetfile==source)
		return FALSE;

	// Target already exists
	if(exists(dequote(targetfile)))
	{
		// Overwrite dialog if necessary
		if (!(overwrite_all | skip_all))
    	{
			FXString label=_("Source : ")+source;
			uplabel->setText(label);
			label=_("Target : ")+targetfile;
			downlabel->setText(label);
			getApp()->repaint();

			FXString msg=_("File ")+targetfile+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
					overwrite = FALSE;
           			break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return FALSE;

		// Use of the standard C function to remove the target
		// avoids to overwrite a non empty directory
		if (::remove(targetfile.text())==-1)
		{
			MessageBox::error(this,MBOX_OK,_("Error"),_("Cannot overwrite the non empty folder %s"), targetfile.text());			
			return FALSE;
		}
	}

	// Copy file or directory
	return File::copyrec(source,targetfile,NULL);
}


// Remove file or directory
FXbool File::remove(const FXString& file)
{
	FXString dirname;
    struct stat status;
	static FXbool ISDIR=FALSE;

    if(lstat(file.text(),&status)==0)
    {
        // It is a directory
		if(S_ISDIR(status.st_mode))
        {
            DIR *dirp=::opendir(file.text());
            if(dirp)
            {			
                struct dirent *dp;
                FXString child;
				
				// Used to display only one progress dialog when deleting a directory
				ISDIR=TRUE;
				
				// Force timeout checking for progress dialog
				if(checkTimeout(timer))
					timer=NULL;

				// Give cancel button an opportunity to be clicked
				if(cancelButton)
					getApp()->runWhileEvents(cancelButton);

				// Set labels for progress dialog
				FXString label=_("Delete folder :")+file;
				uplabel->setText(label);
				dirname=FXFile::directory(FXFile::absolute(file));
				label=_("From : ")+dirname;
				downlabel->setText(label);
				getApp()->repaint();

				// If cancel button was clicked, return
				if (isCancelled)
					return FALSE;

                while((dp=::readdir(dirp))!=NULL)
                {
					if(dp->d_name[0]!='.' || (dp->d_name[1]!='\0' && (dp->d_name[1]!='.' || dp->d_name[2]!='\0')))
                    {
                        child=file;
                        if(!ISPATHSEP(child[child.length()-1]))
                            child.append(PATHSEP);
                        child.append(dp->d_name);
                        if(!File::remove(child))
                            {
                                ::closedir(dirp);
                                return FALSE;
                            }
                    }
                }
                ::closedir(dirp);
            }
            return (::rmdir(file.text())==0);
        }
        else
        {
			// If it was not a directory
			if (!ISDIR)
			{
				// Force timeout checking for progress dialog
				if(checkTimeout(timer))
					timer=NULL;

				// Give cancel button an opportunity to be clicked
				if(cancelButton)
					getApp()->runWhileEvents(cancelButton);

				// Set labels for progress dialog
				FXString label=_("Delete : ")+file;
				uplabel->setText(label);
				dirname=FXFile::directory(FXFile::absolute(file));
				label=_("From : ")+dirname;
				downlabel->setText(label);
				getApp()->repaint();

				// If cancel button was clicked, return
				if (isCancelled)
					return FALSE;
			}
            return (::unlink(file.text())==0);
        }
    }
    return FALSE;
}


// Rename a file or a directory
FXbool File::rename(const FXString& source,const FXString& target)
{
	// Source doesn't exist
    if(!exists(dequote(source)))
		return FALSE;
			
	// Source and target are identical
	if (target==source)
		return FALSE;

	// Target already exists
	if(exists(dequote(target)))
	{
		// Overwrite dialog if necessary
		if (!(overwrite_all | skip_all))
    	{
			FXString msg=_("File ")+target+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
					overwrite = FALSE;
           			break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return FALSE;

		// Use of the standard C function to remove the target
		// avoids to overwrite a non empty directory
		if (::remove(target.text())==-1)
		{
			MessageBox::error(this,MBOX_OK,_("Error"),_("Cannot overwrite the non empty folder %s"), target.text());			
			return FALSE;
		}
	}
       		
	// Rename file using the standard C function
	// This should only work for files that are on the same file system
	if (::rename(source.text(),target.text())==0)
		return TRUE;
	if(errno!=EXDEV)
	{
		MessageBox::error(this,MBOX_OK,_("Error"),_("Cannot rename to target %s"), target.text());	
		return FALSE;
	}

	// If files are on different file systems, use the copy/delete scheme
	// and use a progress bar on top of the progress dialog when file copy is lengthy
	FXString label=_("Source : ")+source;
	uplabel->setText(label);
	label=_("Target : ")+target;
	downlabel->setText(label);
	getApp()->repaint();
	File *f=NULL;
	f=new File(getApp(),_("File move"),COPY);
	f->create();
	FXbool ret=f->copy(source,target);

	// If action is cancelled in progress dialog
	if (f->isCancelled)
	{
		f->hide();
		MessageBox::error(this,MBOX_OK,_("Error"),_("Copy file operation cancelled!"));
		delete f;
		return FALSE;
	}
	delete f;
	if (ret)
		return (remove(source.text())==TRUE);
	else
		return FALSE;
}

// Move files
FXbool File::move(const FXString& source,const FXString& target)
{
	// Source doesn't exist
	if(!exists(dequote(source)))
		return FALSE;

	// Source and target are identical
	if (target==source)
		return FALSE;

	// Target is an existing directory
	FXString targetfile;
	if (isDirectory(target))
		targetfile=target+PATHSEPSTRING+FXFile::name(source);
	else
		targetfile=target;

	// Source and target file are identical
	if (targetfile==source)
		return FALSE;

	// Target file already exists
	if(exists(dequote(targetfile)))
    {
		// Overwrite dialog if necessary
		if (!overwrite_all & !skip_all)
		{
			// Set labels for progress dialog
			FXString label=_("Source : ")+source;
			uplabel->setText(label);
			label=_("Target : ")+target;
			downlabel->setText(label);
			getApp()->repaint();

			FXString msg=_("File ")+targetfile+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;

       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
           			overwrite = FALSE;
					break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return FALSE;
			
		// Use of the standard C function to remove the target
		// avoids to overwrite a non empty directory
		if (::remove(targetfile.text())==-1)
		{
			MessageBox::error(this,MBOX_OK,_("Error"),_("Cannot overwrite the non empty folder %s"), targetfile.text());			
			return FALSE;
		}
	}
	// Rename file
	return File::rename(source,targetfile);
}


// Symbolic Link file
FXbool File::symlink(const FXString& source,const FXString& target)
{
	// Source doesn't exist
    if(!exists(dequote(source)))
		return FALSE;
			
	// Source and target are identical
	if (target==source)
		return FALSE;

	// Target already exists
	if(exists(dequote(target)))
	{
		// Overwrite dialog if necessary
		if (!(overwrite_all | skip_all))
    	{
			FXString msg=_("File ")+target+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
					overwrite = FALSE;
           			break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return FALSE;

		// Use of the standard C function to remove the target
		// Avoids to overwrite a non empty directory
		if (::remove(target.text())==-1)
		{
			MessageBox::error(this,MBOX_OK,_("Error"),_("Cannot overwrite the non empty folder %s"), target.text());			
			return FALSE;
		}
	}
    
	// Create symbolic link using the standard C function
	FXint ret=::symlink(source.text(),target.text());	
	if (ret==0)
		return TRUE;
	else
	{
		MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno));
		return FALSE;
	}		
}


// Chmod a file or directory, recursively or not
// Note : the variable file returns the last processed file
// It can be different from the initial path, if recursive chmod is used
// (Used to fill an error message, if needed) 
int File::chmod(char *path, char *file, mode_t mode, FXbool rec, FXbool dironly)
{		
    struct stat stats;

	// Initialise the file variable with the initial path
	strcpy(file, path);

    if (lstat(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
	{
        if(dironly)
            return 0;
			
		// Force timeout checking for progress dialog
		if(checkTimeout(timer))
			timer=NULL;

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing permissions...");
		uplabel->setText(label);
		label=_("File : ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

        return ::chmod(path,mode);
    }
    else // directory
    {
        if(rec == FALSE)
        {
			// Force timeout checking for progress dialog
			if(checkTimeout(timer))
				timer=NULL;

			// Give cancel button an opportunity to be clicked
			if(cancelButton)
				getApp()->runWhileEvents(cancelButton);

			// Set labels for progress dialog
			FXString label=_("Changing permissions...");
			uplabel->setText(label);
			label=_("Folder : ")+FXString(path);
			downlabel->setText(label);
			getApp()->repaint();

			// If cancel button was clicked, return
			if (isCancelled)
				return -1;

			if(::chmod(path,mode)) // do not change recursively
				return -1;
        }
        else
            return rchmod(path,file,mode,dironly); // recursive change
    }
    return 0;

}


// Recursive chmod for a directory
int File::rchmod(char *path, char *file, mode_t mode, FXbool dironly)
{
    struct stat stats;

	// Initialise the file variable with the initial path
	strcpy(file, path);

    if (lstat(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
    {
        if(dironly)
            return 0;

		// Force timeout checking for progress dialog
		if(checkTimeout(timer))
			timer=NULL;

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing permissions...");
		uplabel->setText(label);
		label=_("File : ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

        return ::chmod(path,mode);
    }

    DIR *dir;
    struct dirent *entry;
    int i, pl = strlen(path);

    if (!(dir = opendir(path)))
        return -1;

    for(i = 0; (entry = readdir(dir)); i++)
	{	
        if (entry->d_name[0] != '.' || (entry->d_name[1] != '\0'
                                        && (entry->d_name[1] != '.' ||
                                            entry->d_name[2] != '\0')))
        {
            int pl1 = pl, l = strlen(entry->d_name);
            char *path1 = (char *)alloca(pl1+l+2);

            strcpy(path1, path);
            if (path1[pl1-1] != '/')
                path1[pl1++] = '/';
            strcpy(path1+pl1, entry->d_name);

			// Modify the file variable with the new path
			strcpy(file, path1);
            if (rchmod(path1,file,mode,dironly))
                return -1;
        }
	}

    if (closedir(dir))
        return -1;

    return ::chmod(path,mode);
}


// Chown a file or directory, recursively or not
// Note : the variable file returns the last processed file
// It can be different from the initial path, if recursive chmod is used
// (Used to fill an error message, if needed) 
int File::chown(char *path, char *file, uid_t uid, gid_t gid, FXbool rec, FXbool dironly)
{
    struct stat stats;

	// Initialise the file variable with the initial path
	strcpy(file, path);

    if (lstat(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
    {
        if(dironly)
            return 0;

		// Force timeout checking for progress dialog
		if(checkTimeout(timer))
			timer=NULL;

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing owner...");
		uplabel->setText(label);
		label=_("File : ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

        if(::chown(path,uid,gid))
            return -1;
    }
    else // directory
    {
        if(rec == FALSE)
        {
			// Force timeout checking for progress dialog
			if(checkTimeout(timer))
				timer=NULL;

			// Give cancel button an opportunity to be clicked
			if(cancelButton)
				getApp()->runWhileEvents(cancelButton);

			// Set labels for progress dialog
			FXString label=_("Changing owner...");
			uplabel->setText(label);
			label=_("Folder : ")+FXString(path);
			downlabel->setText(label);
			getApp()->repaint();

			// If cancel button was clicked, return
			if (isCancelled)
				return -1;

            if(::chown(path,uid,gid)) // do not change recursively
                return -1;
        }
        else
            if(rchown(path,file,uid,gid,dironly)) // recursive change
                return -1;
    }
    return 0;

}


// Recursive chown for a directory
int File::rchown(char *path, char *file, uid_t uid, gid_t gid, FXbool dironly)
{
    struct stat stats;

	// Initialise the file variable with the initial path
	strcpy(file, path);

    if (lstat(path, &stats)) //if it doesn't exist
        return -1;

    if (!S_ISDIR(stats.st_mode)) // file
    {
		if(dironly)
			return 0;

		// Force timeout checking for progress dialog
		if(checkTimeout(timer))
			timer=NULL;

		// Give cancel button an opportunity to be clicked
		if(cancelButton)
			getApp()->runWhileEvents(cancelButton);

		// Set labels for progress dialog
		FXString label=_("Changing owner...");
		uplabel->setText(label);
		label=_("File : ")+FXString(path);
		downlabel->setText(label);
		getApp()->repaint();

		// If cancel button was clicked, return
		if (isCancelled)
			return -1;

		return ::chown(path,uid,gid);
    }

    DIR *dir;
    struct dirent *entry;
    int i, pl = strlen(path);

    if (!(dir = opendir(path)))
        return -1;

    for(i = 0; (entry = readdir(dir)); i++)
        if (entry->d_name[0] != '.' || (entry->d_name[1] != '\0'
                                        && (entry->d_name[1] != '.' ||
                                            entry->d_name[2] != '\0')))
        {
            int pl1 = pl, l = strlen(entry->d_name);
            char *path1 = (char *)alloca(pl1+l+2);

            strcpy(path1, path);
            if (path1[pl1-1] != '/')
                path1[pl1++] = '/';
            strcpy(path1+pl1, entry->d_name);
			strcpy(file, path1);
            if (rchown(path1,file,uid,gid,dironly))
                return -1;
        }

    if (closedir(dir))
        return -1;

    if(::chown(path,uid,gid))
        return -1;

    return 0;
}


// Extract an archive in a specified directory
int File::extract(FXString name, FXString dir,FXString cmd)
{
	// Change to the specified directory
    chdir(dir.text());

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Extract archive"),cmd,30,80);			
	cmdwin->create();

	// The command window object deletes itself after closing the window!

    return 1;
}


// Create an archive
int File::archive(FXString name, FXString cmd)
{
	// Target file already exists
	if(exists(dequote(name)))
    {
		// Overwrite dialog if necessary
		if (!overwrite_all & !skip_all)
		{
			FXString msg=_("File ")+name+_(" already exists. Overwrite?");
       		ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(getApp()->getMainWindow(),_("Confirm Overwrite"),msg);
       		FXuint answer=dlg->execute();
			delete dlg;
       		switch(answer)
       		{
       			case 1:
           			overwrite = TRUE;
           			break;
       			case 2:
           			overwrite_all = TRUE;
           			break;
       			case 3:
           			overwrite = FALSE;
					break;
       			case 4:
           			skip_all = TRUE;
					break;
       		}
		}
		if(!(overwrite | overwrite_all) | skip_all)
			return FALSE;
	}

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Add to archive"),cmd,30,80);			
	cmdwin->create();

	// The command window object deletes itself after closing the window!

    return 1;
}


#if defined(linux)
int File::mount(FXString dir, FXString msg, FXString cmd, unsigned int op)
{
	// Show progress dialog (no timer here)
	show(PLACEMENT_OWNER);
	getApp()->forceRefresh();
	getApp()->flush();

	// Set labels for progress dialog
	uplabel->setText(msg);				
	downlabel->setText(dir.text());
	getApp()->repaint();

	// Give cancel button an opportunity to be clicked
	if(cancelButton)
		getApp()->runWhileEvents(cancelButton);

	// If cancel button was clicked, return
	if (isCancelled)
		return -1;

	// Perform the mount/unmount command
	FILE *pcmd=popen(cmd.text(),"r");
	if(!pcmd)
	{
		MessageBox::error(this,MBOX_OK,_("Error"),_("Failed command : %s"),cmd.text());
		return -1;
	}

	// Get error message if any
	char text[10000]={0};
	FXString buf;
	while(fgets(text,sizeof(text),pcmd))
		buf+=text;
	sprintf(text,"%s",buf.text());
	
	// Close the stream
	if(pclose(pcmd))
	{
		MessageBox::error(this,MBOX_OK,_("Error"),"%s",text);
		return -1;
	}
	// Hide progress dialog
	hide();
	
	// Success message
	if (op==MOUNT)
		MessageBox::information(this,MBOX_OK,_("Success"),_("Folder %s was successfully mounted."),dir.text());
	else
		MessageBox::information(this,MBOX_OK,_("Success"),_("Folder %s was successfully unmounted."),dir.text());
	
	return 1;
}


// Install / Upgrade RPM package
int File::rpmInstall(FXString name, FXString cmd)
{
    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Install/Upgrade RPM"),cmd,10,80);			
	cmdwin->create();

	// The command window object deletes itself after closing the window!

    return 1;
}

// Uninstall RPM package
int File::rpmUninstall(FXString name, FXString cmd)
{
    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp(),_("Uninstall RPM"),cmd,10,80);			
	cmdwin->create();
	
	// Add a message if necessary
	if (cmdwin->getLength()==0)
	{
		FXString msg=_("Package ") + name + _(" uninstalled successfully."); 
		cmdwin->appendText(msg.text());
	}

	// The command window object deletes itself after closing the window!

    return 1;
}

#endif


// Handle cancel button in progress bar dialog
long File::onCmdCancel(FXObject*, FXSelector,void*)
{
	isCancelled=TRUE;
	return 1;
}


// Handle timeout for progress bar
long File::onTimeout(FXObject*, FXSelector,void*)
{
	if (allowPdialog)
		show(PLACEMENT_OWNER);
    getApp()->forceRefresh();
    getApp()->flush();
    return 1;
}

