// File association tables. Taken from the FOX library and slightly modified.

#include "config.h"
#include "i18n.h"

#include <fox/fx.h>
#include <fox/FXPNGIcon.h>

#include "icons.h"
#include "FileDict.h"


#define COMMANDLEN   256
#define EXTENSIONLEN 128
#define MIMETYPELEN  64
#define ICONNAMELEN  256

// You can override the default icon locations to search for your
// particular platform by specifying -DDEFAULTICONPATH="path" on
// the command line.
#ifndef DEFAULTICONPATH
#define DEFAULTICONPATH   "~/.foxicons:/usr/local/share/icons:/usr/share/icons"
#endif


/*******************************************************************************/


// Object implementation
FXIMPLEMENT(IconDict,FXDict,NULL,0)


// Default icon path
const FXchar IconDict::defaultIconPath[]=DEFAULTICONPATH;


// Build icon table
IconDict::IconDict(FXApp* a,const FXString& p):app(a),path(p)
{
    FXTRACE((100,"IconDict::IconDict\n"));
}


// Try and load the icon
void *IconDict::createData(const void* ptr)
{
    register const FXchar *ext;

    // Make sure
    if(!ptr)
        return NULL;

    // Get file extension
    ext=strrchr((const char*)ptr,'.');

    // Determine what type
    if(ext)
    {
        FXIcon *icon=NULL;

        // Create icon of the right type
        if(comparecase(".gif",ext)==0)
        {
            icon=new FXGIFIcon(getApp());
        }
        else if(comparecase(".bmp",ext)==0)
        {
            icon=new FXBMPIcon(getApp());
        }
        else if(comparecase(".xpm",ext)==0)
        {
            icon=new FXXPMIcon(getApp());
        }
        else if(comparecase(".png",ext)==0)
		{
            icon=new FXPNGIcon(getApp());
        }

        // Got icon
        if(icon)
        {

            // Find icon in the icon directory
            FXString iconfile=FXFile::search(path,(const char*)ptr);
            if(!iconfile.empty())
            {
                FXFileStream str;

                FXTRACE((150,"IconDict: found icon in = %s\n",iconfile.text()));

                // Try open the file
                if(str.open(iconfile,FXStreamLoad))
                {

                    FXTRACE((150,"IconDict: loading = %s\n",iconfile.text()));

                    // Load it
                    icon->loadPixels(str);

                    // Done
                    str.close();

                    return icon;
                }
            }

            // Failed, delete the icon
            delete icon;
        }
    }
    return NULL;
}


// Free the icon
void IconDict::deleteData(void* ptr)
{
    delete ((FXIcon*)ptr);
}


// Save data
void IconDict::save(FXStream& store) const
{
    FXDict::save(store);
    store << app;
    store << path;
}


// Load data
void IconDict::load(FXStream& store)
{
    FXDict::load(store);
    store >> app;
    store >> path;
}


// Destructor
IconDict::~IconDict()
{
    FXTRACE((100,"IconDict::~IconDict\n"));
    app=(FXApp*)-1;
    clear();
}


// These registry keys are used for default bindings.
const FXchar FileDict::defaultExecBinding[]="defaultexecbinding";
const FXchar FileDict::defaultDirBinding[]="defaultdirbinding";
const FXchar FileDict::defaultFileBinding[]="defaultfilebinding";


// Object implementation
FXIMPLEMENT(FileDict,FXDict,NULL,0)


// Construct an file-extension association table
FileDict::FileDict(FXApp* a):app(a),settings(&a->reg())
{
    FXTRACE((100,"FileDict::FileDict\n"));
    icons=new IconDict(a,settings->readStringEntry("SETTINGS","iconpath",IconDict::defaultIconPath));
}


// Construct an file-extension association table, and alternative settings database
FileDict::FileDict(FXApp* a,FXSettings* db):app(a),settings(db)
{
    FXTRACE((100,"FileDict::FileDict\n"));
    icons=new IconDict(a,settings->readStringEntry("SETTINGS","iconpath",IconDict::defaultIconPath));
}


// Create new association from extension
void *FileDict::createData(const void* ptr)
{
    register const FXchar *p=(const FXchar*)ptr;
    register FXchar *q;
    FXchar command[COMMANDLEN];
    FXchar extension[EXTENSIONLEN];
    FXchar mimetype[MIMETYPELEN];
    FXchar bigname[ICONNAMELEN];
    FXchar bignameopen[ICONNAMELEN];
    FXchar mininame[ICONNAMELEN];
    FXchar mininameopen[ICONNAMELEN];
    FXFileAssoc *fileassoc;

    FXTRACE((300,"FileDict: adding association: %s\n",(FXchar*)ptr));

    // Make association record
    fileassoc=new FXFileAssoc;

    // Parse command
    for(q=command; *p && *p!=';' && q<command+COMMANDLEN-1; *q++=*p++)
        ;
    *q='\0';

    // Skip section separator
    if(*p==';')
        p++;

    // Parse extension type
    for(q=extension; *p && *p!=';' && q<extension+EXTENSIONLEN-1; *q++=*p++)
        ;
    *q='\0';

    // Skip section separator
    if(*p==';')
        p++;

    // Parse big icon name
    for(q=bigname; *p && *p!=';' && *p!=':' && q<bigname+ICONNAMELEN-1; *q++=*p++)
        ;
    *q='\0';

    // Skip icon separator
    if(*p==':')
        p++;

    // Parse big open icon name
    for(q=bignameopen; *p && *p!=';' && q<bignameopen+ICONNAMELEN-1; *q++=*p++)
        ;
    *q='\0';

    // Skip section separator
    if(*p==';')
        p++;

    // Parse mini icon name
    for(q=mininame; *p && *p!=';' && *p!=':' && q<mininame+ICONNAMELEN-1; *q++=*p++)
        ;
    *q='\0';

    // Skip icon separator
    if(*p==':')
        p++;

    // Parse mini open icon name
    for(q=mininameopen; *p && *p!=';' && q<mininameopen+ICONNAMELEN-1; *q++=*p++)
        ;
    *q='\0';

    // Skip section separator
    if(*p==';')
        p++;

    // Parse mime type
    for(q=mimetype; *p && *p!=';' && q<mimetype+MIMETYPELEN-1; *q++=*p++)
        ;
    *q='\0';

    FXTRACE((300,"FileDict: command=\"%s\" extension=\"%s\" mimetype=\"%s\" big=\"%s\" bigopen=\"%s\" mini=\"%s\" miniopen=\"%s\"\n",command,extension,mimetype,bigname,bignameopen,mininame,mininameopen));

    // Initialize association data
    fileassoc->command=command;
    fileassoc->extension=extension;
    fileassoc->bigicon=NULL;
    fileassoc->miniicon=NULL;
    fileassoc->bigiconopen=NULL;
    fileassoc->miniiconopen=NULL;
    fileassoc->mimetype=mimetype;
    fileassoc->dragtype=0;
    fileassoc->flags=0;

    // Insert icons into icon dictionary
    if(bigname[0])
    {
        fileassoc->bigicon=fileassoc->bigiconopen=icons->insert(bigname);
    }
    if(mininame[0])
    {
        fileassoc->miniicon=fileassoc->miniiconopen=icons->insert(mininame);
    }

    // Add open icons also; we will fall back on the regular icons in needed
    if(bignameopen[0])
    {
        fileassoc->bigiconopen=icons->insert(bignameopen);
    }
    if(mininameopen[0])
    {
        fileassoc->miniiconopen=icons->insert(mininameopen);
    }

    // Return the binding
    return fileassoc;
}


// Delete association
void FileDict::deleteData(void* ptr)
{
    delete ((FileAssoc*)ptr);
}


// Set icon search path
void FileDict::setIconPath(const FXString& path)
{

    // Replace iconpath setting in registry
    settings->writeStringEntry("SETTINGS","iconpath",path.text());

    // Change it in icon dictionary
    icons->setIconPath(path);
}


// Return current icon search path
FXString FileDict::getIconPath() const
{
    return icons->getIconPath();
}


// Replace or add file association
FileAssoc* FileDict::replace(const FXchar* ext,const FXchar* str)
{

    // Replace entry in registry
    settings->writeStringEntry("FILETYPES",ext,str);

    // Replace record
    return (FileAssoc*)FXDict::replace(ext,str);
}


// Remove file association
FileAssoc* FileDict::remove
    (const FXchar* ext)
{

    // Delete registry entry for this type
    settings->deleteEntry("FILETYPES",ext);

    // Remove record
    FXDict::remove
        (ext);

    return NULL;
}


// Find file association
FileAssoc* FileDict::associate(const FXchar* key)
{
    register const FXchar *association;
    register FileAssoc* record;

    FXTRACE((300,"FileDict: trying key: %s\n",key));

    // See if we have an existing record already
    if((record=find(key))!=NULL)
        return record;

    // See if this entry is known in FILETYPES
    association=settings->readStringEntry("FILETYPES",key,"");

    // If not an empty string, make a record for it now
    if(association[0])
        return (FileAssoc*)FXDict::insert(key,association);

    // Not a known file type
    return NULL;
}



// Find file association from registry
FileAssoc* FileDict::findFileBinding(const FXchar* pathname)
{
    register const FXchar *filename=pathname;
    register const FXchar *p=pathname;
    register FileAssoc* record;
    FXTRACE((300,"FileDict: searching file binding for: %s\n",pathname));
    while(*p)
    {
        if(ISPATHSEP(*p))
        {
            filename=p+1;
        }
        p++;
    }
    record=associate(filename);
    if(record)
        return record;
    filename=strchr(filename,'.');
    while(filename)
    {
        record=associate(filename+1);

        // Experimental thumbnail support
        //     if(record){
        //       FXIcon* icon=NULL;
        //       if(record->mimetype=="image/x-xpm"){
        //         FXFileStream str;
        //         if(str.open(pathname,FXStreamLoad)){
        //           icon=new FXXPMIcon(getApp());
        //           icon->loadPixels(str);
        //           str.close();
        //         FXTRACE((140,"FileDict: making thumbnail for: %s\n",pathname));
        //           if((icon->getWidth()>32) || (icon->getHeight()>32)){
        //             if(icon->getWidth()>icon->getHeight()){
        //               icon->scale(32,(32*icon->getHeight())/icon->getWidth());
        //               }
        //             else{
        //               icon->scale((32*icon->getWidth())/icon->getHeight(),32);
        //               }
        //             }
        //           record->bigicon=icon;
        //           }
        //         }
        //       }
        if(record)
            return record;
        filename=strchr(filename+1,'.');
    }
    return associate(defaultFileBinding);
}


// Find directory association from registry
FileAssoc* FileDict::findDirBinding(const FXchar* pathname)
{
    register const FXchar* path=pathname;
    register FileAssoc* record;
    FXTRACE((300,"FileDict: searching dir binding for: %s\n",pathname));
    while(*path)
    {
        record=associate(path);
        if(record)
            return record;
        path++;
        while(*path && !ISPATHSEP(*path))
            path++;
    }
    return associate(defaultDirBinding);
}


// Find executable association from registry
FileAssoc* FileDict::findExecBinding(const FXchar* pathname)
{
    FXTRACE((300,"FileDict: searching exec binding for: %s\n",pathname));
    return associate(defaultExecBinding);
}


// Save data
void FileDict::save(FXStream& store) const
{
    FXDict::save(store);
    store << app;
    store << icons;
}


// Load data
void FileDict::load(FXStream& store)
{
    FXDict::load(store);
    store >> app;
    store >> icons;
}


// Destructor
FileDict::~FileDict()
{
    FXTRACE((100,"FileDict::~FileDict\n"));
    delete icons;
    clear();
    app=(FXApp*)-1;
    icons=(IconDict*)-1;
}
