// This is adapted from 'QueryRPM', a simple RPM viewer/installer found
// in X Win Commander

#include "config.h"
#include "i18n.h"

#include <fox/fx.h>
#include <fox/fxkeys.h>
#include <fox/FXPNGIcon.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "icons.h"
#include "FileDialog.h"
#include "MessageBox.h"
#include "XFileQuery.h"
#include "CommandWindow.h"

// Maximum length of a file name
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

char **args;

// Map
FXDEFMAP(XFileQuery) XFileQueryMap[]={
                                         FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_QUIT,XFileQuery::onCmdQuit),
                                         FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_UNINSTALL,XFileQuery::onCmdUninstall),
                                       	 FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_ABOUT,XFileQuery::onCmdAbout),
                                       	 FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_OPEN,XFileQuery::onCmdOpen),
                                         FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_UPGRADE,XFileQuery::onCmdUpgrade),
                                         FXMAPFUNC(SEL_UPDATE,XFileQuery::ID_DESCRIPTION,XFileQuery::onUpdDescription),
                                         FXMAPFUNC(SEL_UPDATE,XFileQuery::ID_FILELIST,XFileQuery::onUpdFileList),
                                     };


// Object implementation
FXIMPLEMENT(XFileQuery,FXMainWindow,XFileQueryMap,ARRAYNUMBER(XFileQueryMap))

// Constructor
XFileQuery::XFileQuery(FXApp *a, char *file):FXMainWindow(a,"Xfq",NULL,NULL,DECOR_ALL,0,0,600,500,0,0)
{
    FXHotKey hotkey;
    FXButton *btn = NULL;

    FXIcon *bigicon=new FXPNGIcon(getApp(),xfq);
    setIcon(bigicon);

    // Create icons
	FXPNGIcon *fileopenicon=new FXPNGIcon(getApp(),fileopen);
	FXPNGIcon *quiticon=new FXPNGIcon(getApp(),quit);

    // Make menu bar
    menubar=new FXMenubar(this,FRAME_RAISED|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_THICK);

    // Toolbar
    FXHorizontalFrame* toolbarcontainer=new FXHorizontalFrame(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X,0,0,0,0, 0,0,0,0, 0,0);
    toolbar=new FXHorizontalFrame(toolbarcontainer,FRAME_RAISED|PACK_UNIFORM_WIDTH|LAYOUT_SIDE_TOP|LAYOUT_FILL_X,0,0,0,0, 4,4,4,4, 0,0);
    int style=BUTTON_TOOLBAR;

    // File menu
    filemenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&File"),NULL,filemenu);

    // Preferences Menu
    FXMenuPane *prefsmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Preferences"),NULL,prefsmenu);

    // Help menu
    helpmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Help"),NULL,helpmenu);

    // Toolbar button: File open
    btn = new FXButton(toolbar,_("\tOpen package file (Ctrl-O)"),fileopenicon,this,ID_OPEN,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey = (0004<<16) | KEY_o;
    btn->addHotKey(hotkey);

    // Toolbar button: Quit
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
    btn = new FXButton(toolbar,_("\tQuit Xfq (Ctrl-Q, F10)"),quiticon,this,ID_QUIT,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey = (0004<<16) | KEY_q;
    btn->addHotKey(hotkey);
    btn->addHotKey(KEY_F10);

    // File Menu entries
    new FXMenuCommand(filemenu,_("Open file...\tCtrl-O"),fileopenicon,this,ID_OPEN);
    new FXMenuCommand(filemenu,_("Quit\tCtrl-Q"),quiticon,this,ID_QUIT);

    // Preferences menu
    new FXMenuCommand(prefsmenu,_("Toolbar\t\tDisplay toolbar"),NULL,toolbar,FXWindow::ID_TOGGLESHOWN);

    // Help Menu entries
    new FXMenuCommand(helpmenu,_("About X File Query\tF1"),NULL,this,ID_ABOUT,0);

    // Make a tool tip
    new FXTooltip(getApp(),0);

	// Set package name and clean list flags
	if (file)
	{
    	filename=file;
    	desc_clean=TRUE;
    	list_clean=TRUE;
	}
	else
	{
		filename=NULL;
		desc_clean=FALSE;
    	list_clean=FALSE;
	}

    // Buttons
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,0,0,10,10,5,5);

    // Make contents
    contents=new FXHorizontalFrame(this,FRAME_RAISED|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 2,2,2,2, 0,0/*,0,0,0,0, 0,0,0,0*/);

    // Close
    new FXButton(buttons,_("&Close"),NULL,this,XFileQuery::ID_QUIT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Uninstall
    new FXButton(buttons,_("&Uninstall"),NULL,this,XFileQuery::ID_UNINSTALL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Install/Upgrade
    new FXButton(buttons,_("&Install/Upgrade"),NULL,this,XFileQuery::ID_UPGRADE,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Switcher
    FXTabBook *tabbook = new FXTabBook(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_RIGHT);

    // First item is Description
    new FXTabItem(tabbook,_("&Description"),NULL);
    FXVerticalFrame *descr=new FXVerticalFrame(tabbook,FRAME_THICK|FRAME_RAISED);
    FXPacker *p=new FXPacker(descr,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    description=new FXText(p,this,XFileQuery::ID_DESCRIPTION,FRAME_SUNKEN|LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Second item is File List
    new FXTabItem(tabbook,_("&File List"),NULL);
    FXVerticalFrame *flistfr=new FXVerticalFrame(tabbook,FRAME_THICK|FRAME_RAISED);
    p=new FXPacker(flistfr,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    list=new FXTreeList(p,20,this,XFileQuery::ID_FILELIST,LAYOUT_FILL_X|LAYOUT_FILL_Y|TREELIST_SHOWS_LINES);

	// Folder icons for file list
    folder_open=new FXPNGIcon(getApp(),package);
    folder_closed=new FXPNGIcon(getApp(),package);

    // Document icon for file list
    doc=new FXPNGIcon(getApp(),minidoc);
}


// About box
long XFileQuery::onCmdAbout(FXObject*,FXSelector,void*)
{
    FXPNGIcon icon(getApp(),xfq);
    MessageBox about(this,_("About X File Query"),_("X File Query is a simple RPM manager.\n\nCopyright (C) 2002-2004 Roland Baudin (roland65@free.fr)"),&icon,MBOX_OK|DECOR_TITLE|DECOR_BORDER);
    about.execute();
    return 1;
}

// Open package
long XFileQuery::onCmdOpen(FXObject*,FXSelector,void*)
{
    const FXchar *patterns[]=
        {
            _("All Files"),			"*",
            _("RPM source packages"),	"*.src.rpm",
            _("RPM packages"),			"*.rpm",
			NULL
        };
    FileDialog open(this,_("Open Document"));
    open.setFilename(filename);
    open.setPatternList(patterns);
    if(open.execute())
    {
        filename=open.getFilename();
		list_clean=TRUE;
		desc_clean=TRUE;
    }
    return 1;
}

// Install/upgrade RPM package
long XFileQuery::onCmdUpgrade(FXObject*,FXSelector,void*)
{
	getApp()->flush();

	if (strlen(filename.text())==0)
	{
        MessageBox::error(this,MBOX_OK,_("Error"),_("No RPM package loaded"));
		return 0;
	}
		
	// Package name
    FXString package;
	package=FXFile::name(filename);

    // Command to perform
	FXString cmd="rpm -Uvh " +filename;

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp()->getMainWindow(),_("Install/Upgrade RPM"),cmd,10,80);			
	cmdwin->create();

	// The CommandWindow object will delete itself when closed!

    return 1;
}


// Uninstall RPM package based on the package name (version is ignored)
long XFileQuery::onCmdUninstall(FXObject*,FXSelector,void*)
{
	getApp()->flush();
	
	if (strlen(filename.text())==0)
	{
        MessageBox::error(this,MBOX_OK,_("Error"),_("No RPM package loaded"));
		return 0;
	}

	// Get package name
    FXString package;
	package=FXFile::name(filename);
	package=package.extract(0,'-');
    
	// Uninstall command
	FXString cmd="rpm -e " + package;

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(getApp()->getMainWindow(),_("Uninstall RPM"),cmd,10,80);			
	cmdwin->create();

	// The CommandWindow object will delete itself when closed!

    return 1;
}


// Save configuration when quitting
void XFileQuery::saveConfig()
{
    // Write new window size back to registry
    getApp()->reg().writeIntEntry("SETTINGS","width",getWidth());
    getApp()->reg().writeIntEntry("SETTINGS","height",getHeight());

    // Last opened filename
    getApp()->reg().writeStringEntry("SETTINGS","filename",filename.text());

    // Toolbar status
	if(toolbar->shown())
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",TRUE);
    else
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",FALSE);

	// Write registry settings
	getApp()->reg().write();
}


// Quit application
long XFileQuery::onCmdQuit(FXObject*,FXSelector,void*)
{
	// Save settings 
    saveConfig();

    // Exit
	getApp()->exit(0);
    return 1;
}


// Update file list
long XFileQuery::onUpdFileList(FXObject*,FXSelector,void*)
{
	if(!list_clean)
        return 0;
	
    list_clean=FALSE;
    FXString cmd="rpm -qlp '"+filename+"'";
    FILE *rpmcmd=popen(cmd.text(),"r");
    if(!rpmcmd)
    {
        perror("popen");
        exit(1);
    }
    char text[10000]={0};
    FXString str;
    str=FXFile::name(filename);
    strcpy(text,str.text());

	// Clear list
    list->clearItems();

    // First item
	FXTreeItem *topmost;
	topmost=list->addItemFirst(NULL,text,folder_open,folder_closed);
    
	// Next items
	while(fgets(text,sizeof(text),rpmcmd))
    {
        text[strlen(text)-1]='\0';// kill trailing lf
        list->addItemLast(topmost,text,doc,doc);
    }
    list->expandTree(topmost);
    pclose(rpmcmd);
    return 1;
}


// Update description
long XFileQuery::onUpdDescription(FXObject*,FXSelector,void*)
{
    if(!desc_clean)
        return 0;
    desc_clean=FALSE;
    FXString cmd="rpm -qip '"+filename+"'";
    FILE *rpmcmd=popen(cmd.text(),"r");
    if(!rpmcmd)
    {
        perror("popen");
        exit(1);
    }
    char text[10000]={0};
    FXString buf;
    while(fgets(text,sizeof(text),rpmcmd))
        buf+=text;
    if(pclose(rpmcmd))
    {
        MessageBox::error(this,MBOX_OK,_("Error"),_("Query of %s failed!"),filename.text());
        exit(1);
    }
    else
        description->setText(buf.text());

    return 1;
}

void XFileQuery::create()
{
    // Read the Xfe registry
	FXRegistry *reg;
  	reg=new FXRegistry("Xfe",NULL);
	reg->read();
	
	// Set Xfq text font according to the Xfe registry
    FXFontDesc fontdesc;
    FXFont *font = NULL;
    if(fxparsefontdesc(fontdesc,reg->readStringEntry("SETTINGS","textfont","[helvetica] 100 500 1 0 0 0")))
    {
        font=new FXFont(getApp(),fontdesc);
        description->setFont(font);
        list->setFont(font);
    }

	// Create icons
	folder_open->create();
	folder_closed->create();
	doc->create();

	// Wheel scrolling
    FXint wheellines=getApp()->reg().readUnsignedEntry("SETTINGS","wheellines",5);
	getApp()->setWheelLines(wheellines);
	
    // Get size
    width=getApp()->reg().readIntEntry("SETTINGS","width",600);
    height=getApp()->reg().readIntEntry("SETTINGS","height",400);

	// Get toolbar status
    if(getApp()->reg().readUnsignedEntry("SETTINGS","showtoolbar",TRUE)==FALSE)
        toolbar->hide();

    // Reposition window
    position(0,0,width,height);

    handle(this,MKUINT(XFileQuery::ID_FILELIST,SEL_UPDATE),NULL);
    handle(this,MKUINT(XFileQuery::ID_DESCRIPTION,SEL_UPDATE),NULL);
    FXMainWindow::create();
    show();
    description->handle(this,MKUINT(FXText::ID_TOGGLE_EDITABLE,SEL_COMMAND),NULL);
}


// Standard X Window arrow cursor definition
#define left_ptr_width 16
#define left_ptr_height 16
#define left_ptr_x_hot 3
#define left_ptr_y_hot 1
static char left_ptr_bits[] = {
                                  0x00, 0x00, 0x08, 0x00, 0x18, 0x00, 0x38, 0x00, 0x78, 0x00, 0xf8, 0x00,
                                  0xf8, 0x01, 0xf8, 0x03, 0xf8, 0x07, 0xf8, 0x00, 0xd8, 0x00, 0x88, 0x01,
                                  0x80, 0x01, 0x00, 0x03, 0x00, 0x03, 0x00, 0x00};

#define left_ptrmsk_width 16
#define left_ptrmsk_height 16
static char left_ptrmsk_bits[] = {
                                     0x0c, 0x00, 0x1c, 0x00, 0x3c, 0x00, 0x7c, 0x00, 0xfc, 0x00, 0xfc, 0x01,
                                     0xfc, 0x03, 0xfc, 0x07, 0xfc, 0x0f, 0xfc, 0x0f, 0xfc, 0x01, 0xdc, 0x03,
                                     0xcc, 0x03, 0x80, 0x07, 0x80, 0x07, 0x00, 0x03};



// Main function to start the program
int main(int argc,char* argv[])
{
#ifdef HAVE_SETLOCALE
  // Set locale via LC_ALL.
  setlocale (LC_ALL, "");
#endif

#if ENABLE_NLS
  // Set the text message domain.
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif

	XFileQuery *window;
	
	// Global variable
	args=argv;

    // Make application
    FXApp* application=new FXApp("Xfq",NULL);

    // Open display
    application->init(argc,argv);

    // Redefine the default cursor to get the standard X Window arrow cursor
    FXCursor* left_ptr=new FXCursor(application,left_ptr_bits,left_ptrmsk_bits,left_ptr_width,left_ptr_height,left_ptr_x_hot,left_ptr_y_hot);
    application->setDefaultCursor(DEF_ARROW_CURSOR,left_ptr);

    // Read the Xfe registry
	FXRegistry *reg;
  	reg=new FXRegistry("Xfe",NULL);
	reg->read();
	
	// Set Xfq normal font according to the Xfe registry
    FXFontDesc fontdesc;
    FXFont *normalFont = NULL;
    if(fxparsefontdesc(fontdesc,reg->readStringEntry("SETTINGS","font","[helvetica] 100 500 1 0 0 0")))
    {
        normalFont=new FXFont(application,fontdesc);
        application->setNormalFont(normalFont);
    }
	
	// Set Xfq colors according to the Xfe registry
	FXColor basecolor,forecolor,backcolor,selforecolor,selbackcolor,bordercolor;
    basecolor=reg->readColorEntry("SETTINGS","basecolor",FXRGB(189,189,189));
    forecolor=reg->readColorEntry("SETTINGS","forecolor",FXRGB(0,0,0));
    backcolor=reg->readColorEntry("SETTINGS","backcolor",FXRGB(255,255,255));
    selforecolor=reg->readColorEntry("SETTINGS","selforecolor",FXRGB(255,255,255));
    selbackcolor=reg->readColorEntry("SETTINGS","selbackcolor",FXRGB(0,0,132));
    bordercolor=reg->readColorEntry("SETTINGS","bordercolor",FXRGB(0,0,0));
	application->setBaseColor(basecolor);
	application->setForeColor(forecolor);
	application->setBackColor(backcolor);
	application->setSelforeColor(selforecolor);
	application->setSelbackColor(selbackcolor);
	application->setBorderColor(bordercolor);

	// Delete the Xfe registry
	delete reg;

    // Make window
	if (argc==2)
		window=new XFileQuery(application,argv[1]);
	else
		window=new XFileQuery(application,NULL);
		
    // Create it
    application->create();

    // Test the existence of the RedHat package manager (RPM)
	// This is done by checking the existence of the string 'RPM'
	// in the response to the command 'rpm' (with no arguments)
	FXString cmd="rpm 2>&1";
    FILE *rpmcmd=popen(cmd.text(),"r");
    if(!rpmcmd)
    {
        perror("popen");
        exit(1);
    }
    char text[10000]={0};
    FXString buf;
    while(fgets(text,sizeof(text),rpmcmd))
        buf+=text;
    sprintf(text,"%s",buf.text());
	if (strstr(text,"RPM")==NULL)
	{
        MessageBox::error(window,MBOX_OK,_("Error"),_("RedHat package manager (RPM) not found!"));
		exit(1);
	}

    // Run the application
    application->run();

    return 0;
}
