// This code is adapted from 'imageviewer', a demo image viewer found
// in the FOX library and written by Jeroen van der Zijp.

#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <fox-1.4/fx.h>
#include <fox-1.4/fxkeys.h>
#include <fox-1.4/FXPNGImage.h>
#include <fox-1.4/FXJPGImage.h>
#include <fox-1.4/FXTIFImage.h>
#include <fox-1.4/FXPNGIcon.h>

#include "icons.h"
#include "FileList.h"
#include "FileDialog.h"
#include "MessageBox.h"
#include "XFileImage.h"


FXMainWindow *mainWindow;
char **args;

// Predefined zoom factors 
#define NB_ZOOM 24
FXdouble zoomtab[NB_ZOOM]={0.01, 0.025, 0.05, 0.075, 0.10, 0.15, 0.20, 0.30, 0.50, 0.75, 1, \
                        1.5, 2, 3, 4, 5, 7.5, 10, 15, 20, 30, 50, 75, 100};
#define ZOOM_100 10

// Maximum image size (in pixels) for zooming in
#define MAX_IMGSIZE 5120

// Patterns for supported image formats
const FXchar *patterns[]=
{
    _("All Files"),		"*",
    _("GIF Image"),		"*.gif",
    _("BMP Image"),		"*.bmp",
    _("XPM Image"),		"*.xpm",
    _("PCX Image"),		"*.pcx",
    _("ICO Image"),		"*.ico",
    _("RGB Image"),		"*.rgb",
    _("XBM Image"),		"*.xbm",
    _("TARGA Image"), 	"*.tga",
    _("PPM Image"),		"*.ppm",
    _("PNG Image"),		"*.png",
    _("JPEG Image"),	"*.jpg",
    _("TIFF Image"),	"*.tif",
	NULL
};

// Helper function to draw a toolbar separator
void toolbarSeparator(FXToolBar* tb)
{
#define SEP_SPACE	1
#define SEP_HEIGHT	15
    new FXFrame(tb,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,SEP_SPACE);
    new FXVerticalSeparator(tb,LAYOUT_SIDE_TOP|LAYOUT_CENTER_Y|SEPARATOR_GROOVE|LAYOUT_FIX_HEIGHT,0,0,0,SEP_HEIGHT);
    new FXFrame(tb,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,SEP_SPACE);
}

// Map
FXDEFMAP(XFileImage) XFileImageMap[]={
										FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ABOUT,XFileImage::onCmdAbout),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_OPEN,XFileImage::onCmdOpen),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_TITLE,XFileImage::onUpdTitle),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_QUIT,XFileImage::onCmdQuit),
                                        FXMAPFUNC(SEL_SIGNAL,XFileImage::ID_QUIT,XFileImage::onCmdQuit),
                                        FXMAPFUNC(SEL_CLOSE,XFileImage::ID_TITLE,XFileImage::onCmdQuit),
                                        FXMAPFUNC(SEL_DOUBLECLICKED,XFileImage::ID_FILELIST,XFileImage::onCmdFileList),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_RECENTFILE,XFileImage::onCmdRecentFile),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_TOGGLE_HIDDEN,XFileImage::onCmdToggleHidden),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_TOGGLE_THUMBNAILS,XFileImage::onCmdToggleThumbnails),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_SHOW_DETAILS,XFileImage::onCmdShowDetails),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_SHOW_MINI,XFileImage::onCmdShowMini),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_SHOW_BIG,XFileImage::onCmdShowBig),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ROWS,XFileImage::onCmdShowRows),
										FXMAPFUNC(SEL_COMMAND,XFileImage::ID_COLS,XFileImage::onCmdShowCols),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ROTATE_90,XFileImage::onCmdRotate),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ROTATE_270,XFileImage::onCmdRotate),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_MIRROR_HOR,XFileImage::onCmdMirror),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_MIRROR_VER,XFileImage::onCmdMirror),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ZOOM_IN,XFileImage::onCmdZoomIn),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ZOOM_OUT,XFileImage::onCmdZoomOut),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ZOOM_100,XFileImage::onCmdZoom100),
                                        FXMAPFUNC(SEL_COMMAND,XFileImage::ID_ZOOM_WIN,XFileImage::onCmdZoomWin),
	                                    FXMAPFUNC(SEL_COMMAND,XFileImage::ID_PRINT,XFileImage::onCmdPrint),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_PRINT,XFileImage::onUpdImage),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_ROTATE_90,XFileImage::onUpdImage),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_ROTATE_270,XFileImage::onUpdImage),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_MIRROR_HOR,XFileImage::onUpdImage),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_MIRROR_VER,XFileImage::onUpdImage),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_ZOOM_IN,XFileImage::onUpdImage),                                        
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_ZOOM_OUT,XFileImage::onUpdImage),                                        
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_ZOOM_100,XFileImage::onUpdImage),                                        
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_ZOOM_WIN,XFileImage::onUpdImage),                                        
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_SHOW_BIG,XFileImage::onUpdFileView),
										FXMAPFUNC(SEL_UPDATE,XFileImage::ID_SHOW_MINI,XFileImage::onUpdFileView),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_SHOW_DETAILS,XFileImage::onUpdFileView),                                        
 										FXMAPFUNC(SEL_UPDATE,XFileImage::ID_ROWS,XFileImage::onUpdIconView),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_COLS,XFileImage::onUpdIconView),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_TOGGLE_HIDDEN,XFileImage::onUpdToggleHidden),
                                        FXMAPFUNC(SEL_UPDATE,XFileImage::ID_TOGGLE_THUMBNAILS,XFileImage::onUpdToggleThumbnails),
                               			FXMAPFUNC(SEL_UPDATE,XFileImage::ID_LABEL,XFileImage::onUpdLabel),
                                    };


// Object implementation
FXIMPLEMENT(XFileImage,FXMainWindow,XFileImageMap,ARRAYNUMBER(XFileImageMap))


// Make some windows
XFileImage::XFileImage(FXApp* a):FXMainWindow(a,"Xfi ",NULL,NULL,DECOR_ALL,0,0,600,500,0,0)
{
    FXIcon *bigp=new FXPNGIcon(getApp(),xfi);
    setIcon(bigp);

    FXHotKey hotkey;
    FXButton *btn = NULL;

    setTarget(this);
    setSelector(ID_TITLE);

    // Create icons
    uplevelicon=new FXPNGIcon(getApp(),minifolderup);
	FXPNGIcon *fileopenicon=new FXPNGIcon(getApp(),fileopen);
	FXPNGIcon *quiticon=new FXPNGIcon(getApp(),quit);
    FXPNGIcon *rotatelefticon=new FXPNGIcon(getApp(),rotateleft);
    FXPNGIcon *rotaterighticon=new FXPNGIcon(getApp(),rotateright);
    FXPNGIcon *zoominicon=new FXPNGIcon(getApp(),zoomin);
    FXPNGIcon *zoomouticon=new FXPNGIcon(getApp(),zoomout);
   	FXPNGIcon *zoom100icon=new FXPNGIcon(getApp(),zoom100);
    FXPNGIcon *zoomwinicon=new FXPNGIcon(getApp(),zoomwin);
   	FXPNGIcon *mirrorhoricon=new FXPNGIcon(getApp(),fliplr);
    FXPNGIcon *mirrorvericon=new FXPNGIcon(getApp(),flipud);
	FXPNGIcon *printicon=new FXPNGIcon(getApp(),print);	
	FXPNGIcon *showhiddenicon=new FXPNGIcon(getApp(),showhidden);
	FXPNGIcon *hidehiddenicon=new FXPNGIcon(getApp(),hidehidden);
	FXPNGIcon *showthumbicon=new FXPNGIcon(getApp(),showthumb);
	FXPNGIcon *hidethumbicon=new FXPNGIcon(getApp(),hidethumb);
	FXPNGIcon *largeicon=new FXPNGIcon(getApp(),bigicons);
	FXPNGIcon *smallicon=new FXPNGIcon(getApp(),smallicons);
	FXPNGIcon *detailicon=new FXPNGIcon(getApp(),details);

	// Site where to dock
	FXDockSite *docksite=new FXDockSite(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X);

    // Make menu bar
  	FXToolBarShell* dragshell=new FXToolBarShell(this,FRAME_RAISED);
    menubar=new FXMenuBar(docksite,dragshell,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
	new FXToolBarGrip(menubar,menubar,FXMenuBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);

    // Tool bar
  	FXToolBarShell* dragshell1=new FXToolBarShell(this,FRAME_RAISED);
  	toolbar=new FXToolBar(docksite,dragshell1,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
  	new FXToolBarGrip(toolbar,toolbar,FXToolBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);

    // File menu
    filemenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&File"),NULL,filemenu);

    // Image Menu
    imagemenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Image"),NULL,imagemenu);

    // Preferences Menu
    prefsmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Preferences"),NULL,prefsmenu);

    // View menu
    viewmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&View"),NULL,viewmenu);

    // Help menu
    helpmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Help"),NULL,helpmenu);

    // Splitter
    splitter=new FXSplitter(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y|SPLITTER_TRACKING|SPLITTER_VERTICAL|SPLITTER_REVERSED);

    // Sunken border for image widget
    FXHorizontalFrame *imagebox=new FXHorizontalFrame(splitter,FRAME_SUNKEN|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);

    // Make image widget
    imageview=new FXImageView(imagebox,NULL,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Sunken border for file list
    filebox=new FXHorizontalFrame(splitter,LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);

    // Make file list
    FXVerticalFrame* fileframe=new FXVerticalFrame(filebox,FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0, 0,0);
    label=new FXLabel(fileframe," ",NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_RAISED);
    label->setTarget(this);
    label->setSelector(FXSEL(0,XFileImage::ID_LABEL));
    filelist=new FileList(fileframe,this,ID_FILELIST,LAYOUT_FILL_X|LAYOUT_FILL_Y|ICONLIST_MINI_ICONS|ICONLIST_AUTOSIZE);

    // Status bar
	statusbar=new FXHorizontalFrame(fileframe,JUSTIFY_LEFT|LAYOUT_FILL_X|FRAME_RAISED,0,0,0,0, 3,3,0,0);
    new FXToggleButton(statusbar,_("\tShow hidden files (Ctrl-F6)"),_("\tHide hidden files (Ctrl-F6)"),showhiddenicon,hidehiddenicon,this->filelist,
	                   FileList::ID_TOGGLE_HIDDEN,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT);

    new FXToggleButton(statusbar,_("\tShow thumbnails (Ctrl-F7)"),_("\tHide thumbnails (Ctrl-F7)"),showthumbicon,hidethumbicon,this->filelist,
	                   FileList::ID_TOGGLE_THUMBNAILS,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT);
	new FXStatusBar(statusbar,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X);


    // Toolbar button: Open file
    new FXButton(toolbar,_("\tOpen Image (Ctrl-O)\tOpen image file. (Ctrl-O)"),fileopenicon,this,ID_OPEN,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);

    // Toolbar button: Print
	new FXButton(toolbar,_("\tPrint image (Ctrl-P)\tPrint image file. (Ctrl-P)"),printicon,this,ID_PRINT,BUTTON_TOOLBAR|FRAME_RAISED);

    // Toolbar button: Quit
    new FXButton(toolbar,_("\tQuit Xfi (Ctrl-Q)\tQuit Xfi. (Ctrl-Q)"),quiticon,this,ID_QUIT,BUTTON_TOOLBAR|FRAME_RAISED);

	// Separator
	toolbarSeparator(toolbar);

    // Toolbar button: Zoom in
    btn=new FXButton(toolbar,_("\tZoom in (Ctrl+)\tZoom in image. (Ctrl+)"),zoominicon,this,ID_ZOOM_IN,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);
    hotkey=(0004<<16) |  KEY_KP_Add;
    btn->addHotKey(hotkey);

    // Toolbar button: Zoom out
    btn=new FXButton(toolbar,_("\tZoom out (Ctrl-)\tZoom out image. (Ctrl-)"),zoomouticon,this,ID_ZOOM_OUT,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);
    hotkey=(0004<<16) | KEY_KP_Subtract;
    btn->addHotKey(hotkey);

    // Toolbar button: Zoom 100%
    new FXButton(toolbar,_("\tZoom 100% (Ctrl-I)\tZoom 100% image. (Ctrl-I)"),zoom100icon,this,ID_ZOOM_100,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);

    // Toolbar button: Zoom to fit window
    new FXButton(toolbar,_("\tZoom to fit window (Ctrl-W)\tZoom to fit window. (Ctrl-W)"),zoomwinicon,this,ID_ZOOM_WIN,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);

	// Separator
	toolbarSeparator(toolbar);

    // Toolbar button: Rotate left
    new FXButton(toolbar,_("\tRotate left (Ctrl-L)\tRotate left image. (Ctrl-L)"),rotatelefticon,this,ID_ROTATE_90,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);

    // Toolbar button: Rotate right
    new FXButton(toolbar,_("\tRotate right (Ctrl-R)\tRotate right image. (Ctrl-R)"),rotaterighticon,this,ID_ROTATE_270,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);

    // Toolbar button: mirror horizontally
    new FXButton(toolbar,_("\tMirror horizontally (Ctrl-H)\tMirror image horizontally. (Ctrl-H)"),mirrorhoricon,this,ID_MIRROR_HOR,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);

    // Toolbar button: mirror vertically
    new FXButton(toolbar,_("\tMirror vertically (Ctrl-V)\tMirror image vertically. (Ctrl-V)"),mirrorvericon,this,ID_MIRROR_VER,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED);

    // Switch display modes
	toolbarSeparator(toolbar);
    new FXButton(toolbar,_("\tBig icons (F10)\tDisplay directory with big icons. (F10)"),largeicon,this,ID_SHOW_BIG,BUTTON_TOOLBAR|LAYOUT_TOP|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);
    new FXButton(toolbar,_("\tSmall icons (F11)\tDisplay directory with small icons. (F11)"),smallicon,this,ID_SHOW_MINI,BUTTON_TOOLBAR|LAYOUT_TOP|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);
    new FXButton(toolbar,_("\tFull file list (F12)\tDisplay detailed directory listing. (F12)"),detailicon,this,ID_SHOW_DETAILS,BUTTON_TOOLBAR|LAYOUT_TOP|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);

    // File Menu entries
    new FXMenuCommand(filemenu,_("&Open...\tCtrl-O\tOpen image file. (Ctrl-O)"),fileopenicon,this,ID_OPEN);
    new FXMenuCommand(filemenu,_("&Print image...\tCtrl-P\tPrint image file. (Ctrl-P)"),printicon,this,ID_PRINT);

    // Recent file menu; this automatically hides if there are no files
    FXMenuSeparator* sep1=new FXMenuSeparator(filemenu);
    sep1->setTarget(&mrufiles);
    sep1->setSelector(FXRecentFiles::ID_ANYFILES);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_1);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_2);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_3);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_4);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_5);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_6);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_7);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_8);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_9);
    new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_10);
    new FXMenuCommand(filemenu,_("&Clear recent files\t\tClear recent files."),NULL,&mrufiles,FXRecentFiles::ID_CLEAR);
    FXMenuSeparator* sep2=new FXMenuSeparator(filemenu);
    sep2->setTarget(&mrufiles);
    sep2->setSelector(FXRecentFiles::ID_ANYFILES);
    new FXMenuCommand(filemenu,_("&Quit\tCtrl-Q\tQuit Xfi. (Ctrl-Q)"),quiticon,this,ID_QUIT);

    // Image Menu entries
    new FXMenuCommand(imagemenu,_("Zoom &in\tCtrl+\tZoom in image. (Ctrl+)"),zoominicon,this,ID_ZOOM_IN);
    new FXMenuCommand(imagemenu,_("Zoom &out\tCtrl-\tZoom out image. (Ctrl-)"),zoomouticon,this,ID_ZOOM_OUT);
    new FXMenuCommand(imagemenu,_("Zoo&m 100%\tCtrl-I\tZoom image to 100%. (Ctrl-I)"),zoom100icon,this,ID_ZOOM_100);
    new FXMenuCommand(imagemenu,_("Zoom to fit &window\tCtrl-W\tZoomto fit window. (Ctrl-W)"),zoomwinicon,this,ID_ZOOM_WIN);
    new FXMenuCommand(imagemenu,_("Rotate &right\tCtrl-R\tRotate right. (Ctrl-R)"),rotaterighticon,this,ID_ROTATE_270);
    new FXMenuCommand(imagemenu,_("Rotate &left\tCtrl-L\tRotate left. (Ctrl-L)"),rotatelefticon,this,ID_ROTATE_90);
    new FXMenuCommand(imagemenu,_("Mirror &horizontally\tCtrl-H\tMirror horizontally. (Ctrl-H)"),mirrorhoricon,this,ID_MIRROR_HOR);
    new FXMenuCommand(imagemenu,_("Mirror &vertically\tCtrl-V\tMirror vertically. (Ctrl-V)"),mirrorvericon,this,ID_MIRROR_VER);

    // Preferences menu
    new FXMenuCheck(prefsmenu,_("&Toolbar\t\tDisplay toolbar."),toolbar,FXWindow::ID_TOGGLESHOWN);
    new FXMenuCheck(prefsmenu,_("&File list\t\tDisplay file list."),filebox,FXWindow::ID_TOGGLESHOWN);

    // View Menu entries
    new FXMenuCheck(viewmenu,_("&Hidden files\tCtrl-F6\tShow hidden files and directories. (Ctrl-F6)"),this,ID_TOGGLE_HIDDEN);
    new FXMenuCheck(viewmenu,_("&Thumbnails\tCtrl-F7\tShow image thumbnails. (Ctrl-F7)"),this,ID_TOGGLE_THUMBNAILS);
    new FXMenuSeparator(viewmenu);
    new FXMenuRadio(viewmenu,_("&Big icons\tF10\tDisplay directory with big icons (F10)."),this,ID_SHOW_BIG);
    new FXMenuRadio(viewmenu,_("&Small icons\tF11\tDisplay directory with small icons (F11)."),this,ID_SHOW_MINI);
    new FXMenuRadio(viewmenu,_("&Full file list\tF12\tDisplay detailed directory listing (F12)."),this,ID_SHOW_DETAILS);
    new FXMenuSeparator(viewmenu);
    new FXMenuRadio(viewmenu,_("&Rows of icons\t\tView row-wise."),this,ID_ROWS);
    new FXMenuRadio(viewmenu,_("&Columns of icons\t\tView column-wise."),this,ID_COLS);

    // Help Menu entries
    new FXMenuCommand(helpmenu,_("&About X File Image\tF1\tAbout X File Image (F1)."),NULL,this,ID_ABOUT,0);

    // Make a tool tip
    new FXToolTip(getApp(),TOOLTIP_NORMAL);

    // Recent files
    mrufiles.setTarget(this);
    mrufiles.setSelector(ID_RECENTFILE);

    // Initialize file name
    filename="";
	
	// Initialize some flags
	fileview=ID_SHOW_MINI;
	iconview=ID_COLS;
	hiddenfiles=FALSE;	
	thumbnails=FALSE;	
	
	// Initialize zoom to 100%
	indZoom=ZOOM_100;
	zoomval=zoomtab[indZoom];
}


// Clean up
XFileImage::~XFileImage()
{
    delete filemenu;
    delete imagemenu;
    delete helpmenu;
    delete prefsmenu;
    delete viewmenu;
    delete dragshell1;
    delete dragshell2;
    delete uplevelicon;
}


// About box
long XFileImage::onCmdAbout(FXObject*,FXSelector,void*)
{
    FXPNGIcon icon(getApp(),xfi);
    MessageBox about(this,_("About X File Image"),_("X File Image is a simple image viewer.\n\nCopyright (C) 2002-2005 Roland Baudin (roland65@free.fr)"),&icon,MBOX_OK|DECOR_TITLE|DECOR_BORDER);
    about.execute();
    return 1;
}


// Load file
FXbool XFileImage::loadimage(const FXString& file)
{
    struct stat info;
    FILE *fp;

    FXString ext=FXFile::extension(file);

    if(stat(file.text(),&info)!=0)
    {
        MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Non-existing file: %s"),file.text());
        return FALSE;
    }
    else
	{
		fp=fopen(file.text(),"r");
    	if(!fp)
    	{
        	MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Unable to open file: %s"),file.text());
        	return FALSE;
    	}
		else
			fclose(fp);
	}

    if(comparecase(ext,"gif")==0)
    {
        img=new FXGIFImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXGIFImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"bmp")==0)
    {
        img=new FXBMPImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXBMPImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"xpm")==0)
    {
        img=new FXXPMImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXXPMImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
   }
    else if(comparecase(ext,"pcx")==0)
    {
        img=new FXPCXImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXPCXImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"ico")==0 || comparecase(ext,"cur")==0)
    {
        img=new FXICOImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXICOImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"tga")==0)
    {
        img=new FXTGAImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXTGAImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"rgb")==0)
    {
        img=new FXRGBImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"xbm")==0)
    {
        img=new FXXBMImage(getApp(),NULL,NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXXBMImage(getApp(),NULL,NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"ppm")==0)
    {
        img=new FXPPMImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXPPMImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"png")==0)
    {
        img=new FXPNGImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXPNGImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"jpg")==0)
    {
        img=new FXJPGImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXJPGImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }
    else if(comparecase(ext,"tif")==0 || comparecase(ext,"tiff")==0)
    {
        img=new FXTIFImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
        tmpimg=new FXTIFImage(getApp(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
    }

    // Perhaps failed
    if(img==NULL || tmpimg==NULL)
    {
        FXMessageBox::error(this,MBOX_OK,_("Error Loading Image"),_("Unsupported type: %s"),ext.text());
        return FALSE;
    }

    // Load it
    FXFileStream stream;
    if(stream.open(file,FXStreamLoad))
    {
        getApp()->beginWaitCursor();
        img->loadPixels(stream);
        stream.close();

    	if (!FXMEMDUP(&tmpdata,img->getData(),FXColor,img->getWidth()*img->getHeight()))
			throw FXMemoryException(_("Unable to load image"));
		tmpimg->setData(tmpdata,IMAGE_OWNED,img->getWidth(),img->getHeight());	

		img->create();
		tmpimg->create();

    	imageview->setImage(tmpimg);
	
		// Initial zoom and image format
		indZoom=ZOOM_100;
		zoomval=zoomtab[indZoom];
    	getApp()->endWaitCursor();
    }
    filelist->setDirectory(FXFile::directory(file));
    return TRUE;
}



// Open
long XFileImage::onCmdOpen(FXObject*,FXSelector,void*)
{
    FileDialog open(this,_("Open Image"));
	open.setSelectMode(SELECTFILE_EXISTING);
    open.setPatternList(patterns);
  	open.setDirectory(filelist->getDirectory());
    if(open.execute())
    {
        filename=open.getFilename();
        filelist->setCurrentFile(filename);
        mrufiles.appendFile(filename);
        loadimage(filename);
    }
    return 1;
}


// Print the image
long XFileImage::onCmdPrint(FXObject*,FXSelector,void*)
{
	FXPrintDialog dlg(this,_("Print Image"));
	FXPrinter printer;
	if(dlg.execute())
    	dlg.getPrinter(printer);
  	return 1;
}

// Handle toggle hidden command
long XFileImage::onCmdToggleHidden(FXObject* sender,FXSelector sel,void* ptr)
{
	filelist->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_TOGGLE_HIDDEN),ptr);
    return 1;
}

// Update toggle hidden command
long XFileImage::onUpdToggleHidden(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg = FXWindow::ID_UNCHECK;
	hiddenfiles = filelist->showHiddenFiles();
		
	if(hiddenfiles == TRUE)
		msg = FXWindow::ID_CHECK;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);

    // Disable menu item if the file list is not shown
	if(filebox->shown())
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}

// Handle toggle hidden command
long XFileImage::onCmdToggleThumbnails(FXObject* sender,FXSelector sel,void* ptr)
{
	filelist->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_TOGGLE_THUMBNAILS),ptr);
    return 1;
}

// Update toggle hidden command
long XFileImage::onUpdToggleThumbnails(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg = FXWindow::ID_UNCHECK;
	thumbnails = filelist->showThumbnails();
		
	if(thumbnails == TRUE)
		msg = FXWindow::ID_CHECK;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);

    // Disable menu item if the file list is not shown
	if(filebox->shown())
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}

// Show mini icons in file list
long XFileImage::onCmdShowMini(FXObject*,FXSelector,void*)
{
	fileview=ID_SHOW_MINI;
	filelist->handle(this,FXSEL(SEL_COMMAND,FileList::ID_SHOW_MINI_ICONS),NULL);
    return 1;
}

// Show big icons in file list
long XFileImage::onCmdShowBig(FXObject*,FXSelector,void*)
{
	fileview=ID_SHOW_BIG;
	filelist->handle(this,FXSEL(SEL_COMMAND,FileList::ID_SHOW_BIG_ICONS),NULL);
    return 1;
}

// Show details in file list
long XFileImage::onCmdShowDetails(FXObject*,FXSelector,void*)
{
	fileview=ID_SHOW_DETAILS;
	filelist->handle(this,FXSEL(SEL_COMMAND,FileList::ID_SHOW_DETAILS),NULL);
    return 1;
}

// Show icons organized as rows
long XFileImage::onCmdShowRows(FXObject*,FXSelector,void*)
{
	iconview=ID_ROWS;
	filelist->handle(this,FXSEL(SEL_COMMAND,FileList::ID_ARRANGE_BY_ROWS),NULL);
    return 1;
}

// Show icons organized as columns
long XFileImage::onCmdShowCols(FXObject*,FXSelector,void*)
{
	iconview=ID_COLS;
	filelist->handle(this,FXSEL(SEL_COMMAND,FileList::ID_ARRANGE_BY_COLUMNS),NULL);
    return 1;
}

// Update filelist
long XFileImage::onUpdFileView(FXObject* sender,FXSelector sel,void* ptr)
{
    // Update radio buttons
	FXuint msg = FXWindow::ID_UNCHECK;
		
    switch(FXSELID(sel))
    {
    case ID_SHOW_MINI:
		if (fileview==ID_SHOW_MINI)
            msg = FXWindow::ID_CHECK;		
        break;
    
	case ID_SHOW_BIG:
		if (fileview==ID_SHOW_BIG)
		    msg = FXWindow::ID_CHECK;
        break;
    
	case ID_SHOW_DETAILS:
		if (fileview==ID_SHOW_DETAILS)
           	msg = FXWindow::ID_CHECK;		
        break;
    }
    sender->handle(this,FXSEL(SEL_COMMAND,msg),NULL);

    // Disable menus items if the file list is not shown
	if(filebox->shown())
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}

// Update icons view
long XFileImage::onUpdIconView(FXObject* sender,FXSelector sel,void* ptr)
{
    // Update radio buttons
	FXuint msg = FXWindow::ID_UNCHECK;
		
    switch(FXSELID(sel))
    {
    case ID_ROWS:
		if (iconview==ID_ROWS)
            msg = FXWindow::ID_CHECK;		
        break;
    
	case ID_COLS:
		if (iconview==ID_COLS)
		    msg = FXWindow::ID_CHECK;
        break;
    }
    sender->handle(this,FXSEL(SEL_COMMAND,msg),NULL);

    // Disable menus items if the file list is not shown
	if(filebox->shown())
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Quit
long XFileImage::onCmdQuit(FXObject*,FXSelector,void*)
{
	// Save settings 
    saveConfig();

    // Quit
    getApp()->exit(0);
    return 1;
}


// Update title
long XFileImage::onUpdTitle(FXObject* sender,FXSelector,void*)
{
    FXString title="Xfi " + filename;
    FXImage* image=imageview->getImage();
    if(image)
        title+=" (" + FXStringVal(zoomval*100) + "%" ")";
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE),(void*)&title);
    return 1;
}


// Update label (directory path) in file list
long XFileImage::onUpdLabel(FXObject*,FXSelector,void*)
{
    label->setText(filelist->getDirectory());
    return 1;
}


// Open recent file
long XFileImage::onCmdRecentFile(FXObject*,FXSelector,void* ptr)
{
    filename=(FXchar*)ptr;
    filelist->setCurrentFile(filename);
    loadimage(filename);
    return 1;
}


// Command from the tree list
long XFileImage::onCmdFileList(FXObject*,FXSelector,void* ptr)
{
    FXint index=(FXint)(FXival)ptr;
    if(0<=index)
    {
        if(filelist->isItemDirectory(index))
        {
            filelist->setDirectory(filelist->getItemPathname(index));
        }
        else if(filelist->isItemFile(index))
        {
            filename=filelist->getItemPathname(index);
            mrufiles.appendFile(filename);
            loadimage(filename);
        }
    }
    return 1;
}


// Rotate image
long XFileImage::onCmdRotate(FXObject*,FXSelector sel,void*)
{
	getApp()->beginWaitCursor();
    FXImage* image=imageview->getImage();
    switch(FXSELID(sel))
    {
    case ID_ROTATE_90:
        
		// Rotate the actual image
		image->rotate(90);
		
		// Need to also rotate the original image only if the actual size is different
		if (image->getWidth()!=img->getWidth() || image->getHeight()!=img->getHeight())
			img->rotate(90);
        break;

    case ID_ROTATE_270:

		// Rotate the actual image
        image->rotate(270);

		// Need to also rotate the original image only if the actual size is different
		if (image->getWidth()!=img->getWidth() || image->getHeight()!=img->getHeight())
			img->rotate(270);
        break;
    }
    imageview->setImage(image);
	getApp()->endWaitCursor();
    return 1;
}


// Update image
long XFileImage::onUpdImage(FXObject* sender,FXSelector,void*)
{
    if(imageview->getImage())
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
    return 1;
}


// Mirror image
long XFileImage::onCmdMirror(FXObject*,FXSelector sel,void*)
{
	getApp()->beginWaitCursor();
    FXImage* image=imageview->getImage();
    switch(FXSELID(sel))
    {
    case ID_MIRROR_HOR:

		// Mirror the actual image
        image->mirror(TRUE,FALSE);

		// Need to also mirror the original image only if the actual size is different
		if (image->getWidth()!=img->getWidth() || image->getHeight()!=img->getHeight())
        	img->mirror(TRUE,FALSE);
        break;

    case ID_MIRROR_VER:

		// Mirror the actual image
        image->mirror(FALSE,TRUE);

		// Need to also mirror the original image only if the actual size is different
		if (image->getWidth()!=img->getWidth() || image->getHeight()!=img->getHeight())
        	img->mirror(FALSE,TRUE);
        break;
    }
    imageview->setImage(image);
	getApp()->endWaitCursor();
    return 1;
}


// Zoom in image
long XFileImage::onCmdZoomIn(FXObject*,FXSelector,void*)
{
	getApp()->beginWaitCursor();

    // Copy the original image into the actual one
   	if (!FXMEMDUP(&tmpdata,img->getData(),FXColor,img->getWidth()*img->getHeight()))
		throw FXMemoryException(_("Unable to load image"));
	tmpimg->setData(tmpdata,IMAGE_OWNED,img->getWidth(),img->getHeight());	

	// Resize the actual image according to the new zoom factor
	indZoom+=1;
	if (indZoom>NB_ZOOM-1) indZoom=NB_ZOOM-1;
	FXint sx=(FXint)(tmpimg->getWidth()*zoomtab[indZoom]);
	FXint sy=(FXint)(tmpimg->getHeight()*zoomtab[indZoom]);
	
	// Scale only if the actual image size is different
	if (indZoom==ZOOM_100)
		imageview->setImage(img);
	else
	{
		// Maximum zoom allowed
		if (sx>MAX_IMGSIZE || sy > MAX_IMGSIZE)
		{
			indZoom-=1;
			if (indZoom<0) indZoom=0;
			sx=(FXint)(tmpimg->getWidth()*zoomtab[indZoom]);
			sy=(FXint)(tmpimg->getHeight()*zoomtab[indZoom]);
		}

		// Scale image according to the new zoom factor
		tmpimg->scale(sx,sy,1);
		imageview->setImage(tmpimg);
	}

	// Set zoom value for window title
	zoomval=zoomtab[indZoom];
	
	getApp()->endWaitCursor();
    return 1;
}


// Zoom out image
long XFileImage::onCmdZoomOut(FXObject*,FXSelector,void*)
{
	getApp()->beginWaitCursor();

    // Copy the original image into the actual one
   	if (!FXMEMDUP(&tmpdata,img->getData(),FXColor,img->getWidth()*img->getHeight()))
		throw FXMemoryException(_("Unable to load image"));
	tmpimg->setData(tmpdata,IMAGE_OWNED,img->getWidth(),img->getHeight());	

	// Resize the image according to the new zoom factor
	indZoom-=1;
	if (indZoom<0) indZoom=0;
	FXint sx=(FXint)(tmpimg->getWidth()*zoomtab[indZoom]);
	FXint sy=(FXint)(tmpimg->getHeight()*zoomtab[indZoom]);

	// Scale only if the actual image size is different
	if (indZoom==ZOOM_100)
		imageview->setImage(img);
	else
	{
		// Scale image according to the new zoom factor
    	tmpimg->scale(sx,sy,1);
    	imageview->setImage(tmpimg);
	}

	// Set zoom value for window title
	zoomval=zoomtab[indZoom];
	
	getApp()->endWaitCursor();
    return 1;
}


// Zoom to 100%
long XFileImage::onCmdZoom100(FXObject*,FXSelector,void*)
{
	getApp()->beginWaitCursor();
	indZoom=ZOOM_100;
	zoomval=zoomtab[indZoom];	
	imageview->setImage(img);
	getApp()->endWaitCursor();
    return 1;
}


// Zoom to fit window
long XFileImage::onCmdZoomWin(FXObject*,FXSelector,void*)
{
	getApp()->beginWaitCursor();

	// Window and image sizes
	FXint winw=imageview->getWidth();
	FXint winh=imageview->getHeight();
	FXint w=img->getWidth();
	FXint h=img->getHeight();
	
	// Compute zoom factor
	FXdouble zoomwin;	
	if (FXdouble(w)/FXdouble(h)>FXdouble(winw)/FXdouble(winh))
		zoomwin=0.98*(FXdouble)winw/(FXdouble)w;
	else
		zoomwin=0.98*(FXdouble)winh/(FXdouble)h;

	// Find the most approaching predefined zoom
	// This is used in other zoom functions
	for (FXint k=0; k<NB_ZOOM; k++)
	{
 		if (zoomtab[k]>zoomwin)
		{	
			indZoom=k-1;
			break;
		}
	}
	if (indZoom<0) indZoom=0;
	if (indZoom>=NB_ZOOM) indZoom=NB_ZOOM-1;
    	
	// Copy the original image into the actual one
   	if (!FXMEMDUP(&tmpdata,img->getData(),FXColor,img->getWidth()*img->getHeight()))
		throw FXMemoryException(_("Unable to load image"));
	tmpimg->setData(tmpdata,IMAGE_OWNED,img->getWidth(),img->getHeight());	

	// Resize the image according to the new zoom factor
	FXint sx=(FXint)(w*zoomwin);
	FXint sy=(FXint)(h*zoomwin);

	// Scale image according to the new zoom factor
    tmpimg->scale(sx,sy,1);
    imageview->setImage(tmpimg);

	// Set zoom value for window title
	zoomval=zoomwin;
	
	getApp()->endWaitCursor();
    return 1;
}


// Start the ball rolling
void XFileImage::start(FXString startimage)
{
    filename=startimage;
	if(filename != "")
		loadimage(filename);
}


// Create and show window
void XFileImage::create()
{
	// Get size and position
    FXuint ww=getApp()->reg().readUnsignedEntry("SETTINGS","width",600);
    FXuint hh=getApp()->reg().readUnsignedEntry("SETTINGS","height",400);
    FXint xx=getApp()->reg().readIntEntry("SETTINGS","xpos",0);
    FXint yy=getApp()->reg().readIntEntry("SETTINGS","ypos",0);

    FXuint fh=getApp()->reg().readIntEntry("SETTINGS","fileheight",100);
    FXuint fs=getApp()->reg().readIntEntry("SETTINGS","filesshown",TRUE);

    filelist->setDirectory(FXFile::getCurrentDirectory());

    filebox->setHeight(fh);

    // Hide tree if asked for
    if(!fs)
        filebox->hide();

	// Get toolbar status
    if(getApp()->reg().readUnsignedEntry("SETTINGS","showtoolbar",TRUE)==FALSE)
        toolbar->hide();
   
	// Get status bar status
    if(getApp()->reg().readUnsignedEntry("SETTINGS","showstatusbar",TRUE)==FALSE)
        statusbar->hide();

 	// Get hidden file status
	hiddenfiles=getApp()->reg().readUnsignedEntry("SETTINGS","hiddenfiles",0);
	filelist->showHiddenFiles(hiddenfiles);
  
 	// Get thumbnails status
	thumbnails=getApp()->reg().readUnsignedEntry("SETTINGS","thumbnails",0);
	filelist->showThumbnails(thumbnails);
  
	// Get file view and icon view
    fileview=getApp()->reg().readUnsignedEntry("SETTINGS","fileview",ID_SHOW_MINI);
	this->handle(this,FXSEL(SEL_COMMAND,fileview),NULL);
    iconview=getApp()->reg().readUnsignedEntry("SETTINGS","iconview",ID_COLS);
	this->handle(this,FXSEL(SEL_COMMAND,iconview),NULL);

	// Reposition window
    position(xx,yy,ww,hh);

    FXMainWindow::create();
    show();
}


// Save configuration when quitting
void XFileImage::saveConfig()
{
    // Write new window size and position back to registry
	getApp()->reg().writeUnsignedEntry("SETTINGS","width",(FXuint)getWidth());
	getApp()->reg().writeUnsignedEntry("SETTINGS","height",(FXuint)getHeight());
	getApp()->reg().writeIntEntry("SETTINGS","xpos",getX());
	getApp()->reg().writeIntEntry("SETTINGS","ypos",getY());

    // Width of tree
    getApp()->reg().writeIntEntry("SETTINGS","fileheight",filebox->getHeight());

    // Was file box shown
    getApp()->reg().writeIntEntry("SETTINGS","filesshown",filebox->shown());

    // Toolbar status
    if(toolbar->shown())
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",TRUE);
    else
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",FALSE);

	// Hidden files status
	getApp()->reg().writeUnsignedEntry("SETTINGS","hiddenfiles",hiddenfiles);

	// Thumbnails status
	getApp()->reg().writeUnsignedEntry("SETTINGS","thumbnails",thumbnails);

	// File view and icon view
	getApp()->reg().writeUnsignedEntry("SETTINGS","fileview",fileview);
	getApp()->reg().writeUnsignedEntry("SETTINGS","iconview",iconview);

	// Write registry settings
	getApp()->reg().write();
}


// Usage message
#define USAGE_MSG	_("\
Usage: xfi [options] [image] \n\
\n\
    [options] can be any of the following:\n\
\n\
        -h, --help         Print (this) help screen and exit.\n\
        -v, --version      Print version information and exit.\n\
\n\
    [image] is the path to the image file you want to open on start up.\n\
\n")




// Start the whole thing
int main(int argc,char *argv[])
{
	int i;
	FXString startimage="";
	char *appname = "Xfi";
	char *xfename = "Xfe";
	char *vdrname = "XFileExplorer";

#ifdef HAVE_SETLOCALE
  // Set locale via LC_ALL.
  setlocale (LC_ALL, "");
#endif

#if ENABLE_NLS
  // Set the text message domain.
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif

	// Parse basic arguments
	for(i = 1; i < argc; ++i)
	{
    	if(compare(argv[i],"-v")==0 || compare(argv[i],"--version")==0)
		{
			fprintf(stdout,"%s version %s\n",PACKAGE,VERSION);
			exit(0);
		}
    	else if(compare(argv[i],"-h")==0 || compare(argv[i],"--help")==0)
		{
			fprintf(stdout,USAGE_MSG);
			exit(0);
		}
		else
	    {
			// Start image, if any
			startimage=argv[i];
	    }
	}

    args=argv;

    // Make application
    FXApp* application=new FXApp(appname,vdrname);

    // Open display
    application->init(argc,argv);

    // Read the Xfe registry
	FXRegistry* reg=new FXRegistry(xfename,vdrname);
	reg->read();
	
	// Set Xfi normal font according to the Xfe registry
	FXString fontspec;
	fontspec=reg->readStringEntry("SETTINGS","font","Helvetica,100,normal,regular");
	if(!fontspec.empty())
	{
    	FXFont* normalFont=new FXFont(application,fontspec);
        application->setNormalFont(normalFont);
    }

	// Set Xfi colors according to the Xfe registry
	FXColor basecolor,forecolor,backcolor,selforecolor,selbackcolor,bordercolor;
    basecolor=reg->readColorEntry("SETTINGS","basecolor",FXRGB(214,207,198));
    forecolor=reg->readColorEntry("SETTINGS","forecolor",FXRGB(0,0,0));
    backcolor=reg->readColorEntry("SETTINGS","backcolor",FXRGB(255,255,255));
    selforecolor=reg->readColorEntry("SETTINGS","selforecolor",FXRGB(255,255,255));
    selbackcolor=reg->readColorEntry("SETTINGS","selbackcolor",FXRGB(10,36,106));
    bordercolor=reg->readColorEntry("SETTINGS","bordercolor",FXRGB(0,0,0));
	application->setBaseColor(basecolor);
	application->setForeColor(forecolor);
	application->setBackColor(backcolor);
	application->setSelforeColor(selforecolor);
	application->setSelbackColor(selbackcolor);
	application->setBorderColor(bordercolor);

	// Delete the Xfe registry
	delete reg;
    
	// Make window
    XFileImage* window=new XFileImage(application);
	mainWindow=window;
	
	// Handle interrupt to save stuff nicely
    application->addSignal(SIGINT,window,XFileImage::ID_QUIT);
 
    // Create it
    application->create();

    // Start
    window->start(startimage);

   // Run
    return application->run();
}


