/*
 * Copyright (C) 2002-2005 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>

/* put any extra C header you need here... */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/wait.h>
#include <tubo.h>

/* These are required... */
#include "constants.h"
#include "types.h"
#include "primary.h"
#include "gui.h"

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT
LIBXFFM_MODULE




static
const gchar *
get_slocate_string (widgets_t *widgets_p);

static
xfdir_t *
populated_xfdir(widgets_t *widgets_p, const gchar *slocate_string);

G_MODULE_EXPORT
const gchar * 
g_module_check_init(GModule *module){
#ifdef ENABLE_NLS
    bindtextdomain (GETTEXT_PACKAGE,PACKAGE_LOCALE_DIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
#endif
    TRACE("domain=%s", GETTEXT_PACKAGE);
#endif
    return NULL;
}

/****************   void  functions  ***********************/

/**** logical void functions... */

/**
 * module_init:
 * 
 * Use this function to do any initializations the plugin
 * requires upon loading. 
 * *
**/

G_MODULE_EXPORT
void *
module_init(void)
{
    TRACE("initializing slocate plugin");
    return NULL;
}

/**
 * is_root_module:
 * 
 * Logical function.
 * Return a non-null value for plugin to be considered a root level
 * plugin. Yes, you can create plugins which load plugins in turn. The
 * child plugins would not be root level. Look at the bookshelf/bookmarks
 * plugins for an example of this kind of inter-plugin relationship.
 * *
**/

G_MODULE_EXPORT
void *
is_root_module(void)
{
    return GINT_TO_POINTER(1);
}

/**** gchar_p void functions... */
/**
 * exec_name:
 *
 * 
 * If you symlink the xffm executable to the name returned
 * by this function, then upon executing the symbolic
 * link, you will load your plugin directly.
 *
 * XXX currently treeview gui is opened.
 * 

 *
**/

G_MODULE_EXPORT
void *
exec_name(void)
{
    return "xffm-locate";
}

/**
 * module_name:
 *
 * This is the name of the library. It should match whatever
 * you choose in the Makefile, without the "lib" prefix. For example,
 * a plugin located at "libxffm_locate.so.0.0.0" should return
 * "xffm_locate".
 *
**/
G_MODULE_EXPORT
void *
module_name(void)
{
    return (void *)"xffm_locate";
}

G_MODULE_EXPORT
void *
plugin_info(void){
    return _("The SLOCATE plugin allows you to perform \"slocate\" queries and to put the results into the xffm GUI for further manipulation and navigation.");
}


/*******************  natural functions **********************/

G_MODULE_EXPORT
void *
get_dnd_path(void *p)
{
    static gchar *desktop=NULL;
    if (!desktop) {
	desktop=g_strdup_printf("%s%c%s%c%s",
		PACKAGE_DATA_DIR,G_DIR_SEPARATOR,
		"applications",G_DIR_SEPARATOR,
		"Xffm-locate.desktop");
    }
    return (void *)desktop;
}

/**
 * module_icon_id:
 *
 * This is the name of the icon that should be used for your
 * plugin. You can use any mime-icon identifier (see xfmime-edit
 * for a complete list), or a filename to a valid icon. If the 
 * icon is not found, you will get the skull-and-crossbones icon.
 *
**/
G_MODULE_EXPORT
void *
module_icon_id(void * p){
    return "xffm-locate.png";
}

/**
 * module_label:
 *
 * This is the name of the plugin as it should appear in the 
 * label beneath the icon. 
 *
**/
G_MODULE_EXPORT
void *
module_label(void * p){
    return _("Locate");
}



/*******************  rational functions **********************/

/**
 * get_xfdir: 
 *
 * @p: is the entry of the selected element which should be loaded.
 * kasjdflk laskjf asldkfgj asdlfkjsad flk
 * @q: is either the widgets_t pointer for the GUI. 
 * These are provided in case you need them somewhere (like for 
 * printing out to the status or diagnostic windows).
 * This is the principal function. The plugin should populate
 * an xfdir structure and return it. The allocated memory will
 * be freed by xffm so don't free it.
 *
 * p: quuuu
 *
 * @p: whatever
 *
 * @q shit
 *
**/
G_MODULE_EXPORT
void *
get_xfdir (void *p, void *q){
    const gchar *slocate_string;
    gchar *g=g_find_program_in_path("slocate");
    widgets_t *widgets_p=q;

    if (!g) g=g_find_program_in_path("locate");
    if (!g) {
	gchar *g=_("Neither slocate nor locate program could be found.\nPlease install either program\nor make sure your PATH environment is correctly defined");
	xffm_confirm(widgets_p,g,NULL,_("Ok"));
	return NULL;
    }
    g_free(g);
    slocate_string=get_slocate_string(q);
    if (!slocate_string) return NULL;
    TRACE("get the output from: slocate %s",slocate_string);
    return populated_xfdir(q,slocate_string);
}
 



#ifdef NOT_USED_IN_THIS_PLUGIN
/**
 * submodule_name:
 * 
 * If the elements returned by your plugin correspond to
 * other plugins, you should return here what is the
 * name of the plugin that should be used to load each
 * element in turn. The name should follow the rules of
 * the module_name() function.
 * 
**/
G_MODULE_EXPORT
void *
submodule_name(void)
{
    return (void *)"xffm_book";
}
/**
 * is_selectable:
 *
 * Logical function. Determines whether the plugin's icon can
 * be selected or not.
 * 
**/
G_MODULE_EXPORT
void *
is_selectable(void *p){
    return GINT_TO_POINTER(1);
}
/**
 * valid_drop_site:
 *
 * Logical function. 
 * @p is the entry for the element. Should return true if such element
 * is to be a valid drop site. This is only for elements that do not have
 * a valid path in the local file system. If such is the case for your plugin,
 * then you must also provide a function to process the drop. See the
 * smb_list plugin for details on how this can be done.
 *
**/
G_MODULE_EXPORT
void *
valid_drop_site(void *p){
    return NULL;
}

/**
 * private_popup:
 * 
 * This function determines whether the plugin should have
 * a private popup menu. This is not applicable to elements
 * that exist and have a local path (for that use the 
 * extend_popup() function). If this function is not
 * present, or returns null, the popup of the element will
 * correspond to the main menu popup. If function returns
 * true, then you must populate the gtk menu widget and
 * provide callbacks in your plugin. See the frequent/recent
 * plugin for an example of this.
 *
 * @p is the entry of the selected element which should be loaded.
 * @q is either the icon_view_pointer or NULL (if using treeview GUI). 
 * These are provided in case you need them somewhere (like for 
 * printing out to the status or diagnostic windows).
**/
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){

    return GINT_TO_POINTER(1);
}

/**
 * extend_popup:
 *
 * Use this function to add a submenu to the popup menu of a 
 * local file. The submenu widget must be populated and 
 * callbacks provided by the plugin. See the frequent/recent
 * plugin for an example of this.
 *
 * 
 * @p is the entry of the selected element which should be loaded.
 * @q is either the icon_view_pointer or NULL (if using treeview GUI). 
 * These are provided in case you need them somewhere (like for 
 * printing out to the status or diagnostic windows).
 *
**/
G_MODULE_EXPORT
void *
extend_popup(void *p, void *q){
    return GINT_TO_POINTER(1);
}

/**
 * up_entry:
 *
 * If the up element is not the default xffm top, then you should define
 * it with this function. See the smb_list plugin for an example.
 *
 * @en_p is the entry of the parent.
**/
G_MODULE_EXPORT
void *
up_entry(void *en_p,void *en_c){
    record_entry_t *en;
    if (!en_p || !((record_entry_t *)en_p)->path || !strchr(((record_entry_t *)en_p)->path,'/')) return NULL;
    en = copy_entry((record_entry_t *)en_p);
    TRACE("up entry, %s",en->path);
    g_free(en->path);
    TRACE("up entry, freed en->path");
    en->path=up_path(((record_entry_t *)en_p)->path);
    TRACE("and then up path: %s -> %s",((record_entry_t *)en_p)->path,en->path);
    return (void *)en;
}
/**
 * module_monitor:
 *
 * If this function returns non NULL, the corresponding entry will be tagged 
 * for refresh by the GUI monitor function...
 *
**/
G_MODULE_EXPORT
void *
module_monitor(void *p){
    return NULL;
}
/**
 * show_hidden_menuitem:
 *
 * Show the show_hidden_menuitem in popup menu.
**/
G_MODULE_EXPORT
void *
show_hidden_menuitem(void){
    return GINT_TO_POINTER(1);
}

/**
 * preview_images_menuitem:
 *
 * Show the preview_images_menuitem toggle in popup menu.
**/
G_MODULE_EXPORT
void *
preview_images_menuitem(void){
    return GINT_TO_POINTER(1);
}
/**
 * newdirectory_menuitem:
 *
 * Show the newdirectory_menuitem toggle in popup menu.
**/
G_MODULE_EXPORT
void *
newdirectory_menuitem(void){
    return GINT_TO_POINTER(1);
}

/**
 * copy_menuitem:
 *
 * Show the copy_menuitem toggle in popup menu.
**/

G_MODULE_EXPORT
void *
copy_menuitem(void){
    return GINT_TO_POINTER(1);
}

/**
 * cut_menuitem:
 *
 * Show the cut_menuitem toggle in popup menu.
**/
G_MODULE_EXPORT
void *
cut_menuitem(void){
    return GINT_TO_POINTER(1);
}

/**
 * paste_menuitem:
 *
 * Show the paste_menuitem toggle in popup menu.
**/
G_MODULE_EXPORT
void *
paste_menuitem(void){
    return GINT_TO_POINTER(1);
}

/**
 * double_click:
 *
 * What to do with a double click if the element is not
 * a local file. See the smb_list plugin for an example.
 *
 * @p is the entry of the selected element which should be loaded.
 * @q is the widgets_t pointer. 
 * These are provided in case you need them somewhere (like for 
 * printing out to the status or diagnostic windows).
**/
G_MODULE_EXPORT
void *
double_click(void *p, void *q){
    return GINT_TO_POINTER(1);
}

/**
 * on_new_dir:
 *
 * What to do with on_new_dir command if the element is not
 * a local file. See the smb_list plugin for an example.
 *
 * @p is the entry of the selected element which should be loaded.
 * @q is either the icon_view_pointer or NULL (if using treeview GUI). 
 * These are provided in case you need them somewhere (like for 
 * printing out to the status or diagnostic windows).
**/
G_MODULE_EXPORT
void *
on_new_dir(void *p, void *q){
    return GINT_TO_POINTER(1);
}

/**
 * on_remove:
 *
 * What to do with on_remove command if the element is not
 * a local file. See the smb_list plugin for an example.
 *
 * @p is the entry of the selected element which should be loaded.
 * @q is widget_t pointer 
 * **/
G_MODULE_EXPORT
void *
on_remove(void *p, void *q) 
{
   return GINT_TO_POINTER(1);
}


/**
 * parent_module_name:
 *
 * Function used to define the parent_module name (in case
 * of non-root plugins)
 *
 * @p is the element's entry.
 *
 * See smb_list_plugin for an example.
 *
**/

G_MODULE_EXPORT
void *
parent_module_name(void *p){
        return NULL;
}

/** 
 * process_drop:
 * 
 * function used to process drop events when the plugin element is not
 * a local file.
 * 
 * See smb_list_plugin for an example.
 *
**/
G_MODULE_EXPORT
void *
process_drop(void *p, void *q){
   return (GINT_TO_POINTER(1));
}

	
#endif


static
const gchar *
get_slocate_string (widgets_t *widgets_p)
{
        /* create the dialog */
    const gchar *response = NULL;
    static gchar *slocate_string=NULL;
    
    response =gui_get_response(widgets_p, _("locate"),"locate");

    g_free(slocate_string);
    if (response)slocate_string=g_strdup(response);
    else slocate_string=NULL;
    

    if (slocate_string && !strlen(slocate_string)){
	g_free(slocate_string);
	slocate_string=NULL;
    }

    return (const gchar *)slocate_string;
}

/* tubo library method: */
/* first let's define some static variables we shall need
 * to share by reference amongst functions: */
static
xfdir_t 	slocate_xfdir;
static
GList 		*slocate_list=NULL;

static
const gchar 	*slocate_string;
/* Now let's define what we want to do with the 
 * line-to-line output of the slocate command.
 * We want to pipe it all out to the diagnostics
 * window to give the user feed back of the
 * command and its results.
 * 
 * This is done with the following function
 * (which we shall use for both stdout and stderr):
 * */

static int slocate_output(int n, void *data, void *user_data)
{
    widgets_t *widgets_p=user_data;
    char *line;
    if(n) return TRUE;		/* this would mean binary data */
    line = (char *)data;

    if (!widgets_p) return TRUE;

    /* keep users eyes moving: */
    set_progress_generic(widgets_p,-1,-1,1);	    
    while (gtk_events_pending()) gtk_main_iteration();
	    
    /* chop off trailing \n. 
     * * */
    if (strlen(line)) line[strlen(line) - 1] = '\0';

    /* We are only interested in files */
    if (!g_file_test(line,G_FILE_TEST_EXISTS)) return TRUE;
    else {
	gchar *g,*f=g_path_get_basename(line);
	/* and only interested in files with token in basename 
	 * (we could probably use --regexp=<regexp> in command
	 * line instead of this, but I'm lazy for that)*/
	if (!strstr(f,slocate_string)){
	    g_free(f);
	    return TRUE;
	}
	show_text(widgets_p);
	print_diagnostics(widgets_p,NULL, line, NULL);
	TRACE("locate_output: line=%s (%s)",line,f);
	slocate_list=g_list_append(slocate_list,g_strdup(line));
	g=g_strdup_printf(dngettext(GETTEXT_PACKAGE,"%d file found","%d files found",g_list_length(slocate_list)), g_list_length(slocate_list));
	print_status(widgets_p,"xffm/info", g, NULL);
	g_free(g);
	g_free(f);
    }

    return TRUE;
}

/* Next we define what we want to do when the child process
 * (in this case locate) has terminated.
 *
 * This is done in the following function:
 * */
static void slocate_finished(pid_t pid, void *user_data)
{
    widgets_t *widgets_p=user_data;
    GList *tmp;
    gchar *g;
    int status,j;

    /* another wait just in case the one in tubo misses 
     * (shouldn't happen, but it does when there are bugs 
     *  in the OS kernel)*/
    waitpid(pid, &status, WNOHANG);
    

    
    cursor_reset(widgets_p->window);
    if (slocate_list){ 
	    TRACE("locate: %d elements found",g_list_length(slocate_list));
    }
    else {
	print_diagnostics(widgets_p,"xffm/info",_("Nothing found"),"\n",NULL);
	widgets_p->tubo_object=NULL;
	set_progress_generic(widgets_p,-1, -1, -1);
        return;
    }
    cursor_wait(widgets_p->window);
	
    /* allocate space on the xfdir list */
    slocate_xfdir.gl = (dir_t *) malloc(g_list_length(slocate_list) * sizeof(dir_t));
    slocate_xfdir.pathc=g_list_length(slocate_list);
    /* populate xfdir list */
    for (j=0,tmp=slocate_list;tmp; tmp=tmp->next,j++){
	slocate_xfdir.gl[j].pathv=g_path_get_basename((gchar *)(tmp->data));
	slocate_xfdir.gl[j].en = stat_entry((gchar *)(tmp->data),0);
	g_free(tmp->data);
    }
    g_list_free(slocate_list);
    slocate_list=NULL;
    set_progress_generic(widgets_p,-1, -1, -1);
	
    
    g=g_strdup_printf(dngettext(GETTEXT_PACKAGE,"%d file found","%d files found",slocate_xfdir.pathc), slocate_xfdir.pathc);
    print_status(widgets_p,"xffm/info", g, NULL);
    g_free(g);
    
    cursor_reset(widgets_p->window);
    TRACE("slocate_finished(%ld)",(long)pid);  
    /* allow slocate to fork again, and
     * calling routine to return xfdir information
     * to xffm: */
    widgets_p->tubo_object=NULL;
    return;
}

/* and this function is to monitor activity on the stop button */

static gint watch_stop(gpointer data)
{
    widgets_t *widgets_p=data;

    if(!widgets_p || !widgets_p->tubo_object) return FALSE;
    if(widgets_p && widgets_p->stop)
    {
	pid_t Gpid=TuboPID(widgets_p->tubo_object);
	gchar *m=g_strdup_printf("%d", Gpid);
	print_diagnostics(widgets_p,"xffm/warning", strerror(ECANCELLED)," pid=", m, "\n",NULL);
	print_status(widgets_p,"xffm/warning", strerror(ECANCELLED)," pid=", m, NULL);
	g_free(m);
	TuboCancel(widgets_p->tubo_object, NULL, NULL);
	/*usleep(250); if(Gpid) kill(Gpid, SIGHUP);*/
	widgets_p->stop = FALSE;
	set_progress_generic(widgets_p,-1, -1, -1);
	return FALSE;
    }
    set_progress_generic(widgets_p,-1, -1,1);
    return TRUE;
}

static
xfdir_t *
populated_xfdir(widgets_t *widgets_p, const gchar *string){
    gchar *argv[]={"slocate",NULL,NULL};
    gchar *g=g_find_program_in_path("slocate");

    if (!g) argv[0]="locate";
    else g_free(g);
    
    slocate_string=argv[1]=(gchar *)string;
    
    cursor_wait(widgets_p->window);
    print_status(widgets_p,"xffm/info",strerror(EINPROGRESS),"...", NULL);
    
    
    /* just allow one child slocate (per xffm instance) at a time...*/
    if (widgets_p && widgets_p->tubo_object){
	print_status(widgets_p,"xffm/info",strerror(EBUSY),"...", NULL);
	return NULL;
    }

    widgets_p->stop=FALSE;
    show_stop(widgets_p);
	
    /* fork with tubo library */
    widgets_p->tubo_object = Tubo_full(fork_function, 
				(void *)argv, 
				slocate_finished, 
				0, 
				slocate_output, 
				slocate_output,
				widgets_p,15);
    g_timeout_add(260,(GtkFunction) watch_stop, widgets_p);
    /* wait for child to finish */
    while (widgets_p->tubo_object){
	while(gtk_events_pending()) gtk_main_iteration();
	usleep(150);
    }
    hide_stop(widgets_p);
    return &slocate_xfdir;
}


