/*
 * Copyright (C) 2002-4 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <errno.h>
#include <limits.h>
#include <memory.h>
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#elif HAVE_VARARGS_H
#include <varargs.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <dbh.h>

#ifdef XFCE_DISABLE_DEPRECATED
#undef XFCE_DISABLE_DEPRECATED
#endif
#ifdef HAVE_LIBXFCEGUI4
#include <libxfcegui4/libxfcegui4.h>
#endif


#include "constants.h"
#include "types.h"
#include "primary.h"
#include "secondary.h"
#include "actions_lib.h"

#include "tubo.h"

#include "run.i"



/* terminal stuff */

G_MODULE_EXPORT
void
RECENT_add2history(widgets_t *widgets_p, gchar *path){
    if (!function_rational("plugins","xffm_recent",(void *)path,widgets_p,"add2history"))
	function_rational("plugins","xffm_frequent",(void *)path,widgets_p,"add2history");
}

G_MODULE_EXPORT
const gchar *xffm_what_term(void){
    static gchar *term=NULL;
    if (term){
       g_free(term);
       term=NULL;
    }
    term=NULL;
    if (getenv("TERMCMD") && strlen( getenv("TERMCMD"))){
	    if (MIME_is_valid_command(getenv("TERMCMD"))){
		    term = g_strdup(getenv("TERMCMD"));
	    } else {
		    g_warning("TERMCMD=%s: %s",getenv("TERMCMD"),strerror(errno));
	    }
    }
    if (!term) {
	term=g_strdup("xterm");
    }
    if ((strncmp(term,"xterm",strlen("xterm"))==0 || strstr(term,"xfce4-terminal")) 
         && getenv("XFFM_HOLD_XTERM") && strlen( getenv("XFFM_HOLD_XTERM"))){
	gchar *p=term;
	term=g_strconcat(p," -hold",NULL);
	g_free(p);
    }
    return (const gchar *)term;	
}

/*static void new_terminal_query(void){
    show_input(NEW_TERM_INPUT,NULL);
}*/

G_MODULE_EXPORT
void xffm_new_terminal_path(widgets_t *widgets_p,gchar *workdir){
    GError *error=NULL;
    int argc;
    gchar **argv;
  
    g_shell_parse_argv (xffm_what_term(), &argc,&argv,&error);
    if (error){
	gchar *msg = g_strcompress (error->message);
	print_diagnostics(widgets_p,"xfce/error",msg,": ",xffm_what_term(),"\n",NULL);
	g_error_free(error);
	g_free (msg);
	return;
    }
    if (!workdir) workdir=(gchar *)g_get_home_dir();
    xffm_runvwd(widgets_p,workdir,argv);
    g_strfreev (argv);
    return;
}


/* end terminal stuff */

G_MODULE_EXPORT
const 
gchar *
xffm_get_from_open_history(	gchar *path2find, 
				int *in_term){
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *dbh_file=g_build_filename(xdg_dir,OPEN_FILE,NULL);
    GString *gs;
    DBHashTable *d;
    gchar *data;
    gchar *basename=g_path_get_basename(path2find);
    static gchar *in_cmd=NULL;

    g_free(xdg_dir);
    
    if ((d=DBH_open(dbh_file))==NULL){
    	if ((d=DBH_create(dbh_file, 11))==NULL) {
		unlink(dbh_file);
		if ((d=DBH_create(dbh_file, 11))==NULL)	{
		    g_free(basename);
		    g_free(dbh_file);
		    return NULL;
		}
	}
    }
    g_free(dbh_file);
    gs = g_string_new(basename);
    g_free(basename);
    sprintf((char *)DBH_KEY(d), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    data = (gchar *)DBH_DATA(d);

    if (!DBH_load(d)) return NULL;
    

    
    if (*data==0) *in_term = 0;
    else *in_term=1;
    data++;
    g_free(in_cmd);
    in_cmd=g_strdup(data);
    
    DBH_close(d);
    return (const gchar *)in_cmd;
}


G_MODULE_EXPORT
int 
xffm_runvwd(		widgets_t *widgets_p,
			const gchar *workdir,
			char *argv[])
{

    int i;
    GError *error=NULL;
    
    TRACE("xffm_runvwd now, widgets_p=0x%x",(unsigned)widgets_p);
#ifdef TRACE
    {
	gchar **p;
	for (p=argv; *p; p++){
	    TRACE("arg[]=%s",*p);
	}
    }
#endif
    
    if (!argv || !argv[0]) {
	g_warning("!argv || !argv[0]");
	return FALSE;
    }
    if (!workdir || !g_file_test(workdir,G_FILE_TEST_IS_DIR)) {
	/*g_warning("xffm_runvwd(): workdir==NULL");*/
	workdir=g_get_home_dir();
    }
    

    xffm_setenv("PWD",(gchar *)workdir,FALSE);
    chdir(workdir);
    TRACE("TRACE: getenv(PWD)=%s\n",getenv("PWD"));
    
    print_diagnostics(widgets_p,NULL,"$chdir ",workdir,"\n$",argv[0],NULL);
    print_diagnostics(widgets_p,NULL,"$",argv[0],NULL);
    for (i=1;argv[i];i++) {
	print_diagnostics(widgets_p,NULL," ",argv[i],NULL);
    }
    print_diagnostics(widgets_p,NULL,"\n",NULL);

    if (strcmp(argv[0],"sudo")==0)
    {
	gchar *f=g_strdup_printf("sudo %s",argv[1]);
	gchar *g=g_strdup_printf(_("Executing %s"),f);
	print_status(widgets_p,"xfce/info", g, NULL);
	g_free(g);
	g_free(f);
    }
    else 
    {
	gchar *g;
	if (g_file_test(argv[0],G_FILE_TEST_EXISTS)) {
	    gchar *f;
	    f=g_path_get_basename(argv[0]);
	    g=g_strdup_printf(_("Executing %s"),f);
	    g_free(f);
	} else {
	    g=g_strdup_printf(_("Executing %s"),argv[0]);
	}
	print_status(widgets_p,"xfce/info", g, NULL);
	g_free(g);
    }

    if (!g_spawn_async(workdir,argv,NULL,G_SPAWN_SEARCH_PATH,NULL,NULL,NULL,&error)){
	  gchar *msg = g_strcompress (error->message);
	  print_diagnostics(widgets_p,"xfce/error",msg,":\n",argv[0] ,"\n",NULL);
	  g_error_free (error);
	  g_free (msg);
	  chdir (GETWD);
	  xffm_setenv("PWD",(gchar *)GETWD,FALSE);
	  return FALSE;
    }

    /*print_diagnostics(widgets_p,NULL,"$chdir ",GETWD,"\n",NULL);*/
    
    xffm_setenv("PWD",(gchar *)GETWD,FALSE);
    chdir(GETWD);
    return TRUE;

}


G_MODULE_EXPORT
void 
xffm_save_flags(	gchar *in_cmd, 
			gboolean interm, 
			gboolean hold)
{
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *g=g_build_filename(xdg_dir,RUN_FLAG_FILE,NULL);

    g_free(xdg_dir);
    
    if((runflags = DBH_open(g)) == NULL)
    {
	if((runflags = DBH_create(g, 11)) == NULL){
	    g_warning("Cannot create %s\n",g);
	    return;
	}
    }
    gs = g_string_new(in_cmd);
    sprintf((char *)DBH_KEY(runflags), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    /*printf("TRACE: bookmarking with key=%s\n",d_path);*/
    flags = (int *)runflags->data;
    flags[0]=interm;
    flags[1]=hold;
    DBH_set_recordsize(runflags, 2*sizeof(int));
    
    DBH_update(runflags);
    DBH_close(runflags);
    TRACE("flags saved in dbh file for %s\n",in_cmd);
}

G_MODULE_EXPORT
void
xffm_recover_flags(	gchar *in_cmd,
			gboolean *interm,
			gboolean *hold)
{
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *g=g_build_filename(xdg_dir,RUN_FLAG_FILE,NULL);

    g_free(xdg_dir);
    
    if((runflags = DBH_open(g)) == NULL)
    {
	    TRACE("Cannot open %s\n",g);
	    *interm=0;
	    *hold=0;
	    return;
    }
    gs = g_string_new(in_cmd);
    sprintf((char *)DBH_KEY(runflags), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    /*printf("TRACE: bookmarking with key=%s\n",d_path);*/
    flags = (int *)runflags->data;
    DBH_load(runflags);
    *interm = flags[0];
    *hold = flags[1];
    DBH_close(runflags);

    TRACE("flags recovered from dbh file for %s, interm=%d hold=%d\n",in_cmd,*interm,*hold);
}

/* use absolute path to call this function...*/
G_MODULE_EXPORT
int 
xffm_on_run_path(	widgets_t *widgets_p,
			char *in_cmd, 
			const gchar *path, 
			gboolean interm, 
			gboolean remember, 
			gboolean put_in_history,
			gboolean hold)
{
    GError *error=NULL;
    const gchar *command;
    gchar *name=NULL;
    int argc;
    gchar **argv;
    gchar *wd=NULL;
    if (!g_file_test(in_cmd,G_FILE_TEST_EXISTS) && !MIME_is_valid_command(in_cmd)){
	if (!g_file_test(in_cmd,G_FILE_TEST_EXISTS))
	    print_diagnostics(widgets_p,"xfce/error",strerror(ENOENT),": ",in_cmd,"\n",NULL);
	else
	    print_diagnostics(widgets_p,"xfce/error",strerror(ENOEXEC),": ",in_cmd,"\n",NULL);
	return FALSE;
    }

    if(getenv("XFFM_HOLD_XTERM") && strlen( getenv("XFFM_HOLD_XTERM"))) hold=TRUE;
    
    /* why did I ever use a relative path here? Reverting cause it breaks
     * doubleclick on .html files...
     * if (path) name=g_path_get_basename(path); else name=g_strdup("");
     *
     * The reason why is that applications like abiword break with absolute
     * paths if the paths contain spaces in them. So again I revert, testing
     * for doubleclick with .html files...
     *
     * if (path) {name=g_strdup(path);} else {name=g_strdup(""); }
     * */
    if (path && strlen(path)) {
	gchar *t;
	/* strip quotes from dirname */
	if (*path=='\"') wd=g_path_get_dirname(path+1);
	else wd=g_path_get_dirname(path);
	
	/* leave quotes on basename */
	if (*path=='\"') t=g_path_get_basename(path+1);
	else t=g_path_get_basename(path);
	if (strchr(t,'\"')) name=g_strdup_printf("\"%s",t);
	else name=g_strdup_printf("\"%s\"",t);
	g_free(t);	    
    } else name=g_strdup("");
    
    command = MIME_mk_command_line(in_cmd, name, interm, hold);
    TRACE("command=%s",command);
    if (!command){
    	g_free(name);   
	g_free (wd);
	g_warning("!command from %s",in_cmd);
	return FALSE;
    }

    
    TRACE("in_cmd=%s\n",command);
    g_shell_parse_argv (command, &argc,&argv,&error);
    if (error){
	gchar *msg = g_strcompress (error->message);
	TRACE(error->message);
	print_diagnostics(widgets_p,"xfce/error",msg,":\n",command,"\n",NULL);
	g_error_free(error);
	g_free (msg);
	g_free (wd);
	g_free(name); 
	return FALSE;
    }
    TRACE("going to xffm_runvwd %s %s",wd,argv[0]);
    if (wd) chdir(wd);
    if (xffm_runvwd(widgets_p,wd,argv)){
	/* recent history: */
	if (path && strlen(path)){
	    if (!g_file_test(path,G_FILE_TEST_EXISTS)) g_warning("%s does not exist",path);
	    else RECENT_add2history(widgets_p,(gchar *)strip_quotes(path));
	} 
        TRACE("save_to_open_history=%s->%s(%d)",strip_quotes(name),in_cmd,interm);
	save_to_open_history((gchar *)strip_quotes(name),in_cmd,interm);
    }
    g_free(wd); 
    g_strfreev (argv);
    
    TRACE("put_in_history=%d",put_in_history);
    
    if (put_in_history){
      gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
      gchar *f=g_build_filename(xdg_dir,RUN_DBH_FILE,NULL);

      g_free(xdg_dir);
	TRACE("XFC_save_to_history(%s,%s)",f,in_cmd);
      XFC_save_to_history(f,in_cmd);
      xffm_save_flags(in_cmd,interm,hold);
      g_free(f);
    }

    TRACE("path=%s,remember=%d)",path,remember);
    
    if(name && remember)
    {
#if 10
	/* multiple dots? */
	gchar *p=g_path_get_basename(name);
	gchar *foo=NULL;
	if (p && strchr(p,'.') && strchr(p,'.') != strrchr(p,'.') && strlen(strrchr(p,'.'))>1){
	    gchar *m=g_strdup_printf(_("Remember %s or %s?"),strchr(p,'.'),strrchr(p,'.'));
	    if (xffm_confirm (widgets_p,
				     (const gchar *)m,
				     (const gchar *)strchr(p,'.'),
				     (const gchar *)strrchr(p,'.'))){
		foo=g_strdup_printf("%s",strrchr(p,'.')+1);
	    } else {
		foo=g_strdup_printf("%s",strchr(p,'.')+1);
	    }	
	} else {
	    if (p && strchr(p,'.') && strlen(strrchr(p,'.'))>1) foo=g_strdup_printf("%s",strchr(p,'.')+1);
	    else foo=g_strdup(name);
	}
#endif
	if(strlen(name))
	{
	   command = MIME_mk_command_line(in_cmd, NULL, interm, hold);
	
	    TRACE("TRACE: adding %s --> %s\n",name,in_cmd); 
	    print_diagnostics(widgets_p,"xfce/info",foo,"-->",command,"\n",NULL);
	    MIME_add(foo,command);
	}
	g_free(p);
	g_free(foo);
    }
    g_free(name); 
    chdir(GETWD);
    return TRUE;
}


G_MODULE_EXPORT
int 
xffm_on_run(	widgets_t *widgets_p,
		char *in_cmd, 
		record_entry_t * en, 
		gboolean in_terminal, 
		gboolean remember, 
		gboolean put_in_history, 
		gboolean hold)
{
	gchar *path=NULL;
	int result;

	if (en && IS_NETFILE(en->subtype)){
	    path = g_strdup(XF_SMBget_cache_file(en,widgets_p));
	} 
	else if (en) path=g_strdup(en->path);
	else path=g_strdup("");
	
	/*if (en) {
	    gchar *dir=g_path_get_dirname(en->path);
	    path=g_path_get_basename(en->path);
	    chdir(dir);
	}*/
	result = xffm_on_run_path(widgets_p, in_cmd, path, in_terminal, remember, put_in_history, hold);
	g_free(path);
	return result;
}




G_MODULE_EXPORT
void 
xffm_set_run_combo(	widgets_t *widgets_p,
			record_entry_t *en,
			xfc_combo_info_t *combo_info)
{
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *f=g_build_filename(xdg_dir,RUN_DBH_FILE,NULL); 
    gboolean update=FALSE;
    gboolean docheck=TRUE;

    if (access(f,F_OK)!=0){update=TRUE; docheck=FALSE;}
    else {
       struct stat st[2];	    
       if (stat(f,st) < 0) update=TRUE;
       else {
	 gchar *m=g_strconcat(APPLICATION_MIME_FILE,NULL);
	 if (stat(m,st+1) >= 0 && st[0].st_mtime < st[1].st_mtime) update=TRUE;
	 m=g_strconcat(xdg_dir,LOCAL_APPLICATION_MIME_FILE,NULL);
	 if (stat(m,st+1) >= 0 && st[0].st_mtime < st[1].st_mtime) update=TRUE;
       }
    }
    
    XFC_read_history(combo_info,f);

    if (en) {
	const gchar *p=MIME_command(en->path);
    	if (p) combo_info->list = g_list_prepend(combo_info->list,g_strdup(p) );
    }
    XFC_set_combo(combo_info,NULL);
    g_free(f);

    {
      gboolean interm,hold;
      GtkEntry * entry = (GtkEntry *) lookup_widget(widgets_p->window,"input_entry");
      gchar *choice = g_strdup((gchar *)gtk_entry_get_text(entry));
      GtkWidget *check = lookup_widget(widgets_p->window,"checkbutton1");
      xffm_recover_flags(choice,&interm, &hold);
      gtk_toggle_button_set_active ((GtkToggleButton *)check,interm);
      check = lookup_widget(widgets_p->window,"checkbutton3");
      gtk_toggle_button_set_active ((GtkToggleButton *)check,hold);
      if(getenv("XFFM_HOLD_XTERM") && strlen( getenv("XFFM_HOLD_XTERM"))) {
         gtk_toggle_button_set_active ((GtkToggleButton *)check,TRUE);
	 gtk_widget_set_sensitive(check, FALSE);
      } else {
	 gtk_widget_set_sensitive(check, TRUE);
      }
      
      if (!interm) gtk_widget_hide(check); else gtk_widget_show(check); 
      g_free(choice);
    }
    g_free(xdg_dir);

    return;
}


G_MODULE_EXPORT
void 
xffm_double_click_run(	widgets_t *widgets_p, record_entry_t *en)
{
    GError *error=NULL;
    char *a=NULL;
    int argc;
    gchar **argv;
    gboolean interm=FALSE;
    /* interm specified? */
    /* obsolete (from xfapps branch):
     * if (IS_IN_TERM(en->subtype)) interm=TRUE; */
 
    if (!IS_EXE(en->type)) return;
    
    /* is it a script? */
    if (is_a_script(en->path)) interm=TRUE;
    
    if (interm){
	const gchar *the_terminal=xffm_what_term();
	const gchar *exec_flag="-e";
	if (strstr("gnome-terminal",the_terminal) 	|| 
		strstr("gnome2-terminal",the_terminal)	||
		strstr("Terminal",the_terminal)	||
		strstr("terminal",the_terminal)	||
		strstr("xterminal",the_terminal) ) exec_flag="-x";
		
	a = g_strdup_printf("%s %s \"%s\"",the_terminal, exec_flag,en->path);
    } else {
        a=g_strdup_printf("\"%s\"",en->path);
    }
    
    g_shell_parse_argv (a, &argc,&argv,&error);
    if (error){
	gchar *msg = g_strcompress (error->message);
	print_diagnostics(widgets_p,"xfce/error",msg,":\n",a,"\n",NULL);
	g_error_free(error);
	g_free(a);
	g_free (msg);
	return;
    }
    xffm_runvwd(widgets_p,NULL,argv);
    g_strfreev (argv); 
    g_free(a);
    return;
}

G_MODULE_EXPORT
gboolean 
xffm_try_sudo(widgets_t *widgets_p, char *command, char *arg, char *path){
    int i;
    gchar *g;
    g = g_find_program_in_path("sudo");
    if (g){
	char *argument[6];
	static gchar *sudo_prompt=NULL;
       /* try SUDO if available before barfing...*/
	if (sudo_prompt) g_free(sudo_prompt);
	sudo_prompt = g_strconcat("SUDO_PROMPT=","Password:","\n",NULL);
	putenv(sudo_prompt);
	argument[0]="sudo";
	argument[1]="-S";
	argument[2]=command;
	argument[3]=arg;
	argument[4]=path;
	argument[5]=NULL;
	    
	print_diagnostics(widgets_p,"xfce/sudo",NULL);
	for (i=0; argument[i]; i++) print_diagnostics(widgets_p,NULL," ",argument[i],NULL);
	print_diagnostics(widgets_p,NULL,"\n",NULL);
	easy_widgets_p = widgets_p;
	Tubo(fork_function, 
			(void *)argument, 
			easy_finished_function, 
			&childFD, 
			easy_stdout, 
			easy_stderr,0,FALSE);
	g_free(g);
	return TRUE;
    } 
    print_diagnostics(widgets_p,"xfce/error",strerror(EINVAL),": ",command," ",arg," ",path,"\n",NULL);
    
    return FALSE;
}
    

