
/*
 *
 * Edscott Wilson Garcia 2001-2005 for xfce project.
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>

#include <errno.h>
#include <limits.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <regex.h>

#include <gtk/gtk.h>
#include <glib.h>
#include <gmodule.h>

#include "constants.h"

#ifdef HAVE_LIBXFCEGUI4
#include <libxfcegui4/libxfcegui4.h>
#endif

#define CACHE_DIR "xffm","modules"
#define DEFAULT_THEME "Rodent"
#define FALLBACK "hicolor" 

#define VALID_KDEDIR g_getenv("KDEDIR") && strlen(g_getenv("KDEDIR"))
#define KDEDIR g_getenv("KDEDIR"),"share","icons"
#define HOMEDIR  g_get_home_dir(),".icons"
#define SERIAL 41911

typedef struct theme_info_t {
	GHashTable *icon_hash;
	GtkIconFactory *icon_factory;
} theme_info_t;

typedef struct icon_info_t {
    gchar *path;
    int size;  
    gchar *type;
    struct icon_info_t *next;
}icon_info_t;

typedef struct smallhash_t {
    gchar *context;
    GHashTable *hash;
}smallhash_t;

typedef struct themehash_info_t {
    gchar *id;  /* untranslated theme name */
    GHashTable *bighash; /* hash of icon_info_t */
    GList *smallhashes;  /* context hashes to speed access to bighash */ 
}themehash_info_t;

typedef struct cache_info_t {
    long long basedir_sum;
    gchar supported_regex[256];
} cache_info_t;


#include "icon-module.h"

int open_theme(gchar *theme, int size);
void close_theme(void);
const gchar *find_icon_path(const gchar *key,int size,const gchar *context);
gchar *mime_icon_find_pixmap_file (const gchar     *filename);
GdkPixbuf *mime_icon_create_pixbuf (const gchar     *filename);
GtkWidget *mime_icon_create_pixmap (GtkWidget *widget, const gchar   *filename);
GtkIconSet *mime_icon_get_iconset(const gchar *id, GtkWidget *main_window);
void mime_icon_add_iconset(const gchar *tag, GtkIconSet *icon_set);
int mime_icon_load_theme(void);
char **mime_icon_find_themes(gboolean only_valid,gboolean full_paths);
gchar *mime_icon_get_local_xml_file(const gchar *theme_name);
gchar *mime_icon_get_global_xml_file(const gchar *theme_name);
gchar *mime_icon_get_theme_path(const gchar *theme);

#include "icon-module-theme.i"
#include "icon-module-mime.i"
/******************************************************************/

G_MODULE_EXPORT
const gchar *g_module_check_init(GModule *module){
    xfmime_icon_fun = g_new0 (xfmime_icon_functions,1);
    TRACE("g_module_check_init...\n");
    if (!xfmime_icon_fun) return ("unable to create function structure");
    xfmime_icon_fun->mime_icon_find_pixmap_file =mime_icon_find_pixmap_file ;
    xfmime_icon_fun->mime_icon_check4icon_path = mime_icon_check4icon_path ;
    xfmime_icon_fun->mime_icon_get_iconset = mime_icon_get_iconset;
    xfmime_icon_fun->mime_icon_add_iconset = mime_icon_add_iconset;
    xfmime_icon_fun->mime_icon_load_theme = mime_icon_load_theme;
    xfmime_icon_fun->mime_icon_create_pixmap = mime_icon_create_pixmap;
    xfmime_icon_fun->mime_icon_create_pixbuf = mime_icon_create_pixbuf;
    xfmime_icon_fun->mime_icon_find_themes = mime_icon_find_themes;
    xfmime_icon_fun->mime_icon_get_local_xml_file = mime_icon_get_local_xml_file;
    xfmime_icon_fun->mime_icon_get_global_xml_file = mime_icon_get_global_xml_file;
    xfmime_icon_fun->mime_icon_get_theme_path = mime_icon_get_theme_path;
    return NULL;
}

G_MODULE_EXPORT
gchar *
mime_icon_get_local_xml_file(const gchar *theme_name){
    gchar *xdg_dir= xfce_resource_save_location (XFCE_RESOURCE_THEMES,"/",TRUE);
    gchar *base=g_path_get_basename(theme_name);
    gchar *n=g_build_filename(xdg_dir,base,NULL); 
    gchar *mimefile=g_strconcat(n,".mime.xml",NULL);
    g_free(base);
    g_free(n);
    g_free(xdg_dir);
    return mimefile;
}

G_MODULE_EXPORT
gchar *mime_icon_get_theme_path(const gchar *theme){
    gchar **p;
    gchar **sdirs;
    gchar *g=NULL;
    
    xfce_resource_push_path(XFCE_RESOURCE_ICONS, PACKAGE_DATA_DIR "/icons");
    sdirs = xfce_resource_dirs(XFCE_RESOURCE_ICONS);
    xfce_resource_pop_path(XFCE_RESOURCE_ICONS);
    
    /* literal name */
    TRACE("mime_icon_get_theme_path...%s\n",theme);
    for (p=sdirs;*p;p++){
	g=g_build_filename(G_DIR_SEPARATOR_S,*p,theme,NULL);
    TRACE("mime_icon_get_theme_path...%s?\n",g);
	if (g_file_test(g,G_FILE_TEST_IS_DIR)) {
	    goto done;
	}
	g_free(g); g=NULL;
    }
    
    /* literal name not found: is it a translated name? */
    
    TRACE("mime_icon_get_theme_path...2\n");
    for (p=sdirs;*p;p++)
    {
        XfceRc *themefile;
        gchar *themeindex;
	const gchar *themename;
	GDir *dir;
	const gchar *file;
    TRACE("mime_icon_get_theme_path...2.0:%s\n",*p);

	if (!g_file_test(*p,G_FILE_TEST_IS_DIR)) continue;
    
        dir = g_dir_open(*p, 0, NULL);
    TRACE("mime_icon_get_theme_path...2.1\n");
        if(!dir) continue;

    TRACE("mime_icon_get_theme_path...%s\n",*p);
        while((file = g_dir_read_name(dir)))
        {
            themeindex = g_build_path(G_DIR_SEPARATOR_S, *p, file, "index.theme", NULL);
            themefile = xfce_rc_simple_open(themeindex, TRUE);
            g_free(themeindex);
            if(!themefile) continue;

            xfce_rc_set_group(themefile, "Icon Theme");

            if(strcmp(xfce_rc_read_entry(themefile, "Hidden", "false"), "true") == 0)
            {
                xfce_rc_close(themefile);
                continue;
            }

            themename = xfce_rc_read_entry(themefile, "Name", file);
            xfce_rc_close(themefile);

	    if (themename && strcmp(themename,theme)==0){
		/* we got a hit */
		g = g_build_path(G_DIR_SEPARATOR_S, *p, file,  NULL);
		if (g_file_test(g,G_FILE_TEST_IS_DIR)) break;
		g_free(g); g=NULL;
	    }
        }
        g_dir_close(dir);
	if (g) break;
    }
done:
    g_strfreev(sdirs);  
    TRACE("mime_icon_get_global_xml_file...done\n");
    return g;
}

G_MODULE_EXPORT
char **
mime_icon_find_themes(gboolean only_valid,gboolean full_paths)
{
    char **dirs = NULL;
    GList *list = NULL, *li;
    gchar **p;
    gchar **sdirs;
    GDir *gdir;
    const char *file;
    int i, len;

    xfce_resource_push_path(XFCE_RESOURCE_ICONS, PACKAGE_DATA_DIR "/icons");
    sdirs= xfce_resource_dirs(XFCE_RESOURCE_ICONS);
    xfce_resource_pop_path(XFCE_RESOURCE_ICONS);

    for (p=sdirs;*p;p++){	
	gchar **q;

	/* we don't want to process "pixmaps" directories */
	if (strstr(*p,"pixmaps")) continue;
	    
	for(q=sdirs;q!=p;q++){
	    if (strcmp(*q,*p)==0) {
		goto skip_duplicate;
	    }
	}

     if ((gdir = g_dir_open(*p, 0, NULL)) != NULL) {
      while((file = g_dir_read_name(gdir))) {
		char *path = g_build_filename(*p, file, NULL);
		if (only_valid && !is_valid_theme_dir(path)){
		    g_free(path);
		    continue;
		} 
		if(!g_list_find_custom(list, file, (GCompareFunc) strcmp) && g_file_test(path, G_FILE_TEST_IS_DIR))
		{
		    if (!full_paths) list = g_list_append(list, g_strdup(file));
		    else list = g_list_append(list, g_strdup(path));
		}
		g_free(path);
      }
      g_dir_close(gdir);
    } 
skip_duplicate:;
   }
   if (!list) return NULL;
   
   len = g_list_length(list);
   dirs = g_new0(char *, len + 1);
   for(i = 0, li = list; li; li = li->next, i++) {
	/*TRACE("got %s\n",(char *)li->data);*/
	dirs[i] = (char *)li->data;
   }
   g_list_free(list);
   return dirs;
}  

G_MODULE_EXPORT
gchar*
mime_icon_find_pixmap_file                       (const gchar     *filename)
{   
    const gchar *path;
    if (!filename) return NULL;
    path=find_icon_path(filename, 48,"MimeType");
    if (!path) return NULL;
    else return (g_strdup(path));
  /*return (g_strdup(find_icon_path(filename, 48,NULL)));*/
}



G_MODULE_EXPORT
gchar *
mime_icon_get_global_xml_file(const gchar *theme_name){
    gchar *themepath=NULL;
    gchar *mimefile=NULL;
    if (!theme_name) {
	g_warning("theme_name==NULL");
	return NULL;
    }
    TRACE("mime_icon_get_global_xml_file name...%s\n",theme_name);
    themepath=mime_icon_get_theme_path(theme_name);
    if (themepath) {
	gchar *b=g_path_get_basename(themepath);
	mimefile=g_strconcat(PACKAGE_DATA_DIR,G_DIR_SEPARATOR_S,
	        "xffm",G_DIR_SEPARATOR_S,
		 b,".mime.xml",NULL);
	g_free(b);
    }
     
    TRACE("mime_icon_get_global_xml_file...3:%s\n",mimefile);
   /* test for default mimefile (xfce distributed)  */
   if (!mimefile || !g_file_test(mimefile,G_FILE_TEST_EXISTS)) {
        gchar **inherits;
        gchar **p;
	
	TRACE("global file %s not found\n",mimefile);
        g_free(mimefile); mimefile=NULL;
	inherits = mime_icon_get_inherits(theme_name);
	/* test for inherited default mimefiles (xfce distributed) */
	for (p=inherits; p && *p; p++){
	    mimefile=g_strconcat(PACKAGE_DATA_DIR,G_DIR_SEPARATOR_S,
	        "xffm",G_DIR_SEPARATOR_S,
		 *p,".mime.xml",NULL);
	    TRACE("inherits file %s \n",mimefile);
	    if (g_file_test(mimefile,G_FILE_TEST_EXISTS)) break;
	    g_free(mimefile); mimefile=NULL;
	}
	g_strfreev(inherits);  
	
   }
   if (!mimefile || access(mimefile,F_OK)!=0){
	g_warning("No mime file found for theme %s",theme_name);
        if (strcmp("hicolor",theme_name)) {
	    g_free(mimefile); mimefile=NULL;
	    return mime_icon_get_global_xml_file("hicolor");
	} else {
	    return NULL;
	}
   }	
    
   return mimefile;
}

G_MODULE_EXPORT
GdkPixbuf *mime_icon_create_pixbuf                          (const gchar     *filename)
{
  GdkPixbuf *pixbuf;
  GError *error=NULL;
  const gchar *fullpath;


  
  if (!filename || !filename[0]) return NULL;
  if ((fullpath = find_icon_path(filename,48,"MimeType"))==NULL) {
      return NULL;
  }
  /*printf("creating pixbuf for... %s\n",fullpath);*/
  pixbuf = xfce_pixbuf_new_from_file_at_size(fullpath, 48,48,&error);
  if (error) {
      g_warning("failed to create pixbuf:%s", filename);
      g_warning(error->message);
      g_error_free(error);
  }
  return pixbuf;
}

G_MODULE_EXPORT
GtkWidget *mime_icon_create_pixmap (GtkWidget *widget, const gchar     *filename)
{
  const gchar *pathname;
  GtkWidget *pixmap;

  if (!filename || !filename[0]) return gtk_image_new ();
  pathname = (gchar *)find_icon_path(filename, 48, "MimeType");
  /*pathname = (gchar *)find_icon_path(filename, 48, NULL);*/
  

  if (!pathname)
    {
      TRACE ("Couldn't find pixmap file: %s", filename);
      return gtk_image_new ();
    }
  pixmap = gtk_image_new_from_file (pathname);
  if (!pixmap) {
      g_warning("FIXME: this should not happen (gtk<2.4 and no libsvg");
      return gtk_image_new ();
  }
  return pixmap;
}

G_MODULE_EXPORT
GtkIconSet *mime_icon_get_iconset(const gchar *id, GtkWidget *main_window){
    GtkIconSet *icon_set;
    const gchar *iconfile;
    GdkPixbuf *pixbuf;
    gchar *gg=NULL;

    if (!id || !strlen(id)) return NULL;
    TRACE("looking for %s in theme",id);

    if (!icon_hash) return NULL;
    icon_set = gtk_icon_factory_lookup(icon_factory, id);
    if (icon_set) return icon_set;

   

    /* is the id an actual path? (custom icons) */
    if (g_file_test(id,G_FILE_TEST_EXISTS)) {
	GError *error=NULL;
	pixbuf = gdk_pixbuf_new_from_file (id, &error);
	if (error) g_error_free(error);
	if (!pixbuf){
broken_image:
    	TRACE("broken_image!\n");
	    pixbuf = gtk_widget_render_icon(main_window, GTK_STOCK_MISSING_IMAGE, GTK_ICON_SIZE_DIALOG, NULL);   
	}
	goto done;
    }
    else {
	iconfile=(const gchar *) g_hash_table_lookup(icon_hash,id);
	/*TRACE("iconfile %s",iconfile);*/
	if(iconfile && strncmp("gtk-", iconfile, strlen("gtk-")) == 0){
	    TRACE("rendering icon %s...\n",iconfile);
	    pixbuf = gtk_widget_render_icon(main_window, iconfile, GTK_ICON_SIZE_DIALOG, NULL);
	   TRACE("%s rendered pixbuf=0x%x.\n",iconfile,(unsigned)pixbuf);
	    goto done;
	}
	else if (!iconfile && strchr(id,'/')) {
	    gchar *g,*gg;
	    g=g_strdup(id);
	    *strchr(g,'/') = 0;
	    gg=g_strconcat(g,"/","default",NULL);
	    iconfile=(const gchar *) g_hash_table_lookup(icon_hash,gg);
	    g_free(g);
	    g_free(gg);
	}
	if (!iconfile) iconfile=mime_icon_find_pixmap_file(id);
#if 0	
	const gchar *path=mime_icon_find_pixmap_file(id);
	TRACE("path is %s \n",path);
	if (path) 
	    iconfile=path;
	else iconfile=(const gchar *) g_hash_table_lookup(icon_hash,id);
#endif
    }
    
    if (!iconfile || !strlen(iconfile)) {
	    if (strchr(id,'/')){
		    gchar *g=g_strdup(id);
		    *(strchr(g,'/'))=0;
		    gg=g_strconcat(g,"/default",NULL);
		    g_free(g);
    		    icon_set = gtk_icon_factory_lookup(icon_factory, gg);
		    if (icon_set){
			    g_free(gg);
			    return icon_set;
		    }
		    iconfile=g_hash_table_lookup(icon_hash,gg);
	    }
    }
    if (!iconfile || !strlen(iconfile)) {
	TRACE("!iconfile || !strlen(iconfile): %s \n",id);
	goto broken_image;
    }
    TRACE("adding icon to iconfactory: %s->%s",id,iconfile);

    /* full path specified? (as in custom icons) */    
    if (g_file_test(iconfile,G_FILE_TEST_EXISTS)) {
	   GError *error=NULL;
	    TRACE("creating pixbuf");
	   
	   pixbuf = gdk_pixbuf_new_from_file (iconfile, &error);
	    TRACE("created pixbuf");
	   if (error) {
	       g_warning("g_error: %s",error->message);
	       g_error_free(error);
	   }
	   if (!pixbuf) goto broken_image;
    }
    else {
	    TRACE("creating pixbuf with  mime_icon_create_pixbuf");
	    pixbuf=mime_icon_create_pixbuf(iconfile);
	    if (!pixbuf){
		g_warning("problem with %s",iconfile);
		TRACE("problem with %s",iconfile);
	    }
	    else {
		TRACE("created pixbuf OK");
	    }
		
    }
	
done:
    if (!pixbuf) {
	TRACE("!pixbuf");
	return NULL;
    }
    TRACE("creating icon_set");
    icon_set = gtk_icon_set_new_from_pixbuf(pixbuf);
    /*gtk_icon_set_ref (icon_set);*/


    if (!icon_set) {
	TRACE("creating icon_set FAILED");
	return NULL;
    }
    if (gg) {
	    gtk_icon_factory_add(icon_factory, gg, icon_set);
	    g_free(gg);
    } else gtk_icon_factory_add(icon_factory, id, icon_set);
    g_object_unref (G_OBJECT (pixbuf));
    
    return icon_set; 
}

G_MODULE_EXPORT
void mime_icon_add_iconset(const gchar *tag, GtkIconSet *icon_set){
	gtk_icon_factory_add(icon_factory, tag, icon_set);
}

void g_module_unload(GModule *module){
	/* hmm... this really does not free up the resident memory... */
	if (icon_hash){
	   g_hash_table_foreach(icon_hash, clear_iconhash, NULL);
           g_hash_table_destroy(icon_hash);
   	   icon_hash=NULL;
	}
	gtk_icon_factory_remove_default (icon_factory);
	if (xfmime_icon_fun) g_free(xfmime_icon_fun);
	xfmime_icon_fun=NULL;
	close_theme();
}

G_MODULE_EXPORT
int mime_icon_load_theme(void){
    int r;
    TRACE("mime_icon_load_theme...\n");
    r=create_icon_hash();
    if (r) return r;
    g_warning("cannot create icon_hash");
    return r;
}

G_MODULE_EXPORT
xfmime_icon_functions *module_init(void){
    return xfmime_icon_fun;
}


G_MODULE_EXPORT
void
close_theme(void) {
    /* free up space taken by lists and hashtables */
    theme_list = free_theme_list (theme_list);
    base_dirs = free_string_list (base_dirs);    
}

static
gint zombie_killer(gpointer *data){
    TRACE("zombie_killer... pid=%u\n",(unsigned)cache_manager);
  if (cache_manager) {
    int status;
    waitpid(cache_manager, &status, WNOHANG);
    if(WIFEXITED(status)) {
	cache_manager=0;
	TRACE("die zombie...\n"); 
	return FALSE;
    }
  }
  return TRUE;
}

/*
 * open_theme()
 * 
 * if theme is NULL, will open default theme set by gtk.
 * if size <= 0, will not cache. Otherwise, will use a cache
 * with best matches for specified size */

G_MODULE_EXPORT
int 
open_theme(gchar *theme, int size){
    static gchar *last_theme=NULL;

    if (!theme) {
#if GTK_CHECK_VERSION (2,4,0)
	g_object_get (gtk_settings_get_default(), "gtk-icon-theme-name", &theme, NULL);
#else /* gtk < 2.4 */
	{
	    GtkSettings *gsettings = gtk_settings_get_default ();
	    g_object_get (G_OBJECT (gsettings), "gtk-icon-theme-name", &theme, NULL);
	}
#endif
    }

    g_free(requested_theme);
    requested_theme=g_strdup(theme);
    if (!theme) {
        g_warning("no icon theme defined");
        return 0;
    }
    
    if (size > 0){
	g_free(last_theme);
	last_theme=NULL;
	cache_file = get_cache_path(theme,size);
    } 
    if (!cache_file) {
	if (last_theme && strcmp(last_theme,theme)==0) return -1;
	g_free(last_theme);
	last_theme=g_strdup(theme);
	cache_file=NULL;
    }
	
    theme_list = free_theme_list (theme_list);
    base_dirs = get_base_dirs(base_dirs);
#ifdef DEBUG
	showme_themeinfo(theme_list,"theme_list");
#endif

    if (size <= 0) { /* size <= 0, forces no cache used */
	/* this add inherits too */
	if (add_theme(base_dirs,theme) == FALSE) {
	    return 0; 
	}
    }
    else {
	/* if cache already exists, use this one and check
	 * for regeneration (if necessary) in background */
	if (cache_file && g_file_test(cache_file, G_FILE_TEST_EXISTS)){
	    TRACE("cache for size %d exists...\n",size);
	    if ((cache_manager=fork()) == 0) {
		/* give parent a head start */
		sleep(5);
		 TRACE("CHILD: comparing cache information for %s-%d.\n",theme,size);
		if (!compare_cache_info(theme,size)) {
		  TRACE("CHILD: cache does not compare for size %d.\n",size);
		  if (add_theme(base_dirs,theme) == FALSE) {
		    return 0; 
		  }
		  TRACE("CHILD: now generating cache, size=%d, theme=%s...",
		    size,theme);
		  generate_cache(theme,size);
		  close_theme();
		}  
		TRACE("CHILD: exiting now.");
		_exit(123);
	    } else if (cache_manager > 0){
		TRACE("cache manager=%u\n",(unsigned)cache_manager);
		g_timeout_add_full(0, 5000, 
			(GtkFunction) zombie_killer, NULL, NULL);
	    }
	} else {
	    TRACE("cache for size %d does not exist.\n",size);
	    if (add_theme(base_dirs,theme) == FALSE) {
		return 0; 
	    }
	    TRACE("generating cache,  size=%d, theme=%s...",
		    size,theme);
	    generate_cache(theme,size);
	    close_theme();
	}
    }
    
    return 1;
}


G_MODULE_EXPORT
const gchar *
find_icon_path(const gchar *key,int size,const gchar *context){
    gchar *short_key=NULL;
    if (strchr(key,'.')) {
	const gchar *path;
	short_key = g_strdup(key);
	*strrchr(short_key,'.')=0;
	path=find_icon_path_priv(short_key, size, context,TRUE);
	g_free(short_key);
	return path;
    } else return find_icon_path_priv(key, size, context,TRUE);
}

G_MODULE_EXPORT
const gchar *
mime_icon_check4icon_path(const gchar *key){
    gchar *short_key=NULL;
    if (!key||!strlen(key)) return NULL;
    /* shortcircuit absolute paths: */
    if (*key == '/') {
	if (g_file_test(key,G_FILE_TEST_EXISTS)) return key;
	else return NULL;
    }
	    
    if (strchr(key,'.')) {
	const gchar *path;
	short_key = g_strdup(key);
	*strrchr(short_key,'.')=0;
	path=find_icon_path_priv(short_key, 48,"MimeType",FALSE);
	g_free(short_key);
	return path;
    } else return find_icon_path_priv(key, 48,"MimeType",FALSE);
}



