/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "jumptofilewin.h"
#include <xfmedia/xfmedia-playlist.h>
#include <xfmedia/xfmedia-settings.h>

typedef struct
{
    JTFActivateFunc activate_func;
    gpointer callback_data;
    gint column;
    
    GtkWidget *jtf_win;
    GtkWidget *entry;
    GtkWidget *treeview;
    GtkTreeSelection *treesel;
    GtkTreeModel *filter;
    guint idle_id;
    
    gchar **words;
} JtfData;

static gboolean
jtf_window_delete_cb(JtfData *jtfdata)
{
    if(jtfdata->idle_id)
        g_source_remove(jtfdata->idle_id);
    gtk_tree_view_set_model(GTK_TREE_VIEW(jtfdata->treeview), NULL);
    g_object_unref(G_OBJECT(jtfdata->filter));
    if(jtfdata->words)
        g_strfreev(jtfdata->words);
    g_free(jtfdata);
    
    return FALSE;
}

static void
jtf_window_destroy(JtfData *jtfdata)
{
    jtf_window_delete_cb(jtfdata);
    gtk_widget_destroy(jtfdata->jtf_win);
}

static void
jtf_run_callback(JtfData *jtfdata, GtkTreePath *path)
{
    if(jtfdata->activate_func)
        jtfdata->activate_func(path, jtfdata->callback_data);
}

static void
jtf_get_words(JtfData *jtfdata)
{
    gchar *text, *tmp, **words, *str_norm;
    gint i;
    
    if(jtfdata->words)
        g_strfreev(jtfdata->words);
    
    text = gtk_editable_get_chars(GTK_EDITABLE(jtfdata->entry), 0, -1);
    tmp = g_utf8_strdown(text, -1);
    g_free(text);
    words = g_strsplit(tmp, " ", -1);
    g_free(tmp);
    
    for(i = 0; words[i]; i++) {
        str_norm = g_utf8_normalize(words[i], -1, G_NORMALIZE_ALL);
        g_free(words[i]);
        if(!str_norm)
            str_norm = g_strdup("");
        words[i] = str_norm;
    }
    
    jtfdata->words = words;
}

static gboolean
jtf_refilter_idled(gpointer data)
{
    JtfData *jtfdata = data;
    GtkTreePath *path;
    
    gdk_threads_enter();
    
    jtfdata->idle_id = 0;
    
    gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(jtfdata->filter));
    
    gtk_tree_selection_unselect_all(jtfdata->treesel);
    path = gtk_tree_path_new_first();
    gtk_tree_selection_select_path(jtfdata->treesel, path);
    
    if(gtk_tree_selection_get_selected(jtfdata->treesel, NULL, NULL)) {
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(jtfdata->treeview), path,
                NULL, TRUE, 0.0, 0.0);
    }
    
    gtk_tree_path_free(path);
    
    gdk_threads_leave();
    
    return FALSE;
}

static gboolean
jtf_entry_keyrelease_cb(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    JtfData *jtfdata = user_data;
    
    if(evt->keyval == GDK_Escape) {
        jtf_window_destroy(jtfdata);
        return TRUE;
    } else if(evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter) {
        GtkTreeIter itr, child_itr;
        GtkTreePath *path;
        
        if(gtk_tree_selection_get_selected(jtfdata->treesel, NULL, &itr)) {
            GtkTreeModel *child_model;
            gtk_tree_model_filter_convert_iter_to_child_iter(GTK_TREE_MODEL_FILTER(jtfdata->filter),
                    &child_itr, &itr);
            child_model = gtk_tree_model_filter_get_model(GTK_TREE_MODEL_FILTER(jtfdata->filter));
            path = gtk_tree_model_get_path(child_model, &child_itr);
            jtf_run_callback(jtfdata, path);
            gtk_tree_path_free(path);
            jtf_window_destroy(jtfdata);
            
            return TRUE;
        }
    } else if(evt->keyval == GDK_Page_Up || evt->keyval == GDK_Page_Down
            || evt->keyval == GDK_KP_Page_Up || evt->keyval == GDK_KP_Page_Down
            || evt->keyval == GDK_Down || evt->keyval == GDK_KP_Down)
    {
        return TRUE;
    } else {
        if(jtfdata->idle_id)
            g_source_remove(jtfdata->idle_id);
        jtf_get_words(jtfdata);
        jtfdata->idle_id = g_idle_add(jtf_refilter_idled, jtfdata);
    }
    
    return FALSE;
}

static gboolean
jtf_entry_keypress_cb(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    JtfData *jtfdata = user_data;
    
    if(evt->keyval == GDK_Page_Up || evt->keyval == GDK_Page_Down
            || evt->keyval == GDK_KP_Page_Up || evt->keyval == GDK_KP_Page_Down
            || evt->keyval == GDK_Down || evt->keyval == GDK_KP_Down)
    {
        XKeyEvent xevt;
        
        xevt.type = KeyPress;
        xevt.send_event = True;
        xevt.display = GDK_DISPLAY_XDISPLAY(gtk_widget_get_display(jtfdata->treeview));
        xevt.window = GDK_WINDOW_XID(jtfdata->treeview->window);
        xevt.root = GDK_WINDOW_XID(gdk_screen_get_root_window(gtk_widget_get_screen(jtfdata->treeview)));
        xevt.subwindow = None;
        xevt.state = evt->state;
        xevt.keycode = evt->hardware_keycode;
        xevt.same_screen = True;
        
        gtk_widget_grab_focus(jtfdata->treeview);
        gdk_flush();
        gdk_error_trap_push();
        XSendEvent(xevt.display, xevt.window, True, KeyPressMask, (XEvent *)&xevt);
        xevt.type = KeyRelease;
        XSendEvent(xevt.display, xevt.window, True, KeyReleaseMask, (XEvent *)&xevt);
        gdk_error_trap_pop();
    }
    
    return FALSE;
}
    
static gboolean
jtf_treeview_fwd_keypress(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    JtfData *jtfdata = user_data;
    gchar *string, *text;
    gint insert_pos = 0;
    
    if(evt->keyval == GDK_Escape || evt->keyval == GDK_Return
            || evt->keyval == GDK_KP_Enter)
    {
        jtf_entry_keyrelease_cb(jtfdata->entry, evt, jtfdata);
        return TRUE;
    }
    
    string = gdk_keyval_name(evt->keyval);
    if(!strcmp(string, "space"))
        string = " ";
    else if(g_utf8_strlen(string, -1) > 1)
        return FALSE;
    
    text = gtk_editable_get_chars(GTK_EDITABLE(jtfdata->entry), 0, -1);
    if(text) {
        insert_pos = g_utf8_strlen(text, -1);
        g_free(text);
    }
    gtk_editable_insert_text(GTK_EDITABLE(jtfdata->entry), string, 1, &insert_pos);
    
    gtk_widget_grab_focus(jtfdata->entry);
    gtk_editable_set_position(GTK_EDITABLE(jtfdata->entry), -1);
    
    return TRUE;
}

static gboolean
jtf_treeview_buttonpress_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    JtfData *jtfdata = user_data;
    GtkTreeIter itr, child_itr;
    
    if(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS) {
        if(gtk_tree_selection_get_selected(jtfdata->treesel, NULL, &itr)) {
            GtkTreeModel *child_model;
            GtkTreePath *path;
            
            gtk_tree_model_filter_convert_iter_to_child_iter(GTK_TREE_MODEL_FILTER(jtfdata->filter),
                    &child_itr, &itr);
            child_model = gtk_tree_model_filter_get_model(GTK_TREE_MODEL_FILTER(jtfdata->filter));
            path = gtk_tree_model_get_path(child_model, &child_itr);
            jtf_run_callback(jtfdata, path);
            gtk_tree_path_free(path);
            jtf_window_destroy(jtfdata);
            
            return TRUE;
        }
    }
    
    return FALSE;
}

static gboolean
jtf_filter_visible_func(GtkTreeModel *model, GtkTreeIter *itr, gpointer data)
{
    JtfData *jtfdata = data;
    gchar *title = NULL, *tmp, *title_norm, **words;
    gint i;
    gboolean showme = TRUE;
    
    gtk_tree_model_get(model, itr, jtfdata->column, &title, -1);
    if(!title)
        return FALSE;
    tmp = g_utf8_strdown(title, -1);
    g_free(title);
    if(!tmp)
        return FALSE;
    title_norm = g_utf8_normalize(tmp, -1, G_NORMALIZE_ALL);
    g_free(tmp);
    if(!title_norm)
        return FALSE;
    
    if(!jtfdata->words)
        jtf_get_words(jtfdata);
    words = jtfdata->words;
    
    for(i = 0; words[i]; i++) {
        if(strstr(title_norm, words[i]))
            showme = TRUE;
        else {
            showme = FALSE;
            break;
        }
    }
    
    g_free(title_norm);
    
    return showme;
}

static gboolean
jtf_window_configure_cb(GtkWidget *w, GdkEventConfigure *evt,
        gpointer user_data)
{
    xfmedia_settings_set_int("/xfmedia/jtf/win_x", evt->x);
    xfmedia_settings_set_int("/xfmedia/jtf/win_y", evt->x);
    xfmedia_settings_set_int("/xfmedia/jtf/win_width", evt->width);
    xfmedia_settings_set_int("/xfmedia/jtf/win_height", evt->height);
    
    return FALSE;
}

static gboolean
jtf_select_first_initial(gpointer user_data)
{
    JtfData *jtfdata = user_data;
    GtkTreePath *path;
    
    gdk_threads_enter();
    path = gtk_tree_path_new_first();
    gtk_tree_selection_select_path(jtfdata->treesel, path);
    gtk_tree_path_free(path);
    gdk_threads_leave();
    
    return FALSE;
}

void
xfmedia_jump_to_file_show(GtkWindow *parent, GtkTreeModel *model, gint column,
        JTFActivateFunc activate_func, gpointer callback_data)
{
    JtfData *jtfdata = g_new0(JtfData, 1);
    GtkWidget *win, *vbox, *entry, *sw, *treeview, *hbox, *btn;
    GtkTreeModel *filter;
    GtkCellRenderer *render;
    GtkTreeViewColumn *col;
    
    jtfdata->column = column;
    jtfdata->activate_func = activate_func;
    jtfdata->callback_data = callback_data;
    
    jtfdata->jtf_win = win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(win), _("Jump to File"));
    gtk_window_set_transient_for(GTK_WINDOW(win), parent);
    gtk_container_set_border_width(GTK_CONTAINER(win), BORDER);
    gtk_widget_add_events(win, GDK_CONFIGURE);
    g_signal_connect(G_OBJECT(win), "configure-event",
            G_CALLBACK(jtf_window_configure_cb), jtfdata);
    g_signal_connect_swapped(G_OBJECT(win), "delete-event",
            G_CALLBACK(jtf_window_delete_cb), jtfdata);
    
    vbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(win), vbox);
    
    jtfdata->entry = entry = gtk_entry_new();
    gtk_widget_add_events(entry, GDK_KEY_PRESS|GDK_KEY_RELEASE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(entry), "key-press-event",
            G_CALLBACK(jtf_entry_keypress_cb), jtfdata);
    g_signal_connect(G_OBJECT(entry), "key-release-event",
            G_CALLBACK(jtf_entry_keyrelease_cb), jtfdata);
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER,
            GTK_POLICY_ALWAYS);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
            GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(sw);
    gtk_box_pack_start(GTK_BOX(vbox), sw, TRUE, TRUE, 0);
    
    jtfdata->filter = filter = gtk_tree_model_filter_new(GTK_TREE_MODEL(model), NULL);
    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
            jtf_filter_visible_func, jtfdata, NULL);
    
    jtfdata->treeview = treeview = gtk_tree_view_new_with_model(filter);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
    jtfdata->treesel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    gtk_tree_selection_set_mode(jtfdata->treesel, GTK_SELECTION_SINGLE);
    g_idle_add(jtf_select_first_initial, jtfdata);
    
    render = exo_cell_renderer_ellipsized_text_new();
    g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END,
            "ellipsize-set", TRUE, NULL);
    col = gtk_tree_view_column_new_with_attributes("title", render, "text",
            column, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    gtk_widget_add_events(treeview, GDK_KEY_PRESS|GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(treeview), "key-press-event",
            G_CALLBACK(jtf_treeview_fwd_keypress), jtfdata);
    g_signal_connect(G_OBJECT(treeview), "button-press-event",
            G_CALLBACK(jtf_treeview_buttonpress_cb), jtfdata);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    btn = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect_swapped(G_OBJECT(btn), "clicked",
            G_CALLBACK(jtf_window_destroy), jtfdata);
    
    gtk_widget_realize(win);
    gtk_window_resize(GTK_WINDOW(win),
            xfmedia_settings_get_int("/xfmedia/jtf/win_width"),
            xfmedia_settings_get_int("/xfmedia/jtf/win_height"));
    gtk_window_move(GTK_WINDOW(win),
            xfmedia_settings_get_int("/xfmedia/jtf/win_x"),
            xfmedia_settings_get_int("/xfmedia/jtf/win_y"));
    
    gtk_widget_show_all(win);
}
