/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001 - 2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* gui_newlogdialog.c - creation and destruction of the newlog dialog 
 *
 * here we create a new log, if an existing logname is chosen, we open it.
 * If the logname is correct we fire up the logeditor, so we can choose
 * the fields which should be present in the log.
 */

#include <gtk/gtk.h>
#include <string.h>

#include "gui_newlogdialog.h"
#include "gui_logeditordialog.h"
#include "gui_warningdialog.h"
#include "callbacks_mainwindow_menu.h"
#include "support.h"
#include "types.h"
#include "log.h"
#include "utils.h"
#include "gui_utils.h"

extern GList *logwindowlist;
extern GtkWidget *mainnotebook;
extern GtkWidget *mainwindow;
extern preferencestype preferences;
extern statetype state;

/* check for correct characters */
static void
on_newlogentry_changed (GtkEditable * editable, gpointer user_data)
{
	gchar *entry, *end, *j;
	gint len, pos;
	
	entry = gtk_editable_get_chars (editable, 0, -1);
	if ((len = strlen(entry)) > 0)
	{
		end = entry + strlen (entry);
		for (j = entry; j < end; ++j)
		{
			switch (*j)
			{
				case '0' ... '9':
				case 'a' ... 'z':
				case 'A' ... 'Z':
					break;
				default:
					gdk_beep ();
					pos = gtk_editable_get_position (editable);
					gtk_editable_delete_text (editable, pos, pos+1);
					break;
			}
		}
	}
}

void
on_menu_new_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *newlogdialog, *dialog_vbox, *vbox, *label1, *hbox, *label2,
		*newlogentry, *dialog_action_area, *newlogcancelbutton, *newlogokbutton;
	GdkPixbuf *newlogdialog_icon_pixbuf;
	gchar *entry, *filename, *xlogbackupfile;
	logtype *logwindow = NULL;
	gint i, response;
	LOGDB *lp;
	gboolean logopen = FALSE, logexist = FALSE;
	GString *logs;

	newlogdialog = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (newlogdialog), _("xlog - new log"));
	gtk_window_set_position (GTK_WINDOW (newlogdialog), GTK_WIN_POS_MOUSE);
	newlogdialog_icon_pixbuf = create_pixbuf ("xlog.png");
	if (newlogdialog_icon_pixbuf)
		{
			gtk_window_set_icon (GTK_WINDOW (newlogdialog), newlogdialog_icon_pixbuf);
			g_object_unref (newlogdialog_icon_pixbuf);
		}

	dialog_vbox = GTK_DIALOG (newlogdialog)->vbox;
	gtk_widget_show (dialog_vbox);

	vbox = gtk_vbox_new (TRUE, 0);
	gtk_widget_show (vbox);
	gtk_box_pack_start (GTK_BOX (dialog_vbox), vbox, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);

	label1 = gtk_label_new (_("Enter a name for this log"));
	gtk_widget_show (label1);
	gtk_box_pack_start (GTK_BOX (vbox), label1, TRUE, TRUE, 0);
	gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_CENTER);
	gtk_label_set_line_wrap (GTK_LABEL (label1), TRUE);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

	label2 = gtk_label_new_with_mnemonic ("(a-z, A-Z, 0-9)");
	gtk_widget_show (label2);
	gtk_box_pack_start (GTK_BOX (hbox), label2, TRUE, TRUE, 0);
	gtk_label_set_justify (GTK_LABEL (label2), GTK_JUSTIFY_CENTER);
	gtk_label_set_line_wrap (GTK_LABEL (label2), TRUE);

	newlogentry = gtk_entry_new ();
	gtk_widget_show (newlogentry);
	gtk_box_pack_start (GTK_BOX (vbox), newlogentry, FALSE, FALSE, 0);
	gtk_entry_set_max_length (GTK_ENTRY (newlogentry), 15);

	dialog_action_area = GTK_DIALOG (newlogdialog)->action_area;
	gtk_widget_show (dialog_action_area);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_SPREAD);

	newlogcancelbutton = gtk_button_new_from_stock ("gtk-cancel");
	gtk_widget_show (newlogcancelbutton);
	gtk_dialog_add_action_widget (GTK_DIALOG (newlogdialog), newlogcancelbutton, GTK_RESPONSE_CANCEL);
	GTK_WIDGET_SET_FLAGS (newlogcancelbutton, GTK_CAN_DEFAULT);

	newlogokbutton = gtk_button_new_from_stock ("gtk-ok");
	gtk_widget_show (newlogokbutton);
	gtk_dialog_add_action_widget (GTK_DIALOG (newlogdialog), newlogokbutton, GTK_RESPONSE_OK);
	GTK_WIDGET_SET_FLAGS (newlogokbutton, GTK_CAN_DEFAULT);

	g_signal_connect ((gpointer) newlogentry, "changed", G_CALLBACK (on_newlogentry_changed), NULL);

	response = gtk_dialog_run (GTK_DIALOG (newlogdialog));
	state.utf8error = FALSE;
	if (response == GTK_RESPONSE_OK)
	{
		entry = gtk_editable_get_chars (GTK_EDITABLE (newlogentry), 0, -1);

		if (strlen (entry) > 0)
		{
			for (i = 0; i < g_list_length (logwindowlist); i++)
			{		/* see if the log is already open */
				logwindow = g_list_nth_data (logwindowlist, i);
				if (g_ascii_strcasecmp (entry, logwindow->logname) == 0)
				{
					gtk_notebook_set_current_page (GTK_NOTEBOOK(mainnotebook), i);
					set_qsoframe (logwindow);
					logopen = TRUE;
					update_statusbar (_("A log with the same name is already open"));
					break;
				}
			}
			if (!logopen)
			{		/* check if the log exists and open it */
				filename = g_strconcat (entry, ".xlog", 0);
				logs = getlogs (preferences.savedir, "*");
				if (((logs->len) > 0) && (g_strrstr (logs->str, filename)))
				{
					logexist = TRUE;
					filename = g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, entry, ".xlog", NULL);
					lp = log_file_open (filename, TYPE_FLOG);
					if (lp)
					{
						logwindow =	openlog (lp, entry, state.logwindows++);
						log_file_qso_foreach (lp, fillin_list, logwindow);
						log_file_close (lp);
						logwindowlist =	g_list_append (logwindowlist, logwindow);
						gtk_notebook_set_current_page(GTK_NOTEBOOK (mainnotebook), state.logwindows - 1);
						set_qsoframe (logwindow);
						set_tabs_menu ();
						xlogbackupfile =	g_strconcat (filename, ".backup", NULL);
						savelog (logwindow, xlogbackupfile, TYPE_FLOG, 1, logwindow->qsos);
						g_free (xlogbackupfile);
						update_statusbar (_("A log with this name already exists"));
					}
					g_free (filename);
					g_string_free (logs, TRUE);
				}
			}
			if ((!logopen) && (!logexist))
			{		/* create a new log */
				filename = g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, entry, ".xlog", NULL);
				lp = log_file_create (filename, TYPE_FLOG, qso_field_nr, qso_fields, qso_widths);
				if (lp)
				{
					logwindow =	openlog (lp, entry, state.logwindows++);
					log_file_close (lp);
					logwindow->filename = g_strdup (filename);
					logwindowlist =	g_list_append (logwindowlist, logwindow);
					gtk_notebook_set_current_page (GTK_NOTEBOOK(mainnotebook), state.logwindows - 1);
					set_qsoframe (logwindow);
					set_tabs_menu ();
				}	
				g_free (filename);
			}
			g_free (entry);
		}
		else
		{
			gtk_widget_destroy (newlogdialog);
			return;
		}
		on_clearframe_activate (NULL, NULL);
		gtk_widget_destroy (newlogdialog);

		if (logexist && state.utf8error) 
			warningdialog (_("xlog - open log"), 
				_("There were some errors converting from your locale to UTF8, "
				"which is used by GTK+ internally. It is best if you start xlog "
				"from a terminal and see what the errors are. Please check your "
				"language settings and your log fields!"));

		/* now fire up the logeditor */
		if ((!logopen) && (!logexist))
		{
			on_menu_logeditor_activate (NULL, NULL);
		}
	} /* cancel clicked */
	else
		gtk_widget_destroy (newlogdialog);
}
