/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001 - 2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* gui_openlogdialog.c - creation and destruction of the logopen dialog
 *
 * this dialog let's you select any of the supported fileformats
 * (twlog, xlog, editest, trlog (tlf), EDI, adif and cabrillo), checks
 * if a filename is selected and opens the log. If a file is already opened,
 * we jump to the page in the log.
 */

#include <gtk/gtk.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>

#include "gui_openlogdialog.h"
#include "gui_warningdialog.h"
#include "types.h"
#include "log.h"
#include "support.h"
#include "utils.h"
#include "gui_utils.h"

extern statetype state;
extern preferencestype preferences;
extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;
extern GList *logwindowlist;

static GtkWidget *previewscrolledwindow;

/* 
 * in case we open a non-xlog file, we have to know the xlog filename
 */
static gchar *find_logname (const gchar *selected)
{
	gchar *basen, *dirn, **split, *l;

	basen = g_path_get_basename (selected);
	dirn = g_path_get_dirname (selected);

	if (g_strrstr(basen, "."))
	{
		split = g_strsplit (basen, ".", 0);
		l = g_strconcat (dirn, G_DIR_SEPARATOR_S, split[0], ".xlog", 0);
		g_strfreev (split);
	}
	else
	{
		l = g_strconcat (dirn, G_DIR_SEPARATOR_S, basen, ".xlog", 0);
	}
	g_free (basen);
	g_free (dirn);
	return l;
}

/* callback for the preview widget, read the first 10 lines */
static void
cb (GtkFileChooser *chooser, gpointer data)
{
	GtkWidget *preview;
	gchar *filename, buf[80];
	gboolean have_preview = FALSE;
	gint i;
	FILE *in;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;

	preview = GTK_WIDGET (data);
	filename = gtk_file_chooser_get_preview_filename (chooser);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(preview));
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	gtk_text_buffer_delete (buffer, &start, &end);

	in = fopen (filename, "r");
	if (in)
	{
		have_preview = TRUE;
		for (i = 0; i < 10; i++)
		{
			if (fgets (buf, 80, in) == NULL) 
				break;
			else if (g_utf8_validate (buf, -1, NULL))
				gtk_text_buffer_insert_with_tags_by_name 
					(buffer, &start, buf, -1, "tiny", NULL);
		}
	}
	gtk_widget_set_size_request (previewscrolledwindow, 200, -1);
	gtk_file_chooser_set_preview_widget_active (chooser, have_preview);
}

void
on_menu_open_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *fileselection, *combobox, *hbox, *label, *preview;
	GdkPixbuf *fileselection_icon_pixbuf;
	GtkTextBuffer *buffer;
	gchar *filen, *logn, *xlogbackupfile, *temp, *selection;
	gchar *sellog;
	gboolean logexist = FALSE;
	gint i, filetype, response;
	logtype *logwindow;
	LOGDB *lp;

	fileselection = gtk_file_chooser_dialog_new (_("xlog - select a log"),
    	GTK_WINDOW(mainwindow), GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, 
		GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

	/* optionmenu for the file types to be opened */
	hbox = gtk_hbox_new (FALSE, 55);
	label = gtk_label_new (_("Type of log:"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	combobox = gtk_combo_box_new_text ();
	gtk_box_pack_start (GTK_BOX (hbox), combobox, FALSE, FALSE, 0);

	/* Keep entries in same order as enum log_file_type in logfile/logfile.h */
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "xlog");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "twlog");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "adif");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "cabrillo");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "editest");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "trlog");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), "EDI");
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox), 0);
	gtk_widget_set_size_request (combobox, 105, -1);
	gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER(fileselection), hbox);
	gtk_widget_show_all (hbox);

	gtk_window_set_position (GTK_WINDOW (fileselection), GTK_WIN_POS_MOUSE);
	fileselection_icon_pixbuf = create_pixbuf ("xlog.png");
	if (fileselection_icon_pixbuf)
		{
			gtk_window_set_icon (GTK_WINDOW (fileselection), fileselection_icon_pixbuf);
			g_object_unref (fileselection_icon_pixbuf);
		}

	selection = g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, NULL);
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (fileselection), selection);
	g_free (selection);

	/* preview widget */
	previewscrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (previewscrolledwindow),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);
	preview = gtk_text_view_new ();
	gtk_widget_show (preview);
	gtk_container_add (GTK_CONTAINER (previewscrolledwindow), preview);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(preview));
	gtk_text_buffer_create_tag (buffer, "tiny", "size", 8 * PANGO_SCALE, NULL);
	gtk_file_chooser_set_preview_widget (GTK_FILE_CHOOSER (fileselection),
		previewscrolledwindow);
	gtk_widget_set_size_request (previewscrolledwindow, 0, -1);
	g_signal_connect (fileselection, "update-preview", G_CALLBACK (cb), preview);

	gtk_widget_set_sensitive (mainwindow, 0);
	response = gtk_dialog_run (GTK_DIALOG(fileselection));
	state.utf8error = FALSE;
	if (response == GTK_RESPONSE_ACCEPT)
	{
		/* unselect any selected QSO */
		for (i = 0; i < g_list_length (logwindowlist); i++)
		{
			logwindow = g_list_nth_data (logwindowlist, i);
			gtk_tree_selection_unselect_all
				(gtk_tree_view_get_selection(GTK_TREE_VIEW (logwindow->treeview)));
		}

		sellog = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (fileselection));
		filetype = gtk_combo_box_get_active (GTK_COMBO_BOX (combobox));

		/* check by filename and if log exist jump to the page in the notebook */
		filen = find_logname (sellog);
		for (i = 0; i < g_list_length (logwindowlist); i++)
		{
			logwindow = g_list_nth_data (logwindowlist, i);
			if (g_ascii_strcasecmp (filen, logwindow->filename) == 0)
			{
				gtk_notebook_set_current_page (GTK_NOTEBOOK(mainnotebook), i);
				set_qsoframe (logwindow);
				logexist = TRUE;
				update_statusbar (_("A log with the same name is already open"));
				break;
			}
		}

		if (!logexist)
		{
			state.qsos = 0;
			lp = log_file_open (sellog, filetype);
			if (lp)
			{
				logn = logname (g_path_get_basename (sellog));
				logwindow =	openlog (lp, logn, state.logwindows++);
				log_file_qso_foreach (lp, fillin_list, logwindow);
				log_file_close (lp);
				g_free (logn);
				lp->type = filetype;
				if ((lp->type) == TYPE_FLOG)
				{
					xlogbackupfile = g_strconcat (sellog, ".backup", NULL);
					unlink (xlogbackupfile);
					backuplog (sellog, xlogbackupfile);
					g_free (xlogbackupfile);
				}
				logwindow->filename = g_strdup (filen);
				logwindowlist =	g_list_append (logwindowlist, logwindow);
				gtk_notebook_set_current_page 
					(GTK_NOTEBOOK(mainnotebook), state.logwindows - 1);
				set_qsoframe (logwindow);
				set_tabs_menu ();
				temp = g_strdup_printf (_("%d QSO's loaded"), state.qsos);
				update_statusbar (temp);
				g_free (temp);
			}
		}
		g_free (filen);
		g_free (sellog);
	}

	gtk_widget_destroy (fileselection);
	gtk_widget_set_sensitive (mainwindow, 1);

	while (gtk_events_pending ()) gtk_main_iteration ();

	if (state.utf8error) 
		warningdialog (_("xlog - open log"), 
			_("There were some errors converting from your locale to UTF8, "
			"which is used by GTK+ internally. It is best if you start xlog "
			"from a terminal and see what the errors are. Please check your "
			"language settings and your log fields!"));
}
