/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001 - 2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * gui_tracedialog.c - creation and destruction of the trace dialog
 *
 * a dialog is created with a text widget. We restart hamlib with tracing
 * enabled and monitor stderr for changes. If there is a change the textwindow
 * is updated.
 */

#if WANT_HAMLIB

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>

#include "gui_tracedialog.h"
#include "gui_warningdialog.h"
#include "support.h"
#include "types.h"

#include <hamlib/rig.h>
#include "hamlib-utils.h"

typedef struct
	{
		time_t modified; /* Last Modified Time */				
		gchar *filename; /* File Name */
		FILE *fd;				/* File Descriptor */
		unsigned long filesize; /* File Size */
	} fileinfo;

gint saved_fd;
gint tracetimer = -1;
fileinfo finfo;

extern GList *riglist;
extern gchar *xlogdir;
extern preferencestype preferences;
extern GtkWidget *mainwindow;

static void
stop_tracing (void)
{
	gint hamlibresult;

	stop_hamlib ();
	g_source_remove (tracetimer);
	fclose (finfo.fd);
	dup2 (saved_fd, STDERR_FILENO);
	close (saved_fd);
#ifndef __NetBSD__
#ifndef WIN32
	stderr = fdopen (STDERR_FILENO, "w");
#endif
#endif
	hamlibresult = start_hamlib (preferences.radio, preferences.device, RIG_DEBUG_NONE, preferences.polltime);
}

static gboolean
on_tracedialog_delete_event (GtkWidget * widget, GdkEvent * event,
					 gpointer user_data)
{
	stop_tracing ();
	return FALSE;
}

static void
on_tracedialogclosebutton_clicked (GtkButton * button, gpointer user_data)
{
	stop_tracing ();
	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* check if tracefile has changed */
static int fileupdated(void) 
{
	struct stat s;
	gint status;

	if (finfo.filename) 
	{
		status = stat (finfo.filename, &s);
			if (finfo.modified != s.st_mtime) 
				return 1;
	}
	return -1;
}

static gint
updatetrace (gpointer data)
{
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	GtkTextMark *mark;
	FILE *fd;
	gint numread = 0;
	gchar buf[1025];
 
	fflush(stderr);
	if (fileupdated ()) 
		{
			if ((fd = fopen(finfo.filename, "r"))) 
				{ 
					buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (data));
					gtk_text_buffer_get_bounds (buffer, &start, &end);
					fseek (fd, finfo.filesize, SEEK_SET);
					while (!feof(fd))	
						{
							numread = fread(buf, 1, 1024, fd);
							gtk_text_buffer_insert (buffer, &end, buf, numread);
							mark = gtk_text_buffer_get_mark (buffer, "insert");
							gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(data), mark, 0.0, FALSE, 0.0, 1.0);
						}
					fclose (fd);
				}
		}
	return 1;
}

void
on_trace_hamlib_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *tracedialog, *tracelabel, *tracedialog_vbox, *tracedialog_mainvbox,
		*tracedialog_hseparator, *tracedialog_scrolledwindow, *tracedialog_textview,
		*tracedialog_action_area, *tracedialogclosebutton;
	GdkPixbuf *tracedialog_icon_pixbuf;
	gint hamlibresult;
	gchar *tracestr;
	
	if (preferences.hamlib == 0)
	{
		warningdialog ("xlog - trace hamlib", 
			_("Please enable hamlib from the Preferences -> Settings dialog"));
	}
	else
	{
	/* redirect stderr to a file and start the file monitor,
		 stop and start hamlib with tracing enabled */
		tracedialog = gtk_dialog_new ();
		gtk_widget_set_size_request (tracedialog, 400, 300);
		gtk_window_set_title (GTK_WINDOW (tracedialog), _("xlog - trace hamlib"));
		gtk_window_set_position (GTK_WINDOW (tracedialog), GTK_WIN_POS_MOUSE);
		gtk_window_set_destroy_with_parent (GTK_WINDOW (tracedialog), TRUE);
		tracedialog_icon_pixbuf = create_pixbuf ("xlog.png");
		if (tracedialog_icon_pixbuf)
		{
			gtk_window_set_icon (GTK_WINDOW (tracedialog), tracedialog_icon_pixbuf);
			g_object_unref (tracedialog_icon_pixbuf);
		}

		tracedialog_vbox = GTK_DIALOG (tracedialog)->vbox;
		gtk_widget_show (tracedialog_vbox);
	
		tracedialog_mainvbox = gtk_vbox_new (FALSE, 0);
		gtk_widget_show (tracedialog_mainvbox);
		gtk_box_pack_start (GTK_BOX (tracedialog_vbox), tracedialog_mainvbox, TRUE, TRUE, 0);

		tracelabel = gtk_label_new ("");
		gtk_widget_show (tracelabel);
		gtk_box_pack_start (GTK_BOX (tracedialog_mainvbox), tracelabel, FALSE, FALSE, 5);
		gtk_label_set_line_wrap (GTK_LABEL (tracelabel), TRUE);

		tracedialog_hseparator = gtk_hseparator_new ();
		gtk_widget_show (tracedialog_hseparator);
		gtk_box_pack_start (GTK_BOX (tracedialog_mainvbox), tracedialog_hseparator, FALSE, FALSE, 0);

		tracedialog_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
		gtk_widget_show (tracedialog_scrolledwindow);
		gtk_box_pack_start (GTK_BOX (tracedialog_mainvbox), tracedialog_scrolledwindow, TRUE, TRUE, 0);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (tracedialog_scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

		tracedialog_textview = gtk_text_view_new ();
		gtk_widget_show (tracedialog_textview);
		gtk_container_add (GTK_CONTAINER (tracedialog_scrolledwindow), tracedialog_textview);

		tracedialog_action_area = GTK_DIALOG (tracedialog)->action_area;
		gtk_widget_show (tracedialog_action_area);
		gtk_button_box_set_layout (GTK_BUTTON_BOX (tracedialog_action_area), GTK_BUTTONBOX_SPREAD);

		tracedialogclosebutton = gtk_button_new_from_stock ("gtk-close");
		gtk_widget_show (tracedialogclosebutton);
		gtk_dialog_add_action_widget (GTK_DIALOG (tracedialog), tracedialogclosebutton, GTK_RESPONSE_CLOSE);
		GTK_WIDGET_SET_FLAGS (tracedialogclosebutton, GTK_CAN_DEFAULT);

		g_signal_connect ((gpointer) tracedialog, "delete_event",
										G_CALLBACK (on_tracedialog_delete_event),
										NULL);
		g_signal_connect ((gpointer) tracedialogclosebutton, "clicked",
										G_CALLBACK (on_tracedialogclosebutton_clicked),
										NULL);

		saved_fd = dup (STDERR_FILENO);
		finfo.filename = g_strdup_printf ( "%s/hamlib.out", xlogdir);
		stop_hamlib();
		finfo.fd = freopen (finfo.filename, "w", stderr);
		hamlibresult = start_hamlib (preferences.radio, preferences.device, 
			RIG_DEBUG_TRACE, preferences.polltime);
		tracetimer = g_timeout_add (1000, (GSourceFunc) updatetrace, tracedialog_textview);

		tracestr = g_strdup_printf (_("Saving to %s"), finfo.filename);
		gtk_label_set_text (GTK_LABEL (tracelabel), tracestr);
		g_free(tracestr);
		gtk_widget_show (tracedialog);
	}
}
#endif
