/*
 Copyright (C) 2000-2004

 Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

 This file is part of xmds.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmdsintegraterk9ex.cc,v 1.14 2006/05/25 03:19:15 grahamdennis Exp $
*/

/*! @file xmdsintegraterk9ex.cc
  @brief Integrate element parsing classes and methods; fourth order Runge-Kutta in the explicit picture

  More detailed explanation...
*/

#include<xmlbasics.h>
#include<dom3.h>
#include<xmdsutils.h>
#include<xmdsclasses.h>

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateRK9EX public
// **************************************************************************
// **************************************************************************

extern bool debugFlag;

long nxmdsIntegrateRK9EXs=0;  //!< The number of integrate RK9EX objects

// **************************************************************************
xmdsIntegrateRK9EX::xmdsIntegrateRK9EX(
               const xmdsSimulation *const yourSimulation,
               const bool& yourVerboseMode) :
  xmdsIntegrate(yourSimulation,yourVerboseMode,false,false),
  xmdsIntegrateEX(yourSimulation,yourVerboseMode),
  xmdsIntegrateRK9(yourSimulation,yourVerboseMode) {
  if(debugFlag) {
    nxmdsIntegrateRK9EXs++;
    printf("xmdsIntegrateRK9EX::xmdsIntegrateRK9EX\n");
    printf("nxmdsIntegrateRK9EXs=%li\n",nxmdsIntegrateRK9EXs);
  }
};

// **************************************************************************
xmdsIntegrateRK9EX::~xmdsIntegrateRK9EX() {
  if(debugFlag) {
    nxmdsIntegrateRK9EXs--;
    printf("xmdsIntegrateRK9EX::~xmdsIntegrateRK9EX\n");
    printf("nxmdsIntegrateRK9EXs=%li\n",nxmdsIntegrateRK9EXs);
  }
};

// **************************************************************************
void xmdsIntegrateRK9EX::processElement(
          const Element *const yourElement) {
  if(debugFlag) {
    printf("xmdsIntegrateRK9EX::processElement\n");
  }

  if(verbose()) {
    printf("Processing integrate RK9EX element ...\n");
  }

  xmdsIntegrate::processElement(yourElement);
  xmdsIntegrateRK9::processElement(yourElement);
  xmdsIntegrateEX::processElement(yourElement);
};

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateRK9EX private
// **************************************************************************
// **************************************************************************

// **************************************************************************
void xmdsIntegrateRK9EX::writePrototypes(
           FILE *const outfile) const {
  if(debugFlag) {
    printf("xmdsIntegrateRK9EX::writePrototypes\n");
  }

  fprintf(outfile,"// ********************************************************\n");
  fprintf(outfile,"// segment %li (RK9EX) prototypes\n",segmentNumber);
  fprintf(outfile,"\n");

  xmdsIntegrate::writePrototypes(outfile);
  xmdsIntegrateEX::writePrototypes(outfile);
  xmdsIntegrateRK9::writePrototypes(outfile);
};

// **************************************************************************
void xmdsIntegrateRK9EX::writeRoutines(
               FILE *const outfile) const {
  if(debugFlag) {
    printf("xmdsIntegrateRK9EX::writeRoutines\n");
  }

  fprintf(outfile,"// ********************************************************\n");
  fprintf(outfile,"// segment %li (RK9EX) routines\n",segmentNumber);
  fprintf(outfile,"\n");

  xmdsIntegrate::writeRoutines(outfile);
  xmdsIntegrateEX::writeRoutines(outfile);
  xmdsIntegrateRK9::writeRoutines(outfile);
};

// **************************************************************************
void xmdsIntegrateRK9EX::writeSingleStepCode(
               FILE *const outfile,
               const stepCaseEnum& stepCase) const {
  if(debugFlag) {
    printf("xmdsIntegrateRK9EX::writeSingleStepCode\n");
  }

  const char *const fieldName = simulation()->field()->name()->c_str();
  const char *const propDim = simulation()->parameters()->propDimName.c_str();

  list<XMLString> tempVectorNamesList;
  tempVectorNamesList.push_back("main");

  const char* indent = "  ";
  if(simulation()->parameters()->errorCheck) {
    indent = "    ";
  }

  const char* noiseVector="";
  if((simulation()->parameters()->stochastic)&&(!noNoises())) {
    noiseVector=",_noise_vector";
  }

  simulation()->field()->vectors2space(outfile,0,tempVectorNamesList,indent);

  fprintf(outfile,"%s// Step 1 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[0]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akafield_%s_main[_i1] = _%s_main[_i1];\n",indent,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akafield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");
  
  fprintf(outfile,"%s// Step 2 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[1]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akbfield_%s_main[_i1] = _%s_main[_i1] + _b[1][0]*_akafield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akbfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");
  
    fprintf(outfile,"%s// Step 3 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[2]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akcfield_%s_main[_i1] = _%s_main[_i1] + _b[2][0]*_akafield_%s_main[_i1] + _b[2][1]*_akbfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akcfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");
  
    fprintf(outfile,"%s// Step 4 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[3]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akdfield_%s_main[_i1] = _%s_main[_i1] + _b[3][0]*_akafield_%s_main[_i1] + _b[3][1]*_akbfield_%s_main[_i1] + _b[3][2]*_akcfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akdfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");
  
    fprintf(outfile,"%s// Step 5 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[4]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akefield_%s_main[_i1] = _%s_main[_i1] + _b[4][0]*_akafield_%s_main[_i1] + _b[4][1]*_akbfield_%s_main[_i1] + _b[4][2]*_akcfield_%s_main[_i1] + _b[4][3]*_akdfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akefield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");
    
    fprintf(outfile,"%s// Step 6 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[5]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akifield_%s_main[_i1] = _%s_main[_i1] + _b[5][0]*_akafield_%s_main[_i1] + _b[5][3]*_akdfield_%s_main[_i1] + _b[5][4]*_akefield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akifield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");
    
     fprintf(outfile,"%s// Step 7 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[6]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akjfield_%s_main[_i1] = _%s_main[_i1] + _b[6][0]*_akafield_%s_main[_i1] + _b[6][3]*_akdfield_%s_main[_i1] + _b[6][4]*_akefield_%s_main[_i1] + _b[6][5]*_akifield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akjfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 8 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[7]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akbfield_%s_main[_i1] = _%s_main[_i1] + _b[7][0]*_akafield_%s_main[_i1] + _b[7][5]*_akifield_%s_main[_i1] + _b[7][6]*_akjfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akbfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 9 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[8]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akcfield_%s_main[_i1] = _%s_main[_i1] + _b[8][0]*_akafield_%s_main[_i1] + _b[8][5]*_akifield_%s_main[_i1] + _b[8][6]*_akjfield_%s_main[_i1]+ _b[8][7]*_akbfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akcfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 10 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[9]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akdfield_%s_main[_i1] = _%s_main[_i1] + _b[9][0]*_akafield_%s_main[_i1] + _b[9][5]*_akifield_%s_main[_i1] + _b[9][6]*_akjfield_%s_main[_i1]+ _b[9][7]*_akbfield_%s_main[_i1]+ _b[9][8]*_akcfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akdfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 11 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[10]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akefield_%s_main[_i1] = _%s_main[_i1] + _b[10][0]*_akafield_%s_main[_i1] + _b[10][5]*_akifield_%s_main[_i1] + _b[10][6]*_akjfield_%s_main[_i1]+ _b[10][7]*_akbfield_%s_main[_i1] + _b[10][8]*_akcfield_%s_main[_i1] + _b[10][9]*_akdfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akefield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 12 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[11]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akffield_%s_main[_i1] = _%s_main[_i1] + _b[11][0]*_akafield_%s_main[_i1] + _b[11][5]*_akifield_%s_main[_i1] + _b[11][6]*_akjfield_%s_main[_i1]+ _b[11][7]*_akbfield_%s_main[_i1] + _b[11][8]*_akcfield_%s_main[_i1] + _b[11][9]*_akdfield_%s_main[_i1] + _b[11][10]*_akefield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akffield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 13 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[12]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akgfield_%s_main[_i1] = _%s_main[_i1] + _b[12][0]*_akafield_%s_main[_i1] + _b[12][5]*_akifield_%s_main[_i1] + _b[12][6]*_akjfield_%s_main[_i1]+ _b[12][7]*_akbfield_%s_main[_i1] + _b[12][8]*_akcfield_%s_main[_i1] + _b[12][9]*_akdfield_%s_main[_i1] + _b[12][10]*_akefield_%s_main[_i1] + _b[12][11]*_akffield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akgfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 14 \n",indent);
  
  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[13]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");
  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akhfield_%s_main[_i1] = _%s_main[_i1] + _b[13][0]*_akafield_%s_main[_i1] + _b[13][5]*_akifield_%s_main[_i1] + _b[13][6]*_akjfield_%s_main[_i1]+ _b[13][7]*_akbfield_%s_main[_i1] + _b[13][8]*_akcfield_%s_main[_i1] + _b[13][9]*_akdfield_%s_main[_i1] + _b[13][10]*_akefield_%s_main[_i1] + _b[13][11]*_akffield_%s_main[_i1] + _b[13][12]*_akgfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akhfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");

   fprintf(outfile,"%s// Step 15 and 16 combined to reduce memory use \n",indent);
  

  
  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akifield_%s_main[_i1] = _%s_main[_i1] + _b[14][0]*_akafield_%s_main[_i1] + _b[14][5]*_akifield_%s_main[_i1] + _b[14][6]*_akjfield_%s_main[_i1]+ _b[14][7]*_akbfield_%s_main[_i1] + _b[14][8]*_akcfield_%s_main[_i1] + _b[14][9]*_akdfield_%s_main[_i1] + _b[14][10]*_akefield_%s_main[_i1] + _b[14][11]*_akffield_%s_main[_i1] + _b[14][12]*_akgfield_%s_main[_i1] + _b[14][13]*_akhfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  
      if (simulation()->parameters()->useOpenMP) {
   // Continuing the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp for\n"
             "#endif\n");
  }
  
    fprintf(outfile,"\n");
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _akjfield_%s_main[_i1] = (1-_b[15][5]/_b[14][5])*_%s_main[_i1] + (_b[15][0]-_b[14][0]*_b[15][5]/_b[14][5])*_akafield_%s_main[_i1] + (_b[15][5]/_b[14][5])*_akifield_%s_main[_i1] + (_b[15][6]-_b[14][6]*_b[15][5]/_b[14][5])*_akjfield_%s_main[_i1] + (_b[15][7]-_b[14][7]*_b[15][5]/_b[14][5])*_akbfield_%s_main[_i1] + (_b[15][8]-_b[14][8]*_b[15][5]/_b[14][5])*_akcfield_%s_main[_i1] + (_b[15][9]-_b[14][9]*_b[15][5]/_b[14][5])*_akdfield_%s_main[_i1] + (_b[15][10]-_b[14][10]*_b[15][5]/_b[14][5])*_akefield_%s_main[_i1] + (_b[15][11]-_b[14][11]*_b[15][5]/_b[14][5])*_akffield_%s_main[_i1] + (_b[15][12]-_b[14][12]*_b[15][5]/_b[14][5])*_akgfield_%s_main[_i1] + (_b[15][13]-_b[14][13]*_b[15][5]/_b[14][5])*_akhfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);

  
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");



  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[14]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akifield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");


  fprintf(outfile,"\n");
  fprintf(outfile,"%s%s += _a[15]*_step;\n",indent,propDim);
  fprintf(outfile,"\n");

  fprintf(outfile,"%s_active_%s_main = _akjfield_%s_main;\n",indent,fieldName,fieldName);
  fprintf(outfile,"\n");

  if(usesKOperators()) {
    fprintf(outfile,"%s// calculate the co derivative terms\n",indent);
    fprintf(outfile,"%s_segment%li_calculate_co_terms();\n",indent,segmentNumber);
    fprintf(outfile,"\n");
  }

  fprintf(outfile,"%s// a_k=G[a_k,t]\n",indent);
  fprintf(outfile,"%s_segment%li_calculate_delta_a(_step,cycle%s);\n",indent,segmentNumber,noiseVector);
  fprintf(outfile,"\n");
  
    fprintf(outfile,"%s// Take full step\n",indent);
fprintf(outfile,"%s\n",indent);

  if (simulation()->parameters()->useOpenMP) {
   // We begin a parallel region as we have two for loops following each other
   // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
             "#pragma omp parallel\n"
             "{\n"
             "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
             "#endif\n");
 }
  
  if(simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile,"%sfor(long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName);
  }
  else {
    fprintf(outfile,"%sfor(long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n",indent,fieldName,fieldName);
  }
  fprintf(outfile,"%s _%s_main[_i1] += _c[0]*_akafield_%s_main[_i1]+_c[7]*_akbfield_%s_main[_i1]+_c[8]*_akcfield_%s_main[_i1]+_c[9]*_akdfield_%s_main[_i1]+_c[10]*_akefield_%s_main[_i1]+_c[11]*_akffield_%s_main[_i1]+_c[12]*_akgfield_%s_main[_i1]+_c[13]*_akhfield_%s_main[_i1]+_c[14]*_akifield_%s_main[_i1]+_c[15]*_akjfield_%s_main[_i1];\n",indent,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName,fieldName);
  if (simulation()->parameters()->useOpenMP) {
   // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
             "}\n"
             "#endif\n");
  }
  fprintf(outfile,"\n");
  fprintf(outfile,"%s_active_%s_main = _%s_main;\n",indent,fieldName,fieldName  );
  fprintf(outfile,"\n");
  
};

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
