/*
  Copyright (C) 2000-2006

  Code contributed by Greg Collecutt, Joseph Hope, Andrew Reid and Paul Cochrane

  This file is part of xmds.
 
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmlbasics.h 1194 2006-12-18 12:59:46Z paultcochrane $
*/

/*! @file xmlbasics.h
  @brief The most basic XML character and string processing routines.
  Note that only UTF8 encoding is supported here.
*/

/* The most basic XML character and string processing routines.
   Note that only UTF8 encoding is supported here. */

#ifndef STDIO
#include<stdio.h>
#define STDIO
#endif

// ******************************************************************************
// ******************************************************************************
//	XMLException
// ******************************************************************************
// ******************************************************************************

//! XMLException class
class XMLException {

public :

  //! Enumerator containing the exception error codes
  enum {
    UNKNOWN_ENCODING_ERR = 1,
    UNEXPECTED_EOF_ERR	 = 2,
    RANGE_ERR		 = 3,
    INVALID_CHAR_ERR	 = 4,
    UNKNOWN_ERR		 = 0 
  };

  //! The error code?  Possibly?
  unsigned short code;

  //! Constructor of an XMLException object
  XMLException();

  //! Constructor of an XMLException object
  XMLException(
	       const unsigned short& error);
		
  //! Destructor
  ~XMLException();

  //! Gets an xml error
  const char* getError() const;
};

// ******************************************************************************
// ******************************************************************************
//	XMLChar
// ******************************************************************************
// ******************************************************************************

//! Namespace for an xml character
namespace XMLChar {

  //! Determines if character is a character
  bool isChar(
	      const char& ch);

  //! Determines if character is character data
  bool isCharData(
		  const char& ch);

  //! Determines if character is whitespace
  bool isWhiteSpace(
		    const char& ch);

  //! Determines if character is a latin letter
  bool isLatinLetter(
		     const char& ch);

  //! Determines if character is a latin digit
  bool isLatinDigit(
		    const char& ch);

  //! Determines if character is a latin hex digit
  bool isLatinHexDigit(
		       const char& ch);

  //! Determines if character is a letter
  bool isLetter(
		const char& ch);

  //! Determines if character is a base character
  bool isBaseChar(
		  const char& ch);

  //! Determines if character is a digit
  bool isDigit(
	       const char& ch);

  //! Determines if character is a name character
  bool isNameChar(
		  const char& ch);

  //! Determines if character is a NC name character
  bool isNCNameChar(
		    const char& ch);

  //! Determines if character is a public id character
  bool isPubidChar(
		   const char& ch);
};

// ******************************************************************************
// ******************************************************************************
//	XMLString
// ******************************************************************************
// ******************************************************************************

//! XMLString class
class XMLString {

private :

  char* _data;               //!< Internal variable for data
  unsigned long _length;     //!< Internal variable for length of an XML string
  mutable char* _c_str;      //!< Internal variable for a C string
  mutable bool _c_str_valid; //!< Internal variable showing if a C string is valid

public:

  //! Constructor of an XMLString object
  XMLString();

  //! Constructor of an XMLString object
  XMLString(
	    const char* s);

  //! Constructor of an XMLString object
  XMLString(
	    const XMLString& s);

  //! Destructor
  ~XMLString();

  //! Assignment operator for XMLString object
  XMLString& operator=(
		       const XMLString& s);

  //! Assignment operator for XMLString object
  XMLString& operator=(
		       const char* s);

  //! Returns the length of an XMLString
  unsigned long length() const;

  //! Converts an XMLString to a C string
  const char* c_str() const;	

  //! Returns data at index (buh?)
  char data(
	    const unsigned long& index) const;	

  //! Equality operator
  bool operator==(
		  const XMLString& s) const;

  //! Inequality operator
  bool operator!=(
		  const XMLString& s) const;

  //! += operator
  XMLString& operator+=(
			const XMLString& s);

  //! Loads an xml string from file infile
  long loadFromFile(
		    FILE* infile);

  //! Returns a substring of an XML string starting at begin and ending at end_plus_one
  void subString(
		 XMLString& subS,
		 const unsigned long& begin,
		 const unsigned long& end_plus_one) const; 

  //! Inserts string at offset
  void insertString(
		    const unsigned long& offset,
		    const XMLString& s); 

  //! Deletes count data (characters?) starting at offset
  void deleteData(
		  const unsigned long& offset,
		  const unsigned long& count); 

  //! Replaces count data (characters?) starting at offset
  void replaceData(
		   const unsigned long& offset,
		   unsigned long count,
		   const XMLString& s); 

  //! Determines if string has illegal characters
  bool hasIllegalCharacters() const;

  //! Determines if starts with xml
  bool beginsWithxml() const;

  //! Determins if starts with XxMmLl (what is this supposed to mean?)
  bool beginsWithXxMmLl() const;

  //! Determines if is xml (I think)
  bool eqxml() const;

  //! Determines if is xml namespace (I think)
  bool eqxmlns() const;

  //! Determines if is a name
  bool isName() const;

  //! Determines if is a NC name
  bool isNCName() const;

  //! Determines if namespace is well formed
  bool isNSWellFormed() const;

  //! Determines if is a version number
  bool isVersionNum() const;

  //! Determines if is an encoding name
  bool isEncName() const;

  //! Determines if is a public id literal
  bool isPubidLiteral() const;

  //! Determines if is a split namespace name
  bool splitNSName(
		   XMLString& prefix,
		   XMLString& localPart) const;

  //! Determines if is a qualified name
  bool isQualifiedName();

  //! Determines if is all whitespace
  bool isAllWhiteSpace() const;

  //! Determines if is as unsigned long
  bool asULong(
	       unsigned long& outULong) const;

  //! Determines if is as double
  bool asDouble(
		double& outDouble) const;

  //! Changes to Latin alphanumeric (I think)
  void goLatinAlphaNumeric();
  
  //! Changes to Latin alphanumeric with '_' replaced with '0' (for Mathematica)
  void goLatinAlphaNumericNoUnderScore();
};

#define XMLBASICS
