/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: xzoom.c                                                       *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : image zoom routines                                      *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : XMdcImagesZoomIn()              - Zoom image IN          *
 *                XMdcImagesZoomOut()             - Zoom image OUT         *
 *                XMdcImagesZoomCallbackClicked() - Zoom Clicked  callback *
 *                XMdcImagesZoom()                - Display zoomed image   *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: xzoom.c,v 1.10 2004/01/04 22:36:17 enlf Exp $
 */

/*
   Copyright (C) 1997-2004 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "m-depend.h"

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include "xmedcon.h"

/****************************************************************************
                              D E F I N E S
****************************************************************************/

typedef struct ZoomStruct_t {

  float factor;
  Int8 type;
  Uint32 nr;
  GdkPixbuf *im, *cur;
  GtkWidget *darea;

}ZoomStruct;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

#ifdef _WIN32
static void Win32MinDim128(ZoomStruct *zoom, int *arg_w, int *arg_h)
{
  int w=*arg_w, h=*arg_h;
  float factor = zoom->factor;

  while (w < 128 || h < 128) {
    factor *= (float)XMDC_ZOOM_FACTOR;
    w = (int)((float)w * (float)XMDC_ZOOM_FACTOR);
    h = (int)((float)h * (float)XMDC_ZOOM_FACTOR);
  }
  zoom->factor = factor;
  *arg_w = w;
  *arg_h = h;
}
#endif

/* local routine */
static float XMdcGetZoomFactor(Int8 type)
{
  float zoom;

  if (type < 0) zoom = 1./(float)(-type);
  else zoom = (float)type;

  return(zoom); 
}

/* local routine */
static char *XMdcGetStrZoomFactor(float factor)
{
  if (factor <= 0.) factor=1;

  if (factor < 1.) {
    sprintf(xmdcstr,"[1:%d]",(int)(1./factor));
  }else{
    sprintf(xmdcstr,"[%d:1]",(int)factor);
  }

  return(xmdcstr);
}

/* local routine */
static void XMdcSetZoomWindowTitle(GtkWidget *window, ZoomStruct *zoom)
{
  Uint32 nr = zoom->nr + 1;
  float factor = zoom->factor;

  sprintf(mdcbufr,"%u %s",nr,XMdcGetStrZoomFactor(factor));
  gtk_window_set_title(GTK_WINDOW(window),mdcbufr);
}

/* local routine */
static void XMdcRemoveZoomStruct(GtkWidget *window, gpointer data)
{ 
  ZoomStruct *zoom = (ZoomStruct *)data;

  if (zoom != NULL) {
    if (zoom->im  != NULL) gdk_pixbuf_unref(zoom->im); 
    if (zoom->cur != NULL) gdk_pixbuf_unref(zoom->cur);
    MdcFree(zoom); 
  }

}

/* local routine */
static gboolean XMdcImagesZoomCallbackExpose(GtkWidget *widget, 
                         GdkEventExpose *event, gpointer window)
{
  GdkGC *gc = widget->style->fg_gc[GTK_STATE_NORMAL];
  ZoomStruct *zoom;

  zoom = (ZoomStruct *)gtk_object_get_data(GTK_OBJECT(window),"zoomstruct");

  if (event->area.width  <= gdk_pixbuf_get_width(zoom->cur) &&
      event->area.height <= gdk_pixbuf_get_height(zoom->cur)) {

    gdk_pixbuf_render_to_drawable(zoom->cur, widget->window, gc,
                                event->area.x, event->area.y,
                                event->area.x, event->area.y,
                                event->area.width, event->area.height,
                                sRenderSelection.Dither, 0, 0);
  }

  return(TRUE);

}




void XMdcImagesZoomIn(GtkWidget *window)
{
  GdkPixbuf *new;
  ZoomStruct *zoom;
  int w, h;

  zoom = (ZoomStruct *)gtk_object_get_data(GTK_OBJECT(window),"zoomstruct");

  zoom->factor *= (float) XMDC_ZOOM_FACTOR;

  w = (int)((float)gdk_pixbuf_get_width(zoom->im)  * zoom->factor); 
  h = (int)((float)gdk_pixbuf_get_height(zoom->im) * zoom->factor);

#ifdef _WIN32
  Win32MinDim128(zoom, &w, &h);
#endif

  if ((w < (gdk_screen_width() - XMDC_FREE_BORDER)) &&
      (h < (gdk_screen_height()- XMDC_FREE_BORDER))) {

    gtk_window_set_policy(GTK_WINDOW(window),TRUE,TRUE,TRUE);

    new = gdk_pixbuf_scale_simple(zoom->im,w,h,sRenderSelection.Interp); 
    gdk_pixbuf_unref(zoom->cur); zoom->cur = new;

    gtk_drawing_area_size(GTK_DRAWING_AREA(zoom->darea),w,h);

    gdk_pixbuf_render_to_drawable(zoom->cur, zoom->darea->window,
                                  zoom->darea->style->fg_gc[GTK_STATE_NORMAL],
                                  0,0,0,0,w,h,sRenderSelection.Dither,0,0);

    XMdcSetZoomWindowTitle(window,zoom);

    gtk_window_set_policy(GTK_WINDOW(window),FALSE,FALSE,FALSE);

  }else{

    zoom->factor /= (float)XMDC_ZOOM_FACTOR;
 
  }
}

void XMdcImagesZoomOut(GtkWidget *window)
{
  GdkPixbuf *new;
  ZoomStruct *zoom;
  int w, h;

  zoom = (ZoomStruct *)gtk_object_get_data(GTK_OBJECT(window),"zoomstruct");

  if (zoom->factor > XMdcGetZoomFactor(zoom->type)) {

    zoom->factor /= (float)XMDC_ZOOM_FACTOR;

    w = (int)((float)gdk_pixbuf_get_width(zoom->im)  * zoom->factor);
    h = (int)((float)gdk_pixbuf_get_height(zoom->im) * zoom->factor);

#ifdef _WIN32
    Win32MinDim128(zoom, &w, &h);
#endif

    gtk_window_set_policy(GTK_WINDOW(window),TRUE,TRUE,TRUE);

    new = gdk_pixbuf_scale_simple(zoom->im,w,h,sRenderSelection.Interp);
    gdk_pixbuf_unref(zoom->cur); zoom->cur = new;

    gtk_drawing_area_size(GTK_DRAWING_AREA(zoom->darea),w,h);

    gdk_pixbuf_render_to_drawable(zoom->cur, zoom->darea->window,
                                  zoom->darea->style->fg_gc[GTK_STATE_NORMAL],
                                  0,0,0,0,w,h,sRenderSelection.Dither,0,0);
 
    XMdcSetZoomWindowTitle(window,zoom);

    gtk_window_set_policy(GTK_WINDOW(window),FALSE,FALSE,FALSE);

  }
}

gboolean XMdcImagesZoomCallbackClicked(GtkWidget *widget, GdkEventButton *button, GtkWidget *window)
{

  if (button->button == 3)  gtk_widget_destroy(window);

  if (button->button == 1)  XMdcImagesZoomIn(window);

  if (button->button == 2)  XMdcImagesZoomOut(window);

  return(TRUE);

}

void XMdcImagesZoom(GtkWidget *widget, Uint32 nr)
{
  GtkWidget *window;
  GtkWidget *tblbox;
  GtkWidget *darea;
  ZoomStruct *zoom=NULL;
  int w, h, fctr;


  /* allocate structure */
  zoom = (ZoomStruct *)malloc(sizeof(ZoomStruct));
  if (zoom == NULL) {
    XMdcDisplayErr("Couldn't allocate zoom struct"); 
    return;
  }else{
    zoom->nr = my.realnumber[nr];
    zoom->type = sResizeSelection.CurType;
    zoom->factor = XMdcGetZoomFactor(zoom->type);
    zoom->im = NULL; zoom->cur = NULL;
  }

  if (zoom->factor < 1.) {
    fctr = (int)1./zoom->factor;
    w = ((int)my.fi->image[zoom->nr].width  + (fctr-1)) / fctr;
    h = ((int)my.fi->image[zoom->nr].height + (fctr-1)) / fctr;
  }else{
    fctr = (int)zoom->factor;
    w = (int)my.fi->image[zoom->nr].width  * fctr;
    h = (int)my.fi->image[zoom->nr].height * fctr;
  }

#ifdef _WIN32
  Win32MinDim128(zoom, &w, &h);
#endif

  if ( w < (gdk_screen_width() - XMDC_FREE_BORDER) &&
       h < (gdk_screen_height()- XMDC_FREE_BORDER) ) {

    /* get first pixbuf from original size (resizing = degrading image) */
    zoom->im  = XMdcBuildGdkPixbufFI(my.fi,zoom->nr,sGbc.mod.vgbc,MDC_NO);
    zoom->cur = gdk_pixbuf_scale_simple(zoom->im,w,h,sRenderSelection.Interp);

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_policy(GTK_WINDOW(window),TRUE,TRUE,TRUE);
    gtk_container_set_border_width(GTK_CONTAINER(window),1);
    gtk_signal_connect(GTK_OBJECT(window),"destroy",
                       GTK_SIGNAL_FUNC(XMdcRemoveZoomStruct),zoom);
    gtk_signal_connect(GTK_OBJECT(window),"destroy",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);


    tblbox = gtk_table_new(1,1,TRUE);
    gtk_container_add(GTK_CONTAINER(window),tblbox);
    gtk_widget_show(tblbox);

    darea = gtk_drawing_area_new(); zoom->darea = darea;
    gtk_table_attach(GTK_TABLE(tblbox),darea,0,1,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_widget_show(darea);
    gtk_widget_set_events(darea, GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK);
    gtk_drawing_area_size(GTK_DRAWING_AREA(darea),w,h);
    
    gtk_signal_connect(GTK_OBJECT(darea),"button_press_event",
                       GTK_SIGNAL_FUNC(XMdcImagesZoomCallbackClicked),
                       GTK_WIDGET(window));
    gtk_signal_connect(GTK_OBJECT(darea),"expose_event",
                       GTK_SIGNAL_FUNC(XMdcImagesZoomCallbackExpose),
                       GTK_WIDGET(window));

    XMdcShowWidget(window);

    gdk_window_set_cursor (window->window, fleurcursor);

    gtk_object_set_data(GTK_OBJECT(window),"zoomstruct",zoom);

    XMdcSetZoomWindowTitle(window,zoom);

    gtk_window_set_policy(GTK_WINDOW(window),FALSE,FALSE,FALSE);

  }
}

