/*
   XMMS-SID - SIDPlay input plugin for X MultiMedia System (XMMS)

   libSIDPlay v1 support

   Written by Matti "ccr" Hamalainen <ccr@tnsp.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include "xmms-sid.h"

#ifdef HAVE_SIDPLAY1

extern "C" {
#include "xs_sidplay1.h"
#include <stdio.h>
#include <xmms/titlestring.h>
#include "xs_config.h"
#include "xs_support.h"
#include "xs_length.h"
}

#include <sidplay/player.h>
#include <sidplay/myendian.h>
#include <sidplay/fformat.h>


typedef struct {
	emuEngine	*currEng;
	emuConfig	currConfig;
	sidTune		*currTune;
} t_xs_sidplay1;


/*
 * We need to 'export' all this pseudo-C++ crap
 */
extern "C" {


/*
 * Check if we can play the given file
 */
gboolean xs_sidplay1_isourfile(gchar *pcFilename)
{
 sidTune *testTune = new sidTune(pcFilename);

 if (!testTune) return FALSE;
 if (!testTune->getStatus())
	{
	delete testTune;
	return FALSE;
	}

 delete testTune;
 return TRUE;
}


/*
 * Initialize SIDPlay1
 */
gboolean xs_sidplay1_init(t_xs_status *myStatus)
{
 t_xs_sidplay1 *myPlayer;
 assert(myStatus);

 /* Allocate internal structures */
 myPlayer = (t_xs_sidplay1 *) g_malloc0(sizeof(t_xs_sidplay1));
 if (!myPlayer) return FALSE;

 /* Initialize engine */
 myPlayer->currEng = new emuEngine();
 if (!myPlayer->currEng)
	{
	XSERR("Could not initialize libSIDPlay1 emulation engine\n");
	g_free(myPlayer);
	return FALSE;
	}

 /* Verify endianess */
 if (!myPlayer->currEng->verifyEndianess())
	{
	XSERR("Endianess verification failed\n");
	delete myPlayer->currEng;
	g_free(myPlayer);
	return FALSE;
	}

 myStatus->player = myPlayer;
 return TRUE;
}


/*
 * Close SIDPlay1
 */
void xs_sidplay1_close(t_xs_status *myStatus)
{
 t_xs_sidplay1 *myPlayer;
 assert(myStatus);

 /* Free internals */
 myPlayer = (t_xs_sidplay1 *) myStatus->player;
 if (myPlayer->currEng)
	{
	delete myPlayer->currEng;
	myPlayer->currEng = NULL;	
	}

 g_free(myPlayer);
 myStatus->player = NULL;
}


gboolean xs_sidplay1_initsong(t_xs_status *myStatus)
{
 t_xs_sidplay1 *myPlayer = (t_xs_sidplay1 *) myStatus->player;

 if (!myPlayer) return FALSE;

 if (!myPlayer->currTune)
	{
	XSERR("Tune was NULL\n");
	return FALSE;
	}

 if (!myPlayer->currTune->getStatus())
	{
	XSERR("Tune status check failed\n");
	return FALSE;
	}

 return sidEmuInitializeSong(
	*myPlayer->currEng,
	*myPlayer->currTune,
	myStatus->currSong);
}


guint xs_sidplay1_fillbuffer(t_xs_status *myStatus, gchar *audioBuffer, guint audioBufSize)
{
 t_xs_sidplay1 *myPlayer = (t_xs_sidplay1 *) myStatus->player;

 if (!myPlayer) return FALSE;

 sidEmuFillBuffer(
	*myPlayer->currEng,
	*myPlayer->currTune,
	audioBuffer,
	audioBufSize);

 return audioBufSize;
}


gboolean xs_sidplay1_loadsid(t_xs_status *myStatus, gchar *pcFilename)
{
 t_xs_sidplay1 *myPlayer = (t_xs_sidplay1 *) myStatus->player;
 sidTune *newTune;
 assert(myStatus);

 /* Try to get the tune */
 if (!pcFilename) return FALSE;
 newTune = new sidTune(pcFilename);
 if (!newTune) return FALSE;

 /* Get current configuration */
 myPlayer->currEng->getConfig(myPlayer->currConfig);

 /* Configure channels and stuff */
 switch (myStatus->audioChannels) {

	case XS_CHN_AUTOPAN:
		myPlayer->currConfig.channels = SIDEMU_STEREO;
		myPlayer->currConfig.autoPanning = SIDEMU_CENTEREDAUTOPANNING;
		myPlayer->currConfig.volumeControl = SIDEMU_FULLPANNING;
		break;

	case XS_CHN_STEREO:
		myPlayer->currConfig.channels = SIDEMU_STEREO;
		myPlayer->currConfig.autoPanning = SIDEMU_NONE;
		myPlayer->currConfig.volumeControl = SIDEMU_NONE;
		break;

	case XS_CHN_MONO:
	default:
		myPlayer->currConfig.channels = SIDEMU_MONO;
		myPlayer->currConfig.autoPanning = SIDEMU_NONE;
		myPlayer->currConfig.volumeControl = SIDEMU_NONE;
		break;
 }


 /* Memory mode settings */
 switch (xs_cfg.memoryMode) {
	case XS_MPU_BANK_SWITCHING:
		myPlayer->currConfig.memoryMode = MPU_BANK_SWITCHING;
		break;

	case XS_MPU_TRANSPARENT_ROM:
		myPlayer->currConfig.memoryMode = MPU_TRANSPARENT_ROM;
		break;

	case XS_MPU_PLAYSID_ENVIRONMENT:
		myPlayer->currConfig.memoryMode = MPU_PLAYSID_ENVIRONMENT;
		break;

	default:
		myPlayer->currConfig.memoryMode = MPU_BANK_SWITCHING;
		break;
 }


 /* Clockspeed settings */
 switch (xs_cfg.clockSpeed) {
	case XS_CLOCK_NTSC:
		myPlayer->currConfig.clockSpeed = SIDTUNE_CLOCK_NTSC;
		break;

	case XS_CLOCK_PAL:
	default:
		myPlayer->currConfig.clockSpeed = SIDTUNE_CLOCK_PAL;
		break;
 }


 /* Configure rest of the emulation */
 myPlayer->currConfig.mos8580		= xs_cfg.mos8580;
 myPlayer->currConfig.emulateFilter	= xs_cfg.emulateFilters;
 myPlayer->currConfig.filterFs		= xs_cfg.filterFs;
 myPlayer->currConfig.filterFm		= xs_cfg.filterFm;
 myPlayer->currConfig.filterFt		= xs_cfg.filterFt;

 myPlayer->currConfig.bitsPerSample	= myStatus->audioBitsPerSample;
 myPlayer->currConfig.frequency		= myStatus->audioFrequency;

 switch (myStatus->audioBitsPerSample) {
 case XS_RES_8BIT:
	switch (myStatus->audioFormat) {
	case FMT_S8:
		myPlayer->currConfig.sampleFormat	= SIDEMU_SIGNED_PCM;
		break;

	case FMT_U8:
	default:
		myStatus->audioFormat			= FMT_U8;
		myPlayer->currConfig.sampleFormat	= SIDEMU_UNSIGNED_PCM;
		break;
	}
	break;

 case XS_RES_16BIT:
	switch (myStatus->audioFormat) {
	case FMT_U16_NE:
	case FMT_U16_LE:
	case FMT_U16_BE:
		myStatus->audioFormat			= FMT_U16_NE;
		myPlayer->currConfig.sampleFormat       = SIDEMU_UNSIGNED_PCM;
		break;

	case FMT_S16_NE:
	case FMT_S16_LE:
	case FMT_S16_BE:
	default:
		myStatus->audioFormat			= FMT_S16_NE;
		myPlayer->currConfig.sampleFormat       = SIDEMU_SIGNED_PCM;
		break;
	}
	break;
 }

 /* Now set the emulator configuration */
 if (!myPlayer->currEng->setConfig(myPlayer->currConfig))
	{
	XSERR("Emulator engine configuration failed!\n");
	return FALSE;
	}

 /* Initialize status information */
 myPlayer->currTune = newTune;

 return TRUE;
}


/*
 * Delete INTERNAL information
 */
void xs_sidplay1_deletesid(t_xs_status *myStatus)
{
 t_xs_sidplay1 *myPlayer;
 assert(myStatus);

 myPlayer = (t_xs_sidplay1 *) myStatus->player;
 if (!myPlayer) return;

 if (myPlayer->currTune)
	{
	delete myPlayer->currTune;
	myPlayer->currTune = NULL;
	}
}


/*
 * Return song information
 */
#define TFUNCTION1	xs_sidplay1_filetitle
#define TFUNCTION2	xs_sidplay1_getsidinfo
#define TTUNEINFO	sidTuneInfo
#define TTUNE		sidTune
#include "xs_sidplay.h"

}	/* extern "C" */
#endif	/* HAVE_SIDPLAY1 */
