#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <stdio.h>

#include "xs_curve.h"
#include <gtk/gtkdrawingarea.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>


#define GET_X(i)	curve->ctlpoints[i].x
#define GET_Y(i)	curve->ctlpoints[i].y


#define RADIUS		3	/* radius of the control points */
#define RADIUS2		(RADIUS * 2)

#define MIN_DISTANCE	8	/* min distance between control points */

#define GRAPH_MASK	(GDK_EXPOSURE_MASK |		\
			GDK_POINTER_MOTION_MASK |	\
			GDK_POINTER_MOTION_HINT_MASK |	\
			GDK_ENTER_NOTIFY_MASK |		\
			GDK_BUTTON_PRESS_MASK |		\
			GDK_BUTTON_RELEASE_MASK |	\
			GDK_BUTTON1_MOTION_MASK)

enum {
	ARG_0,
	ARG_CURVE_TYPE,
	ARG_MIN_X,
	ARG_MAX_X,
	ARG_MIN_Y,
	ARG_MAX_Y
};

static GtkDrawingAreaClass *parent_class = NULL;

static void xs_curve_class_init(XSCurveClass * class);
static void xs_curve_init(XSCurve * curve);
static void xs_curve_set_arg(GtkObject * object, GtkArg * arg, guint arg_id);
static void xs_curve_get_arg(GtkObject * object, GtkArg * arg, guint arg_id);
static void xs_curve_finalize(GtkObject * object);
static gint xs_curve_graph_events(GtkWidget * widget, GdkEvent * event, XSCurve * c);
static void xs_curve_size_graph(XSCurve * curve);


GtkType xs_curve_get_type(void)
{
	static GtkType curve_type = 0;

	if (!curve_type) {
		static const GtkTypeInfo curve_info = {
			"XSCurve",
			sizeof(XSCurve),
			sizeof(XSCurveClass),
			(GtkClassInitFunc) xs_curve_class_init,
			(GtkObjectInitFunc) xs_curve_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		curve_type = gtk_type_unique(GTK_TYPE_DRAWING_AREA, &curve_info);
	}
	return curve_type;
}


static void xs_curve_class_init(XSCurveClass *class)
{
	GtkObjectClass *object_class;

	parent_class = gtk_type_class(GTK_TYPE_DRAWING_AREA);

	object_class = (GtkObjectClass *) class;

	object_class->set_arg = xs_curve_set_arg;
	object_class->get_arg = xs_curve_get_arg;
	object_class->finalize = xs_curve_finalize;

	gtk_object_add_arg_type("XSCurve::min_x", GTK_TYPE_FLOAT, GTK_ARG_READWRITE, ARG_MIN_X);
	gtk_object_add_arg_type("XSCurve::max_x", GTK_TYPE_FLOAT, GTK_ARG_READWRITE, ARG_MAX_X);
	gtk_object_add_arg_type("XSCurve::min_y", GTK_TYPE_FLOAT, GTK_ARG_READWRITE, ARG_MIN_Y);
	gtk_object_add_arg_type("XSCurve::max_y", GTK_TYPE_FLOAT, GTK_ARG_READWRITE, ARG_MAX_Y);
}


static void xs_curve_init(XSCurve *curve)
{
	gint old_mask;

	curve->pixmap = NULL;
	curve->height = 0;
	curve->grab_point = -1;

	curve->num_ctlpoints = 0;
	curve->ctlpoints = NULL;

	curve->min_x = 0.0;
	curve->max_x = 2048.0;
	curve->min_y = 0.0;
	curve->max_y = 22000.0;

	old_mask = gtk_widget_get_events(GTK_WIDGET(curve));
	gtk_widget_set_events(GTK_WIDGET(curve), old_mask | GRAPH_MASK);
	gtk_signal_connect(GTK_OBJECT(curve), "event", (GtkSignalFunc) xs_curve_graph_events, curve);
	xs_curve_size_graph(curve);
}


static void xs_curve_set_arg(GtkObject *object, GtkArg *arg, guint arg_id)
{
	XSCurve *curve = XS_CURVE(object);

	switch (arg_id) {
	case ARG_MIN_X:
		xs_curve_set_range(curve, GTK_VALUE_FLOAT(*arg), curve->max_x, curve->min_y, curve->max_y);
		break;
	case ARG_MAX_X:
		xs_curve_set_range(curve, curve->min_x, GTK_VALUE_FLOAT(*arg), curve->min_y, curve->max_y);
		break;
	case ARG_MIN_Y:
		xs_curve_set_range(curve, curve->min_x, curve->max_x, GTK_VALUE_FLOAT(*arg), curve->max_y);
		break;
	case ARG_MAX_Y:
		xs_curve_set_range(curve, curve->min_x, curve->max_x, curve->min_y, GTK_VALUE_FLOAT(*arg));
		break;
	}
}


static void xs_curve_get_arg(GtkObject *object, GtkArg *arg, guint arg_id)
{
	XSCurve *curve = XS_CURVE(object);

	switch (arg_id) {
	case ARG_MIN_X:
		GTK_VALUE_FLOAT(*arg) = curve->min_x;
		break;
	case ARG_MAX_X:
		GTK_VALUE_FLOAT(*arg) = curve->max_x;
		break;
	case ARG_MIN_Y:
		GTK_VALUE_FLOAT(*arg) = curve->min_y;
		break;
	case ARG_MAX_Y:
		GTK_VALUE_FLOAT(*arg) = curve->max_y;
		break;
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}


static int xs_project(gfloat value, gfloat min, gfloat max, int norm)
{
	return (norm - 1) * ((value - min) / (max - min)) + 0.5;
}


static gfloat xs_unproject(gint value, gfloat min, gfloat max, int norm)
{
	return value / (gfloat) (norm - 1) * (max - min) + min;
}


static inline void xs_cubic_coeff(gfloat x1, gfloat y1,
			gfloat x2, gfloat y2,
			gfloat k1, gfloat k2,
			gfloat *a, gfloat *b,
			gfloat *c, gfloat *d)
{
	gfloat dx = x2 - x1, dy = y2 - y1;

	*a = ((k1 + k2) - 2 * dy / dx) / (dx * dx);
	*b = ((k2 - k1) / dx - 3 * (x1 + x2) * (*a)) / 2;
	*c = k1 - (3 * x1 * (*a) + 2 * (*b)) * x1;
	*d = y1 - ((x1 * (*a) + (*b)) * x1 + (*c)) * x1;
}

#define x(val) val->x
#define y(val) val->y

static void xs_curve_draw(XSCurve *curve, gint width, gint height)
{
	gfloat res = 10.0f;
	GtkStateType state;
	GtkStyle *style;
	gint i;
	t_xs_point *p0, *p1, *p2, *p3;

	if (!curve->pixmap)
		return;

	state = GTK_STATE_NORMAL;
	if (!GTK_WIDGET_IS_SENSITIVE(GTK_WIDGET(curve)))
		state = GTK_STATE_INSENSITIVE;

	style = GTK_WIDGET(curve)->style;

	/* Clear the pixmap */
	gtk_paint_flat_box(style, curve->pixmap,
		GTK_STATE_NORMAL, GTK_SHADOW_NONE,
		NULL, GTK_WIDGET(curve), "curve_bg",
		0, 0,
		width + RADIUS2,
		height + RADIUS2);

	
	/* Draw the grid */
	for (i = 0; i < 5; i++) {
		gdk_draw_line(curve->pixmap, style->dark_gc[state],
			RADIUS,		i * (height / 4.0) + RADIUS,
			width + RADIUS,	i * (height / 4.0) + RADIUS);

		gdk_draw_line(curve->pixmap, style->dark_gc[state],
			i * (width / 4.0) + RADIUS, RADIUS,
			i * (width / 4.0) + RADIUS, height + RADIUS);
	}

	
	/* Draw the spline/curve itself */
	p0 = curve->ctlpoints;
	p1 = p0; p1++;
	p2 = p1; p2++;
	p3 = p2; p3++;

	/* Draw each curve segment */
	fprintf(stderr, "npoints = %d\n", curve->num_ctlpoints);
	for (i = 0; i < curve->num_ctlpoints; i++, ++p0, ++p1, ++p2, ++p3) {
		gfloat k1, k2, a, b, c, d, x;
		
		/* p1 and p2 equal; single point */
		if (x(p1) == x(p2))
			continue;
		
		/* Both end points repeated; straight line */
		if (x(p0) == x(p1) && x(p2) == x(p3)) {
			k1 = k2 = (y(p2) - y(p1)) / (x(p2) - x(p1));
		}
		/* p0 and p1 equal; use f''(x1) = 0 */
		else if (x(p0) == x(p1)) {
			k2 = (y(p3) - y(p1)) / (x(p3) - x(p1));
			k1 = (3 * (y(p2) - y(p1)) / (x(p2) - x(p1)) - k2) / 2;
		}
		/* p2 and p3 equal; use f''(x2) = 0 */
		else if (x(p2) == x(p3)) {
			k1 = (y(p2) - y(p0)) / (x(p2) - x(p0));
			k2 = (3 * (y(p2) - y(p1)) / (x(p2) - x(p1)) - k1) / 2;
		}
		/* Normal curve */
		else {
			k1 = (y(p2) - y(p0)) / (x(p2) - x(p0));
			k2 = (y(p3) - y(p1)) / (x(p3) - x(p1));
		}

		xs_cubic_coeff(x(p1), y(p1), x(p2), y(p2), k1, k2, &a, &b, &c, &d);
		
		fprintf(stderr, "--seg[%1.3f, %1.3f]=>[%1.3f, %1.3f]--\n",
			x(p1), y(p1), x(p2), y(p2));
		
		for (x = x(p1); x <= x(p2); x += res) {
			gfloat y = ((a * x + b) * x + c) * x + d;
			gint qx, qy;
			qx = RADIUS + xs_project(x, curve->min_x, curve->max_x, width);
			qy = RADIUS + xs_project(y, curve->min_y, curve->max_y, height);

			fprintf(stderr, "[%1.3f, %1.3f] -> %d, %d\n", x, y, qx, qy);
			
			gdk_draw_point(curve->pixmap, style->fg_gc[state],
				RADIUS + xs_project(x, curve->min_x, curve->max_x, width),
				RADIUS + xs_project(y, curve->min_y, curve->max_y, height)
				);

		}
		
		fprintf(stderr, "-------\n");
	}
	
	/* Draw control points */
	for (i = 0; i < curve->num_ctlpoints; ++i) {
		gint x, y;

		if (GET_X(i) < curve->min_x || GET_Y(i) < curve->min_y ||
			GET_X(i) >= curve->max_x || GET_Y(i) >= curve->max_y)
			continue;

		x = xs_project(GET_X(i), curve->min_x, curve->max_x, width);
		y = xs_project(GET_Y(i), curve->min_y, curve->max_y, height);

		gdk_draw_arc(curve->pixmap, style->fg_gc[state], TRUE,
			x, y, RADIUS2, RADIUS2, 0, 360 * 64);
	}
	
	/* Draw pixmap in the widget */
	gdk_draw_pixmap(GTK_WIDGET(curve)->window,
			style->fg_gc[state], curve->pixmap,
			0, 0, 0, 0,
			width + RADIUS2,
			height + RADIUS2);
}

#undef x
#undef y


static gint xs_curve_graph_events(GtkWidget *widget, GdkEvent *event, XSCurve *curve)
{
	GdkCursorType new_type = curve->cursor_type;
	GdkEventButton *bevent;
	GdkEventMotion *mevent;
	GtkWidget *w;
	gint i, width, height, x, y, tx, ty, cx, closest_point = 0, min_x;
	guint distance;

	w = GTK_WIDGET(curve);
	width = w->allocation.width - RADIUS2;
	height = w->allocation.height - RADIUS2;

	if ((width < 0) || (height < 0))
		return FALSE;

	/* get the pointer position */
	gdk_window_get_pointer(w->window, &tx, &ty, NULL);
	x = CLAMP((tx - RADIUS), 0, width - 1);
	y = CLAMP((ty - RADIUS), 0, height - 1);

	min_x = curve->min_x;

	distance = ~0U;
	for (i = 0; i < curve->num_ctlpoints; ++i) {
		cx = xs_project(GET_X(i), min_x, curve->max_x, width);
		if ((guint) abs(x - cx) < distance) {
			distance = abs(x - cx);
			closest_point = i;
		}
	}
	
	/* Act based on event type */
	switch (event->type) {
	case GDK_CONFIGURE:
		if (curve->pixmap)
			gdk_pixmap_unref(curve->pixmap);
		curve->pixmap = 0;

		/* fall through */

	case GDK_EXPOSE:
		if (!curve->pixmap) {
			curve->pixmap = gdk_pixmap_new(w->window,
			w->allocation.width, w->allocation.height, -1);
		}
		xs_curve_draw(curve, width, height);
		break;

	case GDK_BUTTON_PRESS:
		gtk_grab_add(widget);

		bevent = (GdkEventButton *) event;
		new_type = GDK_TCROSS;

		if (distance > MIN_DISTANCE) {
			/* insert a new control point */
			if (curve->num_ctlpoints > 0) {
				cx = xs_project(GET_X(closest_point), min_x, curve->max_x, width);
				if (x > cx) closest_point++;
			}
			
			curve->num_ctlpoints++;
			
			curve->ctlpoints = g_realloc(curve->ctlpoints, curve->num_ctlpoints * sizeof(*curve->ctlpoints));
			for (i = curve->num_ctlpoints - 1; i > closest_point; --i) {
				memcpy(curve->ctlpoints + i, curve->ctlpoints + i - 1, sizeof(*curve->ctlpoints));
			}
		}
		
		curve->grab_point = closest_point;
		GET_X(curve->grab_point) = xs_unproject(x, min_x, curve->max_x, width);
		GET_Y(curve->grab_point) = xs_unproject(y, curve->min_y, curve->max_y, height);

		xs_curve_draw(curve, width, height);
		break;

	case GDK_BUTTON_RELEASE:
		{
		gint src, dst;
		
		gtk_grab_remove(widget);

		/* delete inactive points: */
		for (src = dst = 0; src < curve->num_ctlpoints; ++src)
		if (GET_X(src) >= min_x) {
			memcpy(curve->ctlpoints + dst, curve->ctlpoints + src, sizeof(*curve->ctlpoints));
			dst++;
		}

		if (dst < src) {
			curve->num_ctlpoints -= (src - dst);
			if (curve->num_ctlpoints <= 0) {
				curve->num_ctlpoints = 1;
				GET_X(0) = min_x;
				GET_Y(0) = curve->min_y;
				xs_curve_draw(curve, width, height);
			}
			curve->ctlpoints = g_realloc(curve->ctlpoints, curve->num_ctlpoints * sizeof(*curve->ctlpoints));
		}

		new_type = GDK_FLEUR;
		curve->grab_point = -1;
		}
		break;

	case GDK_MOTION_NOTIFY:
		mevent = (GdkEventMotion *) event;

		if (curve->grab_point == -1) {
			/* if no point is grabbed...  */
			if (distance <= MIN_DISTANCE)
				new_type = GDK_FLEUR;
			else
				new_type = GDK_TCROSS;
		} else {
			gint leftbound, rightbound;
			
			/* drag the grabbed point  */
			new_type = GDK_TCROSS;
			
			leftbound = -MIN_DISTANCE;
			if (curve->grab_point > 0) {
				leftbound = xs_project(
					GET_X(curve->grab_point-1),
					min_x, curve->max_x, width);
			}

			rightbound = width + RADIUS2 + MIN_DISTANCE;
			if (curve->grab_point + 1 < curve->num_ctlpoints) {
				rightbound = xs_project(
					GET_X(curve->grab_point+1),
					min_x, curve->max_x, width);
			}

			if ((tx <= leftbound) || (tx >= rightbound) ||
				(ty > height + RADIUS2 + MIN_DISTANCE) || (ty < -MIN_DISTANCE)) {
				GET_X(curve->grab_point) = min_x - 1.0;
			} else {
				GET_X(curve->grab_point) =
					xs_unproject(x, min_x, curve->max_x, width);
				GET_Y(curve->grab_point) =
					xs_unproject(y, curve->min_y, curve->max_y, height);
			}
			
			xs_curve_draw(curve, width, height);
		}

		if (new_type != (GdkCursorType) curve->cursor_type) {
			GdkCursor *cursor;

			curve->cursor_type = new_type;

			cursor = gdk_cursor_new(curve->cursor_type);
			gdk_window_set_cursor(w->window, cursor);
			gdk_cursor_destroy(cursor);
		}
		break;

	default:
		break;
	}
	
	return FALSE;
}


static void xs_curve_size_graph(XSCurve *curve)
{
	gint width, height;
	gfloat aspect;

	width = (curve->max_x - curve->min_x) + 1;
	height = (curve->max_y - curve->min_y) + 1;
	aspect = width / (gfloat) height;

	if (width > gdk_screen_width() / 4)
		width = gdk_screen_width() / 4;

	if (height > gdk_screen_height() / 4)
		height = gdk_screen_height() / 4;

	if (aspect < 1.0)
		width = height * aspect;
	else
		height = width / aspect;

	gtk_drawing_area_size(GTK_DRAWING_AREA(curve), width + RADIUS2, height + RADIUS2);
}


void xs_curve_reset(XSCurve *curve)
{
	if (curve->ctlpoints)
		g_free(curve->ctlpoints);

	curve->num_ctlpoints = 4;
	curve->ctlpoints = g_malloc(curve->num_ctlpoints * sizeof(curve->ctlpoints[0]));

	GET_X(0) = curve->min_x;
	GET_Y(0) = curve->min_y;

	GET_X(1) = curve->min_x;
	GET_Y(1) = curve->min_y;

	GET_X(2) = curve->max_x;
	GET_Y(2) = curve->max_y;

	GET_X(3) = curve->max_x;
	GET_Y(3) = curve->max_y;

	if (curve->pixmap) {
		gint width, height;

		width = GTK_WIDGET(curve)->allocation.width - RADIUS2;
		height = GTK_WIDGET(curve)->allocation.height - RADIUS2;
		xs_curve_draw(curve, width, height);
	}
}


void xs_curve_set_range(XSCurve *curve, gfloat min_x, gfloat max_x, gfloat min_y, gfloat max_y)
{
	curve->min_x = min_x;
	curve->max_x = max_x;
	curve->min_y = min_y;
	curve->max_y = max_y;

	xs_curve_size_graph(curve);
	xs_curve_reset(curve);
}


GtkWidget *xs_curve_new(void)
{
	return gtk_type_new(xs_curve_get_type());
}


static void xs_curve_finalize(GtkObject *object)
{
	XSCurve *curve;

	g_return_if_fail(object != NULL);
	g_return_if_fail(XS_IS_CURVE(object));

	curve = XS_CURVE(object);

	if (curve->pixmap)
		gdk_pixmap_unref(curve->pixmap);

	if (curve->ctlpoints)
		g_free(curve->ctlpoints);

	(*GTK_OBJECT_CLASS(parent_class)->finalize) (object);
}

