/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include "singit_macros.h"
#include "singit_config.h"
#include "singit_displayer_plugin.h"
#include "singit_wgt_karaoke.h"

#include "displayer_fullscreen_status.h"

#define FULL_SCREEN_HIDE_MOUSE_DELAY 3000
#define BLACK_BACKGROUND 1

static GtkWidget *displayer_full_screen_window = NULL;

void displayer_fullscreen_init(void);
void displayer_fullscreen_show(SingitSong *cur_song);
void displayer_fullscreen_hide(SingitSong *cur_song);
void displayer_fullscreen_toggle(SingitSong *cur_song);
void displayer_fullscreen_finish(void);
void displayer_fullscreen_set_time(guint time,SingitSong *cur_song, GList *real_next);
void displayer_fullscreen_config_update(const SingitConfigData *singit_config);

DisplayerPlugin fullscreen_dp =
{
	NULL,
	NULL,
	0,

	"Fullscreen displayer",

	displayer_fullscreen_init,
	displayer_fullscreen_show,
	NULL,
	NULL,
	displayer_fullscreen_finish,
	NULL,
	displayer_fullscreen_config_update,

	displayer_fullscreen_set_time,

	NULL,
	NULL,
	NULL,
	NULL,

	NULL,
	NULL
};

DIPSPLAYER_SYMBOL(libdisplayer_fullscreen, fullscreen_dp)

static void fullscreen_set_jumping_ball()
{
	if (full_screen_status.guess_synced_lyrics) {
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
			(singit_song_guess_sync_lyrics(full_screen_status.song) &&
			full_screen_status.show_jumping_ball));
	}
	else {
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
			full_screen_status.show_jumping_ball);
	}
}

static void clear_mouse_cursor(GtkWidget *widget, gint clear)
{
	if (clear) {
		GdkCursor *cursor;
		GdkPixmap *p;

		p = gdk_bitmap_create_from_data(widget->window, "\0\0\0", 1, 1);

		cursor = gdk_cursor_new_from_pixmap(p, p,
			&widget->style->fg[GTK_STATE_ACTIVE],
			&widget->style->bg[GTK_STATE_ACTIVE],
			0, 0);

		gdk_window_set_cursor (widget->window, cursor);

		if (cursor) gdk_cursor_destroy (cursor);
		gdk_pixmap_unref(p);
	}
	else { gdk_window_set_cursor (widget->window, NULL); }
}

static gint fullscreen_hide_mouse_cb(gpointer data)
{
	if (full_screen_status.hide_mouse_id == -1) { return FALSE; }

	clear_mouse_cursor(displayer_full_screen_window, TRUE);

	full_screen_status.hide_mouse_id = -1;

	return FALSE;
}

static void fullscreen_hide_mouse_disable()
{
	if (full_screen_status.hide_mouse_id != -1) {
		gtk_timeout_remove(full_screen_status.hide_mouse_id);
		full_screen_status.hide_mouse_id = -1;
	}
}

static void fullscreen_hide_mouse_reset()
{
	fullscreen_hide_mouse_disable();
	full_screen_status.hide_mouse_id =
		gtk_timeout_add(FULL_SCREEN_HIDE_MOUSE_DELAY, fullscreen_hide_mouse_cb, NULL);
}

static void fullscreen_mouse_moved(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	clear_mouse_cursor(displayer_full_screen_window, FALSE);
	fullscreen_hide_mouse_reset();
}

static gboolean displayer_fullscreen_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	gint volume;

        switch(event->keyval) {
                case GDK_Escape: // exit plugin
			fullscreen_dp.disable(&fullscreen_dp);
                        break;
                case GDK_b:
                        xmms_remote_playlist_next(fullscreen_dp.xmms_session);
                        break;
                case GDK_v:
                        xmms_remote_stop(fullscreen_dp.xmms_session);
                        break;
                case GDK_x:
                        xmms_remote_play(fullscreen_dp.xmms_session);
                        break;
		case GDK_y:
			if (full_screen_status.toggle_yz)
				{ xmms_remote_playlist_prev(fullscreen_dp.xmms_session); }
			break;
                case GDK_z:
			if (!full_screen_status.toggle_yz)
				{ xmms_remote_playlist_prev(fullscreen_dp.xmms_session); }
                        break;
		case GDK_KP_Add:
			volume = xmms_remote_get_main_volume(fullscreen_dp.xmms_session);
			if (volume < 100) { volume++; }
			xmms_remote_set_main_volume(fullscreen_dp.xmms_session, volume);
			break;
		case GDK_KP_Subtract:
			volume = xmms_remote_get_main_volume(fullscreen_dp.xmms_session);
			if (volume > 0) { volume--; }
			xmms_remote_set_main_volume(fullscreen_dp.xmms_session, volume);
			break;
		case GDK_KP_Enter:
			if (full_screen_status.jump_to_song == 0) { break; }
			if (xmms_remote_get_playlist_pos(fullscreen_dp.xmms_session) != (full_screen_status.jump_to_song-1))
				{ xmms_remote_set_playlist_pos(fullscreen_dp.xmms_session, (full_screen_status.jump_to_song-1)); }
		case GDK_KP_Decimal:
			full_screen_status.jump_to_song = 0;
			break;
		case GDK_KP_0:
		case GDK_KP_1:
		case GDK_KP_2:
		case GDK_KP_3:
		case GDK_KP_4:
		case GDK_KP_5:
		case GDK_KP_6:
		case GDK_KP_7:
		case GDK_KP_8:
		case GDK_KP_9:
			full_screen_status.jump_to_song = (full_screen_status.jump_to_song*10) + (event->keyval - GDK_KP_0);
			if (full_screen_status.jump_to_song > xmms_remote_get_playlist_length(fullscreen_dp.xmms_session))
				{ full_screen_status.jump_to_song = (event->keyval - GDK_KP_0); }
			break;
		case GDK_f:
			if (event->state & GDK_CONTROL_MASK) {
				fullscreen_dp.disable(&fullscreen_dp);
			}
			break;
                default:
                        break;
        }

        return TRUE;
}

static gint displayer_fullscreen_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	fullscreen_dp.disable(&fullscreen_dp);
	return (FALSE);
}

void displayer_fullscreen_init(void)
{
	gint w;
	gint h;
//	GtkStyle *style;

	#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_init]\n"));
	#endif

	g_return_if_fail(displayer_full_screen_window == NULL);

	displayer_fullscreen_status_init();

	w = gdk_screen_width();
	h = gdk_screen_height();

	displayer_full_screen_window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_container_border_width(GTK_CONTAINER(displayer_full_screen_window), 0);
	gtk_signal_connect(GTK_OBJECT(displayer_full_screen_window), "delete_event", GTK_SIGNAL_FUNC(displayer_fullscreen_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_full_screen_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_full_screen_window);
	gtk_signal_connect(GTK_OBJECT(displayer_full_screen_window), "key_press_event", GTK_SIGNAL_FUNC(displayer_fullscreen_keypress), NULL);
	gtk_window_set_title(GTK_WINDOW(displayer_full_screen_window), _("SingIt full screen mode"));
	gtk_widget_set_usize(displayer_full_screen_window, w, h);
	gtk_widget_set_uposition(displayer_full_screen_window, 0, 0);

	full_screen_status.karaoke_widget = singit_karaoke_widget_new();
	gtk_container_add(GTK_CONTAINER(displayer_full_screen_window), full_screen_status.karaoke_widget);
	singit_karaoke_widget_set_show_empty_lines (
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget), TRUE);
	singit_karaoke_widget_set_lines (SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget), 7);
	singit_karaoke_widget_set_toplines (SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget), 3);
	gtk_signal_connect(GTK_OBJECT(full_screen_status.karaoke_widget),
		"key_press_event", GTK_SIGNAL_FUNC(displayer_fullscreen_keypress), NULL);
	gtk_widget_show(full_screen_status.karaoke_widget);

//	style = gtk_style_copy(gtk_widget_get_style(full_screen_status.karaoke_widget));
//	gtk_style_ref(style);
//	style->bg[GTK_STATE_NORMAL] = style->black;
//	gtk_widget_set_style(full_screen_status.karaoke_widget, style);
//	gtk_style_unref(style);

	gtk_widget_realize(displayer_full_screen_window);

	singit_karaoke_widget_set_optimal_font(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget));

	gtk_signal_connect(GTK_OBJECT(full_screen_status.karaoke_widget), "motion_notify_event",
		GTK_SIGNAL_FUNC(fullscreen_mouse_moved), NULL);
}

void displayer_fullscreen_show(SingitSong *cur_song)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_show]\n"));
	#endif

	if (displayer_full_screen_window && singit_song_lyrics_found(cur_song)) {
		if (!GTK_WIDGET_VISIBLE(displayer_full_screen_window)) {
			gtk_widget_show(displayer_full_screen_window);
			clear_mouse_cursor(displayer_full_screen_window, TRUE);
			gtk_window_set_modal(GTK_WINDOW(displayer_full_screen_window), TRUE);
			gdk_keyboard_grab(displayer_full_screen_window->window, TRUE, GDK_CURRENT_TIME);
			gtk_grab_add(displayer_full_screen_window);
			gtk_widget_grab_focus(displayer_full_screen_window);
		}
		else { gdk_window_raise(displayer_full_screen_window->window); }
	}
}

void displayer_fullscreen_hide(SingitSong *cur_song)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_hide]\n"));
	#endif

	if (displayer_full_screen_window) {
		if (GTK_WIDGET_VISIBLE(displayer_full_screen_window)) {
			fullscreen_hide_mouse_disable();
			gdk_keyboard_ungrab(GDK_CURRENT_TIME);

			gtk_grab_remove(displayer_full_screen_window);
			gtk_widget_hide(displayer_full_screen_window);
		}
	}
}

void displayer_fullscreen_toggle(SingitSong *cur_song)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_toggle]\n"));
	#endif

	if (displayer_full_screen_window) {
		if (gdk_window_is_visible(displayer_full_screen_window->window))
			{ displayer_fullscreen_hide(cur_song); }
		else { displayer_fullscreen_show(cur_song); }
	}
}

void displayer_fullscreen_set_time(guint time, SingitSong *cur_song, GList *real_next)
{
	SingitSong *my_song;

	if (displayer_full_screen_window) {
		my_song = singit_song_attach(cur_song);
		if (my_song == NULL) 
			{ return; }

		if ((my_song != full_screen_status.song) ||
			(strcmp(my_song->song_filename, full_screen_status.song->song_filename) != 0))
		{
			singit_song_detach(&full_screen_status.song);
			full_screen_status.song = my_song;

			GDK_THREADS_ENTER();
			singit_karaoke_widget_freeze
				(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget));
			singit_karaoke_widget_set_song
				(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget), my_song);
			singit_karaoke_widget_set_time
				(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget), time);
			if (full_screen_status.find_optimal_font) {
				singit_karaoke_widget_set_optimal_font
					(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget));
			}

			fullscreen_set_jumping_ball();
			singit_karaoke_widget_thaw
				(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget));
			GDK_THREADS_LEAVE();
		}
		else {
			GDK_THREADS_ENTER();
			singit_karaoke_widget_set_time
				(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget), time);
			singit_song_detach(&my_song);
			GDK_THREADS_LEAVE();
		}
	}
}

void displayer_fullscreen_finish(void)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_finish]\n"));
	#endif

	if (displayer_full_screen_window)
		{ gtk_widget_destroy(displayer_full_screen_window); }

	displayer_fullscreen_status_finish();
}

void displayer_fullscreen_config_update(const SingitConfigData *singit_config)
{
	full_screen_status.print_debug = singit_config->debugEnable;

	#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_config_update]\n"));
	#endif

	full_screen_status.toggle_yz = singit_config->toggleYZ;
	full_screen_status.guess_synced_lyrics = singit_config->guessSyncLyrics;
	full_screen_status.show_jumping_ball = singit_config->showJumpingBall;
	full_screen_status.find_optimal_font = singit_config->findOptimalFont;

	singit_karaoke_widget_freeze
		(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget));

	// Update colors
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
		skwc_background, singit_config->backgroundColor);
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
		skwc_normal_text, singit_config->normalTextColor);
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
		skwc_normal_bar, singit_config->normalBarColor);
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
		skwc_progress_text, singit_config->progressTextColor);
	singit_karaoke_widget_set_color(
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
		skwc_progress_bar, singit_config->progressBarColor);

	singit_karaoke_widget_set_font(
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
		singit_config->lyricFontName);
	singit_karaoke_widget_set_time_indicators(
		SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget),
		singit_config->showTimeIndicator);
	fullscreen_set_jumping_ball();
	if (full_screen_status.find_optimal_font) {
		singit_karaoke_widget_set_optimal_font
			(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget));
	}
	singit_karaoke_widget_thaw
		(SINGIT_KARAOKE_WIDGET(full_screen_status.karaoke_widget));
}
