/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include <stdlib.h>
#include <stdio.h>

#include <xmms/xmmsctrl.h>
#include <xmms/util.h>
#include <xmms/titlestring.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_syswm.h>
#include <SDL/SDL_ttf.h>

#include "singit_displayer_plugin.h"
#include "displayer_sdl.h"

#include "singit_config.h"

#include "singit_dialogs.h"

#include "sdl_handlers.h"
#include "sdl_helpers.h"
#include "sdl_skd_handlers.h"

#include "displayer_sdl_config.h"
#include "displayer_sdl_config_dlg.h"

#include "displayer_sdl_lyrics.h"

#include "SDL_prim.h"

static void displayer_sdl_init(void);
static void displayer_sdl_finish(void);

static void displayer_sdl_show(SingitSong *cur_song);
static void displayer_sdl_hide(SingitSong *cur_song);
static void displayer_sdl_toggle(SingitSong *cur_song);

static void displayer_sdl_set_time(guint time, SingitSong *cur_song, GList *real_next);
static void displayer_sdl_config_update(const SingitConfigData *singit_config);

static void displayer_sdl_playback_start(void);
static void displayer_sdl_playback_stop(void);

static void displayer_sdl_configure(void);

extern SingitConfigGen *displayer_sdl_config;

DisplayerPlugin sdl_dp =
{
	NULL,
	NULL,
	0,

	"SDL displayer",

	displayer_sdl_init,
	displayer_sdl_show,
	displayer_sdl_hide,
	NULL,
	displayer_sdl_finish,
	NULL,
	displayer_sdl_config_update,

	displayer_sdl_set_time,

	NULL,
	displayer_sdl_configure,
	displayer_sdl_playback_start,
	displayer_sdl_playback_stop,

	displayer_sdl_render_pcm,
	NULL
};

DIPSPLAYER_SYMBOL(libdisplayer_sdl, sdl_dp)

static void displayer_sdl_config_update_event(SingitConfigGen *scg, SDLConfigData *cfg_data, gpointer user_data);

gint displayer_sdl_disable_func(gpointer data)
{
	GDK_THREADS_ENTER();
	sdl_dp.disable(&sdl_dp);
	GDK_THREADS_LEAVE();
	return (FALSE);
}

static void displayer_sdl_process_events(guint time)
{
	SDL_Event sdl_event;

	if (displayer_sdl_status.sdl_inuse == TRUE)
		{ return; }

	sdl_lyrics_set_text_colors(getPaletteColor(255),
		getPaletteColor(128), getPaletteColor(0));

	if (displayer_sdl_status.screen != SDL_GetVideoSurface())
		{ return; }

//	__SDL_PRIM_LOCKSURFACE(displayer_sdl_status.screen);
	sdl_lyrics_render_text(time);
	displayer_sdl_render();
	displayer_sdl_draw_text();
	SDL_UpdateRect(displayer_sdl_status.screen, 0, 0, 0, 0);
//	__SDL_PRIM_UNLOCKSURFACE(displayer_sdl_status.screen);

	if (!displayer_sdl_status.kill_render_thread && (SDL_PollEvent(&sdl_event) > 0)) {
		switch (sdl_event.type) {
		case SDL_KEYDOWN:
			sdl_handler_key_down(&sdl_event);
			break;
		case SDL_QUIT:
			// Prevent double shutdown
			if (displayer_sdl_status.shutdown == TRUE)
				break;
			displayer_sdl_status.shutdown = TRUE;
			sdl_helper_blanking(BLANK_RESET);
			if (displayer_sdl_status.kill_render_thread == FALSE) {
				gtk_idle_add(displayer_sdl_disable_func, NULL);
			}
			break;
		case SDL_VIDEORESIZE:
			// After a resize directly redraw the screen so return
			sdl_handler_resize(&sdl_event);
			return;
		case SDL_ACTIVEEVENT: // Application visibility event structure
			sdl_handler_active(&sdl_event);
			break;
		case SDL_VIDEOEXPOSE:
			// Since the screen is rendered quite often expose
			// events don't need to be handled expicit
			break;
		}
	}

	singit_framerate_counter_new_frame(displayer_sdl_status.sfc);
}

static void *displayer_sdl_event_pthread_func(void *args)
{
	singit_framerate_counter_set_limit(displayer_sdl_status.sfc, 32);

	while (!displayer_sdl_status.kill_render_thread)
	{
		displayer_sdl_process_events(0);
	}

	singit_framerate_counter_set_limit(displayer_sdl_status.sfc, 0);

#	ifdef CODEDEBUG
	DEBUG(("  Quit render thread\n"));
#	endif

	displayer_sdl_status.kill_render_thread = FALSE;
	displayer_sdl_status.render_thread = 0;

	pthread_exit(0);
}

static gboolean sdl_render_pthread_quit()
{
	if (displayer_sdl_status.render_thread != 0)
	{
#ifdef CODEDEBUG
		DEBUG(("  Kill render thread\n"));
#endif
		displayer_sdl_status.kill_render_thread = TRUE;
		pthread_join(displayer_sdl_status.render_thread, NULL);
		displayer_sdl_status.kill_render_thread = FALSE;
		displayer_sdl_status.render_thread = 0;
		return TRUE;
	}
	return FALSE;
}

static gboolean set_sdl_render_pthread()
{
	gboolean result = FALSE;

	gboolean need_thread =
		((displayer_sdl_status.xmms_playing == FALSE) ||
		 (displayer_sdl_status.singit_playing == FALSE) ||
		 (xmms_remote_is_paused(sdl_dp.xmms_session) == TRUE));

	if ((displayer_sdl_status.render_thread == 0) && (need_thread == TRUE))
	{
#ifdef CODEDEBUG
		DEBUG(("  Start render thread\n"));
#endif
		if (pthread_create(&displayer_sdl_status.render_thread, NULL,
			displayer_sdl_event_pthread_func, NULL) != 0)
		{
			displayer_sdl_status.render_thread = 0;
			g_print("Unable to create sdl render thread\n");
		}
		else { result = TRUE; }
	}
	else if (need_thread == FALSE)
		{ result = sdl_render_pthread_quit(); }

	return result;
}

static void displayer_sdl_init(void)
{
	SDLConfigData *scd;

#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_init]\n"));
#	endif

	displayer_sdl_status_init();

	if (sdl_handler_init() == FALSE) {
		gtk_idle_add (displayer_sdl_disable_func, NULL);
		return;
	}
	sdl_skd_init();
	sdl_lyrics_init();
	sdl_lyrics_set_optimize_font(TRUE);

	if (!singit_config_gen_attach(displayer_sdl_config)) {
		displayer_sdl_config = displayer_sdl_config_new();
		singit_config_gen_load(SINGIT_CONFIG_GEN(displayer_sdl_config));
	}

	scd = GET_DSCD(displayer_sdl_config);
	if (!scd) {
		singit_config_gen_detach(SINGIT_CONFIG_GEN(displayer_sdl_config));
		gtk_idle_add (displayer_sdl_disable_func, NULL);
		return;
	}

	if (!((scd->lyricFontName != NULL) &&
		sdl_lyrics_set_font(scd->lyricFontName, scd->lyricFontSize, TRUE)))
	{
		sdl_lyrics_set_font(SINGIT_DATA_DIR "/vixar.ttf", 32, TRUE);
	}

	gtk_signal_connect(GTK_OBJECT(displayer_sdl_config), "update",
		GTK_SIGNAL_FUNC(displayer_sdl_config_update_event), NULL);

	displayer_sdl_status.xmms_playing = FALSE;

	displayer_sdl_status.sfc = SINGIT_FRAMERATE_COUNTER
		(singit_framerate_counter_new(15));
	singit_framerate_counter_set_ticktype(displayer_sdl_status.sfc, SDL_TICKS);
	singit_framerate_counter_start(displayer_sdl_status.sfc);

	set_sdl_render_pthread();
}

static void displayer_sdl_finish(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_finish]\n"));
#	endif

	displayer_sdl_status.shutdown = TRUE;

	sdl_render_pthread_quit();

	if (displayer_sdl_status.sdl_inuse == FALSE) {
		sdl_skd_finish();
		sdl_lyrics_finish();
	}

	displayer_sdl_status_finish();

	if (displayer_sdl_config != NULL)
		singit_config_gen_detach(SINGIT_CONFIG_GEN(displayer_sdl_config));
}

static void displayer_sdl_show(SingitSong *cur_song)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_show]\n"));
#	endif

//	if (!xmms_remote_is_paused(sdl_dp.xmms_session) &&
//		(set_sdl_render_pthread()))
	if (set_sdl_render_pthread() == TRUE)
	{
		sdl_helper_map(TRUE);
	}
}

static void displayer_sdl_hide(SingitSong *cur_song)
{
	gboolean hide;

#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_hide]\n"));
#	endif

	hide = ((GET_DSCD(displayer_sdl_config)->hideIfNotFound && !displayer_sdl_status.is_fullscreen) ||
		(GET_DSCD(displayer_sdl_config)->hideIfNotFoundFullscreen && displayer_sdl_status.is_fullscreen));

	if (hide) { sdl_helper_map(FALSE); }
	set_sdl_render_pthread();
}

static void displayer_sdl_toggle(SingitSong *cur_song)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_toggle]\n"));
#	endif
}

static void displayer_sdl_set_time(guint time, SingitSong *cur_song, GList *real_next)
{
	SingitSong *my_song;

	my_song = singit_song_attach(cur_song);

	if (my_song != displayer_sdl_status.song)
	{
		singit_song_detach(&displayer_sdl_status.song);
		displayer_sdl_status.song = singit_song_attach(my_song);

		displayer_sdl_status.singit_playing =
			singit_song_lyrics_found(my_song);

		displayer_sdl_status.xmms_playing =
			xmms_remote_is_playing(sdl_dp.xmms_session);

		sdl_lyrics_set_song(my_song);
	}

	set_sdl_render_pthread();

	if (displayer_sdl_status.screen == SDL_GetVideoSurface())
	{
		singit_karaoke_data_set_time(displayer_sdl_status.skd, time);
		displayer_sdl_status.time = time;
		if (displayer_sdl_status.render_thread == 0)
			{ displayer_sdl_process_events(time); }
	}

	singit_song_detach(&my_song);
}

static void displayer_sdl_playback_start(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_playback_start]\n"));
#	endif

	displayer_sdl_status.xmms_playing = TRUE;
}

static void displayer_sdl_playback_stop(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_playback_stop]\n"));
#	endif

	displayer_sdl_status.xmms_playing = FALSE;
	set_sdl_render_pthread();
}

static void displayer_sdl_configure(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_configure]\n"));
#	endif

	if (!displayer_sdl_status.is_fullscreen) {
		displayer_sdl_config_show();
	}
}

static void displayer_sdl_config_update_event(SingitConfigGen *scg, SDLConfigData *cfg_data, gpointer user_data)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_config_update_event]\n"));
#	endif

	if (!((cfg_data->lyricFontName != NULL) &&
		sdl_lyrics_set_font(cfg_data->lyricFontName, cfg_data->lyricFontSize, FALSE)))
	{
		sdl_lyrics_set_font(SINGIT_DATA_DIR "/vixar.ttf", 32, FALSE);
	}

	sdl_lyrics_set_optimize_font(cfg_data->optimizeFontSize);
}

static void displayer_sdl_config_update(const SingitConfigData *singit_config)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_config_update]\n"));
	#endif

	displayer_sdl_status.toggle_yz = singit_config->toggleYZ;

	if (displayer_sdl_status.render_thread != 0)
		singit_framerate_counter_set_limit(displayer_sdl_status.sfc,
			singit_config->framerateLimit);
}
