/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <pthread.h>
#include <gtk/gtk.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#ifdef HAVE_SCHED_SETSCHEDULER
#include <sched.h>
#endif

#include "singit_displayer_plugin.h"
#include "singit_debug.h"
#include "singit_config_gen.h"
#include "singit_config.h"
#include "singit_config_private.h"
#include "singit_song.h"
#include "singit_song_private.h"
#include "dlg_singit_config.h"

#include "singit_main.h"
#include "singit_plugin_scanner.h"
#include "editor_singit_main.h"
#include "dlg_singit_about.h"

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
#include "singit_tag_manager.h"
#endif

#if ((defined CODEDEBUG) && (defined USE_MTRACE))
#include <mcheck.h>
#endif

extern VisPlugin singit_vp;
extern SingitConfigGen *singit_config;

static void singit_main_set_timeout(void);

void singit_main_init(gboolean initialize_plugins)
{
#	ifdef CODEDEBUG
	DEBUG(8, ("singit_main.c [singit_main_init]\n"));
#	endif

	singit_status.attachments++;
	if (singit_status.attachments <= 1) {

#		if ((defined CODEDEBUG) && (defined USE_MTRACE))
		g_print("singit_main.c [singit_main_init] : mtrace\n");
		mtrace();
#		endif

		singit_status.singit_sound_precalcs = g_malloc(sizeof(SingitSoundPrecalcs));

		pthread_mutex_init(&singit_status.config_rw_mutex, NULL);
		pthread_mutex_init(&singit_status.lyrics_rw_mutex, NULL);

#		ifdef ENABLE_NLS
		setlocale (LC_ALL, "");
		bindtextdomain (PACKAGE, LOCALE_DIR);
#		endif

		if (!singit_config_gen_attach(singit_config)) {
			singit_config_new();
			singit_config_gen_load(singit_config);
		}

		plugins_init();

		singit_status.singit_framerate_counter =
			SINGIT_FRAMERATE_COUNTER
			(singit_framerate_counter_new(15));
		singit_framerate_counter_set_ticktype
			(singit_status.singit_framerate_counter, GLIB_TICKS);
	}

	if (initialize_plugins) {
		plugins_initialize();
		singit_status.initialize_plugins = TRUE;
	}
	singit_main_set_timeout();
}

void singit_main_finish(gboolean finish_plugins)
{
#	ifdef CODEDEBUG
	DEBUG(8, ("singit_main.c [singit_main_finish]\n"));
#	endif

	singit_status.attachments--;
	if (singit_status.attachments > 0) {
		if (finish_plugins) {
			plugins_finalize();
		}
		singit_main_set_timeout();
		return;
	}

	if (singit_status.display_thread || singit_status.check_thread) {

		singit_status.kill_threads = TRUE;

		pthread_join(singit_status.check_thread, NULL);
		pthread_join(singit_status.display_thread, NULL);

		singit_status.kill_threads = FALSE;
	}

	plugins_finish();

	singit_config_hide();
	singit_about_hide();
#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
	singit_tag_manager_hide();
#endif
	singit_editor_hide();

	singit_config_gen_detach(SINGIT_CONFIG_GEN(singit_config));

	singit_main_status_finish();

#	if ((defined CODEDEBUG) && (defined USE_MTRACE))
	g_print("singit_main.c [singit_main_finish] : muntrace\n");
	muntrace();
#	endif
}

static void singit_main_set_timeout(void)
{
#	ifdef CODEDEBUG
	DEBUG(5, ("singit_main.c [singit_main_set_timeout]: "));
#	endif

	if (!is_dis_plugin_running(-1) && singit_status.display_thread &&
		!singit_editor_is_realized()) {

#	ifdef CODEDEBUG
		DEBUG(5, ("finish\n"));
#	endif
		singit_status.kill_threads = TRUE;

		pthread_join(singit_status.check_thread, NULL);
		pthread_join(singit_status.display_thread, NULL);

		singit_status.kill_threads = FALSE;
		singit_status.next_lyric_line = NULL;
	}
	else {
		if ((is_dis_plugin_running(-1) || singit_editor_is_realized()) &&
			!singit_status.display_thread)
		{
#		ifdef CODEDEBUG
			DEBUG(5, ("start\n"));
#		endif
			if (pthread_create(&singit_status.check_thread, NULL,
				check_thread_func, NULL))
			{
				singit_status.check_thread = 0;
				g_print("Unable to create check thread\n");
			}

			if (pthread_create(&singit_status.display_thread, NULL,
				display_thread_func, NULL))
			{
				singit_status.display_thread = 0;
				g_print("Unable to create display thread\n");
			}
		}
#	ifdef CODEDEBUG
		else {
			DEBUG(5, ("unchanged\n"));
		}
#	endif
	}
}

// * This is the function to  start a plugin from an other plugin
// * Because of the locking it has to run as a GTK idle loop
// * Otherwise key handling inside the *set_time call would lock the plugin
// * (It took me one afternoon to fix it - so I'm quite sure about it ;-(
static gint enable_plugin_real(gpointer data)
{
#	ifdef CODEDEBUG
	DEBUG(8, ("singit_main.c [enable_plugin_real]\n"));
#	endif

	GDK_THREADS_ENTER();
	set_dis_plugin_status(GPOINTER_TO_INT(data), TRUE);
	config_dis_plugins_rescan();
	GDK_THREADS_LEAVE();

	return FALSE;
}

void enable_plugin(gint plugin)
{
#	ifdef CODEDEBUG
	DEBUG(8, ("singit_main.c [enable_plugin]\n"));
#	endif

	gtk_idle_add(enable_plugin_real, GINT_TO_POINTER(plugin));
}
