/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>

#include "singit_debug.h"

#include "singit_song.h"
#include "singit_song_private.h"

#include "singit_tools.h"
#include "singit_file_info.h"

const gint tag_length[] = { 7, 7, 11, 10 };

static inline gint get_timetag_type(gchar* tag)
{
	if ((tag[0] == '[') && (isdigit(tag[1])) && (isdigit(tag[2])) &&
		(tag[3] == ':') && (isdigit(tag[4])) && (isdigit(tag[5]))) {
		if (tag[6] == ']') { return 1; }
		if ((tag[6] == ':') && (isdigit(tag[7])) && 
			(isdigit(tag[8])) && (isdigit(tag[9])) &&
			(tag[10] == ']'))
		{ 
			return 2; 
		}
		if ((tag[6] == '.') && (isdigit(tag[7])) && 
			(isdigit(tag[8])) && (tag[9] == ']'))
		{
			return 3;
		}
	}
	return 0;
}

gint singit_song_get_timetag_type(gchar* tag)
{
	guint length = strlen(tag);
	if (length < 7) { return 0; }
	if ((tag[6] == ':') && (length < 11)) { return 0; }
	if ((tag[6] == '.') && (length < 10)) { return 0; }
	return get_timetag_type(tag);
}

inline gboolean extrakt_timetag_information(gchar* tag, guint* time)
{
	gboolean result = TRUE;

	switch (get_timetag_type(tag)) {
	case 1:
		if (time != NULL) {
			tag[3] = '\0';
			tag[6] = '\0';
			*time = (atoi(&tag[1]) * 60 + atoi(&tag[4])) * 1000;
			tag[3] = ':';
			tag[6] = ']';
		}
		break;
	case 2:
		if (time != NULL) {
			tag[3] = '\0';
			tag[6] = '\0';
			tag[10] = '\0';
			*time = (atoi(&tag[1]) * 60 + atoi(&tag[4])) * 1000 + atoi(&tag[7]);
			tag[3] = ':';
			tag[6] = ':';
			tag[10] = ']';
		}
		break;
	case 3:
		if (time != NULL) {
			tag[3] = '\0';
			tag[6] = '\0';
			tag[9] = '\0';
			*time = (atoi(&tag[1]) * 60 + atoi(&tag[4])) * 1000 + atoi(&tag[7]) * 10;
			tag[3] = ':';
			tag[6] = '.';
			tag[9] = ']';
		}
		break;
	default:
		result = FALSE;
		break;
	}

	return result;
}

static inline gchar* extract_token(SingitSong* song, gchar* line, guint line_nr)
{
	const gchar old_tags[4][5] = {
		"[ar:",
		"[ti:",
		"[la:",
		"[by:"
	};
	gchar *start_tag, *end_tag = line;
	gint copy_chars;

	LToken *token = NULL;
	gint time = 0, outlen = 0, tokenlen = 0;
	guint stringlen;
	gchar *new_line = NULL, *new_pos = NULL;

	stringlen = strlen(line);
	if (stringlen <= 0) { return g_strdup(line); }

	start_tag = strstr(line, "[");
	if (start_tag == NULL) { return g_strdup(line); }

	new_pos = new_line = g_new(gchar, stringlen + 1);

	while (start_tag != NULL)
	{
		if (end_tag < start_tag)
		{
			copy_chars = start_tag - end_tag;
			if (copy_chars > 0) {
				memcpy(new_pos, end_tag, copy_chars);
				new_pos += copy_chars;
			}
		}

		end_tag = strstr(start_tag, "]");
		// We have no end tag -> break the loop
		if (end_tag == NULL) 
			{ break; }

		tokenlen = end_tag - start_tag;
		switch (tokenlen) {
		case 6:
		case 9:
		case 10:
			if (extrakt_timetag_information(start_tag, &time) == TRUE) {
#ifdef CODEDEBUG
				DEBUG(8, ("."));
#endif
				tokenlen++;
				outlen += tokenlen;

				token = g_malloc(sizeof(LToken));
				token->time = time;
				token->line = line_nr;
				token->pos = start_tag + tokenlen - line - outlen;

				song->first_token = g_list_append(song->first_token, token);
				end_tag++;
				start_tag = strstr(end_tag, "[");
				break;
			}
		default:
			if ((end_tag - start_tag > 5) && (start_tag == line)) {
				tokenlen = 0;
				while (tokenlen < 4) {
					if (strncmp(&old_tags[tokenlen][0], start_tag, 4) == 0) {
#ifdef CODEDEBUG
						DEBUG(8, ("o"));
#endif
						// switch (time) {
						//default:
						//}
						tokenlen = 4;
						return NULL;
					}
					tokenlen++;
				}
			}
			end_tag++;
			copy_chars = end_tag - start_tag;
			memcpy(new_pos, start_tag, copy_chars);
			new_pos += copy_chars;
			start_tag = strstr(end_tag, "[");
			break;
		}
	}

	//If we have no end tag copy from the start one to the end of the string
	if (end_tag == NULL) {
		copy_chars = strlen(start_tag);
		memcpy(new_pos, start_tag, copy_chars);
		new_pos += copy_chars;
	}
	else if (end_tag[0] != '\0') {
		copy_chars = strlen(end_tag);
		memcpy(new_pos, end_tag, copy_chars);
		new_pos += copy_chars;
	}
	new_pos[0] = '\0';

	return new_line;
}

gboolean singit_song_read_text_stream(SingitSong *song, gchar *buffer)
{
	GSList *string_list = NULL, *slist;

  	gchar *s;
  	guint n = 0;
	const gchar *delimiter = "\n";
	guint len;
	gchar *new_string, *line;
	guint delimiter_len = -1;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_song.c [singit_song_read_text_stream]\n"));
	DEBUG(8, ("        'l' = line / '.' = tag / 'o' = old special tag\n"));
#endif

	if ((buffer == NULL) || (song == NULL)) 
		{ return FALSE; }

  	s = strstr (buffer, delimiter);
  	if (s) {
      		delimiter_len = strlen (delimiter);
		len = s - buffer;
		if (buffer[len-1] == 13) {
			song->delimiter = g_strdup("    ");
			song->delimiter[0] = 13;
			song->delimiter[1] = '\n';
			song->delimiter[2] = '\0';
		}
		do {
#ifdef CODEDEBUG
			DEBUG(8, ("l"));
#endif
			len = s - buffer;
			if (buffer[len-1] == 13) { len--; }
			new_string = g_new (gchar, len + 1);
			strncpy (new_string, buffer, len);
			new_string[len] = '\0';
			line = extract_token(song, g_strstrip(new_string), n);
			if (line != NULL) {
				string_list = g_slist_prepend(string_list, line);
				n++;
			}
			g_free(new_string);
			buffer = s + delimiter_len;
			s = strstr (buffer, delimiter);
		}
      		while (s);
	}
	if (*buffer) {
		len = strlen(buffer);
#ifdef CODEDEBUG
		DEBUG(8, ("l"));
#endif
		if (buffer[len-1] == 13) {
			len--;
			new_string = g_new (gchar, len + 1);
			strncpy (new_string, buffer, len);
			new_string[len] = 0;
			string_list = g_slist_prepend
				(string_list, g_strdup(extract_token(song, new_string, n)));
			g_free(new_string);
		}
		else
			{ string_list = g_slist_prepend (string_list, g_strdup (extract_token(song, buffer, n))); }
		n++;
	}

	song->lyric_lines = n;
	song->lyrics = g_new (gchar*, n+1);

	song->lyrics[n--] = NULL;
	for (slist = string_list; slist; slist = slist->next)
		song->lyrics[n--] = slist->data;
	g_slist_free (string_list);

	if (song->first_token != NULL) {
		song->first_token = g_list_sort(song->first_token, compare_token_by_time);
		song->first_token = g_list_first(song->first_token);
		song->last_token = g_list_last(song->first_token);

#ifdef CODEDEBUG
		DEBUG(8, ("\nTagTime-Gap: %.2i:%.2i - %.2i:%.2i\n", tTime(song->first_token) / 60000,
			tTime(song->first_token) % 60000 / 1000, tTime(song->last_token) / 60000,
			tTime(song->last_token) % 60000 / 1000));
#endif
	}
	else
		{ song->active_token = song->last_token = song->first_token; }

	song->lyric_type = LT_TEXT;

#ifdef CODEDEBUG
	DEBUG(8, ("\n"));
#endif
	return TRUE;
}

gboolean singit_song_write_text_stream(SingitSong *song, gchar **buffer, gint tag_type)
{
	gchar **stream, *tmp = NULL, *connected = NULL;
	GList *item;
	gchar timeToken[12];
	gint i, offset = 0, last_line = -1;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_song.c [singit_song_write_text_stream]\n"));
#endif

	if (!song || !buffer) { return FALSE; }
	if (!singit_song_text_found(song)) { return FALSE; }

	timeToken[7] = '\0';
	timeToken[11] = '\0';
	stream = g_new(gchar*, song->lyric_lines + 1);
	stream[song->lyric_lines] = NULL;
	for (i = 0; i < (gint) song->lyric_lines; i++)
		{ stream[i] = g_strdup(song->lyrics[i]); }
	item = song->first_token;

	if ((tag_type != 3) && (tag_type != 2))
		{ tag_type = 1; }
		
	while (item) {
		switch (tag_type) {
		case 3:
			sprintf(timeToken, "[%.2i:%.2i.%.2i]", tTime(item) / 60000, 
				(tTime(item) / 1000) % 60, (tTime(item) % 1000) / 10);
			break;
		case 2:
			sprintf(timeToken, "[%.2i:%.2i:%.3i]", tTime(item) / 60000, 
				(tTime(item) / 1000) % 60, tTime(item) % 1000);
			break;
		default:
			sprintf(timeToken, "[%.2i:%.2i]", tTime(item) / 60000, 
				(tTime(item) / 1000) % 60);
			break;
		}
		if ((gint) tLine(item) != last_line) { offset = 0; }
		else { offset += tag_length[tag_type]; }

		last_line = tLine(item);
		tmp = stream[tLine(item)];
		connected = tools_insert_string
			(stream[tLine(item)], &timeToken[0], tPos(item) + offset);
		if (connected) {
			stream[tLine(item)] = connected;
			g_free(tmp);
		}
		item = g_list_next(item);
	}

	*buffer = g_strjoinv("\n", stream);
	return TRUE;
}

gboolean singit_song_save_to_text_file(SingitSong *song, gchar *filename, gint tag_type)
{
	FILE *file;
        gchar *buffer, *useFilename;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_song_text.c [singit_song_save_to_text_file]\n"));
#endif

	if (!song) { return FALSE; }
	if (!singit_song_text_found(song)) { return FALSE; }

	if (filename) { useFilename = filename; }
	else { useFilename = song->lyric_filename; }

        if (!(file = fopen(useFilename, "w"))) { return FALSE; }

        singit_song_write_text_stream(song, &buffer, tag_type);
        if (fwrite(buffer, 1, strlen(buffer), file) != strlen(buffer))
        {
                g_free(buffer);
                fclose(file);
                return FALSE;
        }
        fclose(file);
	g_free(buffer);

	if (!song->lyric_filename) { song->lyric_filename = g_strdup(filename); }

	return TRUE;
}

gboolean singit_song_load_from_text_file(SingitSong *song, gchar *filename)
{
	FILE *file;
        gchar *buffer;
        struct stat stats;
	gboolean result;
	SingitSong *my_song;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_song_text.c [singit_song_load_from_text_file] : "));
#endif

	my_song = singit_song_attach(song);
	if (my_song == NULL) 
		{ return FALSE; }

	if ((stat(filename, &stats) == -1) ||
		((file = fopen(filename, "r")) == NULL))
	{
		singit_song_detach(&my_song);
#ifdef CODEDEBUG
		DEBUG(8, ("Unable to read file\n"));
#endif
		return FALSE;
	}

	if ((stats.st_size > 100000) || (stats.st_size < 1)) {
		fclose(file);
		singit_song_detach(&my_song);
#ifdef CODEDEBUG
		DEBUG(8, ("Wrong size (%i)\n", (guint) stats.st_size));
#endif
		return FALSE;
	}

	buffer = g_malloc(stats.st_size + 1);
	if (fread(buffer, 1, stats.st_size, file) != (guint) stats.st_size)
	{
		g_free(buffer);
		fclose(file);
		singit_song_detach(&my_song);
#ifdef CODEDEBUG
		DEBUG(8, ("Buffered read failed\n"));
#endif
		return FALSE;
	}
	fclose(file);
	buffer[stats.st_size] = '\0';

	singit_file_info_reset((SingitFileInfo*) my_song->file_info, TRUE);

	result = singit_song_read_text_stream(my_song, buffer);
	if (result) {
		my_song->lyric_filename = g_strdup(filename);
		my_song->lyric_type = LT_TEXT;
	}
	else { my_song->lyric_type = LT_NONE; }

	g_free(buffer);

	singit_song_detach(&my_song);
	return result;
}
