/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include <ogg/ogg.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>

#include "singit_debug.h"
#include "singit_config.h"
#include "singit_config_gen.h"
#include "singit_macros.h"
#include "singit_config_private.h"
#include "singit_tools.h"

//#include "list_genres.h"
//#include "list_languages.h"
#include "singit_tag_manager_private.h"

extern SingitConfigGen *singit_config;

static GtkWidget *oggvorbis_file_info_values_label;

static void fill_oggvorbis_info_page(OggVorbis_File *vf)
{
	vorbis_info *vi;
	vorbis_comment *comment = NULL;
	gchar *value_str;
	FILE *fh;

	gint time, minutes, seconds, bitrate, rate, channels, filesize;

	if (oggvorbis_file_info_values_label == NULL)
		{ return; }

	comment = ov_comment(vf, -1);
	if ((vi = ov_info(vf, 0)) != NULL)
	{
		bitrate = vi->bitrate_nominal/1000;
		rate = vi->rate;
		channels = vi->channels;
	}
	else
	{
		bitrate = 0;
		rate = 0;
		channels = 0;
	}

	time = ov_time_total(vf, -1);
	minutes = time / 60;
	seconds = time % 60;
	fh = (FILE*) (vf->datasource);
	fseek(fh, 0, SEEK_END);
	filesize = ftell(fh);

	value_str = g_strdup_printf
		("\n%d\n\n%d kbps\n%d Hz\n%d\n%d:%.2d\n\n%d B",
		vi->version, bitrate, rate, channels, minutes,
		seconds, filesize);

	gtk_label_set_text(GTK_LABEL(oggvorbis_file_info_values_label), value_str);
	g_free(value_str);
}

OggVorbis_File* get_oggvorbis_file_info(gchar* file_name, gboolean *is_writeable)
{
	FILE *fh;
	OggVorbis_File *vf;
	gboolean can_write;

	g_return_val_if_fail(file_name != NULL, NULL);

	// Mode 'b' is ignored on most systems, but some may require binary access
	if ((fh = fopen(file_name, "r+b")) == NULL) {
		if ((fh = fopen(file_name, "rb")) == NULL)
			{ return NULL; }
		else { can_write = FALSE; }
	}
	else { can_write = TRUE; }

	vf = g_malloc(sizeof(OggVorbis_File));
	if (ov_open(fh, vf, NULL, 0) != 0) {
		fclose(fh);
		g_free(vf);
		return NULL;
	}

	if (is_writeable != NULL)
		{ *is_writeable = can_write; }

	// once the ov_open succeeds, the stream belongs to vorbisfile.a.
	// ov_clear will also "fclose" it !!!!
	return vf;
}


void append_oggvorbis_info_page
	(GtkWidget *notebook, GtkWidget *window, OggVorbis_File *vf)
{
	GtkWidget *oggvorbis_file_info_table, *oggvorbis_file_info_label,
		*oggvorbis_file_info_properties_label;

	gchar *cat_str = NULL;

	g_return_if_fail(notebook != NULL);
	g_return_if_fail(window != NULL);
	g_return_if_fail(vf != NULL);

	g_return_if_fail(GTK_IS_NOTEBOOK(notebook));

	oggvorbis_file_info_table = gtk_table_new (1, 2, FALSE);
	create_std_wgt_description(oggvorbis_file_info_table,
		"oggvorbis_file_info_table", TRUE, window);
	gtk_container_set_border_width(GTK_CONTAINER(oggvorbis_file_info_table), 1);
	gtk_table_set_row_spacings(GTK_TABLE(oggvorbis_file_info_table), 3);
	gtk_table_set_col_spacings(GTK_TABLE(oggvorbis_file_info_table), 2);

	oggvorbis_file_info_label = gtk_label_new (_("ogg vorbis file info"));
	create_std_wgt_description(oggvorbis_file_info_label,
		"oggvorbis_file_info_label", TRUE, window);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
		oggvorbis_file_info_table, oggvorbis_file_info_label);

	cat_str = g_strconcat("Ogg Vorbis\n    ", _("Version"), ":\n\n",
		_("Nominal bit rate"), ":\n", _("Sample rate"), ":\n",
		_("Channels"), ":\n", _("Length"), ":\n\n", _("Filesize"),
		":",  NULL);

	oggvorbis_file_info_properties_label = gtk_label_new(cat_str);
	gtk_label_set_justify(GTK_LABEL(oggvorbis_file_info_properties_label), GTK_JUSTIFY_LEFT);
	g_free(cat_str);
	create_std_wgt_description(oggvorbis_file_info_properties_label,
		"oggvorbis_file_info_properties_label", TRUE, window);
	gtk_table_attach(GTK_TABLE (oggvorbis_file_info_table),
		oggvorbis_file_info_properties_label, 0, 1, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 2);

	oggvorbis_file_info_values_label = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(oggvorbis_file_info_values_label), GTK_JUSTIFY_LEFT);
	create_std_wgt_description(oggvorbis_file_info_values_label,
		"oggvorbis_file_info_values_label", TRUE, window);
	gtk_table_attach(GTK_TABLE (oggvorbis_file_info_table),
		oggvorbis_file_info_values_label, 1, 2, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 2);

	fill_oggvorbis_info_page(vf);
}
