#include "math.h"
#include "resample.h"

/* periodic symmetric extension of src */
#define REFOLD(J,L)   { if(J<0) J =-J; J = J % (2*L-1); if(J>=L) J=(2*L-1)-J;}
/* border extension of src */
#define BORD(J,L)   { if(J<0) J=0; else if (J>L) J=L; }
/* zero extension of src */
#define ZERO(J,L)   { if(J<0 || J>L) continue; }

/* enforce boundary condition <-> array bounds */
//#define BC(J,L) REFOLD(J,L)



#undef  MAX
#define MAX(a, b)  (((a) > (b)) ? (a) : (b))

#undef  MIN
#define MIN(a, b)  (((a) < (b)) ? (a) : (b))

#undef  ABS
#define ABS(a)     (((a) < 0) ? -(a) : (a))

#undef  CLAMP
#define CLAMP(x, low, high)  (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))


#include "stdio.h"
#include "math.h"


/* these lines should help if you are using  braindead MS C compiler */
#ifdef MSVC 
#define inline 
#define static 
#warning static and inline are disabled 
#endif

// WARNING this is not ANTIALIASED in any way but is very simple and fast

static inline
void
resample_array_inv_noaa_
  (const double *F, 
   const PIXEL_TYPE *src, int s_len, int s_stride, 
         PIXEL_TYPE *dst, int d_len, int d_stride)
{
  int i,p=0,j;
  for(i=0;i<d_len;i++) {
    j=F[i];
    BORD(j,s_len)
    dst[p]=src[j* s_stride];
    p+= d_stride;
  }
}


/*********** antialiased using bilinear interpixel**************
 * works well when the scale is not reduced and/or when there are not
 * very fine grains; it is much faster; it is approximate in many points
 * For animations it is better to use antialiased
 */

static
void
resample_array_inv_bilin
  (const double *F, 
   const PIXEL_TYPE *src, int s_len, int s_stride, 
         PIXEL_TYPE *dst, int d_len, int d_stride)
{
  int i,p=0,j,nj,oj;
  double v;
  oj=floor(F[0]);
  BORD(oj,s_len);
  for(i=0;i<d_len;i++) {
    v=0;
    j=floor(F[i]);
    if(j>=0 && j <s_len) {
#ifdef SEEMS_UNNEEDED
      int c;
      /* I have tested the code and this seems to introduce ripples;
      * I don't want to improve it, use AA if you really want AA*/
      if( oj>=j+2) {
	c=oj-j+1;
	if(oj>=s_len) oj=s_len-1;
	while (oj>=j) { v+=src[oj* s_stride]; oj--;  }
	v/=c;
      }else 
	if ( oj+2<=j) {
	  c=j-oj+1;
	  if(oj<0) oj=0;
	  while (oj<=j) { v+=src[oj* s_stride]; oj++;  }
	  v/=c;
	} else 
#endif
	  {      //if( ox-x<=1 && x-ox<=1) 
	    double dj=F[i]-j;
	    nj=j+1;
	    if(nj>=s_len) nj=s_len-1;
	    v=src[j* s_stride]*(1-dj) + src[nj* s_stride]*dj;
	  }
    }
    dst[p]=v;
    p+= d_stride;
    oj=j;
  }
}
/*********** antialiased using convolutional kernels***************/

#if 256 >= (PIXEL_MAX-PIXEL_MIN)
#include "sinc_256.h"
#else
#include "sinc_1024.h"
#endif


static double sinc(const double x)
{
  if (x<0.0001 && x > -0.0001)
    return 1;
  else
    {
      double tmp = M_PI * x  ;
      return sin(tmp) / tmp;
    }
}

static double sinc_by_table(const double x)
{
  if (x < -4 || x > 4) {
    double tmp = M_PI * x  ;
    return sin(tmp) / tmp;
  }  else
    return sinc_table [(int)((ABS(x))*SINC_TABLE_UNIT )];
}


static double lanczos(const double x)
{
  if (x < -2 || x > 2)
    return 0;
  else
    return sinc(x);
}

static double lanczos4(const double x)
{
  if (x < -4 || x > 4)
    return 0;
  else
    return sinc(x);
}

static double triangle(const double x)
{
  if (x<-1) return 0;
  else if(x<0) return x+1;
  else if(x<1) return 1-x;
  else return 0;
}


/*************  choice of antialiasing kernel **************/
/* optimizations  for the case of any kernel  based on sinc
   suggested by lvalero, oberger 05/05/2004 */
#define KERNEL_sinc_fast

/* if the above is undefined, then this will be used instead */
#define KERNEL lanczos

/* this is the half of the width of the kernel 
 for lanczos4, it must be 4, otherwise 2
*/
#define KERNEL_WIDTH 2
/***********end  choice of antialiasing kernel **************/



#if defined(KERNEL_sinc_fast) && ( SINC_TABLE_UNIT  < (PIXEL_MAX-PIXEL_MIN) )
#warning the kernel is based on the sinc(); the function sinc is replaced by a table; but the table is not precise enough for your type of pixels
#endif





/*TABLE HYPER  PRECISION 
adds TYH bits of precision to table indexes without needing a 
table that big, and without overusing 'double' variables;
does not add any visible difference, but has a small computational cost
*/
#if (SINC_TABLE_UNIT*KERNEL_WIDTH) < (1<<11)
#define TYH 4
#elif (SINC_TABLE_UNIT*KERNEL_WIDTH) < (1<<13)
#define TYH 2
#else
#define TYH 0
#endif


static void resample_array_inv_conv
  (const double *F, 
   const PIXEL_TYPE *src, int s_len, int s_stride, 
         PIXEL_TYPE *dst, int d_len, int d_stride)
{
  int i,p=0,j, firstj,lastj;
  double x,px,c,s,v,dx;
#ifdef KERNEL_sinc_fast
  int index, indexbound, increment;
#endif
  px=F[1];
  for(i=0;i<d_len;i++) {
    v=0; c=0;
    x=F[i];
    //FIXED! READ OUT OF ARRAY!!!
    //nx=F[i+1];
    dx=ABS(x-px);
    if(dx<1) dx=1;

    firstj=floor(x-dx*KERNEL_WIDTH);
    lastj=ceil(x+dx*KERNEL_WIDTH);

    j=firstj;

    /* note that in the past I had  reversed the usage of the filter;
       this does not really make any difference as long as the filter is
       symmetric... but I want to be more precise     */

#ifdef KERNEL_sinc_fast
    index=((  (((double)firstj-x) / dx)  )* (double)(SINC_TABLE_UNIT << TYH));
    increment= ( (SINC_TABLE_UNIT << TYH) / dx  );
    indexbound=( (KERNEL_WIDTH*SINC_TABLE_UNIT) << TYH);
    /*roughly  
           index ~ -indexbound;
      but if it is approximated to that, then the warping loses 
      sub-pixel precision, and there are visible artifacts
    */
    // this condition is always true
    //if (index >=  -indexbound) 
    { j++;  index += increment; }
#endif

#ifndef KERNEL_sinc_fast
    /* this is the convolution with a generic kernel */
    for ( ; j<=lastj  ; j++ ) {
      s=KERNEL( ((double)j-x) / dx );
      if ( j>=0 && j<s_len)
	v+=s* (double)src[j* s_stride];
      c+=s;
    }
#else
    /* this is how it is computed very fast for a kernel based on sinc, such as lanczos */
    for ( ;  index<=0 ; ) { 
      s=sinc_table[ -(index) >> TYH]; 
      if ( j>=0 && j<s_len) 
	v+=s* (double)src[j*s_stride];
      c+=s; 
      j++;
      index+=increment;
    }
    for ( ;  index <= indexbound  ; ) {
      s=sinc_table[ (index) >> TYH];
      if ( j>=0 && j<s_len) 
	v+=s* (double)src[j*s_stride];
      c+=s; 
      j++;
      index+=increment;
    }
#endif

    if(c<0.0001 && c > -0.0001)
      fprintf(stderr,"%s:%s:%d: too small c=%f\n",__FILE__,__FUNCTION__,__LINE__,c);
    else
      v/=c;
    dst[p]=CLAMP(v,PIXEL_MIN,PIXEL_MAX);
    
    p+= d_stride;
    px=x;
  }
}




void (*resample_array_inv)(const double *F, 
   const PIXEL_TYPE *src, int s_len, int s_stride, 
   PIXEL_TYPE *dst, int d_len, int d_stride)= resample_array_inv_conv;


void
mesh_resample_use_aa(int f)
{
  if (f)
    resample_array_inv=resample_array_inv_conv;
  else
    resample_array_inv=resample_array_inv_bilin;
}







/************************ extra stuff   *************************************/

#ifdef ARTIFACTS

/* A Mennucc: this routine that follows was provided by lvalero and oberger;
it uses  integers for every double variable,
and directly accesses the sinc table; so it is very very fast;
unfortunately if x and dx are integers, then warping is not precise
to the subpixel , and there are visible artifacts

I have rewritten resample_array_inv_conv (above)
so that it directly accesses the sinc_table , and it uses integers,
but so that it is precise to subpixel

in my tests I could achieve these results

old routine     9.55 sec
new routine     3.50 sec
lvalero,oberger 2.86 sec 

note that this routine needs the old style sinc table which is not provided
any more 
*/
#include "sinc_table.h"




/* this test was provided by lvalero, oberger 05/05/2004 :*/

/*With one image 2048*2048 in 16 bits, Athlon 1.4Ghz + parhelia 128 Mo + 785 Mo RAM
 we have :
for resample_array_inv_conv (optimised) 3s
for resample_array_inv_bilin 2.5s
for resample_array_inv_noaa_ 1.4s*/     



//VERSION OPTIMISEE MODIF lvalero, oberger 05/05/2004
static void resample_array_inv_conv (const double *F, const PIXEL_TYPE *src, int s_len, int s_stride, PIXEL_TYPE *dst, int d_len, int d_stride)
{
  int i, p = 0, j, lastj ;
  int x, nx, px, dx ;
  double c, s, v ;
  int firstj,index_stride,index,increment ;
  
  for( i = 0 ; i < d_len ; ++i ) 
  {
   v=0; c=0;
   x=(int)F[i];
   nx=(int)F[i+1];
   px=(int)F[i-1];
   //dx=ABS(nx-px)/2.;
   dx=ABS(nx-px) >> 1 ;

   if(dx<1) dx=1;

   //lastj=ceil(x+dx+dx);
   lastj = x+dx+dx ;
   
   // optimisation pour v+=s* (double)src[j*s_stride]; => on economise une multiplication
   // par defaut, g met floor pour passer en int, mais ce peut etre mauvais
   //firstj=floor(x-dx-dx);
   firstj = x-dx-dx;

   //index_stride = floor(firstj*s_stride) ;
   index_stride = firstj*s_stride ;
   // en deroulant la boucle avec j = x-dx-dx on a :
   //
   // (x-j)/dx = (x-x+dx+dx)/dx = (2*dx)/dx = 2  <- etape 0
   // (x-(j+1))/dx = (x-(x+1)+dx+dx)/dx  = 2-(1/dx)  <- etape 1
   // (x-(j+2))/dx = (x-(x+2)+dx+dx)/dx  = 2-(2/dx)  <- etape 2
   // (x-(j+3))/dx = (x-(x+3)+dx+dx)/dx  = 2-(3/dx)  <- etape 3
   // ...
   // (x-lastj)/dx = (x-x-dx-dx)/dx  = -2  <- etape fin
   //
   // Donc on a pour l'etape n :
   //
   // ((2 - (n/dx)) + 4)*100 = (2+4)*100 - (n/dx)*100
   //           = (2+4)*100 - n*(1/dx)*100
   //           = (2+4)*100 - n*(100/dx)
   
   // avec le commentaire precedent, on s'apercoit que la premiere valeur est :
   //index = (2+4)*100 ;
   index = 600 ;
   // et l'increment devient :
   //increment = floor(100./dx) ;//  correspond a (1/dx)*100, mais on racle une multiplication
   increment = 100/dx ;
   
   for ( j = firstj ; j <= lastj  ; ++j )
   {
    // avec cette optimisation il convient de changer le test : laisser les operations
    // comme elle sont ecrites, le compilo fera le calcul
    //if ( index < ((-2+4)*100) /*|| index > ((2+4)*100)*/ )
    if ( index < (200) /*|| index > ((2+4)*100)*/ )
     s = 0 ;    
    else
     s=sinc_table [index];
    if ( j>=0 && j<s_len)
     v+=s* (double)src[index_stride];
     c+=s;
    
    index -= increment ;
    ++index_stride ;
   }
   if (!(c<0.0001 && c > -0.0001)) v /=c ;
   dst[p]=(PIXEL_TYPE)CLAMP(v,0,255);
   p+= d_stride;    
  }
}



#endif
