// This file is a part of the xMule Project.
//
// Copyright (c) 2004, by
//
// Carlo Wood, Run on IRC <carlo@alinoe.com>
// RSA-1024 0x624ACAD5 1997-01-26                    Sign & Encrypt
// Fingerprint16 = 32 EC A7 B6 AC DB 65 A6  F6 F6 55 DD 1C DC FF 61
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef wxString_H
#define wxString_H

#ifndef wxString
#include <string>
#include <algorithm>
#endif
#include <wx/string.h>

#ifndef wxString

//
// class wxString
//
// This is meant as a replacement for wxString.  The reason for that is
// that the current implementation of wxString is very inefficient, we
// want to use std::string instead.
//
// The class should be completely transparent however, so it should support
// all methods that wxString has and most of all support automatic conversion
// to wxString when it is used to call a libwx function that demands a
// wxString.
//

class wxString {
#if wxUSE_UNICODE
    typedef std::wstring string_type;
#else
    typedef std::string string_type;
#endif

public:
    // Standard types
    typedef string_type::traits_type traits_type;
    typedef string_type::allocator_type allocator_type;
    typedef string_type::iterator iterator;
    typedef string_type::const_iterator const_iterator;
    typedef string_type::size_type size_type;
    typedef string_type::difference_type difference_type;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
    typedef std::reverse_iterator<iterator> reverse_iterator;
    typedef allocator_type::pointer pointer;
    typedef allocator_type::const_pointer const_pointer;
    typedef allocator_type::reference reference;
    typedef allocator_type::const_reference const_reference;
    typedef allocator_type::value_type value_type;

    // npos
    static const size_type npos = static_cast<size_type>(-1);

private:
    // The underlying string.
    string_type M_string;

public:
    // First all standard constructors.
    wxString(void) { }
    explicit wxString(string_type::allocator_type const& al) : M_string(al) { }
    wxString(string_type const& right) : M_string(right) { }
    wxString(string_type const& right, size_type roff, size_type count = npos) : M_string(right, roff, count) { }
    wxString(string_type const& right, size_type roff, size_type count, allocator_type const& al) : M_string(right, roff, count, al) { }
    wxString(value_type const* ptr, size_type count) : M_string(ptr, count) { }
    wxString(value_type const* ptr, size_type count, allocator_type const& al) : M_string(ptr, count, al) { }
    wxString(value_type const* ptr) : M_string(ptr) { }
    wxString(value_type const* ptr, allocator_type const& al) : M_string(ptr, al) { }
    wxString(size_type count, value_type ch) : M_string(count, ch) { }
    wxString(size_type count, value_type ch, allocator_type const& al) : M_string(count, ch, al) { }
    template <class InIt>
        wxString(InIt first, InIt last) : M_string(first, last) { }
    template <class InIt>
        wxString(InIt first, InIt last, allocator_type const& al) : M_string(first, last, al) { }

    // Automatic conversion to a string_type reference.
    operator string_type const& () const { return M_string; }
    operator string_type& () { return M_string; }

    // Copy from a wxString
    wxString(wxString const& right) : M_string(right.GetData(), right.length()) { }

    // Standard assignment operators.
    wxString& operator=(string_type const& right) { M_string = right; return *this; }
    wxString& operator=(value_type const* ptr) { M_string = ptr; return *this; }
    wxString& operator=(value_type ch) { M_string = ch; return *this; }
    // Assignment from a wxString
    wxString& operator=(wxString const& right) { M_string.assign(right.GetData(), right.length()); return *this; }


    // Automatic conversion to wxString when we really can't do otherwise :(
    operator wxString const () const { return wxString(M_string.data(), M_string.length()); }
    operator wxString () { return wxString(M_string.data(), M_string.length()); }

    // wxString specific functions that possibly convert the resulting data.
#if wxUSE_UNICODE
    wxCharBuffer const mb_str(wxMBConv& conv = wxConvLibc) const { return conv.cWC2MB(M_string.c_str()); }
#else
    wxChar const* mb_str(void) const { return M_string.c_str(); }
    wxChar const* mb_str(wxMBConv& WXUNUSED(conv)) const { return M_string.c_str(); }
#endif

    // Then support some wxString specific functions in a more standard way.
    int Cmp(wxChar const* psz) const;
    int CmpNoCase(wxChar const* psz) const;
    bool IsSameAs(wxChar const* psz, bool compareWithCase = true) const;
    bool IsSameAs(wxChar c, bool compareWithCase = true) const;
    // And some extra support for direct wxString's (since we don't automatically convert to wxChar*).
    int CmpNoCase(wxString const& psz) const;

    int PrintfV(wxChar const* pszFormat, va_list argptr);
    int Printf(wxChar const* pszFormat, ...);
    static wxString FormatV(wxChar const* pszFormat, va_list argptr);
    static wxString Format(wxChar const* pszFormat, ...);

    friend std::ostream& operator<<(std::ostream& os, wxString const& str)
	{ return os << str.M_string; }

public:
    // Conversion of standard methods.
    iterator begin() { return M_string.begin(); }
    const_iterator begin() const { return M_string.begin(); }
    iterator end() { return M_string.end(); }
    const_iterator end() const { return M_string.end(); }
    reverse_iterator rbegin() { return M_string.rbegin(); }
    const_reverse_iterator rbegin() const { return M_string.rbegin(); }
    reverse_iterator rend() { return M_string.rend(); }
    const_reverse_iterator rend() const { return M_string.rend(); }
    const_reference at(size_type off) const { return M_string.at(off); }
    reference at(size_type off) { return M_string.at(off); }
    const_reference operator[](size_type off) const { return M_string.operator[](off); }
    reference operator[](size_type off) { return M_string.operator[](off); }
    void push_back(value_type ch) { M_string.push_back(ch); }
    const value_type *c_str() const { return M_string.c_str(); }
    const value_type *data() const { return M_string.data(); }
    size_type length() const { return M_string.length(); }
    size_type size() const { return M_string.size(); }
    size_type max_size() const { return M_string.max_size(); }
    void resize(size_type newsize, value_type ch = value_type()) { M_string.resize(newsize, ch); }
    size_type capacity() const { return M_string.capacity(); }
    void reserve(size_type count = 0) { M_string.reserve(count); }
    bool empty() const { return M_string.empty(); }
    wxString& operator+=(const string_type& right) { M_string.operator+=(right); return *this; }
    wxString& operator+=(const value_type *ptr) { M_string.operator+=(ptr); return *this; }
    wxString& operator+=(value_type ch) { M_string.operator+=(ch); return *this; }
    wxString& append(const string_type& right) { M_string.append(right); return *this; }
    wxString& append(const string_type& right, size_type roff, size_type count) { M_string.append(right,roff, count); return *this; }
    wxString& append(const value_type *ptr, size_type count) { M_string.append(ptr, count); return *this; }
    wxString& append(const value_type *ptr) { M_string.append(ptr); return *this; }
    wxString& append(size_type count, value_type ch) { M_string.append(count, ch); return *this; }
    template<class InIt>
        wxString& append(InIt first, InIt last) { M_string.append(first, last); return *this; }
    wxString& assign(const string_type& right) { M_string.assign(right); return *this; }
    wxString& assign(const string_type& right, size_type roff, size_type count) { M_string.assign(right, roff, count); return *this; }
    wxString& assign(const value_type *ptr, size_type count) { M_string.assign(ptr, count); return *this; }
    wxString& assign(const value_type *ptr) { M_string.assign(ptr); return *this; }
    wxString& assign(size_type count, value_type ch) { M_string.assign(count, ch); return *this; }
    template<class InIt>
        wxString& assign(InIt first, InIt last) { M_string.assign(first, last); return *this; }
    wxString& insert(size_type off, const string_type& right) { M_string.insert(off, right); return *this; }
    wxString& insert(size_type off, const string_type& right, size_type roff, size_type count) { M_string.insert(off, right, roff, count); return *this; }
    wxString& insert(size_type off, const value_type *ptr, size_type count) { M_string.insert(off, ptr, count); return *this; }
    wxString& insert(size_type off, const value_type *ptr) { M_string.insert(off, ptr); return *this; }
    wxString& insert(size_type off, size_type count, value_type ch) { M_string.insert(off, count, ch); return *this; }
    iterator insert(iterator where, value_type ch = value_type()) { return M_string.insert(where, ch); }
    void insert(iterator where, size_type count, value_type ch) { M_string.insert(where, count, ch); }
    template<class InIt>
        void insert(iterator where, InIt first, InIt last) { M_string.insert(where, first, last); }
    wxString& erase(size_type off = 0, size_type count = npos) { M_string.erase(off, count); return *this; }
    iterator erase(iterator where) { return M_string.erase(where); }
    iterator erase(iterator first, iterator last) { return M_string.erase(first, last); }
    void clear() { M_string.clear(); }
    wxString& replace(size_type off, size_type n0, const string_type& right) { M_string.replace(off, n0, right); return *this; }
    wxString& replace(size_type off, size_type n0, const string_type& right, size_type roff, size_type count) { M_string.replace(off, n0, right, roff, count); return *this; }
    wxString& replace(size_type off, size_type n0, const value_type *ptr, size_type count) { M_string.replace(off, n0, ptr, count); return *this; }
    wxString& replace(size_type off, size_type n0, const value_type *ptr) { M_string.replace(off, n0, ptr); return *this; }
    wxString& replace(size_type off, size_type n0, size_type count, value_type ch) { M_string.replace(off, n0, count, ch); return *this; }
    wxString& replace(iterator first, iterator last, const string_type& right) { M_string.replace(first, last, right); return *this; }
    wxString& replace(iterator first, iterator last, const value_type *ptr, size_type count) { M_string.replace(first, last, ptr, count); return *this; }
    wxString& replace(iterator first, iterator last, const value_type *ptr) { M_string.replace(first, last, ptr); return *this; }
    wxString& replace(iterator first, iterator last, size_type count, value_type ch) { M_string.replace(first, last, count, ch); return *this; }
    template<class InIt>
        wxString& replace(iterator first, iterator last, InIt first2, InIt last2) { M_string.replace(first, last, first2, last2); return *this; }
    size_type copy(value_type *ptr, size_type count, size_type off = 0) const { return M_string.copy(ptr, count, off); }
    void swap(string_type& right) { M_string.swap(right); }

    size_type find(const string_type& right, size_type off = 0) const { return M_string.find(right, off); }
    size_type find(const value_type *ptr, size_type off, size_type count) const { return M_string.find(ptr, off, count); }
    size_type find(const value_type *ptr, size_type off = 0) const { return M_string.find(ptr, off); }
    size_type find(value_type ch, size_type off = 0) const { return M_string.find(ch, off); }
    size_type rfind(const string_type& right, size_type off = npos) const { return M_string.rfind(right, off); }
    size_type rfind(const value_type *ptr, size_type off, size_type count = npos) const { return M_string.rfind(ptr, off, count); }
    size_type rfind(const value_type *ptr, size_type off = npos) const { return M_string.rfind(ptr, off); }
    size_type rfind(value_type ch, size_type off = npos) const { return M_string.rfind(ch, off); }
    size_type find_first_of(const string_type& right, size_type off = 0) const { return M_string.find_first_of(right, off); }
    size_type find_first_of(const value_type *ptr, size_type off, size_type count) const { return M_string.find_first_of(ptr, off, count); }
    size_type find_first_of(const value_type *ptr, size_type off = 0) const { return M_string.find_first_of(ptr, off); }
    size_type find_first_of(value_type ch, size_type off = 0) const { return M_string.find_first_of(ch, off); }
    size_type find_last_of(const string_type& right, size_type off = npos) const { return M_string.find_last_of(right, off); }
    size_type find_last_of(const value_type *ptr, size_type off, size_type count = npos) const { return M_string.find_last_of(ptr, off, count); }
    size_type find_last_of(const value_type *ptr, size_type off = npos) const { return M_string.find_last_of(ptr, off); }
    size_type find_last_of(value_type ch, size_type off = npos) const { return M_string.find_last_of(ch, off); }
    size_type find_first_not_of(const string_type& right, size_type off = 0) const { return M_string.find_first_not_of(right, off); }
    size_type find_first_not_of(const value_type *ptr, size_type off, size_type count) const { return M_string.find_first_not_of(ptr, off, count); }
    size_type find_first_not_of(const value_type *ptr, size_type off = 0) const { return M_string.find_first_not_of(ptr, off); }
    size_type find_first_not_of(value_type ch, size_type off = 0) const { return M_string.find_first_not_of(ch, off); }
    size_type find_last_not_of(const string_type& right, size_type off = npos) const { return M_string.find_last_not_of(right, off); }
    size_type find_last_not_of(const value_type *ptr, size_type off, size_type count) const { return M_string.find_last_not_of(ptr, off, count); }
    size_type find_last_not_of(const value_type *ptr, size_type off = npos) const { return M_string.find_last_not_of(ptr, off); }
    size_type find_last_not_of(value_type ch, size_type off = npos) const { return M_string.find_last_not_of(ch, off); }
    wxString substr(size_type off = 0, size_type count = npos) const { return M_string.substr(off, count); }
    int compare(const string_type& right) const { return M_string.compare(right); }
    int compare(size_type off, size_type n0, const string_type& right) const { return M_string.compare(off, n0, right); }
    int compare(size_type off, size_type n0, const string_type& right, size_type roff, size_type count) const { return M_string.compare(off, n0, right, roff, count); }
    int compare(const value_type *ptr) const { return M_string.compare(ptr); }
    int compare(size_type off, size_type n0, const value_type *ptr) const { return M_string.compare(off, n0, ptr); }
    int compare(size_type off, size_type n0, const value_type *ptr, size_type count) const { return M_string.compare(off, n0, ptr, count); }
    allocator_type get_allocator() const { return M_string.get_allocator(); }
};

inline int wxString::PrintfV(wxChar const* pszFormat, va_list argptr)
{
  *this = wxString::FormatV(pszFormat, argptr);
  return this->length();
}

inline int wxString::Printf(wxChar const* pszFormat, ...)
{
  va_list argptr;
  va_start(argptr, pszFormat);

  int iLen = PrintfV(pszFormat, argptr);

  va_end(argptr);

  return iLen;
}

inline wxString wxString::FormatV(wxChar const* pszFormat, va_list argptr)
{
    wxString s;
    s.PrintfV(pszFormat, argptr);
    return s;
}

inline wxString wxString::Format(wxChar const* pszFormat, ...)
{
    va_list argptr;
    va_start(argptr, pszFormat);

    wxString s;
    s.PrintfV(pszFormat, argptr);

    va_end(argptr);

    return s;
}

// Predicates.
struct caseInsensitiveCompare {
    bool operator() (wxChar c1, wxChar c2) const { return std::tolower(c1) == std::tolower(c2); }
};

inline int wxString::Cmp(wxChar const* psz) const
{
    std::pair<string_type::const_iterator, wxChar const*> res = std::mismatch(this->begin(), this->end(), psz);
    return *res.second == 0 ? (res.first == this->end() ? 0 : 1) :
            ((res.first == this->end() || *res.first < *res.second) ? -1 : 1);
}

inline int wxString::CmpNoCase(wxChar const* psz) const
{
    static caseInsensitiveCompare pred;
    std::pair<string_type::const_iterator, wxChar const*> res = std::mismatch(this->begin(), this->end(), psz, pred);
    return *res.second == 0 ? (res.first == this->end() ? 0 : 1) :
            ((res.first == this->end() || std::tolower(*res.first) < std::tolower(*res.second)) ? -1 : 1);
}

inline int wxString::CmpNoCase(wxString const& psz) const
{
    static caseInsensitiveCompare pred;
    std::pair<string_type::const_iterator, string_type::const_iterator> res = std::mismatch(this->begin(), this->end(), psz.begin(), pred);
    return res.second == this->end() ? (res.first == this->end() ? 0 : 1) :
            ((res.first == this->end() || std::tolower(*res.first) < std::tolower(*res.second)) ? -1 : 1);
}

inline bool wxString::IsSameAs(wxChar const* psz, bool compareWithCase) const
{
    std::pair<string_type::const_iterator, wxChar const*> res;
    if (compareWithCase)
	res = std::mismatch(this->begin(), this->end(), psz);
    else
    {
	static caseInsensitiveCompare pred;
	res = std::mismatch(this->begin(), this->end(), psz, pred);
    }
    return *res.second == 0 && res.first == this->end();
}

inline bool wxString::IsSameAs(wxChar c, bool compareWithCase) const
{
    return this->length() == 1 &&
            (compareWithCase ? (c == *this->data()) : (std::tolower(c) == std::tolower(*this->data())));
}

#endif // wxString

#endif // wxString_H
