/* EBox.c */

#define NDEBUG 1

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef NDEBUG
#include <stdio.h>
#endif
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xaw/BoxP.h>
#include <X11/Xaw/XawInit.h>
#include "VBox.h"
#include "VBoxP.h"

/*
 * Class Methods
 */
static void VBoxChangeManaged(Widget);
static XtGeometryResult VBoxGeometryManager(Widget, XtWidgetGeometry*,
					      XtWidgetGeometry*);
static XtGeometryResult VBoxQueryGeometry(Widget, XtWidgetGeometry*,
					    XtWidgetGeometry*);
static void VBoxResize(Widget);


#define UNUSED __attribute__ ((unused))
#if __GNUC__ > 3
#define SENTINEL __attribute__ ((sentinel(0)))
#else
#define SENTINEL
#endif

/*
 * Initialization
 */

static XtResource vbox_resources[] = {
	{ XtNresize,	XtCResize,	XtRBoolean,
		sizeof(Boolean),	XtOffsetOf(VBoxRec, vbox.resize),
		XtRImmediate,		(XtPointer)False
	}
};

VBoxClassRec vboxClassRec = {
  /* core */
  {
    (WidgetClass)&boxClassRec,		/* superclass */
    "VBox",				/* class_name */
    sizeof(VBoxRec),			/* widget_size */
    NULL,				/* class_initialize */
    NULL,				/* class_part_init */
    False,				/* class_inited */
    NULL,				/* initialize */
    NULL,				/* initialize_hook */
    XtInheritRealize,			/* realize */
    NULL,				/* actions */
    0,					/* num_actions */
    vbox_resources,	/* resources */
    XtNumber(vbox_resources),		/* num_resources */
    NULLQUARK,				/* xrm_class */
    True,				/* compress_motion */
    True,				/* compress_exposure */
    True,				/* compress_enterleave */
    False,				/* visible_interest */
    NULL,				/* destroy */
    VBoxResize,				/* resize */
    XtInheritExpose,			/* expose */
    NULL,				/* set_values */
    NULL,				/* set_values_hook */
    XtInheritSetValuesAlmost,		/* set_values_almost */
    NULL,				/* get_values_hook */
    NULL,				/* accept_focus */
    XtVersion,				/* version */
    NULL,				/* callback_private */
    NULL,				/* tm_table */
    VBoxQueryGeometry,			/* query_geometry */
    XtInheritDisplayAccelerator,	/* display_accelerator */
    NULL,				/* extension */
  },
  /* composite */
  {
    VBoxGeometryManager,		/* geometry_manager */
    VBoxChangeManaged,			/* change_managed */
    XtInheritInsertChild,		/* insert_child */
    XtInheritDeleteChild,		/* delete_child */
    NULL,				/* extension */
  },
  /* box */
  {
    NULL,				/* extension */
  },
  /* vbox */
  {
    NULL,				/* extension */
  },
};

WidgetClass vboxWidgetClass = (WidgetClass)&vboxClassRec;


#define REQUESTS_X(gr) (((gr)->request_mode & CWX ) != 0)
#define REQUESTS_Y(gr) (((gr)->request_mode & CWY ) != 0)
#define REQUESTS_HEIGHT(gr) (((gr)->request_mode & CWHeight ) != 0)
#define REQUESTS_WIDTH(gr) (((gr)->request_mode & CWWidth ) != 0)
#define REQUESTS_BORDER(gr) (((gr)->request_mode & CWBorderWidth ) != 0)
#define QUERY_ONLY(gr) (((gr)->request_mode & XtCWQueryOnly ) != 0)
#define setmax(var,val) {if( var < val ) var = val;}
#ifndef XtX
#define XtX(w)            (((RectObj)w)->rectangle.x)
#endif
#ifndef XtY
#define XtY(w)            (((RectObj)w)->rectangle.y)
#endif
#ifndef XtWidth
#define XtWidth(w)        (((RectObj)w)->rectangle.width)
#endif
#ifndef XtHeight
#define XtHeight(w)       (((RectObj)w)->rectangle.height)
#endif
#ifndef XtBorderWidth
#define XtBorderWidth(w)  (((RectObj)w)->rectangle.border_width)
#endif

/*
 * Calculate preferred size, given constraining box, caching it in the widget
 */
static XtGeometryResult
VBoxQueryGeometry(Widget wid, XtWidgetGeometry *constraint,
		    XtWidgetGeometry *preferred)
{
    VBoxWidget w = (VBoxWidget)wid;
    Dimension width, height;
    XtWidgetGeometry suggested;
    Cardinal i;

#ifndef NDEBUG
    fprintf(stderr, "VBoxQueryGeometry(%p, ", wid);
    if( REQUESTS_X(constraint) )
    	fprintf(stderr, "x=%hu ", constraint->x);
    if( REQUESTS_Y(constraint) )
    	fprintf(stderr, "y=%hu ", constraint->y);
    if( REQUESTS_WIDTH(constraint) )
    	fprintf(stderr, "width=%hu ", constraint->width);
    if( REQUESTS_HEIGHT(constraint) )
    	fprintf(stderr, "height=%hu ", constraint->height);
    if( REQUESTS_BORDER(constraint) )
    	fprintf(stderr, "border=%hu ", constraint->border_width);
    if( QUERY_ONLY(constraint) )
    	fprintf(stderr, "query-only ");
    fprintf(stderr, ")\n");
#endif // NDEBUG
    constraint->request_mode &= CWWidth | CWHeight;

    if (constraint->request_mode == 0)
	/* parent isn't going to change w or h, so nothing to re-compute */
    return (XtGeometryYes);

    if (constraint->request_mode == w->box.last_query_mode
	&& (!(constraint->request_mode & CWWidth)
	  || constraint->width == w->box.last_query_width)
	&& (!(constraint->request_mode & CWHeight)
	  || constraint->height == w->box.last_query_height)) {
	/* same query; current preferences are still valid */
	preferred->request_mode = CWWidth | CWHeight;
	preferred->width = w->box.preferred_width;
	preferred->height = w->box.preferred_height;
	if (constraint->request_mode == preferred->request_mode 
			&& constraint->width == preferred->width
			&& constraint->height == preferred->height)
		return XtGeometryYes;
	else
		return XtGeometryAlmost;
    }

	w->box.last_query_mode = constraint->request_mode;
	w->box.last_query_width = constraint->width;
	w->box.last_query_height= constraint->height;

	if( w->composite.num_children == 0 ) {
		/* without children we accept everthing */
		return XtGeometryYes;
	}

	suggested.request_mode = CWHeight;
	suggested.width = constraint->width;
	width = 0;
	height = 0;
	if( REQUESTS_WIDTH(constraint) )
		suggested.request_mode |= CWWidth;

	for (i = 0; i < w->composite.num_children; i++) {
		Widget widget = w->composite.children[i];
    		XtWidgetGeometry wanted;
		XtGeometryResult r;

		suggested.height = widget->core.height;

#ifndef NDEBUG
    fprintf(stderr, " Calling XtQueryGeometry(%p,", widget);
    if( REQUESTS_WIDTH(&suggested) )
	    fprintf(stderr, " width=%hu", suggested.width);
    if( REQUESTS_HEIGHT(&suggested) )
	    fprintf(stderr, " height=%hu", suggested.height);
#endif
		r = XtQueryGeometry(widget, &suggested, &wanted);
		switch( r ) {
			case XtGeometryYes:
				if( REQUESTS_WIDTH(&suggested) )
					setmax(width, suggested.width);
				height += suggested.height;
				break;
			case XtGeometryAlmost:
				if( REQUESTS_WIDTH(&wanted) )
					setmax(width, wanted.width);
				if( REQUESTS_HEIGHT(&wanted) )
					height += wanted.height;
				else
					height += widget->core.height;
				break;
			case XtGeometryNo:
			default:
				setmax(width, widget->core.width);
				height += widget->core.height;
				break;
		}
	}
	preferred->request_mode = 0;
	if( width > constraint->width ) {
		preferred->width = width;
		preferred->request_mode |= CWWidth;
	} else
		preferred->width = constraint->width;
	w->box.preferred_width = preferred->width;

	preferred->request_mode |= CWHeight;
	w->box.preferred_height = preferred->height = height;

	if ((~constraint->request_mode & preferred->request_mode) == 0
			&& constraint->width == preferred->width
			&& constraint->height == preferred->height)
		return XtGeometryYes;
	else
		return XtGeometryAlmost;
}

/*
 * Actually layout the vbox. Do not care for height, though.
 */
static void
VBoxResize(Widget wid)
{
	VBoxWidget vb = (VBoxWidget)wid;
	Cardinal i;
	Dimension x,y,w,h,b;

#ifndef NDEBUG
    fprintf(stderr, "VBoxResize(%p) {\n", wid);
#endif
	x = 0; y = 0;
	w = XtWidth(vb);
	for (i = 0; i < vb->composite.num_children; i++) {
		Widget widget = vb->composite.children[i];
		if( !widget->core.managed )
			continue;
		h = XtHeight(widget);
		b = XtBorderWidth(widget);
    		if( h == 0 )
			h = 1;
    		if( w == 0 )
			w = 1;
#ifndef NDEBUG
    fprintf(stderr, " Calling XtConfigureWidget(%p, %hd, %hd, %hu, %hu, %hu);\n",
		    widget, x, y, w, h, b);
#endif
		XtConfigureWidget(widget, x, y, w, h, b);
		y += h + 2*b;
	}
	vb->box.preferred_height = y;
#ifndef NDEBUG
    fprintf(stderr, "} // VBoxResize(%p);\n", wid);
#endif
}

static void
VBoxResizeExcept(VBoxWidget vb, Widget exclude, Dimension eh, Dimension eb)
{
	Cardinal i;
	Dimension x,y,w,h,b;

#ifndef NDEBUG
    fprintf(stderr, "VBoxResizeExcept(%p) {\n", vb);
#endif
	x = 0; y = 0;
	w = XtWidth(vb);
	for (i = 0; i < vb->composite.num_children; i++) {
		Widget widget = vb->composite.children[i];
		if( !widget->core.managed )
			continue;
		if( widget == exclude ) {
			y += eh + 2*eb;
		} else {
			h = XtHeight(widget);
			b = XtBorderWidth(widget);
#ifndef NDEBUG
    fprintf(stderr, " Calling XtConfigureWidget(%p, %hd, %hd, %hu, %hu, %hu);\n",
		    widget, x, y, w, h, b);
#endif
			XtConfigureWidget(widget, x, y, w, h, b);
			y += h + 2*b;
		}
	}
	vb->box.preferred_height = y;
#ifndef NDEBUG
    fprintf(stderr, "} // VBoxResize(%p);\n", vb);
#endif
}

static void
VBoxResizeExcept2(VBoxWidget vb, Widget exclude, Dimension eh, Dimension eb)
{
	Cardinal i;
	Dimension x,y,w,h,b;

#ifndef NDEBUG
    fprintf(stderr, "VBoxResizeExcept2(%p) {\n", vb);
#endif
	x = 0; y = 0;
	w = XtWidth(vb);
	for (i = 0; i < vb->composite.num_children; i++) {
		Widget widget = vb->composite.children[i];
		if( !widget->core.managed )
			continue;
		if( widget == exclude ) {
			h = eh;
			b = eb;
		} else {
			h = XtHeight(widget);
			b = XtBorderWidth(widget);
		}
#ifndef NDEBUG
    fprintf(stderr, " Calling XtConfigureWidget(%p, %hd, %hd, %hu, %hu, %hu);\n",
		    widget, x, y, w, h, b);
#endif
		XtConfigureWidget(widget, x, y, w, h, b);
		y += h + 2*b;
	}
	vb->box.preferred_height = y;
#ifndef NDEBUG
    fprintf(stderr, "} // VBoxResize2(%p);\n", vb);
#endif
}


/*
 * Geometry Manager
 */
static XtGeometryResult
VBoxGeometryManager(Widget w, XtWidgetGeometry *request,
		      XtWidgetGeometry *reply)
{
	VBoxWidget vb = (VBoxWidget)w->core.parent;
	XtGeometryResult result = XtGeometryYes;
	Dimension newwidth, newheight;
	XtWidgetGeometry parentAsk, parentReply;
	Boolean secondTry; Boolean secondNoTry;

#ifndef NDEBUG
	fprintf(stderr, "VBoxGeometryManager(%p (parent %p), ", w, vb);
	if( REQUESTS_X(request) )
		fprintf(stderr, "x=%hu ", request->x);
	if( REQUESTS_Y(request) )
		fprintf(stderr, "y=%hu ", request->y);
	if( REQUESTS_WIDTH(request) )
		fprintf(stderr, "width=%hu ", request->width);
	if( REQUESTS_HEIGHT(request) )
		fprintf(stderr, "height=%hu ", request->height);
	if( REQUESTS_BORDER(request) )
		fprintf(stderr, "border=%hu ", request->border_width);
	if( QUERY_ONLY(request) )
		fprintf(stderr, "query-only ");
	fprintf(stderr, ")\n");
#endif // NDEBUG
	if( reply != NULL ) {
		*reply = *request;
		reply->request_mode = 0;
	}

	// TODO: does this include width and height?
	if( REQUESTS_BORDER(request) &&
			request->border_width != XtBorderWidth(w) ) {
		if( reply == NULL )
			return XtGeometryNo;
		result = XtGeometryAlmost;
		reply->border_width =  XtBorderWidth(w);
		reply->request_mode |= CWBorderWidth;
	}
	if( REQUESTS_X(request) && request->x != XtX(w) ) {
		if( reply == NULL )
			return XtGeometryNo;
		result = XtGeometryAlmost;
		reply->x = XtX(w);
		reply->request_mode |= CWX;
	}
	if( REQUESTS_Y(request) && request->y != XtY(w) ) {
		if( reply == NULL )
			return XtGeometryNo;
		result = XtGeometryAlmost;
		reply->y = XtY(w);
		reply->request_mode |= CWY;
	}
	if( !REQUESTS_HEIGHT(request) )
		request->height = XtHeight(w);
	if( !REQUESTS_WIDTH(request) )
		request->width = XtWidth(w);
	if( !REQUESTS_BORDER(request) )
		reply->border_width = request->border_width = XtBorderWidth(w);
	newwidth = 0;
	if( (request->request_mode & (CWBorderWidth | CWWidth)) != 0 ) {
		newwidth = 2*request->border_width + request->width;
		if( newwidth == XtWidth(vb) )
			newwidth = 0;
		else if( !vb->vbox.resize ) {
			if( reply == NULL )
				return XtGeometryNo;
			if( request->border_width >= XtWidth(vb)/2 ) {
				reply->border_width = XtWidth(vb)/2;
				if( reply->border_width > 0 && (XtWidth(vb)&1) ==0 )
					reply->border_width--;
				reply->request_mode |= CWBorderWidth;
			}
			if( REQUESTS_BORDER(request) )
				reply->request_mode |= CWBorderWidth;
			reply->width = XtWidth(vb) - 2*reply->border_width;
			reply->request_mode |= CWWidth;
			result = XtGeometryAlmost;
			newwidth = 0;
		}
	}
	if( newwidth != 0 && newwidth < XtWidth(vb) ) {
		// TODO: ask every child if it is happy with a shrink
		/* until then just reject horiz shrinks: */
		if( reply == NULL )
			return XtGeometryNo;
		if( request->border_width >= XtWidth(vb)/2 ) {
			reply->border_width = XtWidth(vb)/2;
			if( reply->border_width > 0 && (XtWidth(vb)&1) ==0 )
				reply->border_width--;
			reply->request_mode |= CWBorderWidth;
		}
		if( REQUESTS_BORDER(request) )
			reply->request_mode |= CWBorderWidth;
		reply->width = XtWidth(vb) - 2*reply->border_width;
		reply->request_mode |= CWWidth;
		result = XtGeometryAlmost;
		newwidth = 0;
	}

	newheight = 0;
	if( (request->request_mode & (CWBorderWidth | CWHeight)) != 0 ) {
		Dimension h = XtHeight(w) + 2*XtBorderWidth(w);
		Dimension newh = request->height + 2*request->border_width;
		if( newh <= h ) {
			newheight = vb->box.preferred_height - (h-newh);
		} else if( newh > h ) {
			newheight = vb->box.preferred_height + (newh-h);
		}
		if( newheight == XtHeight(vb) )
			newheight = 0;
	}
	parentAsk.request_mode = 0;
	if( newheight != 0 ) {
		parentAsk.request_mode |= CWHeight;
		parentAsk.height = newheight;
	} else // does this have to be set or is it a bug in viewport to need it?
		parentAsk.height = XtHeight(vb);
	if( newwidth != 0 ) {
		parentAsk.request_mode |= CWWidth;
		parentAsk.width = newwidth;
	} else // dito
		parentAsk.width = XtWidth(vb);
	if( parentAsk.request_mode == 0 ) {
		if( result == XtGeometryAlmost && reply != NULL )
			reply->request_mode |= request->request_mode;
		return result;
	}

	/* box needs resizing to fit, ask parent for it */

	if( result == XtGeometryAlmost || QUERY_ONLY(request) )
		parentAsk.request_mode |= XtCWQueryOnly;
	secondTry = False; secondNoTry = False;
	do switch( XtMakeGeometryRequest((Widget)vb, &parentAsk, &parentReply) ) {
		case XtGeometryNo:
		default:
			if( newheight != 0 ) {
				/* try again without a height setting */
				parentAsk.height = XtHeight(vb);
				parentAsk.request_mode &= ~(CWHeight|XtCWQueryOnly);
				if( secondNoTry )
					return XtGeometryNo;
				secondNoTry = True;
				continue;
			}
			return XtGeometryNo;
		case XtGeometryYes:
			if( result == XtGeometryYes && !QUERY_ONLY(request) ) {
				if( 0==1 ) 
					VBoxResizeExcept(vb,w,
							request->height,
							request->border_width);
				else {
					VBoxResizeExcept2(vb,w,
							request->height,
							request->border_width);
					return XtGeometryDone;
				}
			}
			if( result == XtGeometryAlmost && reply != NULL )
				reply->request_mode |= request->request_mode;
			return result;
		case XtGeometryAlmost:
			if( REQUESTS_WIDTH(&parentReply) &&
					parentReply.width != newwidth ) {
				if( reply == NULL )
					return XtGeometryNo;
				if( request->border_width >= parentReply.width/2 ) {
					reply->border_width = parentReply.width/2;
					if( reply->border_width > 0 && (parentReply.width&1) ==0 )
						reply->border_width--;
					reply->request_mode |= CWBorderWidth;
				}
				if( REQUESTS_BORDER(request) )
					reply->request_mode |= CWBorderWidth;
				reply->width = parentReply.width - 2*reply->border_width;
				reply->request_mode |= CWWidth;
				reply->request_mode |= request->request_mode;
				result = XtGeometryAlmost;
			}
			if( REQUESTS_HEIGHT(&parentReply) &&
					parentReply.height != newheight ) {
				/* do not care for height */
				/* tell client to fix other problems first */
				if( result == XtGeometryAlmost ) {
					reply->request_mode |= request->request_mode;
					return result;
				}
				/* if it was only a query, tell it works */
				if( QUERY_ONLY(request) )
					return XtGeometryYes;
				/* otherwise choose a height parent likes
				 * and let things leave the window */
				parentAsk.height = parentReply.height;
				parentAsk.request_mode &= ~(CWHeight|XtCWQueryOnly);
				if( secondTry ) //should not happen, but does
					return XtGeometryNo;
				secondTry = True;
			} else /* strange situation, no idea what to do */
				return XtGeometryNo;
			break;
	} while (True);
	/* NOT REACHED */
}

static void
VBoxChangeManaged(Widget w)
{
#ifndef NDEBUG
    fprintf(stderr, "VBoxChangeManaged(%p) {\n", w);
#endif // NDEBUG
    // TODO: query geometry and try to resize...
    VBoxResize(w);
#ifndef NDEBUG
    fprintf(stderr, "} // VBoxChangeManaged(%p);\n", w);
#endif // NDEBUG
}
