/*-
# X-BASED PANEX(tm)
#
#  Panex.c
#
###
#
#  Copyright (c) 1996 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Panex */

#include "PanexP.h"

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wpanex.ini"
#endif

#define SECTION "setup"

static const char *pyramidColorString[MAXSTACKS - 1] =
{
	"255 0 255",
	"255 255 0"
};
static char pyramidColorChar[MAXSTACKS - 1] =
{'M', 'Y'};
#else

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesPanex(Widget current, Widget request, Widget renew);
static void QuitPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void DestroyPanex(Widget old);
static void ResizePanex(PanexWidget w);
static void SizePanex(PanexWidget w);
static void InitializePanex(Widget request, Widget renew);
static void ExposePanex(Widget renew, XEvent * event, Region region);
static void HidePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void SelectPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ReleasePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void GetPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void WritePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ClearPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ClearPanexMaybe(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ClearPanex2(PanexWidget w, XEvent * event, char **args, int nArgs);
static void UndoPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void SolvePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void IncrementPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void DecrementPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ModePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void EnterPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void LeavePanex(PanexWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsPanex[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>r: Clear()\n\
 <KeyPress>c: Clear()\n\
 <Btn3Down>: ClearMaybe()\n\
 <Btn3Down>(2+): Clear2()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>m: Mode()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPanex[] =
{
	{(char *) "Quit", (XtActionProc) QuitPanex},
	{(char *) "Hide", (XtActionProc) HidePanex},
	{(char *) "Select", (XtActionProc) SelectPanex},
	{(char *) "Release", (XtActionProc) ReleasePanex},
	{(char *) "Clear", (XtActionProc) ClearPanex},
	{(char *) "ClearMaybe", (XtActionProc) ClearPanexMaybe},
	{(char *) "Clear2", (XtActionProc) ClearPanex2},
	{(char *) "Get", (XtActionProc) GetPanex},
	{(char *) "Write", (XtActionProc) WritePanex},
	{(char *) "Undo", (XtActionProc) UndoPanex},
	{(char *) "Solve", (XtActionProc) SolvePanex},
	{(char *) "Increment", (XtActionProc) IncrementPanex},
	{(char *) "Decrement", (XtActionProc) DecrementPanex},
	{(char *) "Mode", (XtActionProc) ModePanex},
	{(char *) "Enter", (XtActionProc) EnterPanex},
	{(char *) "Leave", (XtActionProc) LeavePanex}
};

static XtResource resourcesPanex[] =
{
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.username),
	 XtRString, (caddr_t) "nobody"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNtileColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.tileColor),
	 XtRString, (caddr_t) "gray75" /*XtDefaultBackground*/},
	{XtNtileBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNpyramidColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.pyramidName[0]),
	 XtRString, (caddr_t) "magenta"},
	{XtNpyramidColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.pyramidName[1]),
	 XtRString, (caddr_t) "yellow"},
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(PanexWidget, core.width),
	 XtRString, (caddr_t) "400"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(PanexWidget, core.height),
	 XtRString, (caddr_t) "200"},
	{XtNtiles, XtCTiles, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.tiles),
	 XtRString, (caddr_t) "10"}, /* DEFAULTTILES */
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.mode),
	 XtRString, (caddr_t) "1"}, /* DEFAULTMODE */
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.delay),
	 XtRString, (caddr_t) "300"}, /* DEFAULTDELAY */
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(PanexWidget, panex.select), XtRCallback, (caddr_t) NULL}
};

PanexClassRec panexClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Panex",	/* class name */
		sizeof (PanexRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePanex,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPanex,	/* actions */
		XtNumber(actionsListPanex),	/* num actions */
		resourcesPanex,	/* resources */
		XtNumber(resourcesPanex),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyPanex,	/* destroy */
		(XtWidgetProc) ResizePanex,	/* resize */
		(XtExposeProc) ExposePanex,	/* expose */
		(XtSetValuesFunc) SetValuesPanex,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPanex,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass panexWidgetClass = (WidgetClass) & panexClassRec;

#ifndef HAVE_USLEEP
#if !defined( VMS ) || defined( XVMSUTILS ) ||  ( __VMS_VER >= 70000000 )
#ifdef USE_XVMSUTILS
#include <X11/unix_time.h>
#endif
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#endif
#endif
#if defined(SYSV) || defined(SVR4)
#ifdef LESS_THAN_AIX3_2
#include <sys/poll.h>
#else /* !LESS_THAN_AIX3_2 */
#include <poll.h>
#endif /* !LESS_THAN_AIX3_2 */
#endif /* defined(SYSV) || defined(SVR4) */

/* not static in case usleep found in system include */
int
usleep(unsigned int usec)
{
#if (defined (SYSV) || defined(SVR4)) && !defined(__hpux)
#if defined(HAVE_NANOSLEEP)
	{
		struct timespec rqt;

		rqt.tv_nsec = 1000 * (usec % (unsigned int) 1000000);
		rqt.tv_sec = usec / (unsigned int) 1000000;
		return nanosleep(&rqt, (struct timespec *) NULL);
	}
#else
	(void) poll(
#if defined(__cplusplus) || defined(c_plusplus)
		(pollfd *) /* guess */
#else
		(void *)
#endif
		0, (int) 0, (usec / 1000));	/* ms resolution */
#endif
#else
#ifdef VMS
	long        timadr[2];

	if (usec != 0) {
		timadr[0] = -usec * 10;
		timadr[1] = -1;

		sys$setimr(4, &timadr, 0, 0, 0);
		sys$waitfr(4);
	}
#else
	struct timeval time_out;

#if 0
	/* (!defined(AIXV3) && !defined(__hpux)) */
	extern int  select(int, fd_set *, fd_set *, fd_set *, struct timeval *);

#endif

	time_out.tv_usec = usec % (unsigned int) 1000000;
	time_out.tv_sec = usec / (unsigned int) 1000000;
	(void) select(0, (void *) 0, (void *) 0, (void *) 0, &time_out);
#endif
#endif
	return 0;
}
#endif

void
SetPanex(PanexWidget w, int reason)
{
	panexCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(PanexWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->panex.fontInfo) {
		XUnloadFont(XtDisplay(w), w->panex.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->panex.fontInfo);
	}
	if ((w->panex.fontInfo = XLoadQueryFont(display,
			w->panex.font)) == NULL) {
		(void) sprintf(buf,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->panex.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->panex.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Cannot open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->panex.fontInfo) {
		w->panex.letterOffset.x = XTextWidth(w->panex.fontInfo, "8", 1)
			/ 2;
		w->panex.letterOffset.y = w->panex.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->panex.letterOffset.x = 3;
		w->panex.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "panex.log"
#endif

static int  startPositions[MAXMODES][MAXSTACKS] =
{
	{0, -1, -1},
	{0, -1, 1}
};
static int  finishPositions[MAXMODES][MAXSTACKS] =
{
	{-1, -1, 0},
	{1, -1, 0}
};

static Point trapazoidUnit[5] =
{
	{0, 0},
	{1, 1},
	{-3, 0},
	{1, -1},
	{2, 0}
};

#if (!defined(WINVER) || (WINVER <= 0x030a)) /* if X or WINDOWS 3.1 or less */
static void
Sleep(unsigned int cMilliseconds)
{
#if (defined(WINVER) && (WINVER <= 0x030a))
	unsigned long time_out = GetTickCount() + cMilliseconds;

	while (time_out > GetTickCount());
#else
	(void) usleep(cMilliseconds * 1000);
#endif
}
#endif

void
intCat(char ** string, const char * var1, const int var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + 21))) {
        	DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%d", var1, var2);
}

void
stringCat(char ** string, const char * var1, const char * var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + strlen(var2) + 1))) {
        	DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%s", var1, var2);
}

static void
CheckTiles(PanexWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->panex.tiles < MINTILES) {
		intCat(&buf1, "Number of tiles out of bounds, use at least ",
			MINTILES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTTILES);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->panex.tiles = DEFAULTTILES;
	}
	if (w->panex.mode < HANOI || w->panex.mode > PANEX) {
		intCat(&buf1,
			"Mode is in error, use 0 for Hanoi, 1 for Panex, defaulting to ",
			DEFAULTMODE);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->panex.mode = DEFAULTMODE;
	}
	if (w->panex.delay < 0) {
		intCat(&buf1, "Delay can not be negative (",
			w->panex.delay);
		stringCat(&buf2, buf1, "), taking absolute value");
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		w->panex.delay = -w->panex.delay;
	}
}

#if 0
/* Only applies to Panex */
static int  middlePositions[MAXSTACKS] =
{-1, 0, 1};

Boolean
CheckMiddle(PanexWidget w)
{
	int         stack, loc;
	PanexLoc    i;

	for (stack = 0; stack < MAXSTACKS; stack++)
		if ((i.stack = middlePositions[stack]) >= 0)
			for (loc = 1; loc <= w->panex.tiles; loc++) {
				i.loc = loc - 1;
				if (w->panex.tileOfPosition[stack][loc].stack != i.stack ||
				    w->panex.tileOfPosition[stack][loc].loc != i.loc)
					return False;
			}
	return True;
}
#endif

Boolean
CheckSolved(PanexWidget w)
{
	int         stack, loc;
	PanexLoc    i;

	for (stack = 0; stack < MAXSTACKS; stack++)
		if ((i.stack = finishPositions[w->panex.mode][stack]) >= 0)
			for (loc = 1; loc <= w->panex.tiles; loc++) {
				i.loc = loc - 1;
				if (w->panex.tileOfPosition[stack][loc].stack != i.stack ||
				    w->panex.tileOfPosition[stack][loc].loc != i.loc)
					return False;
			}
	return True;
}

#ifdef DEBUG
void
PrintStacks(PanexWidget w)
{
	int         stack, position;

	(void) printf("top: where are the tiles in the stack\n");
	for (position = 0; position <= w->panex.tiles; position++)
		for (stack = 0; stack < MAXSTACKS; stack++) {
			printf("%d,%d", w->panex.tileOfPosition[stack][position].stack,
			       w->panex.tileOfPosition[stack][position].loc);
			if (stack + 1 == MAXSTACKS)
				(void) printf("\n");
			else
				(void) printf(" | ");
		}
}

void
PrintTiles(PanexWidget w)
{
	int         stack, position;

	(void) printf("pot: which stack are the tiles in\n");
	for (position = 0; position < w->panex.tiles; position++)
		for (stack = 0; stack <= w->panex.mode; stack++) {
			printf("%d,%d", w->panex.positionOfTile[stack][position].stack,
			       w->panex.positionOfTile[stack][position].loc);
			if (stack == w->panex.mode)
				(void) printf("\n");
			else
				(void) printf(" | ");
		}
}
#endif

#ifdef WINVER
static void
POLYGON(PanexWidget w, GC color, GC lineColor, const POINT * poly, int n,
		Boolean origin)
{
	/* CoordModePrevious -> CoordModeOrigin */
	POINT      *temp = NULL;
	int         pt;

	if (!origin) {
		if (!(temp = (POINT *) malloc(sizeof (POINT) * n))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		temp[0] = poly[0];
		for (pt = 1; pt < n; pt++) {
			temp[pt].x = temp[pt - 1].x + poly[pt].x,
			temp[pt].y = temp[pt - 1].y + poly[pt].y;
		}
	}
	w->panex.hPen = CreatePen(PS_SOLID, 1, lineColor);
	w->panex.hOldPen = (HPEN) SelectObject(w->core.hDC, w->panex.hPen);
	w->panex.hBrush = CreateSolidBrush(color);
	w->panex.hOldBrush = (HBRUSH) SelectObject(w->core.hDC,
		w->panex.hBrush);
	(void) Polygon(w->core.hDC, (origin) ? poly : temp, n);
	(void) SelectObject(w->core.hDC, w->panex.hOldBrush);
	(void) DeleteObject(w->panex.hBrush);
	(void) SelectObject(w->core.hDC, w->panex.hOldPen);
	(void) DeleteObject(w->panex.hPen);
	if (!origin) {
		free(temp);
	}
}
#endif

static void
fill3DRect(PanexWidget w, GC gc, GC darkerGC, GC brighterGC,
		int x, int y, int width, int height, Boolean raised)
{
	GC currentGC = (raised) ? gc : darkerGC;

	if (width > 2 && height > 2)
		FILLRECTANGLE(w, currentGC,
			x + 1, y + 1, width - 2, height - 2);
	currentGC = (raised) ? brighterGC : darkerGC;
	FILLRECTANGLE(w, currentGC,
		x, y, 1, height);
	if (width > 1)
		FILLRECTANGLE(w, currentGC,
			x + 1, y, width - 1, 1);
	FILLRECTANGLE(w, currentGC,
		x + 1, y, 1, height);
	if (width > 2)
		FILLRECTANGLE(w, currentGC,
			x + 2, y + 1, width - 2, 1);
	currentGC = (raised) ? darkerGC : gc;
	if (width > 1 && height > 1)
		FILLRECTANGLE(w, currentGC,
			x + 1, y + height - 1, width - 1, 1);
	if (width > 1 && height > 1)
		FILLRECTANGLE(w, currentGC,
			x + width - 1, y, 1, height - 1);
	if (width > 3 && height > 2)
		FILLRECTANGLE(w, currentGC,
			x + 2, y + height - 2, width - 3, 1);
	if (width > 2 && height > 3)
		FILLRECTANGLE(w, currentGC,
			x + width - 2, y + 1, 1, height - 3);
}

static void
drawShadow(PanexWidget w, GC gc, int startX, int startY,
		int sizeX, int sizeY)
{
	FILLRECTANGLE(w, gc,
		       startX, startY, sizeX, 1);
	FILLRECTANGLE(w, gc,
		       startX, startY, 1, sizeY);
}

static void
DrawPyramid(PanexWidget w, int color, int i, int j, int size,
		int offsetPosition)
{
	Point      trapazoidList[5];
	int         k;
	GC          pyramidGC;

	for (k = 0; k <= 4; k++) {
		if (ABS(trapazoidUnit[k].x) == 3)
			trapazoidList[k].x = (size + 1) *
				SIGN(trapazoidUnit[k].x) *
				w->panex.tileSize.x / (w->panex.tiles + 1);
		else if (ABS(trapazoidUnit[k].x) == 2)
			trapazoidList[k].x = size * SIGN(trapazoidUnit[k].x) *
				w->panex.tileSize.x / (w->panex.tiles + 1);
		else
			trapazoidList[k].x = w->panex.tileSize.x /
				(2 * (w->panex.tiles + 1));
		trapazoidList[k].y = (w->panex.tileSize.y - 4) *
			trapazoidUnit[k].y;
	}
	k = w->panex.delta.x + i * w->panex.pos.x + w->panex.puzzleOffset.x +
		w->panex.tileSize.x / 2 + offsetPosition + 1;
	trapazoidList[0].x = trapazoidList[4].x / 2 + k;
	trapazoidList[0].y = j * w->panex.pos.y + w->panex.delta.y +
		w->panex.puzzleOffset.y + offsetPosition + 2;
	if (w->panex.mono) {
		pyramidGC = (offsetPosition) ? w->panex.tileGC :
		       	w->panex.borderGC;
	} else {
#ifndef WINVER
		if ((w->panex.pyramidColor[color] == w->panex.borderColor)
				&& offsetPosition)
			pyramidGC = w->panex.tileGC;
		else if ((w->panex.pyramidColor[color] == w->panex.tileColor)
				&& !offsetPosition)
			pyramidGC = w->panex.borderGC;
		else
#endif
			pyramidGC = w->panex.pyramidGC[color];

	}
	POLYGON(w, pyramidGC, pyramidGC, trapazoidList, 4, False);
	if (w->panex.mono) {

		char        buf[2];

		buf[0] =
#ifdef WINVER
			w->panex.pyramidChar[color];
#else
			w->panex.pyramidName[color][0];
#endif
		DRAWTEXT(w, (offsetPosition) ? w->panex.borderGC :
			    w->panex.tileGC,
			    k - w->panex.letterOffset.x, trapazoidList[0].y +
		  w->panex.tileSize.y / 2 + w->panex.letterOffset.y, buf, 1);
	}
}

static void
DrawTile(PanexWidget w, int i, int j, Boolean erase, int offsetPosition)
{
	int         dx, dy;
	GC          tileGC, borderGC;

	if (erase) {
		tileGC = w->panex.inverseGC;
		borderGC = w->panex.borderGC;
	} else {
		tileGC = w->panex.tileGC;
		borderGC = w->panex.borderGC;
	}
	dx = i * w->panex.pos.x + w->panex.delta.x + w->panex.puzzleOffset.x;
	dy = j * w->panex.pos.y + w->panex.delta.y + w->panex.puzzleOffset.y;
	if (erase)  {
		/* Draw Slots */
		FILLRECTANGLE(w, tileGC,
			dx + offsetPosition, dy + offsetPosition,
			w->panex.tileSize.x, w->panex.tileSize.y);
		if (j == 0) {
			if (i == 0) {
				FILLRECTANGLE(w, borderGC,
					dx + w->panex.tileSize.x / 3 + offsetPosition,
					dy + w->panex.tileSize.y / 3 - 1,
					2 * w->panex.tileSize.x / 3 + 2,
					w->panex.tileSize.y / 3);
			} else if (i == MAXSTACKS - 1) {
				FILLRECTANGLE(w, borderGC,
					dx + offsetPosition,
					dy + w->panex.tileSize.y / 3 - 1,
					2 * w->panex.tileSize.x / 3 - 1,
					w->panex.tileSize.y / 3 +
				       	offsetPosition);
			} else {
				FILLRECTANGLE(w, borderGC,
					dx + offsetPosition,
					dy + w->panex.tileSize.y / 3 - 1,
					w->panex.tileSize.x + 1,
					w->panex.tileSize.y / 3);
			}
			FILLRECTANGLE(w, borderGC,
				dx + w->panex.tileSize.x / 3,
				dy + w->panex.tileSize.y / 3 - 1,
				w->panex.tileSize.x / 3,
				w->panex.tileSize.y - w->panex.tileSize.y / 3 +
				2 + offsetPosition - 1);
		} else if (j == w->panex.tiles) {
			FILLRECTANGLE(w, borderGC,
				dx + w->panex.tileSize.x / 3, dy,
				w->panex.tileSize.x / 3,
				2 * w->panex.tileSize.y / 3 + 2);
		} else {
			FILLRECTANGLE(w, borderGC,
				dx + w->panex.tileSize.x / 3, dy,
				w->panex.tileSize.x / 3,
				w->panex.tileSize.y + offsetPosition);
		}
	} else {
		if (offsetPosition != 0) {
			drawShadow(w, w->panex.tileDarkerGC,
				dx, dy,
				w->panex.tileSize.x, w->panex.tileSize.y);
		}
		dx += offsetPosition;
		dy += offsetPosition;

		fill3DRect(w, tileGC,
			w->panex.tileDarkerGC, w->panex.tileBrighterGC,
			dx, dy, w->panex.tileSize.x, w->panex.tileSize.y,
			offsetPosition == 0);
		DrawPyramid(w,
			    w->panex.tileOfPosition[i][j].stack, i, j,
			    w->panex.tileOfPosition[i][j].loc, offsetPosition);
	}
}

void
DrawAllTiles(PanexWidget w)
{
	int         i, j;

	for (i = 0; i < MAXSTACKS; i++)
		for (j = 0; j <= w->panex.tiles; j++)
			if (w->panex.tileOfPosition[i][j].stack >= 0)
				DrawTile(w, i, j, False, FALSE);
}

static void
DrawSlots(PanexWidget w)
{
	int         i, dx, dy, y, sizey;

	sizey = w->panex.tileSize.y / 3;
	if (sizey == 0)
		sizey = 1;
	dx = w->panex.tileSize.x / 3 + w->panex.delta.x +
		w->panex.puzzleOffset.x;
	y = (w->panex.tiles + 1) * w->panex.pos.y + w->panex.delta.y;
	dy = w->panex.delta.y / 2 + w->panex.puzzleOffset.y + sizey;
	FILLRECTANGLE(w, w->panex.borderGC, dx, dy,
	  w->panex.tileSize.x / 3 + (MAXSTACKS - 1) * w->panex.pos.x,
		       w->panex.tileSize.y / 3);
	for (i = 0; i < MAXSTACKS; i++) {
		FILLRECTANGLE(w, w->panex.borderGC, dx, dy,
		w->panex.tileSize.x / 3, y - 1 - 2 * w->panex.tileSize.y / 3);
		dx += w->panex.pos.x;
	}
}

static void
EraseFrame(PanexWidget w, Boolean focus)
{
	DRAWRECTANGLE(w, (focus) ? w->panex.frameGC : w->panex.inverseGC,
		0, 0, w->core.width - 1, w->core.height - 1);
	FILLRECTANGLE(w, w->panex.inverseGC,
		1, 1, w->core.width - 2, w->core.height - 2);
}

static void
DrawFrame(PanexWidget w, Boolean focus)
{
	DRAWRECTANGLE(w, (focus) ? w->panex.frameGC : w->panex.borderGC,
		0, 0, w->core.width - 1, w->core.height - 1);
}

int
TopOfStack(PanexWidget w, int stack)
{
	int         i;

	for (i = 0; i <= w->panex.tiles; i++)
		if (w->panex.tileOfPosition[stack][i].stack >= 0)
			return i;
	return -1;
}

static int
RequestMove(PanexWidget w, int fromStack, int fromPosition, int toStack)
{
	int         i;

	/* Do not have to check above stack since it is the top one */
	if (toStack > fromStack)
		for (i = fromStack + 1; i <= toStack; i++) {
			if (w->panex.tileOfPosition[i][0].stack >= 0)
				return (-1);
	} else			/* Already ruled out toStack == fromStack */
		for (i = fromStack - 1; i >= toStack; i--) {
			if (w->panex.tileOfPosition[i][0].stack >= 0)
				return (-1);
		}
	i = TopOfStack(w, toStack);
	i = (i == -1) ? w->panex.tiles : i - 1;
	if (w->panex.mode == HANOI) {
		if (i == w->panex.tiles || w->panex.tileOfPosition[toStack][i + 1].loc >
		    w->panex.tileOfPosition[fromStack][fromPosition].loc)
			return i;
		else
			return -2;
	} else {
		if (i > w->panex.tileOfPosition[fromStack][fromPosition].loc + 1)
			return (w->panex.tileOfPosition[fromStack][fromPosition].loc + 1);
		else
			return i;
	}
}

static int
SelectTile(PanexWidget w, int x)
{
	int         i;

	x -= w->panex.puzzleOffset.x;
	i = (x - w->panex.delta.x / 2) / w->panex.pos.x;
	if (i < 0)
		i = 0;
	else if (i >= MAXSTACKS)
		i = MAXSTACKS - 1;
	/*
	   y -= w->panex.puzzleOffset.y;
	   j = (y - w->panex.delta.y / 2) / w->panex.pos.y;
	   if (j < 0)
	   j = 0;
	   else if (j > w->panex.tiles)
	   j = w->panex.tiles;
	 */
	return i;
}


static void
SlideTile(PanexWidget w, int fromStack, int fromPosition, int toStack, int toPosition)
{
	PanexLoc    top;
	int         currentStack = fromStack, currentPosition = fromPosition;
	int         nextStack = fromStack, nextPosition = fromPosition;

	while (currentPosition > 0) {
		DrawTile(w, currentStack, currentPosition, True, FALSE);
		top = w->panex.tileOfPosition[currentStack][currentPosition];
		nextPosition = currentPosition - 1;
		w->panex.tileOfPosition[nextStack][nextPosition] = top;
		w->panex.positionOfTile[top.stack][top.loc].stack = nextStack;
		w->panex.positionOfTile[top.stack][top.loc].loc = nextPosition;
		w->panex.tileOfPosition[currentStack][currentPosition].stack = -1;
		currentPosition = nextPosition;
		DrawTile(w, currentStack, currentPosition, False, FALSE);
#ifndef WINVER
		XFlush(XtDisplay(w));
#endif
		Sleep((unsigned int) w->panex.delay / (w->panex.tiles + MAXSTACKS - 1));
	}
	while (currentStack != toStack) {
		DrawTile(w, currentStack, currentPosition, True, FALSE);
		top = w->panex.tileOfPosition[currentStack][currentPosition];
		nextStack = (currentStack < toStack) ? currentStack + 1 :
			currentStack - 1;
		w->panex.tileOfPosition[nextStack][nextPosition] = top;
		w->panex.positionOfTile[top.stack][top.loc].stack = nextStack;
		w->panex.positionOfTile[top.stack][top.loc].loc = nextPosition;
		w->panex.tileOfPosition[currentStack][currentPosition].stack = -1;
		currentStack = nextStack;
		DrawTile(w, currentStack, currentPosition, False, FALSE);
#ifndef WINVER
		XFlush(XtDisplay(w));
#endif
		Sleep((unsigned int) w->panex.delay / (w->panex.tiles + MAXSTACKS - 1));
	}
	while (currentPosition < toPosition) {
		DrawTile(w, currentStack, currentPosition, True, FALSE);
		top = w->panex.tileOfPosition[currentStack][currentPosition];
		nextPosition = currentPosition + 1;
		w->panex.tileOfPosition[nextStack][nextPosition] = top;
		w->panex.positionOfTile[top.stack][top.loc].stack = nextStack;
		w->panex.positionOfTile[top.stack][top.loc].loc = nextPosition;
		w->panex.tileOfPosition[currentStack][currentPosition].stack = -1;
		currentPosition = nextPosition;
		DrawTile(w, currentStack, currentPosition, False, FALSE);
#ifndef WINVER
		XFlush(XtDisplay(w));
#endif
		Sleep((unsigned int) w->panex.delay / (w->panex.tiles + MAXSTACKS - 1));
	}
}

static int
MoveTile(PanexWidget w, int fromStack, int fromPosition, int toStack)
{
	int         toPosition;

	if ((toPosition = RequestMove(w, fromStack, fromPosition, toStack)) >= 0)
		SlideTile(w, fromStack, fromPosition, toStack, toPosition);
	return toPosition;
}

static void
ResetTiles(PanexWidget w)
{
	int         stack, loc;
	PanexLoc    i;

	w->panex.currentStack = -1;
	for (stack = 0; stack < MAXMODES; stack++) {
		if (w->panex.positionOfTile[stack]) {
			free(w->panex.positionOfTile[stack]);
			w->panex.positionOfTile[stack] = NULL;
		}
		if (startLoc[stack]) {
			free(startLoc[stack]);
			startLoc[stack] = NULL;
		}
	}
	for (stack = 0; stack <= w->panex.mode; stack++) {
		if (!(w->panex.positionOfTile[stack] = (PanexLoc *)
		      malloc(sizeof (PanexLoc) * w->panex.tiles))) {
        		DISPLAY_ERROR("Not enough memory, exiting.");
		}
		if (!(startLoc[stack] = (PanexLoc *)
		      malloc(sizeof (PanexLoc) * w->panex.tiles))) {
        		DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (stack = 0; stack < MAXSTACKS; stack++) {
		if (w->panex.tileOfPosition[stack])
			free(w->panex.tileOfPosition[stack]);
		if (!(w->panex.tileOfPosition[stack] = (PanexLoc *)
		      malloc(sizeof (PanexLoc) * (w->panex.tiles + 1)))) {
        		DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (stack = 0; stack <= w->panex.mode; stack++) {
		i.stack = 2 * stack;
		for (loc = 0; loc < w->panex.tiles; loc++) {
			i.loc = loc + 1;
			w->panex.positionOfTile[stack][loc] = i;
		}
	}
	for (stack = 0; stack < MAXSTACKS; stack++) {
		i.stack = -1;
		i.loc = -1;
		w->panex.tileOfPosition[stack][0] = i;
		if ((i.stack = startPositions[w->panex.mode][stack]) >= 0)
			for (loc = 1; loc <= w->panex.tiles; loc++) {
				i.loc = loc - 1;
				w->panex.tileOfPosition[stack][loc] = i;
		} else
			for (loc = 1; loc <= w->panex.tiles; loc++)
				w->panex.tileOfPosition[stack][loc] = i;
	}
	FlushMoves(w);
	w->panex.started = False;
}

static void
GetTiles(PanexWidget w)
{
	FILE       *fp;
	int         c, i, tiles, mode, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &tiles);
	if (tiles >= MINTILES) {
		for (i = w->panex.tiles; i < tiles; i++) {
			SetPanex(w, PANEX_INC);
		}
		for (i = w->panex.tiles; i > tiles; i--) {
			SetPanex(w, PANEX_DEC);
		}
	} else {
		stringCat(&buf1, name, " corrupted: tiles ");
		intCat(&buf2, buf1, tiles);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINTILES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &mode);
	if (mode >= HANOI && mode <= PANEX) {
		if (w->panex.mode != mode) {
			SetPanex(w, PANEX_MODE);
		}
	} else {
		stringCat(&buf1, name, " corrupted: mode ");
		intCat(&buf2, buf1, mode);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, HANOI);
		free(buf1);
		stringCat(&buf1, buf2, " and ");
		free(buf2);
		intCat(&buf2, buf1, PANEX);
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
	}
#ifdef WINVER
	ResetTiles(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetPanex(w, PANEX_RESTORE);
	SetStartPosition(w);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	if (mode == 0)
		(void) printf("%s: mode hanoi, tiles %d, moves %d.\n",
			name, tiles, moves);
	else
		(void) printf("%s: mode panex, tiles %d, moves %d.\n",
			name, tiles, moves);
	free(lname);
	free(fname);
	w->panex.cheat = True; /* Assume the worst. */
}

static void
WriteTiles(PanexWidget w)
{
	FILE       *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "tiles%c %d\n", SYMBOL, w->panex.tiles);
	(void) fprintf(fp, "mode%c %d\n", SYMBOL, w->panex.mode);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearTiles(PanexWidget w)
{
	if (w->panex.currentStack >= 0)
		return;
	w->panex.cheat = False;
	EraseFrame(w, w->panex.focus);
	ResetTiles(w);
	DrawSlots(w);
	DrawAllTiles(w);
	SetPanex(w, PANEX_RESET);
}

static void
UndoTiles(PanexWidget w)
{
	if (MadeMoves() && w->panex.currentStack < 0) {
		int         fromStack, fromPosition, toStack;

		GetMove(&toStack, &fromStack);
		if ((fromPosition = TopOfStack(w, fromStack)) < 0 ||
		    MoveTile(w, fromStack, fromPosition, toStack) < 0) {
			char *buf1, *buf2;

                        intCat(&buf1, "Move from ", fromStack);
                        stringCat(&buf2, buf1, " to ");
                        free(buf1);
                        intCat(&buf1, buf2, toStack);
                        free(buf2);
                        stringCat(&buf2, buf1, " can not be made");
                        free(buf1);
			DISPLAY_WARNING(buf2);
                        free(buf2);
		} else {
			SetPanex(w, PANEX_UNDO);
		}
	}
}

static void
SolveTiles(PanexWidget w)
{
	/* Cheat and Reset To Start */
	EraseFrame(w, w->panex.focus);
	ResetTiles(w);
	DrawSlots(w);
	DrawAllTiles(w);
	SetPanex(w, PANEX_RESET);
	SolveTilesFromStart(w);
}

static void
IncrementTiles(PanexWidget w)
{
	SetPanex(w, PANEX_INC);
}

static void
DecrementTiles(PanexWidget w)
{
	if (w->panex.tiles <= MINTILES)
		return;
	SetPanex(w, PANEX_DEC);
}

static void
ModeTiles(PanexWidget w)
{
	SetPanex(w, PANEX_MODE);
}

static void
MoveNoTiles(PanexWidget w)
{
	SetPanex(w, PANEX_IGNORE);
}

int
MovePanex(PanexWidget w, int fromStack, int fromPosition, int toStack)
{
	int         toPosition;

	if ((toPosition = MoveTile(w, fromStack, fromPosition, toStack)) >= 0) {
		SetPanex(w, PANEX_MOVED);
		PutMove(fromStack, toStack);
	}
	return toPosition;
}

#define FACTOR 0.7

#ifdef WINVER
#define MAXINTENSITY 0xFF
static int
brighter(const int light)
{
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);
	int temp = light;

	if (temp < i)
		temp = i;
	return MIN(temp / FACTOR, MAXINTENSITY);
}

static int
darker(const int light)
{
	 return (int) (light * FACTOR);
}


static void
SetValuesPanex(PanexWidget w)
{
	struct tagColor {
		int         red, green, blue;
	} color;
	char        szBuf[80], buf[20], charbuf[2];
	int         pyramid;

	w->panex.tiles = GetPrivateProfileInt(SECTION, "tiles",
		DEFAULTTILES, INIFILE);
	w->panex.mode = GetPrivateProfileInt(SECTION, "mode",
		DEFAULTMODE, INIFILE);
	w->panex.delay = GetPrivateProfileInt(SECTION, "delay",
		DEFAULTDELAY, INIFILE);
	w->panex.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->panex.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.frameGC = RGB(color.red, color.green, color.blue);
	/* gray75 */
	(void) GetPrivateProfileString(SECTION, "tileColor", "191 191 191",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.tileGC = RGB(color.red, color.green, color.blue);
	w->panex.tileBrighterGC = RGB(brighter(color.red),
		brighter(color.green), brighter(color.blue));
	w->panex.tileDarkerGC = RGB(darker(color.red),
		darker(color.green), darker(color.blue));
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "tileBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.inverseGC = RGB(color.red, color.green, color.blue);
	for (pyramid = 0; pyramid < MAXSTACKS - 1; pyramid++) {
		(void) sprintf(buf, "pyramidColor%d", pyramid);
		(void) GetPrivateProfileString(SECTION,
			buf, pyramidColorString[pyramid],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->panex.pyramidGC[pyramid] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "pyramidChar%d", pyramid);
		charbuf[0] = pyramidColorChar[pyramid];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->panex.pyramidChar[pyramid] = szBuf[0];
	}
	(void) GetPrivateProfileString(SECTION, "name", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->panex.username, szBuf);
	 w->panex.username[80] = 0;
}

void
DestroyPanex(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else
#define MAXINTENSITY 0xFFFF

static Pixel
brighter(PanexWidget w, Pixel pixel)
{
	XColor color;
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	if (color.red < i)
		color.red = i;
	if (color.green < i)
		color.green = i;
	if (color.blue < i)
		color.blue = i;
	color.red = (unsigned short) MIN(color.red / FACTOR, MAXINTENSITY);
	color.green = (unsigned short) MIN(color.green / FACTOR, MAXINTENSITY);
	color.blue = (unsigned short) MIN(color.blue / FACTOR, MAXINTENSITY);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static Pixel
darker(PanexWidget w, Pixel pixel)
{
	XColor color;

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	color.red = (unsigned short) (color.red * FACTOR);
	color.green = (unsigned short) (color.green * FACTOR);
	color.blue = (unsigned short) (color.blue * FACTOR);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static void
GetColor(PanexWidget w, int pyramid)
{
	XGCValues   values;
	XtGCMask    valueMask;
	XColor      colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->panex.reverse) {
		values.background = w->panex.foreground;
	} else {
		values.background = w->panex.background;
	}
	if (!w->panex.mono) {
		if (XAllocNamedColor(XtDisplay(w),
				  DefaultColormapOfScreen(XtScreen(w)),
			  w->panex.pyramidName[pyramid], &colorCell, &rgb)) {
			values.foreground = w->panex.pyramidColor[pyramid] =
				colorCell.pixel;
			if (w->panex.pyramidGC[pyramid])
				XtReleaseGC((Widget) w,
					w->panex.pyramidGC[pyramid]);
			w->panex.pyramidGC[pyramid] = XtGetGC((Widget) w,
				valueMask, &values);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->panex.pyramidName[pyramid]);
			stringCat(&buf2, buf1,
				"\" is not defined for pyramid ");
			free(buf1);
			intCat(&buf1, buf2, pyramid);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->panex.reverse) {
		values.background = w->panex.foreground;
		values.foreground = w->panex.background;
	} else {
		values.background = w->panex.background;
		values.foreground = w->panex.foreground;
	}
	if (w->panex.pyramidGC[pyramid])
		XtReleaseGC((Widget) w, w->panex.pyramidGC[pyramid]);
	w->panex.pyramidGC[pyramid] = XtGetGC((Widget) w, valueMask, &values);
}

static void
SetAllColors(PanexWidget w)
{
	XGCValues   values;
	XtGCMask    valueMask;
	int         pyramid;

	valueMask = GCForeground | GCBackground;
	if (w->panex.reverse) {
		values.background = w->panex.foreground;
		values.foreground = w->panex.background;
	} else {
		values.foreground = w->panex.background;
		values.background = w->panex.foreground;
	}
	if (w->panex.inverseGC)
		XtReleaseGC((Widget) w, w->panex.inverseGC);
	w->panex.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->panex.mono) {
		if (w->panex.reverse) {
			values.foreground = w->panex.background;
			values.background = w->panex.foreground;
		} else {
			values.foreground = w->panex.foreground;
			values.background = w->panex.background;
		}
	} else {
		values.foreground = w->panex.frameColor;
		values.background = w->panex.borderColor;
	}
	if (w->panex.frameGC)
		XtReleaseGC((Widget) w, w->panex.frameGC);
	w->panex.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->panex.reverse) {
		if (w->panex.mono) {
			values.background = w->panex.foreground;
			values.foreground = w->panex.background;
		} else {
			values.background = w->panex.tileColor;
			values.foreground = w->panex.borderColor;
		}
	} else {
		if (w->panex.mono) {
			values.foreground = w->panex.foreground;
			values.background = w->panex.background;
		} else {
			values.foreground = w->panex.tileColor;
			values.background = w->panex.borderColor;
		}
	}
	if (w->panex.tileGC)
		XtReleaseGC((Widget) w, w->panex.tileGC);
	w->panex.tileGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->panex.mono) {
		values.foreground = brighter(w, w->panex.tileColor);
	}
	if (w->panex.tileBrighterGC)
		XtReleaseGC((Widget) w, w->panex.tileBrighterGC);
	w->panex.tileBrighterGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->panex.mono) {
		values.foreground = darker(w, w->panex.tileColor);
	}
	if (w->panex.tileDarkerGC)
		XtReleaseGC((Widget) w, w->panex.tileDarkerGC);
	w->panex.tileDarkerGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->panex.reverse) {
		if (w->panex.mono) {
			values.foreground = w->panex.foreground;
			values.background = w->panex.background;
		} else {
			values.foreground = w->panex.tileColor;
			values.background = w->panex.borderColor;
		}
	} else {
		if (w->panex.mono) {
			values.background = w->panex.foreground;
			values.foreground = w->panex.background;
		} else {
			values.background = w->panex.tileColor;
			values.foreground = w->panex.borderColor;
		}
	}
	if (w->panex.borderGC)
		XtReleaseGC((Widget) w, w->panex.borderGC);
	w->panex.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (pyramid = 0; pyramid <= w->panex.mode; pyramid++)
		GetColor(w, pyramid);
	if (w->panex.fontInfo)
		XSetFont(XtDisplay(w), w->panex.borderGC,
			w->panex.fontInfo->fid);
}

static      Boolean
SetValuesPanex(Widget current, Widget request, Widget renew)
{
	PanexWidget c = (PanexWidget) current, w = (PanexWidget) renew;
	Boolean     redraw = False;
	Boolean     redrawTiles = False, setColors = False;
	int         pyramid;

	CheckTiles(w);
	for (pyramid = 0; pyramid <= w->panex.mode; pyramid++) {
		if (strcmp(w->panex.pyramidName[pyramid],
				c->panex.pyramidName[pyramid])) {
			setColors = True;
			break;
		}
	}
	if (w->panex.font != c->panex.font ||
			w->panex.borderColor != c->panex.borderColor ||
			w->panex.reverse != c->panex.reverse ||
			w->panex.mono != c->panex.mono ||
	    		setColors) {
		loadFont(w);
		SetAllColors(w);
		redrawTiles = True;
	} else if (w->panex.background != c->panex.background ||
	    w->panex.foreground != c->panex.foreground ||
	    w->panex.borderColor != c->panex.borderColor ||
	    w->panex.tileColor != c->panex.tileColor ||
	    w->panex.reverse != c->panex.reverse ||
	    w->panex.mono != c->panex.mono ||
	    setColors) {
		SetAllColors(w);
		redrawTiles = True;
	}
	if (w->panex.tiles != c->panex.tiles) {
		EraseFrame(c, False);
		SizePanex(w);
		redraw = True;
	} else if (w->panex.mode != c->panex.mode) {
		ResetTiles(w);
		redraw = True;
	} else if (w->panex.puzzleOffset.x != c->panex.puzzleOffset.x ||
		   w->panex.puzzleOffset.y != c->panex.puzzleOffset.y) {
		EraseFrame(c, False);
		ResizePanex(w);
		redraw = True;
	}
	if (w->panex.menu != -1) {
		switch (w->panex.menu) {
		case 0:
			w->panex.menu = -1;
			GetTiles(w);
			break;
		case 1:
			w->panex.menu = -1;
			WriteTiles(w);
			break;
		case 3:
			w->panex.menu = -1;
			ClearTiles(w);
			break;
		case 4:
			w->panex.menu = -1;
			UndoTiles(w);
			break;
		case 5:
			w->panex.menu = -1;
			SolveTiles(w);
			break;
		case 6:
			w->panex.menu = -1;
			IncrementTiles(w);
			break;
		case 7:
			w->panex.menu = -1;
			DecrementTiles(w);
			break;
		case 8:
			w->panex.menu = -1;
			ModeTiles(w);
			break;
		default:
			w->panex.menu = -1;
			break;
		}
	}
	if (redrawTiles && !redraw && XtIsRealized(renew) &&
		       renew->core.visible) {
		EraseFrame(w, w->panex.focus);
		DrawSlots(w);
		DrawAllTiles(w);
	}
	return (redraw);
}

static void
QuitPanex(PanexWidget w, XEvent * event, char **args, int nArgs)
{
	Display *display = XtDisplay(w);

	if (w->panex.fontInfo) {
		XUnloadFont(display, w->panex.fontInfo->fid);
		XFreeFont(display, w->panex.fontInfo);
	}
	XtCloseDisplay(display);
	exit(0);
}

static void
DestroyPanex(Widget old)
{
	PanexWidget w = (PanexWidget) old;
	int         pyramid;

	for (pyramid = 0; pyramid <= w->panex.mode; pyramid++)
		XtReleaseGC(old, w->panex.pyramidGC[pyramid]);
	XtReleaseGC(old, w->panex.frameGC);
	XtReleaseGC(old, w->panex.tileGC);
	XtReleaseGC(old, w->panex.borderGC);
	XtReleaseGC(old, w->panex.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->panex.select);
}
#endif

#ifndef WINVER
static
#endif
void
ResizePanex(PanexWidget w)
{
#ifdef WINVER
	RECT        rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->panex.delta.x = 8;
	w->panex.delta.y = 2;
	w->panex.pos.x = MAX(((int) w->core.width - w->panex.delta.x) /
			     MAXSTACKS, w->panex.delta.x);
	w->panex.pos.y = MAX(((int) w->core.height - 2 * w->panex.delta.y - 3) /
			     (w->panex.tiles + 1), w->panex.delta.y);
	w->panex.width = w->panex.pos.x * MAXSTACKS + w->panex.delta.x + 2;
	w->panex.height = w->panex.pos.y * (w->panex.tiles + 1) +
		w->panex.delta.y + 3;
	w->panex.puzzleOffset.x = ((int) w->core.width - w->panex.width + 2) / 2;
	w->panex.puzzleOffset.y = ((int) w->core.height - w->panex.height + 2) / 2;
	/* Make the following odd */
	w->panex.tileSize.x = (((w->panex.pos.x - w->panex.delta.x) >> 1) << 1) + 1;
	w->panex.tileSize.y = w->panex.pos.y - w->panex.delta.y + 1;
	w->panex.letterOffset.x = 3;
	w->panex.letterOffset.y = 4;
}

#ifndef WINVER
static
#endif
void
SizePanex(PanexWidget w)
{
	ResetTiles(w);
	ResizePanex(w);
}

#ifndef WINVER
static
#endif
void
InitializePanex(
#ifdef WINVER
PanexWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int         stack;
#ifdef WINVER
	SetValuesPanex(w);
#else
	PanexWidget w = (PanexWidget) renew;

	w->panex.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->panex.mono);
	for (stack = 0; stack <= w->panex.mode; stack++)
		w->panex.pyramidGC[stack] = NULL;
	w->panex.fontInfo = NULL;
	w->panex.frameGC = NULL;
	w->panex.tileGC = NULL;
	w->panex.borderGC = NULL;
	w->panex.inverseGC = NULL;
#endif
	w->panex.focus = False;
	loadFont(w);
	for (stack = 0; stack < MAXSTACKS; stack++) {
		w->panex.tileOfPosition[stack] = NULL;
		w->panex.positionOfTile[stack] = NULL;
	}
	CheckTiles(w);
	InitMoves();
	w->panex.cheat = False;
	SizePanex(w);
#ifdef WINVER
	brush = CreateSolidBrush(w->panex.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
#else
	(void) SRAND(getpid());
	SetAllColors(w);
#endif
}

void
ExposePanex(
#ifdef WINVER
PanexWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	PanexWidget w = (PanexWidget) renew;

	if (!w->core.visible)
		return;
#endif
	EraseFrame(w, w->panex.focus);
	DrawSlots(w);
	DrawAllTiles(w);
}

void
HidePanex(
#ifdef WINVER
const PanexWidget w
#else
PanexWidget w, XEvent * event, char **args, int nArgs
#endif
)
{
	SetPanex(w, PANEX_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectPanex(PanexWidget w
#ifdef WINVER
, const int x
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int         stack;
#ifndef WINVER
        int x = event->xbutton.x;
#endif

	if (CheckSolved(w)) {
		MoveNoTiles(w);
	} else if (((stack = SelectTile(w, x)) >= 0) &&
	    ((w->panex.currentPosition = TopOfStack(w, stack)) >= 0)) {
		w->panex.currentStack = stack;
		DrawTile(w, stack, w->panex.currentPosition, False, TRUE);
	} else
		w->panex.currentStack = -1;
}

#ifndef WINVER
static
#endif
void
ReleasePanex(PanexWidget w
#ifdef WINVER
, const int x
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int         toStack, toPosition;
#ifndef WINVER
        int x = event->xbutton.x;
#endif

	if (w->panex.currentStack < 0)
		return;
	DrawTile(w, w->panex.currentStack, w->panex.currentPosition,
		True, TRUE);
	DrawTile(w, w->panex.currentStack, w->panex.currentPosition,
		False, FALSE);
	if ((toStack = SelectTile(w, x)) >= 0 &&
	    toStack != w->panex.currentStack) {
		if ((toPosition = MovePanex(w, w->panex.currentStack,
				w->panex.currentPosition, toStack)) >= 0) {
			if (CheckSolved(w)) {
				SetPanex(w, PANEX_SOLVED);
			}
		} else if (toPosition == -1) {
			SetPanex(w, PANEX_BLOCKED);
		} else {
			SetPanex(w, PANEX_ILLEGAL);
		}
	}
	w->panex.currentStack = -1;
}

void
GetPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetTiles(w);
}

void
WritePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WriteTiles(w);
}

void
ClearPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearTiles(w);
}

#ifndef WINVER
void
ClearPanexMaybe(PanexWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (w->panex.started) {
		/* Check if one really wants to destroy calculations. */
		SetPanex(w, PANEX_RESET_QUERY);
	}
#endif
}

void
ClearPanex2(PanexWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifndef HAVE_MOTIF
	ClearTiles(w);
#endif
}
#endif

void
UndoPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoTiles(w);
}

void
SolvePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolveTiles(w);
}

void
IncrementPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	IncrementTiles(w);
}

void
DecrementPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	DecrementTiles(w);
}

void
ModePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ModeTiles(w);
}

void
EnterPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->panex.focus = True;
	DrawFrame(w, w->panex.focus);
}

void
LeavePanex(PanexWidget w
#ifndef WINVER
, XEvent * event,  char **args, int nArgs
#endif
)
{
	w->panex.focus = False;
	DrawFrame(w, w->panex.focus);
}
