/*-
# X-BASED PANEX(tm)
#
#  PanexS.c
#
###
#
#  Taken from Some Results on the Panex Puzzle by Mark Manasse &
#  Danny Sleator of AT&T Bell Laboratories, and Victor K. Wei
#  of Bell Communications Research.
#  Thanks to Nick Baxter <nickb@baxterweb.com> for debugging level n > 4
#  and vTrick.
#  Though most code by Rene Jansen <rene.j.jansen@bigfoot.com> is now
#  removed much inspiration was gained by his efforts implementing
#  an algorithm from Quantum January/February 1996 by Vladimir
#  Dubrovsky.
#  Break ability taken from the X puzzle by Don Bennett, HP Labs
#
#  Copyright (c) 1996 - 2003	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Solver file for Panex */

#ifndef WINVER
#define JMP
#ifdef JMP
#include <setjmp.h> /* longjmp ... interrupt */
#endif
#endif
#include "PanexP.h"

static Boolean SolvingFlag = False;
/* TODO Interruptability for Windows */
#ifdef JMP
static Boolean AbortSolvingFlag = False;
static jmp_buf solve_env;

static void
AbortSolving(void)
{
	if (SolvingFlag)
		AbortSolvingFlag = True;
}

static void
ProcessButton(XButtonEvent *event)
{
	AbortSolving();
}

static void
ProcessVisibility(XVisibilityEvent *event)
{
	if (event->state != VisibilityUnobscured) AbortSolving();
}

static void
GetNextEvent(PanexWidget w, XEvent *event)
{
	if (!XCheckMaskEvent(XtDisplay(w), VisibilityChangeMask, event))
		(void) XNextEvent(XtDisplay(w), event);
}

static void
ProcessEvent(XEvent *event)
{
	switch(event->type) {
		case KeyPress:
		case ButtonPress:
			ProcessButton(&event->xbutton);
			break;
		case VisibilityNotify:
			ProcessVisibility(&event->xvisibility);
			break;
		default:
			break;
	}
}

static void
ProcessEvents(PanexWidget w)
{
	XEvent event;

	while(XPending(XtDisplay(w))) {
		GetNextEvent(w, &event);
		ProcessEvent(&event);
	}
}
#endif

#ifdef DEBUG
static int counter; /* debugging */
#endif

#if !defined(SIMPLE_ALG) && !defined(QUANTUM_ALG) && !defined(QUANTUM_JANSEN_ALG) && !defined(BAXTER_ALG) && !defined(ATT_ALG)
#if 0
#define SIMPLE_ALG
#define QUANTUM_ALG
#define QUANTUM_JANSEN_ALG
#define BAXTER_ALG
#endif
#define ATT_ALG
#endif

static      Boolean
MoveATile(PanexWidget w, int fromStack, int toStack)
{
	int         fromPosition;

#ifdef JMP
	if (XPending(XtDisplay(w)))
		ProcessEvents(w);
	if (SolvingFlag && AbortSolvingFlag)
		longjmp(solve_env, 1);
#endif
	if ((fromPosition = TopOfStack(w, fromStack)) < 0 ||
	    MovePanex(w, fromStack, fromPosition, toStack) < 0) {
		(void) printf("move from %d to %d can not be made.",
			      fromStack, toStack);
		return False;
	}
	return True;
}

static void
SolveHanoiMode(PanexWidget w)
{
	int         triangleTurn;
	int         triangleDirection;
	int         triangleStack;
	int         fromStack, toStack, temp1Stack, temp2Stack;
	int         temp1Loc, temp2Loc;

	if (SolvingFlag)
		return;
#ifdef JMP
	if (!setjmp(solve_env))
#endif
	{
		triangleTurn = TRUE;
		triangleDirection = (w->panex.tiles & 1) ? 2 : 1;
		triangleStack = 0;
		while (!CheckSolved(w)) {
			SolvingFlag = True;

			if (triangleTurn) {
				fromStack = triangleStack;
				toStack = (triangleStack + triangleDirection) %
					 MAXSTACKS;
				triangleStack = toStack;
				triangleTurn = FALSE;
			} else {
				temp1Stack = (triangleStack + 1) % MAXSTACKS;
				temp2Stack = (triangleStack + 2) % MAXSTACKS;
				temp1Loc = TopOfStack(w, temp1Stack);
				temp2Loc = TopOfStack(w, temp2Stack);
				if (temp1Loc < 0 && temp2Loc < 0)
					return;
				else if (temp1Loc < 0) {
					fromStack = temp2Stack;
					toStack = temp1Stack;
				} else if (temp2Loc < 0) {
					fromStack = temp1Stack;
					toStack = temp2Stack;
				} else if (w->panex.tileOfPosition[temp1Stack][temp1Loc].loc <
				 w->panex.tileOfPosition[temp2Stack][temp2Loc].loc) {
					fromStack = temp1Stack;
					toStack = temp2Stack;
				} else {
					fromStack = temp2Stack;
					toStack = temp1Stack;
				}
				triangleTurn = TRUE;
			}
			if (!MoveATile(w, fromStack, toStack))
				return;
		}
	}
#ifdef JMP
	else {
		DrawAllTiles(w);
	}
	AbortSolvingFlag = False;
#endif
	SolvingFlag = False;
	w->panex.cheat = True; /* Assume the worst. */
	SetPanex(w, PANEX_COMPUTED);
}

#ifdef DEBUG
static void PrintStacks(PanexWidget w)
{
	int position, stack;

	for (position = 0; position <= w->panex.tiles; position++)
		for (stack = 0; stack < MAXSTACKS; stack++) {
			char tile = '-';

			if (w->panex.tileOfPosition[stack][position].stack >= 0) {
				if (w->panex.tileOfPosition[stack][position].stack == 0)
					tile = 'A';
				else
					tile = 'a';
				tile += (char) w->panex.tileOfPosition[stack][position].loc;
			}
			(void) printf("%d", tile);
			if (stack == MAXSTACKS - 1)
				(void) printf("\n");
			else
				(void) printf(" ");
		}
}
#endif

/*
	Recursive algorithm starts at level 5, shown by computer checking
	to be the shortest algorithm, i.e. the upper bound U(n), to at
	least to level 8.
	Lower bounds discussed by Manasse, Sleator, & Wei is given as L(n).
	T(n) = move a tower of hight n
	X(n) = transfer or swap towers
	S(n) = sink a n level piece in center stack
	SQ(n) = move a single piece (for Quantum and Baxter alg.)
	DQ(n) = move a 2 n and n-1 and reorder (for Quantum and Baxter alg.)
	WB(n) = solve but one piece n in center (Baxter alg.)

	n	T(n)	X(n)	U(n)	L(n)
	1	1	3
	2	3	13
	3	9	42
	4	24	128
	5	58	343	343	320
	6	143	881	881	796
	7	345	2189	2189	1944
	8	836	5359	5359	4716
	9	2018	?	13023	11408
	10	4875	?	31537	27564
*/

#ifdef DEBUG
static Boolean LEFT = False;
static Boolean RIGHT = True;
static Boolean NORMAL = False;
static Boolean INVERSE = True;
#endif

/* Randomize to make it interesting */
static void centerSwap(PanexWidget w)
{
	if (NRAND(2) == 1) {
		(void) MoveATile(w, 1, 0);
		(void) MoveATile(w, 1, 2);
		(void) MoveATile(w, 0, 1);
		(void) MoveATile(w, 2, 1);
	} else {
		(void) MoveATile(w, 1, 2);
		(void) MoveATile(w, 1, 0);
		(void) MoveATile(w, 2, 1);
		(void) MoveATile(w, 0, 1);
	}
}

static void sideSwap(PanexWidget w, Boolean side)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last  = 2;
	}
	if (NRAND(2) == 1) {
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, first);
	} else {
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, first);
	}
}

/*-
	n	T(n)	TBar(n)	S(n)	SBar(n)
	1	1	1	-	-
	2	3	2	2	2
	3	9	9	8	8
	4	24	23	22	22
	5	58	58	57	57
	6	143	142	141	141
	7	345	345	344	344
	8	836	835	834	834
	9	2018	2018	2017	2017
	10	4875	4874	4873	4873
*/

/* Changes to TBar, SBar, V to use the "vTrick" that        */
/* swaps B and b, rather than calling LinvTBar(3) and       */
/* RinvTBar(3).                                             */
/*                                                          */
/* All calls to TBar and SBar require a new parameter,      */
/* vTrick. This will always be False, except when called    */
/* from V, and when carefully passed down from within       */
/* TBar and SBar. When True, the last call to LInvTBar(3)   */
/* in LInvTBar(n-1) and the first call to RinvTBar(3)       */
/* in RInvSBar(n-1) are suppressed. V added the replacement */
/* 19 moves (with 18 moves removed).                        */

static void T(PanexWidget w, int n, Boolean side, Boolean inv);
static void TBar(PanexWidget w, int n,
	Boolean side, Boolean inv, Boolean vTrick, Boolean qTrick);
static void S(PanexWidget w, int n, Boolean side, Boolean inv);
static void SBar(PanexWidget w, int n,
	Boolean side, Boolean inv, Boolean vTrick);

static void T(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
		}
	if (inv) {
		if (n == 1) {
			(void) MoveATile(w, 1, first);
		} else if (n == 2) {
			(void) MoveATile(w, 1, last);
			TBar(w, n, side, inv, False, False);
		} else if (n == 3) {
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, 1, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
			T(w, 2, side, inv);
		} else if (n >= 4) {
			T(w, n - 2, side, inv); /* Paper says invert but... */
			centerSwap(w);
			if (((n & 1) == 0) && (NRAND(2) == 1)) {
				S(w, n - 1, side, inv);
				T(w, n - 1, side, inv);
			} else {
				SBar(w, n - 1, side, inv, False);
				TBar(w, n - 1, side, inv, False, False);
			}
		}
	} else {
		if (n == 1) {
			(void) MoveATile(w, first, 1);
		} else if (n == 2) {
			TBar(w, n, side, inv, False, False);
			(void) MoveATile(w, last, 1);
		} else if (n == 3) {
			T(w, 2, side, inv);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, 1);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, first, 1);
		} else if (n >= 4) {
			if (((n & 1) == 0) && (NRAND(2) == 1)) {
				T(w, n - 1, side, inv);
				S(w, n - 1, side, inv);
			} else {
				TBar(w, n - 1, side, inv, False, False);
				SBar(w, n - 1, side, inv, False);
			}
			centerSwap(w);
			T(w, n - 2, side, inv); /* Paper says invert but... */
		}
	}
}

static void TBar(PanexWidget w, int n,
	Boolean side, Boolean inv, Boolean vTrick, Boolean qTrick)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		if (n == 1) {
			(void) MoveATile(w, last, first);
		} else if (n == 2) {
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, first);
		} else if (n == 3) {
			if (!vTrick) {
				if (!qTrick)
					(void) MoveATile(w, last, first);
				(void) MoveATile(w, 1, first);
				(void) MoveATile(w, 1, last);
				(void) MoveATile(w, first, 1);
				(void) MoveATile(w, first, 1);
				(void) MoveATile(w, last, first);
				T(w, 2, side, inv);
			}
		} else if (n >= 4) {
			TBar(w, n - 2, side, inv, False, qTrick); /* Paper says invert but... */
			centerSwap(w);
			if (((n & 1) == 0) && (NRAND(2) == 1) && !vTrick) {
				S(w, n - 1, side, inv);
				T(w, n - 1, side, inv);
			} else {
				SBar(w, n - 1, side, inv, False);
				TBar(w, n - 1, side, inv, vTrick, False);
			}
		}
	} else {
		if (n == 1) {
			(void) MoveATile(w, first, last);
		} else if (n == 2) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, 1);
		} else if (n == 3) {
			if (!vTrick) {
				T(w, 2, side, inv);
				(void) MoveATile(w, first, last);
				(void) MoveATile(w, 1, first);
				(void) MoveATile(w, 1, first);
				(void) MoveATile(w, last, 1);
				(void) MoveATile(w, first, 1);
				if (!qTrick)
					(void) MoveATile(w, first, last);
			}
		} else if (n >= 4) {
			if (((n & 1) == 0) && (NRAND(2) == 1) && ! vTrick) {
				T(w, n - 1, side, inv);
				S(w, n - 1, side, inv);
			} else {
				TBar(w, n - 1, side, inv, vTrick, False);
				SBar(w, n - 1, side, inv, False);
			}
			centerSwap(w);
			TBar(w, n - 2, side, inv, False, qTrick); /* Paper says invert but... */
		}
	}
}

static void S(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		if (n == 2) {
			SBar(w, n, side, inv, False);
			(void) MoveATile(w, last, 1);
		} else if (n == 3) {
			sideSwap(w, side);
			(void) MoveATile(w, 1, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
		} else if (n >= 4) {
			if (((n & 1) == 1) && (NRAND(2) == 1)) {
				T(w, n - 2, side, !inv);
				S(w, n - 1, side, !inv);
			} else {
				TBar(w, n - 2, side, !inv, False, False);
				SBar(w, n - 1, side, !inv, False);
			}
			centerSwap(w);
			S(w, n - 1, side, inv);
		}
	} else {
		if (n == 2) {
			(void) MoveATile(w, 1, last);
			SBar(w, n, side, inv, False);
		} else if (n == 3) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, 1);
			sideSwap(w, side);
		} else if (n >= 4) {
			S(w, n - 1, side, inv);
			centerSwap(w);
			if (((n & 1) == 1) && (NRAND(2) == 1)) {
				S(w, n - 1, side, !inv);
				T(w, n - 2, side, !inv);
			} else {
				SBar(w, n - 1, side, !inv, False);
				TBar(w, n - 2, side, !inv, False, False);
			}
		}
	}
}

static void SBar(PanexWidget w, int n,
	Boolean side, Boolean inv, Boolean vTrick)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		if (n == 2) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
		} else if (n == 3) {
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, 1);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
		} else if (n >= 4) {
			if (((n & 1) == 1) && (NRAND(2) == 1) && !vTrick) {
				T(w, n - 2, side, !inv);
				S(w, n - 1, side, !inv);
			} else {
				TBar(w, n - 2, side, !inv, vTrick, False);
				SBar(w, n - 1, side, !inv, False);
			}
			centerSwap(w);
			SBar(w, n - 1, side, inv, False);
		}
	} else {
		if (n == 2) {
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
		} else if (n == 3) {
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, 1, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, 1, first);
		} else if (n >= 4) {
			SBar(w, n - 1, side, inv, False);
			centerSwap(w);
			if (((n & 1) == 1) && (NRAND(2) == 1) && !vTrick) {
				S(w, n - 1, side, !inv);
				T(w, n - 2, side, !inv);
			} else {
				SBar(w, n - 1, side, !inv, False);
				TBar(w, n - 2, side, !inv, vTrick, False);
			}
		}
	}
}

static void W(PanexWidget w, Boolean side, Boolean inv) {
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, last);
	} else {
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
	}
}

static void VTrick(PanexWidget w, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		W(w, side, inv);
	} else {
		W(w, side, inv);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, first);
	}
}

static void V(PanexWidget w, int n, Boolean side, Boolean inv)
{
	if (inv) {
		SBar(w, n, !side, !inv, True);
		VTrick(w, side, inv);
		TBar(w, n - 1, side, !inv, True, False);
	} else {
		TBar(w, n - 1, side, !inv, True, False);
		VTrick(w, side, inv);
		SBar(w, n, !side, !inv, True);
	}
}

static void Y(PanexWidget w, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
	} else {
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
	}
}

static void YBar(PanexWidget w, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
	} else {
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
	}
}

static void Z(PanexWidget w, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		centerSwap(w);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, 1);
	} else {
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, first);
		centerSwap(w);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, last);
	}
}

static void SQ(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		if (n == 1) {
			(void) MoveATile(w, 1, first);
		} else if (n == 2) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, first);
		} else if (n >= 3) {
			if (((n & 1) == 0) && (NRAND(2) == 1)) {
				S(w, n, side, inv);
				T(w, n - 1, side, inv);
			} else {
				SBar(w, n, side, inv, False);
				TBar(w, n - 1, side, inv, False, False);
			}
		}
	} else {
		if (n == 1) {
			(void) MoveATile(w, first, 1);
		} else if (n == 2) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
		} else if (n >= 3) {
			if (((n & 1) == 0) && (NRAND(2) == 1)) {
				T(w, n - 1, side, inv);
				S(w, n, side, inv);
			} else {
				TBar(w, n - 1, side, inv, False, False);
				SBar(w, n, side, inv, False);
			}
		}
	}
}

static void DQ(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (n <= 1)
		return;
	if (inv) {
		if (n == 2) {
			centerSwap(w);
			(void) MoveATile(w, 1, last);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, first);
		} else if (n >= 3) {
			centerSwap(w);
			if (((n & 1) == 0) && (NRAND(2) == 1)) {
				S(w, n - 1, side, inv);
				T(w, n - 1, side, inv);
			} else {
				SBar(w, n - 1, side, inv, False);
				TBar(w, n - 1, side, inv, False, False);
			}
		}
	} else {
		if (n == 2) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, 1);
			centerSwap(w);
		} else if (n >= 3) {
			if (((n & 1) == 0) && (NRAND(2) == 1)) {
				T(w, n - 1, side, inv);
				S(w, n - 1, side, inv);
			} else {
				TBar(w, n - 1, side, inv, False, False);
				SBar(w, n - 1, side, inv, False);
			}
			centerSwap(w);
		}
	}
}

static void WB(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		if (n == 1) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
		} else if (n == 2) {
			sideSwap(w, side);
			(void) MoveATile(w, last, 1);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, last);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, first);
		} else if (n >= 3) {
			SQ(w, n, side, !inv);
			SQ(w, n - 1, !side, !inv);
			centerSwap(w);
			SQ(w, n - 1, !side, inv);
			SQ(w, n - 1, side, !inv);
			SQ(w, n - 2, !side, !inv);
			centerSwap(w);
			SQ(w, n - 2, !side, inv);
			WB(w, n - 2, side, inv);
			DQ(w, n, side, inv);
		}
	} else {
		if (n == 1) {
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
		} else if (n == 2) {
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, 1);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, 1, last);
			sideSwap(w, side);
		} else if (n >= 3) {
			DQ(w, n, side, inv);
			WB(w, n - 2, side, inv);
			SQ(w, n - 2, !side, inv);
			centerSwap(w);
			SQ(w, n - 2, !side, !inv);
			SQ(w, n - 1, side, !inv);
			SQ(w, n - 1, !side, inv);
			centerSwap(w);
			SQ(w, n - 1, !side, !inv);
			SQ(w, n, side, !inv);
		}
	}
}

#if defined(QUANTUM_ALG) || defined(QUANTUM_JANSEN_ALG)
static void BQ(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		SQ(w, n - 1, side, !inv);
		centerSwap(w);
		SQ(w, n - 1, side, inv);
		SQ(w, n - 1, !side, !inv);
	} else {
		SQ(w, n - 1, !side, !inv);
		SQ(w, n - 1, side, inv);
		centerSwap(w);
		SQ(w, n - 1, side, !inv);
	}
}
#endif

static void XSimple(PanexWidget w, int n, Boolean side, Boolean inv)
{
	/* Yeah, it works but it takes a long time */
	if (inv) {
		XSimple(w, n - 1, side, inv);
		TBar(w, n - 1, !side, !inv, False, False);
		SBar(w, n, !side, !inv, False);
		TBar(w, n - 1, side, !inv, False, False);
		SBar(w, n, side, !inv, False);
		SBar(w, n, !side, inv, False);
		TBar(w, n - 1, !side, inv, False, False);
		SBar(w, n, side, inv, False);
		TBar(w, n - 1, side, inv, False, False);
	} else {
		TBar(w, n - 1, side, inv, False, False);
		SBar(w, n, side, inv, False);
		TBar(w, n - 1, !side, inv, False, False);
		SBar(w, n, !side, inv, False);
		SBar(w, n, side, !inv, False);
		TBar(w, n - 1, side, !inv, False, False);
		SBar(w, n, !side, !inv, False);
		TBar(w, n - 1, !side, !inv, False, False);
		XSimple(w, n - 1, side, inv);
	}
}

#ifdef BAXTER_ALG
/* From Nick Baxter */
static void XBaxter(PanexWidget w, int n, Boolean side, Boolean inv)
{
	if (inv) {
		SQ(w, n, !side, !inv);
		WB(w, n, side, inv);
	} else {
		WB(w, n, side, inv);
		SQ(w, n, !side, !inv);
	}
}
#endif

#if defined(QUANTUM_ALG) || defined(QUANTUM_JANSEN_ALG)
/* From Quantum */
static void XQuantum(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last, i;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (inv) {
		SQ(w, n, side, !inv);
		SQ(w, n, !side, !inv);

#ifdef QUANTUM_ALG
		/* Original Algorithm */
		for (i = n; i >= 3; i--)
			BQ(w, i, side, inv);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, first);
#else /* QUANTUM_JANSEN_ALG */
		/* Rene Jansen's one move improvement */
		for (i = n; i >= 4; i--)
			BQ(w, i, side, inv);
		sideSwap(w, side);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
#endif
		(void) MoveATile(w, last, first);
		if ((n & 1) == 1)
			(void) MoveATile(w, 1, last);
		else
			(void) MoveATile(w, 1, first);
/* qTrick combine next (first, last) move to save a move if n odd */
		TBar(w, n, !side, inv, False, ((n & 1) == 1));
	} else {
		TBar(w, n, !side, inv, False, ((n & 1) == 1));
/* qTrick combine previous (last, first) move to save a move if n odd */
		if ((n & 1) == 1)
			(void) MoveATile(w, last, 1);
		else
			(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, last);

#ifdef QUANTUM_ALG
		/* Original Algorithm */
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, last, first);
		for (int i = 3; i <= n; i++)
			BQ(w, i, inv);
#else /* QUANTUM_JANSEN_ALG */
		/* Rene Jansen's one move improvement */
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		(void) MoveATile(w, last, first);
		(void) MoveATile(w, 1, last);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, first, 1);
		(void) MoveATile(w, first, last);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, 1, first);
		(void) MoveATile(w, last, 1);
		sideSwap(w, side);
		for (i = 4; i <= n; i++)
			BQ(w, i, side, inv);
#endif
		SQ(w, n, !side, !inv);
		SQ(w, n, side, !inv);
	}
}
#endif

static void X(PanexWidget w, int n, Boolean side, Boolean inv)
{
	int first, last, i;

	if (side) {
		first = 2;
		last  = 0;
	} else {
		first = 0;
		last = 2;
	}
	if (n == 1) {
		if (inv) {
			T(w, 1, !side, !inv);
			(void) MoveATile(w, first, last);
			T(w, 1, side, inv);
		} else {
			T(w, 1, side, inv);
			(void) MoveATile(w, last, first);
			T(w, 1, !side, !inv);
		}
	} else if (n == 2) {
		if (inv) {
			TBar(w, 2, !side, !inv, False, False);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, last, 1);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, last);
			T(w, 2, side, inv);
		} else {
			T(w, 2, side, inv);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, 1, last);
			(void) MoveATile(w, first, last);
			(void) MoveATile(w, first, 1);
			(void) MoveATile(w, last, first);
			(void) MoveATile(w, 1, first);
			(void) MoveATile(w, last, first);
			TBar(w, 2, !side, !inv, False, False);
		}
	} else if (n == 3) {
		if (inv) {
			Z(w, side, inv);
			Y(w, side, inv);
			TBar(w, 2, !side, inv, False, False);
			SBar(w, 3, side, inv, False);
			TBar(w, 2, side, inv, False, False);
		} else {
			TBar(w, 2, side, inv, False, False);
			SBar(w, 3, side, inv, False);
			TBar(w, 2, !side, inv, False, False);
			Y(w, side, inv);
			Z(w, side, inv);
		}
	} else if (n == 4) {
		if (inv) {
			TBar(w, 3, !side, !inv, False, False);
			SBar(w, 3, !side, !inv, False);
			centerSwap(w);
			S(w, 3, !side, inv);
			W(w, side, inv);
			SBar(w, 4, side, !inv, False);
			SBar(w, 4, !side, inv, False);
			TBar(w, 2, !side, inv, False, False);
			TBar(w, 2, side, !inv, False, False);
			YBar(w, side, inv);
			TBar(w, 2, !side, inv, False, False);
			centerSwap(w);
			SBar(w, 3, side, inv, False);
			TBar(w, 3, side, inv, False, False);
		} else {
			TBar(w, 3, side, inv, False, False);
			SBar(w, 3, side, inv, False);
			centerSwap(w);
			TBar(w, 2, !side, inv, False, False);
			YBar(w, side, inv);
			TBar(w, 2, side, !inv, False, False);
			TBar(w, 2, !side, inv, False, False);
			SBar(w, 4, !side, inv, False);
			SBar(w, 4, side, !inv, False);
			W(w, side, inv);
			S(w, 3, !side, inv);
			centerSwap(w);
			SBar(w, 3, !side, !inv, False);
			TBar(w, 3, !side, !inv, False, False);
		}
	} else if (n >= 5) {
		if (inv) {
			TBar(w, n - 1, !side, !inv, False, False);
			V(w, n, side, inv);
			SBar(w, n, side, !inv, False);
			SBar(w, n, !side, inv, False);
			TBar(w, 2, !side, inv, False, False);
			TBar(w, 2, side, !inv, False, False);
			YBar(w, side, inv);
			TBar(w, 2, !side, inv, False, False);
			TBar(w, 2, side, !inv, False, False);
			for (i = n - 5; i >= 1; i--) {
				SBar(w, n - 1 - i, side, !inv, False);
				SBar(w, n - 1 - i, !side, inv, False);
				TBar(w, n - 2 - i, !side, inv, False, False);
				TBar(w, n - 2 - i, side, !inv, False, False);
			}
			SBar(w, n - 1, side, !inv, False);
			SBar(w, n - 1, !side, inv, False);
			TBar(w, n - 2, !side, inv, False, False);
			centerSwap(w);
			SBar(w, n - 1, side, inv, False);
			TBar(w, n - 1, side, inv, False, False);
		} else {
			TBar(w, n - 1, side, inv, False, False);
			SBar(w, n - 1, side, inv, False);
			centerSwap(w);
			TBar(w, n - 2, !side, inv, False, False);
			SBar(w, n - 1, !side, inv, False);
			SBar(w, n - 1, side, !inv, False);
			for (i = 1; i <= n - 5; i++) {
				TBar(w, n - 2 - i, side, !inv, False, False);
				TBar(w, n - 2 - i, !side, inv, False, False);
				SBar(w, n - 1 - i, !side, inv, False);
				SBar(w, n - 1 - i, side, !inv, False);
			}
			TBar(w, 2, side, !inv, False, False);
			TBar(w, 2, !side, inv, False, False);
			YBar(w, side, inv);
			TBar(w, 2, side, !inv, False, False);
			TBar(w, 2, !side, inv, False, False);
			SBar(w, n, !side, inv, False);
			SBar(w, n, side, !inv, False);
			V(w, n, side, inv);
			TBar(w, n - 1, !side, !inv, False, False);
		}
	}
}

static void
SolvePanexMode(PanexWidget w)
{
	if (SolvingFlag)
		return;
#ifdef JMP
	if (!setjmp(solve_env))
#endif
	{
		SolvingFlag = True;

#ifdef WINVER
		(void) SRAND(time(NULL));
#else
		(void) SRAND(getpid());
#endif
		if (w->panex.tiles > 0) {
			Boolean side = (NRAND(2) == 1);
			Boolean inv = (NRAND(2) == 1);
#ifdef DEBUG
			/* good for debugging */
#if 0
			Boolean side = LEFT;
#else
			Boolean side = RIGHT;
#endif
#if 0
			Boolean side = NORMAL;
#else
			Boolean side = INVERSE;
#endif
#endif

#ifndef ATT_ALG
/* Other, less efficient algorithms */
			if (w->panex.tiles >= 3) {
#ifdef SIMPLE_ALG
				XSimple(w, w->panex.tiles, side, inv);
#else
#ifdef BAXTER_ALG
				XBaxter(w, w->panex.tiles, side, inv);
#else /* QUANTUM_ALG || QUANTUM_JANSEN_ALG */
				XQuantum(w, w->panex.tiles, side, inv);
#endif
#endif
			} else
#else
			X(w, w->panex.tiles, side, inv);
#endif
		}
	}
#ifdef JMP
	else {
		DrawAllTiles(w);
	}
	AbortSolvingFlag = False;
#endif
	SolvingFlag = False;
	w->panex.cheat = True; /* Assume the worst. */
	SetPanex(w, PANEX_COMPUTED);
#ifdef DEBUG
	counter = 0;
#endif
}

void
SolveTilesFromStart(PanexWidget w)
{
	if (w->panex.mode == PANEX)
		SolvePanexMode(w);
	else
		SolveHanoiMode(w);
}
