/*-
# MOTIF/X-BASED PANEX(tm)
#
#  xpanex.c
#
###
#
#  Copyright (c) 1996 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5.3: 96/04/30 Xt/Motif
*/

#ifdef WINVER
#include "PanexP.h"
#include "wpanex.h"
#define TITLE "wpanex"

static PanexRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->panex.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Panex.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/panex.xpm"
#endif
#include "icons/panex.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1.1\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
	"I got some help from Rene Jansen <rene.j.jansen@bigfoot.com> "
	"and Nick Baxter <nickb@baxterweb.com>.\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]  "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-tile {color}] [-pyramid{0|1} {color}]\n"
	"[-tiles {int}] [-{mode {int}|hanoi|panex}] [-delay msecs] "
	"[-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"The original Panex puzzle is from the Japanese Magic Company "
	"estimated from the 1980's.  Mathematicians at Bell\n"
	"Laboratories estimated the number of moves to swap 2 columns "
	"of order 10 to be  27,564 <= N <= 31,537.  It came\n"
	"in two varieties: one with a magenta and a yellow pyramid "
	"of order 10 on silver tiles; in the gold version, the\n"
	"gold version pieces of each color look alike (i.e. no "
	"pyramid is drawn on them), this is a little harder.\n"
	"The original Tower of Hanoi puzzle is the invention of "
	"Edouard Lucas and temple of Benares of the god Brahma is\n"
	"also his invention.\n"
};
static const char featuresHelp[] = {
	"Press \"mouse-left\" button to move a tile in the top tile "
	"of a column.  Release \"mouse-left\" button on\n"
	"another column to move the tile to that column.  It will "
	"not move if blocked.\n"
	"Click \"mouse-right\" button, or press \"C\", \"c\", \"R\","
	"or \"r\" keys, to clear/reset the puzzle.\n"
	"Press \"M\" or \"m\" keys to toggle between hanoi "
	"(one pyramid column) and panex (two pyramid columns) modes\n"
	"(they each have different rules).\n"
	"In Hanoi, one can not place  larger trapezoid on a smaller "
	"trapezoid.  Here the goal is to move the pile from\n"
	"the left peg to the rightmost peg.\n"
	"In Panex, a tile can not go lower that its initial starting "
	"point.  Here the goal is to exchange the 2 piles.\n"
	"Press \"I\" or \"i\" keys to increase the number of tiles.\n"
	"Press \"D\" or \"d\" keys to decrease the number of tiles.\n"
	"Press \"S\" or \"s\" keys to auto-solve.  Unfortunately, its "
	"only implemented from the starting position.\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"Press \"Esc\" key to hide program.\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"Unlike other puzzles in the collection there is no way to "
	"move pieces without drag and drop.\n"
};
static const char referencesHelp[] = {
	"Mark Manasse & Danny Sleator of AT&T Bell Laboratories  "
	"and Victor K. Wei of Bell Communications Research,\n"
	"Some Results on the Panex Puzzle, Murray Hill, NJ, 1985 "
	"20 pp. (unpublished).\n"
	"Vladimir Dubrovsky,  Nesting Puzzles  Part 1: Moving oriental "
	" Quantum  January/February  1996  pp 55-57,\n"
	"50-51.\n"
	"L.  E.  Horden,  Sliding  Piece  Puzzles  (Recreations  in  "
	"Mathematics  Series),  Oxford University Press  1986,\n"
	"pp 144, 145.\n"
	"Jerry Slocum & Jack Botermans,  Puzzles Old & New (How to Make "
	"and  Solve Them),  University of Washington\n"
	"Press, Seattle 1987, p 135.\n"
};
#endif

#ifndef SCOREFILE
#define SCOREFILE "panex.scores"
#endif

#define MAXTILES 10
#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord panexRecord[MAXMODES][MAXTILES - MINTILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
#define MULTSPEED 20
static Widget moves, record, message, modes[MAXMODES], tile, speed;
static char buff[21];
static const char *modeString[] =
{
	"Hanoi", "Panex"
};
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget clearDialog;
#else
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap panexIcon = None;
static Widget topLevel, panex;
static Arg arg[3];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int tiles, int moves, char *msg)
{
	(void) sprintf(titleDsp, "%s: %d@ (%d/%s) - %s",
		progDsp, tiles, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int i, mode;

	for (mode = 0; mode < MAXMODES; mode++)
		for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
			panexRecord[mode][i].score = NEVER;
			(void) strncpy(panexRecord[mode][i].name, NOACCESS,
				USERNAMELEN);
		}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, mode, n;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (mode = 0; mode < MAXMODES; mode++)
		for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
			(void) fscanf(fp, "%d %s\n", &n, username);
			if (n <= panexRecord[mode][i].score ||
			    panexRecord[mode][i].score <= NEVER) {
				panexRecord[mode][i].score = n;
				(void) strncpy(panexRecord[mode][i].name,
					username, USERNAMELEN);
			}
		}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, mode;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (mode = 0; mode < MAXMODES; mode++) {
			for (i = 0; i < MAXTILES - MINTILES + 1; i++)
				(void) fprintf(fp, "%d %s\n",
					panexRecord[mode][i].score,
					panexRecord[mode][i].name);
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(int tiles, int mode)
{
	int i = tiles - MINTILES, j = mode - HANOI;

	if (tiles > MAXTILES)
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	else if (panexRecord[j][i].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			panexRecord[j][i].score, panexRecord[j][i].name);

	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, int tiles, int mode)
{
	int i = tiles - MINTILES, j = mode - HANOI;

	if (tiles <= MAXTILES && (counter < panexRecord[j][i].score ||
	    panexRecord[j][i].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		panexRecord[j][i].score = counter;
		(void) strncpy(panexRecord[j][i].name, usernameDsp, USERNAMELEN);
		WriteRecords();
		PrintRecord(tiles, mode);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
	PanexWidget w, HBRUSH brush
#else
	Widget w
#endif
)
{
	int tiles, mode;
	char *username;

#ifdef WINVER
	InitializePanex(w, brush);

	tiles = w->panex.tiles;
	mode = w->panex.mode;
	username = w->panex.username;
#else
	int delay;

	XtVaGetValues(w,
		XtNuserName, &username,
		XtNtiles, &tiles,
		XtNmode, &mode,
		XtNdelay, &delay, NULL);
#ifdef HAVE_MOTIF
	if (tiles > MAXTILES)
		XtVaSetValues(tile, XmNmaximum, tiles, NULL);
	XmScaleSetValue(tile, tiles);
	XmToggleButtonSetState(modes[mode - HANOI], True, False);
	XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay / MULTSPEED - 1);
#endif
#endif
	/* SET_STARTED(w, False); */
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
		/* The NOACCESS is not necessary, but it may stop silliness. */
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", "guest");
#endif
	}
	PrintRecord(tiles, mode);
#ifndef HAVE_MOTIF
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		tiles, movesDsp, messageDsp);
#endif

}

#ifdef WINVER
void
SetPanex(PanexWidget w, int reason)
#else
static void
CallbackPanex(Widget w, caddr_t clientData, panexCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int tiles, mode;
	Boolean cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	tiles = w->panex.tiles;
	mode = w->panex.mode;
	cheat = w->panex.cheat;
#else
	XtVaGetValues(w,
		XtNtiles, &tiles,
		XtNmode, &mode,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case PANEX_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case PANEX_RESET_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(clearDialog);
#else
			XtSetArg(arg[0], XtNmenu, 3); /* menu choice */
			XtSetValues(panex, arg, 1);
#endif
			break;
#endif
		case PANEX_RESTORE:
		case PANEX_RESET:
			movesDsp = 0;
			break;
		case PANEX_ILLEGAL:
			(void) strncpy(messageDsp, "Illegal move", MESSAGELEN);
			break;
		case PANEX_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case PANEX_SPACE:
#if 0
			/* Too annoying */
			(void) strncpy(messageDsp, "Spaces can not slide",
				MESSAGELEN));
#endif
			break;
		case PANEX_IGNORE:
			(void) strncpy(messageDsp, "Clear to start",
				MESSAGELEN);
			break;
		case PANEX_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case PANEX_SOLVED:
			if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, tiles, mode))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case PANEX_DEC:
			movesDsp = 0;
			tiles--;
			PrintRecord(tiles, mode);
#ifdef WINVER
			w->panex.tiles = tiles;
#else
			XtSetArg(arg[0], XtNtiles, tiles);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tile, tiles);
			if (tiles >= MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
#endif
#endif
			break;
		case PANEX_INC:
			movesDsp = 0;
			tiles++;
			PrintRecord(tiles, mode);
#ifdef WINVER
			w->panex.tiles = tiles;
#else
			XtSetArg(arg[0], XtNtiles, tiles);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			if (tiles > MAXTILES)
				XtVaSetValues(tile, XmNmaximum, tiles, NULL);
			XmScaleSetValue(tile, tiles);
#endif
#endif
			break;
		case PANEX_MODE:
			movesDsp = 0;
			mode = (mode == PANEX) ? HANOI : PANEX;
			PrintRecord(tiles, mode);
#ifdef WINVER
			w->panex.mode = mode;
#else
			XtSetArg(arg[0], XtNmode, mode);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(modes[mode - HANOI], True,
				True);
#endif
#endif
			break;
		case PANEX_COMPUTED:
			SET_STARTED(w, False);
			break;
		case PANEX_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
	}
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		tiles, movesDsp, messageDsp);
#endif

}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.panex.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterPanex(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.panex.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeavePanex(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyPanex(brush);
			break;
		case WM_SIZE:
			ResizePanex(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposePanex(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ClearPanex(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			SelectPanex(&widget, LOWORD(lParam));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ReleasePanex(&widget, LOWORD(lParam));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetPanex(&widget);
					ResizePanex(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WritePanex(&widget);
					break;
				case IDM_EXIT:
					DestroyPanex(brush);
					break;
				case IDM_HIDE:
					HidePanex(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearPanex(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoPanex(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolvePanex(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_MODE:
					(void) ModePanex(&widget);
					SizePanex(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DEC:
				case IDM_INC:
					((LOWORD(wParam) == IDM_DEC) ?
						DecrementPanex(&widget) : IncrementPanex(&widget));
					SizePanex(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance,
	HINSTANCE hPrevInstance,
	LPSTR lpCmdLine,
	int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-tile {color}] [-pyramid{0|1} {color}]\n");
	(void) fprintf(stderr,
		"\t[-tiles {int}] [-{mode {int}|hanoi|panex}] [-delay msecs]\n");
	(void) fprintf(stderr,
		"\t[-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*panex.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*panex.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*panex.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*panex.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*panex.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*panex.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*panex.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*panex.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*panex.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*panex.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*panex.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-pyramid0", (char *) "*panex.pyramidColor0", XrmoptionSepArg, NULL},
	{(char *) "-pyramid1", (char *) "*panex.pyramidColor1", XrmoptionSepArg, NULL},
	{(char *) "-tiles", (char *) "*panex.tiles", XrmoptionSepArg, NULL},
	{(char *) "-mode", (char *) "*panex.mode", XrmoptionSepArg, NULL},
	{(char *) "-hanoi", (char *) "*panex.mode", XrmoptionNoArg, (char *) "0"},
	{(char *) "-panex", (char *) "*panex.mode", XrmoptionNoArg, (char *) "1"},
	{(char *) "-delay", (char *) "*panex.delay", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*panex.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*panex.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackPanexClear(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 3); /* menu choice */
		XtSetValues(panex, arg, 1);
	}
}

static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int tiles = cbs->value, mode, old;

	XtVaGetValues(panex,
		XtNtiles, &old,
		XtNmode, &mode, NULL);
	if (old == tiles)
		return;
	XtVaSetValues(panex,
		XtNtiles, tiles, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, mode);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
ModeToggle(Widget w, int mode, XmToggleButtonCallbackStruct * cbs)
{
	int tiles, old;

	XtVaGetValues(panex,
		XtNtiles, &tiles,
		XtNmode, &old, NULL);
	if (old == mode)
		return;
	XtVaSetValues(panex,
		XtNmode, mode + HANOI, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(tiles, mode + HANOI);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MULTSPEED * (MAXSPEED + MINSPEED - cbs->value - 1);
	int oldDelay;

	XtVaGetValues(panex,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(panex, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(panex, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(panex, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d\n", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget panel, mainPanel, controlPanel;
	Widget movesRowCol, radioRowCol, tileRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, solveString;
	XmString incrementString, decrementString, mode;
	unsigned int i;
#endif

	progDsp = argv[0];

	topLevel = XtInitialize(argv[0], "Panex",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) panex_xpm, &panexIcon, NULL,
			&xpmAttributes);
	}
	if (panexIcon == (Pixmap) NULL)
#endif
		panexIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) panex_bits,
			panex_width, panex_height);
	XtSetArg(arg[0], XtNiconPixmap, panexIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	panel = XtVaCreateManagedWidget("panel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(panel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	solveString = XmStringCreateSimple((char *) "Solve");
	incrementString = XmStringCreateSimple((char *) "Increment");
	decrementString = XmStringCreateSimple((char *) "Decrement");
	mode = XmStringCreateSimple((char *) "Mode");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, mode, 'M', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(solveString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(mode);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	clearDialog = createQuery(topLevel,
		(char *) "Are you sure you want to start over?",
		(char *) "Clear Query",
		(XtCallbackProc) CallbackPanexClear);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, panel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Clear", 6, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	radioRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN,
		XmNradioBehavior, True, NULL);
	for (i = 0; i < XtNumber(modeString); i++) {
		modes[i] = XtVaCreateManagedWidget(modeString[i],
			xmToggleButtonGadgetClass, radioRowCol,
			XmNradioBehavior, True, NULL);
		XtAddCallback(modes[i], XmNvalueChangedCallback,
			(XtCallbackProc) ModeToggle, (XtPointer) i);
	}
	tileRowCol = XtVaCreateManagedWidget("tileRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tileRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size", 5,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
	speed = XtVaCreateManagedWidget("speed",
		xmScaleWidgetClass, tileRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 8,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED + MINSPEED - DEFAULTDELAY / MULTSPEED - 1,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);

	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Panex!",
		xmLabelWidgetClass, messageRowCol, NULL);

	panex = XtCreateManagedWidget("panex",
		panexWidgetClass, mainPanel, NULL, 0);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	panex = XtCreateManagedWidget("panex",
		panexWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(panex, XtNselectCallback,
		(XtCallbackProc) CallbackPanex, (XtPointer) NULL);
	Initialize(panex);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(panex), (unsigned int) AnyButton, AnyModifier,
		XtWindow(panex), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(panex),
		XCreateFontCursor(XtDisplay(panex), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
