/*-
# X-BASED PANEX(tm)
#
#  Panex.c
#
###
#
#  Copyright (c) 1996 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Panex */

#include "file.h"
#include "rngs.h"
#include "sound.h"
#include "PanexP.h"

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wpanex.ini"
#endif

#define SECTION "setup"

static const char *pyramidColorString[MAXSTACKS - 1] =
{
	"255 0 255",
	"255 255 0"
};
static char pyramidColorChar[MAXSTACKS - 1] =
{'M', 'Y'};
#else

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesPanex(Widget current, Widget request, Widget renew);
static void QuitPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void DestroyPanex(Widget old);
static void ResizePanex(PanexWidget w);
static void SizePanex(PanexWidget w);
static void InitializePanex(Widget request, Widget renew);
static void ExposePanex(Widget renew, XEvent * event, Region region);
static void HidePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void SelectPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void MotionPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ReleasePanex(PanexWidget w, XEvent * event, char **args,
	int nArgs);
static void GetPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void WritePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ClearPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void ClearPanexMaybe(PanexWidget w, XEvent * event, char **args,
	int nArgs);
static void ClearPanex2(PanexWidget w, XEvent * event, char **args, int nArgs);
static void UndoPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void SolvePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void SpeedPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void SlowPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void SoundPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void IncrementPanex(PanexWidget w, XEvent * event, char **args,
	int nArgs);
static void DecrementPanex(PanexWidget w, XEvent * event, char **args,
	int nArgs);
static void ModePanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void EnterPanex(PanexWidget w, XEvent * event, char **args, int nArgs);
static void LeavePanex(PanexWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsPanex[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <Btn1Down>: Select()\n\
 <Btn1Motion>: Motion()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>r: Clear()\n\
 <KeyPress>c: Clear()\n\
 <Btn3Down>: ClearMaybe()\n\
 <Btn3Down>(2+): Clear2()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 <KeyPress>@: Sound()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>m: Mode()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPanex[] =
{
	{(char *) "Quit", (XtActionProc) QuitPanex},
	{(char *) "Hide", (XtActionProc) HidePanex},
	{(char *) "Select", (XtActionProc) SelectPanex},
	{(char *) "Motion", (XtActionProc) MotionPanex},
	{(char *) "Release", (XtActionProc) ReleasePanex},
	{(char *) "Clear", (XtActionProc) ClearPanex},
	{(char *) "ClearMaybe", (XtActionProc) ClearPanexMaybe},
	{(char *) "Clear2", (XtActionProc) ClearPanex2},
	{(char *) "Get", (XtActionProc) GetPanex},
	{(char *) "Write", (XtActionProc) WritePanex},
	{(char *) "Undo", (XtActionProc) UndoPanex},
	{(char *) "Solve", (XtActionProc) SolvePanex},
	{(char *) "Speed", (XtActionProc) SpeedPanex},
	{(char *) "Slow", (XtActionProc) SlowPanex},
	{(char *) "Sound", (XtActionProc) SoundPanex},
	{(char *) "Increment", (XtActionProc) IncrementPanex},
	{(char *) "Decrement", (XtActionProc) DecrementPanex},
	{(char *) "Mode", (XtActionProc) ModePanex},
	{(char *) "Enter", (XtActionProc) EnterPanex},
	{(char *) "Leave", (XtActionProc) LeavePanex}
};

static XtResource resourcesPanex[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(PanexWidget, core.width),
	 XtRString, (caddr_t) "400"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(PanexWidget, core.height),
	 XtRString, (caddr_t) "200"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNtileColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.tileColor),
	 XtRString, (caddr_t) "gray75" /*XtDefaultBackground*/},
	{XtNpyramidColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.pyramidName[0]),
	 XtRString, (caddr_t) "magenta"},
	{XtNpyramidColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.pyramidName[1]),
	 XtRString, (caddr_t) "yellow"},
	{XtNtileBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PanexWidget, panex.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.delay),
	 XtRString, (caddr_t) "10"}, /* DEFAULTDELAY */
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNtiles, XtCTiles, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.tiles),
	 XtRString, (caddr_t) "10"}, /* DEFAULTTILES */
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.mode),
	 XtRString, (caddr_t) "1"}, /* DEFAULTMODE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(PanexWidget, panex.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(PanexWidget, panex.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PanexWidget, panex.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(PanexWidget, panex.select), XtRCallback, (caddr_t) NULL}
};

PanexClassRec panexClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Panex",	/* class name */
		sizeof (PanexRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePanex,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPanex,	/* actions */
		XtNumber(actionsListPanex),	/* num actions */
		resourcesPanex,	/* resources */
		XtNumber(resourcesPanex),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyPanex,	/* destroy */
		(XtWidgetProc) ResizePanex,	/* resize */
		(XtExposeProc) ExposePanex,	/* expose */
		(XtSetValuesFunc) SetValuesPanex,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPanex,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass panexWidgetClass = (WidgetClass) & panexClassRec;

void
SetPanex(PanexWidget w, int reason)
{
	panexCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(PanexWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->panex.fontInfo) {
		XUnloadFont(XtDisplay(w), w->panex.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->panex.fontInfo);
	}
	if ((w->panex.fontInfo = XLoadQueryFont(display,
			w->panex.font)) == NULL) {
		(void) sprintf(buf,
			"Can not open %s font.\nAttempting %s font as alternate\n",
			w->panex.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->panex.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Can not open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->panex.fontInfo) {
		w->panex.letterOffset.x = XTextWidth(w->panex.fontInfo, "8", 1)
			/ 2;
		w->panex.letterOffset.y = w->panex.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->panex.letterOffset.x = 3;
		w->panex.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "panex.log"
#endif

static int startPositions[MAXMODES][MAXSTACKS] =
{
	{0, -1, -1},
	{0, -1, 1}
};
static int finishPositions[MAXMODES][MAXSTACKS] =
{
	{-1, -1, 0},
	{1, -1, 0}
};

static Point trapazoidUnit[5] =
{
	{0, 0},
	{1, 1},
	{-3, 0},
	{1, -1},
	{2, 0}
};

#define MULTDELAY 48

static void
CheckTiles(PanexWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->panex.tiles < MINTILES) {
		intCat(&buf1, "Number of tiles out of bounds, use at least ",
			MINTILES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTTILES);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->panex.tiles = DEFAULTTILES;
	}
	if (w->panex.mode < HANOI || w->panex.mode > PANEX) {
		intCat(&buf1,
			"Mode is in error, use 0 for Hanoi, 1 for Panex, defaulting to ",
			DEFAULTMODE);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->panex.mode = DEFAULTMODE;
	}
	if (w->panex.delay < 0) {
		intCat(&buf1, "Delay can not be negative (",
			w->panex.delay);
		stringCat(&buf2, buf1, "), taking absolute value");
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		w->panex.delay = -w->panex.delay;
	}
}

static int
CartesianX(PanexWidget w, int stack)
{
	return stack * w->panex.pos.x + w->panex.delta.x + w->panex.puzzleOffset.x;
}

static int
CartesianY(PanexWidget w, int loc)
{
	return loc * w->panex.pos.y + w->panex.delta.y + w->panex.puzzleOffset.y;
}

#if 0
/* Only applies to Panex */
static int middlePositions[MAXSTACKS] =
{-1, 0, 1};

Boolean
CheckMiddle(PanexWidget w)
{
	int stack, loc;
	PanexLoc i;

	for (stack = 0; stack < MAXSTACKS; stack++)
		if ((i.stack = middlePositions[stack]) >= 0)
			for (loc = 1; loc <= w->panex.tiles; loc++) {
				i.loc = loc - 1;
				if (w->panex.tileOfPosition[stack][loc].stack != i.stack ||
						w->panex.tileOfPosition[stack][loc].loc != i.loc)
					return False;
			}
	return True;
}
#endif

Boolean
CheckSolved(PanexWidget w)
{
	int stack, loc;
	PanexLoc i;

	for (stack = 0; stack < MAXSTACKS; stack++)
		if ((i.stack = finishPositions[w->panex.mode][stack]) >= 0)
			for (loc = 1; loc <= w->panex.tiles; loc++) {
				i.loc = loc - 1;
				if (w->panex.tileOfPosition[stack][loc].stack != i.stack ||
						w->panex.tileOfPosition[stack][loc].loc != i.loc)
					return False;
			}
	return True;
}

#ifdef DEBUG
void
PrintStacks(PanexWidget w)
{
	int stack, position;

	(void) printf("top: where are the tiles in the stack\n");
	for (position = 0; position <= w->panex.tiles; position++)
		for (stack = 0; stack < MAXSTACKS; stack++) {
			printf("%d,%d", w->panex.tileOfPosition[stack][position].stack,
				w->panex.tileOfPosition[stack][position].loc);
			if (stack + 1 == MAXSTACKS)
				(void) printf("\n");
			else
				(void) printf(" | ");
		}
}

void
PrintTiles(PanexWidget w)
{
	int stack, position;

	(void) printf("pot: which stack are the tiles in\n");
	for (position = 0; position < w->panex.tiles; position++)
		for (stack = 0; stack <= w->panex.mode; stack++) {
			printf("%d,%d", w->panex.positionOfTile[stack][position].stack,
				w->panex.positionOfTile[stack][position].loc);
			if (stack == w->panex.mode)
				(void) printf("\n");
			else
				(void) printf(" | ");
		}
}
#endif

static void
fill3DRect(PanexWidget w, Pixmap dr, GC gc, GC darkerGC, GC brighterGC,
		int x, int y, int width, int height, Boolean raised)
{
	GC currentGC = (raised) ? gc : darkerGC;

	if (width > 2 && height > 2)
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y + 1, width - 2, height - 2);
	currentGC = (raised) ? brighterGC : darkerGC;
	FILLRECTANGLE(w, dr, currentGC,
		x, y, 1, height);
	if (width > 1)
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y, width - 1, 1);
	FILLRECTANGLE(w, dr, currentGC,
		x + 1, y, 1, height);
	if (width > 2)
		FILLRECTANGLE(w, dr, currentGC,
			x + 2, y + 1, width - 2, 1);
	currentGC = (raised) ? darkerGC : gc;
	if (width > 1 && height > 1)
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y + height - 1, width - 1, 1);
	if (width > 1 && height > 1)
		FILLRECTANGLE(w, dr, currentGC,
			x + width - 1, y, 1, height - 1);
	if (width > 3 && height > 2)
		FILLRECTANGLE(w, dr, currentGC,
			x + 2, y + height - 2, width - 3, 1);
	if (width > 2 && height > 3)
		FILLRECTANGLE(w, dr, currentGC,
			x + width - 2, y + 1, 1, height - 3);
}

static void
drawShadow(PanexWidget w, Pixmap dr, GC gc, int startX, int startY,
		int sizeX, int sizeY)
{
	FILLRECTANGLE(w, dr, gc, startX, startY, sizeX, 1);
	FILLRECTANGLE(w, dr, gc, startX, startY, 1, sizeY);
}

static void
DrawPyramid(PanexWidget w, Pixmap dr, int color, int i, int j, int size,
		int offsetPosition)
{
	Point trapazoidList[5];
	int k;
	GC pyramidGC;

	for (k = 0; k <= 4; k++) {
		if (ABS(trapazoidUnit[k].x) == 3)
			trapazoidList[k].x = (size + 1) *
				SIGN(trapazoidUnit[k].x) *
				w->panex.tileSize.x / (w->panex.tiles + 1);
		else if (ABS(trapazoidUnit[k].x) == 2)
			trapazoidList[k].x = size * SIGN(trapazoidUnit[k].x) *
				w->panex.tileSize.x / (w->panex.tiles + 1);
		else
			trapazoidList[k].x = w->panex.tileSize.x /
				(2 * (w->panex.tiles + 1));
		trapazoidList[k].y = (w->panex.tileSize.y - 5) *
			trapazoidUnit[k].y;
	}
	k = w->panex.delta.x + i * w->panex.pos.x + w->panex.puzzleOffset.x +
		w->panex.tileSize.x / 2 + offsetPosition + 1;
	trapazoidList[0].x = trapazoidList[4].x / 2 + k;
	trapazoidList[0].y = j * w->panex.pos.y + w->panex.delta.y +
		w->panex.puzzleOffset.y + offsetPosition + 2;
	if (w->panex.mono) {
		pyramidGC = (offsetPosition) ? w->panex.tileGC :
			w->panex.borderGC;
	} else {
#ifndef WINVER
		if ((w->panex.pyramidColor[color] == w->panex.borderColor)
				&& offsetPosition)
			pyramidGC = w->panex.tileGC;
		else if ((w->panex.pyramidColor[color] == w->panex.tileColor)
				&& !offsetPosition)
			pyramidGC = w->panex.borderGC;
		else
#endif
			pyramidGC = w->panex.pyramidGC[color];

	}
	POLYGON(w, dr, pyramidGC, pyramidGC, trapazoidList, 4, True, False);
	if (w->panex.mono) {

		char buf[2];

		buf[0] =
#ifdef WINVER
			w->panex.pyramidChar[color];
#else
			w->panex.pyramidName[color][0];
#endif
		DRAWTEXT(w, dr, (offsetPosition) ? w->panex.borderGC :
			w->panex.tileGC,
			k - w->panex.letterOffset.x, trapazoidList[0].y +
			w->panex.tileSize.y / 2 + w->panex.letterOffset.y, buf, 1);
	}
}

static void
FillRectClip(PanexWidget w, Pixmap dr, GC gc, int dx, int dy, int sx, int sy,
		int wdx, int wdy, int wsx, int wsy)
{
	int ndx = wdx, nsx = wsx, ndy = wdy, nsy = wsy;

	/* w is the clipping window */
	if (dx + sx < wdx || dx > wdx + wsx ||
			dy + sy < wdy || dy > wdy + wsy ||
			sx <= 0 || sy <= 0 || wsx <= 0 || wsy <= 0) {
		return;
	}
	if (dx > wdx) {
		ndx = dx;
		nsx = wsx - dx + wdx;
	}
	if (dy > wdy) {
		ndy = dy;
		nsy = wsy - dy + wdy;
	}
	if (ndx + nsx > dx + sx) {
		nsx = sx + dx - ndx;
	}
	if (ndy + nsy > dy + sy) {
		nsy = sy + dy - ndy;
	}
	FILLRECTANGLE(w, dr, gc, ndx, ndy, nsx, nsy);
}

static void
DrawSlot(PanexWidget w, int stack, int x, int y, int width, int height)
{
	Pixmap dr = 0;
	GC gc = w->panex.inverseGC;
	int sizex = w->panex.tileSize.x / 3;
	int sizey = w->panex.tileSize.y / 3;

	FILLRECTANGLE(w, dr, gc, x, y, width, height);
	FillRectClip(w, dr, w->panex.borderGC,
		sizex + w->panex.delta.x +
		w->panex.puzzleOffset.x,
		w->panex.delta.y / 2 + w->panex.puzzleOffset.y + sizey,
		sizex + (MAXSTACKS - 1) * w->panex.pos.x,
		sizey,
		x, y, width, height);
	FillRectClip(w, dr, w->panex.borderGC,
		sizex + w->panex.delta.x +
		w->panex.puzzleOffset.x + stack * w->panex.pos.x,
		w->panex.delta.y / 2 + w->panex.puzzleOffset.y + sizey,
		sizex,
		(w->panex.tiles + 1) * w->panex.pos.y + w->panex.delta.y -
		1 - 2 * sizey,
		x, y, width, height);
	FillRectClip(w, dr, w->panex.borderGC,
		sizex + w->panex.delta.x +
		w->panex.puzzleOffset.x + w->panex.pos.x,
		w->panex.delta.y / 2 + w->panex.puzzleOffset.y + sizey,
		sizex,
		(w->panex.tiles + 1) * w->panex.pos.y + w->panex.delta.y -
		1 - 2 * sizey,
		x, y, width, height);
}

static void
DrawTile(PanexWidget w, int i, int j, Boolean erase, int pressedOffset,
		int offsetX, int offsetY)
{
	int dx, dy, sx, sy;
	int tileI = w->panex.tileOfPosition[i][j].stack;
	int tileJ = w->panex.tileOfPosition[i][j].loc;
	Pixmap dr = 0;

	dx = CartesianX(w, i) + pressedOffset + offsetX;
	dy = CartesianY(w, j) + pressedOffset + offsetY;
	sx = CartesianX(w, tileI);
	sy = CartesianY(w, tileJ);
	if (erase) {
		/* Draw Slots */
		FILLRECTANGLE(w, dr, w->panex.inverseGC,
			dx, dy,
			w->panex.tileSize.x + 1,
			w->panex.tileSize.y + 1 - pressedOffset);
		if (j == 0) {
			if (i == 0) {
				FILLRECTANGLE(w, dr, w->panex.borderGC,
					dx + w->panex.tileSize.x / 3,
					dy + w->panex.tileSize.y / 3 - 1 -
					pressedOffset,
					2 * w->panex.tileSize.x / 3 + 2,
					w->panex.tileSize.y / 3);
			} else if (i == MAXSTACKS - 1) {
				FILLRECTANGLE(w, dr, w->panex.borderGC,
					dx,
					dy + w->panex.tileSize.y / 3 - 1 -
					pressedOffset,
					2 * w->panex.tileSize.x / 3 - 1,
					w->panex.tileSize.y / 3 +
					pressedOffset);
			} else {
				FILLRECTANGLE(w, dr, w->panex.borderGC,
					dx,
					dy + w->panex.tileSize.y / 3 - 1 -
					pressedOffset,
					w->panex.tileSize.x + 1,
					w->panex.tileSize.y / 3);
			}
			FILLRECTANGLE(w, dr, w->panex.borderGC,
				dx + w->panex.tileSize.x / 3 - pressedOffset,
				dy + w->panex.tileSize.y / 3 - 1,
				w->panex.tileSize.x / 3,
				w->panex.tileSize.y - w->panex.tileSize.y / 3 +
				2 - pressedOffset);
		} else if (j == w->panex.tiles) {
			FILLRECTANGLE(w, dr, w->panex.borderGC,
				dx + w->panex.tileSize.x / 3 - pressedOffset,
				dy - pressedOffset,
				w->panex.tileSize.x / 3,
				2 * w->panex.tileSize.y / 3 + 2);
		} else {
			FILLRECTANGLE(w, dr, w->panex.borderGC,
				dx + w->panex.tileSize.x / 3 - pressedOffset,
				dy - pressedOffset,
				w->panex.tileSize.x / 3,
				w->panex.tileSize.y + pressedOffset + 1);
		}
		return;
	}
#ifdef WINVER
	w->core.hOldBitmap = (HBITMAP) SelectObject(w->core.memDC,
		w->panex.bufferTiles[pressedOffset]);
	BitBlt(w->core.hDC,
		dx, dy,
		w->panex.tileSize.x, w->panex.tileSize.y,
		w->core.memDC,
		sx, sy,
		SRCCOPY);
	SelectObject(w->core.memDC, w->core.hOldBitmap);
#else
	XSetGraphicsExposures(XtDisplay(w), w->panex.tileGC, False);
	XCopyArea(XtDisplay(w),
		w->panex.bufferTiles[pressedOffset],
		XtWindow(w),
		w->panex.tileGC,
		sx, sy,
		w->panex.tileSize.x, w->panex.tileSize.y,
		dx, dy);
#endif
}


static void
DrawBufferedTile(PanexWidget w, int i, int j, int pressedOffset)
{
	Pixmap *dr;
	int dx, dy;
	GC tileGC;

	dr = &(w->panex.bufferTiles[pressedOffset]);
	tileGC = w->panex.tileGC;
	dx = CartesianX(w, i) + pressedOffset;
	dy = CartesianY(w, j) + pressedOffset;
	if (pressedOffset != 0) {
		drawShadow(w, *dr, w->panex.tileDarkerGC,
			dx - pressedOffset, dy - pressedOffset,
			w->panex.tileSize.x, w->panex.tileSize.y);
	}
	fill3DRect(w, *dr, tileGC,
		w->panex.tileDarkerGC, w->panex.tileBrighterGC,
		dx, dy, w->panex.tileSize.x, w->panex.tileSize.y,
		pressedOffset == 0);
	if (pressedOffset != 0) {
		FILLRECTANGLE(w, *dr, w->panex.tileGC,
			dx + w->panex.tileSize.x - 2, dy - 1, 1, 3);
		FILLRECTANGLE(w, *dr, w->panex.tileGC,
			dx - 1, dy + w->panex.tileSize.y - 2, 3, 1);
	}
	DrawPyramid(w, *dr,
		i, i, j, j, pressedOffset);
}

static void
DrawAllBufferedTiles(const PanexWidget w)
{
	int i, j, l;

	for (i = 0; i <= w->panex.mode; i++)
		for (j = 0; j < w->panex.tiles; j++)
			for (l = 0; l < 2; l++)
				DrawBufferedTile(w, i, j, l);
}

void
DrawAllTiles(PanexWidget w)
{
	int i, j;

	for (i = 0; i <= w->panex.mode; i++)
		for (j = 0; j < w->panex.tiles; j++)
			DrawTile(w,
				w->panex.positionOfTile[i][j].stack,
				w->panex.positionOfTile[i][j].loc,
			       	False, FALSE, 0, 0);
}

static void
AnimateSlide(PanexWidget w, int currentStack, int currentPosition,
		int nextStack, int nextPosition, int fast)
{
	int diffStack = nextStack - currentStack;
	int diffPosition = nextPosition - currentPosition;
	int dir = 0, spaces = 0, inc;
	int gapI = 0, moveI = 0, space;
	int dx, dy;
	int ix = 0, iy = 0;

	if (diffPosition > 0) {
		dir = BOTTOM;
		spaces = diffPosition;
		moveI = w->panex.pos.y;
	} else if (diffPosition < 0) {
		dir = TOP;
		spaces = -diffPosition;
		moveI = w->panex.pos.y;
	} else if (diffStack > 0) {
		dir = RIGHT;
		spaces = diffStack;
		moveI = w->panex.pos.x;
	} else if (diffStack < 0) {
		dir = LEFT;
		spaces = -diffStack;
		moveI = w->panex.pos.x;
	}
		space = spaces * moveI;
		if ((dir == RIGHT) || (dir == LEFT)) {
			gapI = w->panex.pos.x * fast / w->panex.numSlices;
		} else if ((dir == TOP) || (dir == BOTTOM)) {
			gapI = w->panex.pos.y * fast / w->panex.numSlices;
		}
		if (gapI == 0)
			gapI++;
		FLUSH(w);
		initTimer(w->panex.oldTime);
		for (inc = 0; inc < space + gapI; inc += gapI) {
			if (inc > space) {
				inc = space;
			}
			/* Calculate deltas */
			dx = CartesianX(w, currentStack);
			dy = CartesianY(w, currentPosition);

			if ((dir == RIGHT) || (dir == LEFT)) {
				ix = ((dir == RIGHT) ? inc : -inc);
				iy = 0;
				DrawSlot(w, currentStack,
					dx, dy + w->panex.tileSize.y,
					w->panex.tileSize.x + 1, 1);
			} else if ((dir == TOP) || (dir == BOTTOM)) {
				ix = 0;
				iy = ((dir == BOTTOM) ? inc : -inc);
			}
			DrawTile(w, currentStack, currentPosition, False,
				FALSE, ix, iy);
			/* Erase old slivers */
			ix += dx;
			iy += dy;
			if (inc != 0)
			switch (dir) {
			case TOP:
				DrawSlot(w, currentStack,
					ix, iy + w->panex.tileSize.y,
					w->panex.tileSize.x + 1, gapI);
				FLUSH(w);
				useTimer(&(w->panex.oldTime), w->panex.delay);
				break;
			case RIGHT:
				DrawSlot(w, currentStack,
					ix - gapI, iy,
					gapI, w->panex.tileSize.y + 1);
				FLUSH(w);
				useTimer(&(w->panex.oldTime), 3 * w->panex.delay);
				break;
			case BOTTOM:
				DrawSlot(w, currentStack,
					ix, iy - gapI,
					w->panex.tileSize.x + 1, gapI);
				FLUSH(w);
				useTimer(&(w->panex.oldTime), w->panex.delay);
				break;
			case LEFT:
				DrawSlot(w, currentStack,
					ix + w->panex.tileSize.x, iy,
					gapI + 1, w->panex.tileSize.y + 1);
				FLUSH(w);
				useTimer(&(w->panex.oldTime), 3 * w->panex.delay);
				break;
			}
		}
}

static void
DrawSlots(PanexWidget w, Pixmap dr)
{
	int i, dx, dy, y, sizex, sizey;

	sizex = w->panex.tileSize.x / 3;
	sizey = w->panex.tileSize.y / 3;
	if (sizey == 0)
		sizey = 1;
	dx = sizex + w->panex.delta.x + w->panex.puzzleOffset.x;
	y = (w->panex.tiles + 1) * w->panex.pos.y + w->panex.delta.y;
	dy = w->panex.delta.y / 2 + w->panex.puzzleOffset.y + sizey;
	FILLRECTANGLE(w, dr, w->panex.borderGC, dx, dy,
		sizex + (MAXSTACKS - 1) * w->panex.pos.x,
		sizey);
	for (i = 0; i < MAXSTACKS; i++) {
		FILLRECTANGLE(w, dr, w->panex.borderGC, dx, dy,
			sizex, y - 1 - 2 * sizey);
		dx += w->panex.pos.x;
	}
}

static void
EraseFrame(PanexWidget w, Pixmap dr, Boolean focus)
{
	DRAWRECTANGLE(w, dr, (focus) ? w->panex.frameGC : w->panex.inverseGC,
		0, 0, w->core.width - 1, w->core.height - 1);
	FILLRECTANGLE(w, dr, w->panex.inverseGC,
		1, 1, w->core.width - 2, w->core.height - 2);
}

static void
DrawFrame(PanexWidget w, Pixmap dr, Boolean focus)
{
	DRAWRECTANGLE(w, dr, (focus) ? w->panex.frameGC : w->panex.borderGC,
		0, 0, w->core.width - 1, w->core.height - 1);
}

static void
MoveNoTiles(PanexWidget w)
{
	SetPanex(w, PANEX_IGNORE);
}

int
TopOfStack(PanexWidget w, int stack, int start)
{
	int i;

	for (i = start; i <= w->panex.tiles; i++)
		if (w->panex.tileOfPosition[stack][i].stack >= 0)
			return i;
	return -1;
}

static int
RequestMove(PanexWidget w, int fromStack, int fromPosition, int toStack,
		Boolean limbo)
{
	int i;

	/* Do not have to check above stack since it is the top one */
	if (toStack > fromStack) {
		for (i = fromStack + 1; i <= toStack; i++) {
			if (w->panex.tileOfPosition[i][0].stack >= 0)
				return (-1);
		}
		i = TopOfStack(w, toStack, 0);
	} else if (toStack < fromStack) {
		for (i = fromStack - 1; i >= toStack; i--) {
			if (w->panex.tileOfPosition[i][0].stack >= 0)
				return (-1);
		}
		i = TopOfStack(w, toStack, 0);
	} else {
		/* Ruled out toStack == fromStack, except for motion */
		i = TopOfStack(w, toStack, 1);
	}
	i = (i == -1) ? w->panex.tiles : i - 1;
	if (w->panex.mode == HANOI) {
		if (i == w->panex.tiles || limbo ||
				w->panex.tileOfPosition[toStack][i + 1].loc >=
				w->panex.tileOfPosition[fromStack][fromPosition].loc)
			return i;
		else
			return -2;
	} else {
		if (i > w->panex.tileOfPosition[fromStack][fromPosition].loc + 1)
			return (w->panex.tileOfPosition[fromStack][fromPosition].loc + 1);
		else
			return i;
	}
}

static int
SelectTile(PanexWidget w, int x)
{
	int i;

	x -= w->panex.puzzleOffset.x;
	i = (x - w->panex.delta.x / 2) / w->panex.pos.x;
	if (i < 0)
		i = 0;
	else if (i >= MAXSTACKS)
		i = MAXSTACKS - 1;
#if 0
	y -= w->panex.puzzleOffset.y;
	j = (y - w->panex.delta.y / 2) / w->panex.pos.y;
	if (j < 0)
		j = 0;
	else if (j > w->panex.tiles)
		j = w->panex.tiles;
#endif
	return i;
}

static void
VirtualMove(PanexWidget w, int currentStack, int currentPosition,
		int nextStack, int nextPosition)
{
	PanexLoc top;

	top = w->panex.tileOfPosition[currentStack][currentPosition];
	w->panex.tileOfPosition[nextStack][nextPosition] = top;
	w->panex.positionOfTile[top.stack][top.loc].stack = nextStack;
	w->panex.positionOfTile[top.stack][top.loc].loc = nextPosition;
	w->panex.tileOfPosition[currentStack][currentPosition].stack = -1;
}

static void
DiscreteMoves(PanexWidget w, int currentStack, int currentPosition,
		int nextStack, int nextPosition)
{
	DrawTile(w, currentStack, currentPosition, True, FALSE, 0, 0);
	VirtualMove(w, currentStack, currentPosition, nextStack, nextPosition);
	DrawTile(w, nextStack, nextPosition, False, FALSE, 0, 0);
	FLUSH(w);
	Sleep((unsigned int) MULTDELAY * w->panex.delay /
		(w->panex.tiles + MAXSTACKS - 1));
}

static void
SlideTile(PanexWidget w, int fromStack, int fromPosition,
		int toStack, int toPosition, int fast)
{
	int currentStack = fromStack, currentPosition = fromPosition;
	int nextStack = fromStack, nextPosition = fromPosition;

	if (currentPosition > 0) {
		if (fast != INSTANT && w->panex.delay > 0) {
			nextPosition = 0;
			AnimateSlide(w, currentStack, currentPosition,
				nextStack, nextPosition, fast);
			VirtualMove(w, currentStack, currentPosition,
				nextStack, nextPosition);
			currentPosition = nextPosition;
		} else {
			while (currentPosition > 0) {
				nextPosition = currentPosition - 1;
				DiscreteMoves(w, currentStack, currentPosition,
					nextStack, nextPosition);
				currentPosition = nextPosition;
			}
		}
#ifdef USE_SOUND
		if (w->panex.sound) {
			playSound((char *) MOVESOUND);
		}
#endif
	}
	if (currentStack != toStack) {
		if (fast != INSTANT && w->panex.delay > 0) {
			nextStack = toStack;
			AnimateSlide(w, currentStack, currentPosition,
				nextStack, nextPosition, fast);
			VirtualMove(w, currentStack, currentPosition,
				nextStack, nextPosition);
			currentStack = nextStack;
		} else {
			while (currentStack != toStack) {
				nextStack = (currentStack < toStack) ?
					currentStack + 1 : currentStack - 1;
				nextPosition = currentPosition;
				DiscreteMoves(w, currentStack, currentPosition,
					nextStack, nextPosition);
				currentStack = nextStack;
			}
		}
#ifdef USE_SOUND
		if (w->panex.sound) {
			playSound((char *) MOVESOUND);
		}
#endif
	}
	if (currentPosition < toPosition) {
		if (fast != INSTANT && w->panex.delay > 0) {
			nextPosition = toPosition;
			AnimateSlide(w, currentStack, currentPosition,
				nextStack, nextPosition, fast);
			VirtualMove(w, currentStack, currentPosition,
				nextStack, nextPosition);
			currentPosition = nextPosition;
		} else {
			while (currentPosition < toPosition) {
				nextPosition = currentPosition + 1;
				DiscreteMoves(w, currentStack, currentPosition,
					nextStack, nextPosition);
				currentPosition = nextPosition;
			}
		}
#ifdef USE_SOUND
		if (w->panex.sound) {
			playSound((char *) MOVESOUND);
		}
#endif
	}
}

static int
MoveTile(PanexWidget w, int fromStack, int fromPosition, int toStack, int fast)
{
	int toPosition;

	if ((toPosition = RequestMove(w, fromStack, fromPosition, toStack,
			False)) >= 0)
		SlideTile(w, fromStack, fromPosition, toStack, toPosition,
			fast);
	return toPosition;
}

static Boolean
MoveTileToLimbo(PanexWidget w, int fromStack, int fromPosition, int toStack,
		Boolean limbo)
{
	Boolean aMove;

	aMove = (RequestMove(w, fromStack, fromPosition, toStack, limbo) >= 0);
	if (aMove)
		SlideTile(w, fromStack, fromPosition, toStack, 0, NORMAL);
	return aMove;
}

static void
ResetTiles(PanexWidget w)
{
	int stack, loc;
	PanexLoc i;

	w->panex.currentStack = -1;
	for (stack = 0; stack < MAXMODES; stack++) {
		if (w->panex.positionOfTile[stack]) {
			free(w->panex.positionOfTile[stack]);
			w->panex.positionOfTile[stack] = NULL;
		}
		if (startLoc[stack]) {
			free(startLoc[stack]);
			startLoc[stack] = NULL;
		}
	}
	for (stack = 0; stack <= w->panex.mode; stack++) {
		if (!(w->panex.positionOfTile[stack] = (PanexLoc *)
				malloc(sizeof (PanexLoc) * w->panex.tiles))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		if (!(startLoc[stack] = (PanexLoc *)
				malloc(sizeof (PanexLoc) * w->panex.tiles))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (stack = 0; stack < MAXSTACKS; stack++) {
		if (w->panex.tileOfPosition[stack])
			free(w->panex.tileOfPosition[stack]);
		if (!(w->panex.tileOfPosition[stack] = (PanexLoc *)
				malloc(sizeof (PanexLoc) *
				(w->panex.tiles + 1)))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (stack = 0; stack <= w->panex.mode; stack++) {
		i.stack = 2 * stack;
		for (loc = 0; loc < w->panex.tiles; loc++) {
			i.loc = loc + 1;
			w->panex.positionOfTile[stack][loc] = i;
		}
	}
	for (stack = 0; stack < MAXSTACKS; stack++) {
		i.stack = -1;
		i.loc = -1;
		w->panex.tileOfPosition[stack][0] = i;
		if ((i.stack = startPositions[w->panex.mode][stack]) >= 0)
			for (loc = 1; loc <= w->panex.tiles; loc++) {
				i.loc = loc - 1;
				w->panex.tileOfPosition[stack][loc] = i;
		} else
			for (loc = 1; loc <= w->panex.tiles; loc++)
				w->panex.tileOfPosition[stack][loc] = i;
	}
	FlushMoves(w);
	w->panex.started = False;
	w->panex.cheat = False;
}

static void
GetTiles(PanexWidget w)
{
	FILE *fp;
	int c, i, tiles, mode, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &tiles);
	if (tiles >= MINTILES) {
		for (i = w->panex.tiles; i < tiles; i++) {
			SetPanex(w, PANEX_INC);
		}
		for (i = w->panex.tiles; i > tiles; i--) {
			SetPanex(w, PANEX_DEC);
		}
	} else {
		stringCat(&buf1, name, " corrupted: tiles ");
		intCat(&buf2, buf1, tiles);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINTILES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &mode);
	if (mode >= HANOI && mode <= PANEX) {
		if (w->panex.mode != mode) {
			SetPanex(w, PANEX_MODE);
		}
	} else {
		stringCat(&buf1, name, " corrupted: mode ");
		intCat(&buf2, buf1, mode);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, HANOI);
		free(buf1);
		stringCat(&buf1, buf2, " and ");
		free(buf2);
		intCat(&buf2, buf1, PANEX);
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
	}
#ifdef WINVER
	ResetTiles(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetPanex(w, PANEX_RESTORE);
	SetStartPosition(w);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	if (mode == 0)
		(void) printf("%s: mode hanoi, tiles %d, moves %d.\n",
			name, tiles, moves);
	else
		(void) printf("%s: mode panex, tiles %d, moves %d.\n",
			name, tiles, moves);
	free(lname);
	free(fname);
	w->panex.cheat = True; /* Assume the worst. */
}

static void
WriteTiles(PanexWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "tiles%c %d\n", SYMBOL, w->panex.tiles);
	(void) fprintf(fp, "mode%c %d\n", SYMBOL, w->panex.mode);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearTiles(PanexWidget w)
{
	if (w->panex.currentStack >= 0)
		return;
	EraseFrame(w, 0, w->panex.focus);
	ResetTiles(w);
	DrawSlots(w, 0);
	DrawAllTiles(w);
	SetPanex(w, PANEX_RESET);
}

static void
UndoTiles(PanexWidget w)
{
	if (MadeMoves() && w->panex.currentStack < 0) {
		int fromStack, fromPosition, toStack;

		GetMove(&toStack, &fromStack);
		if ((fromPosition = TopOfStack(w, fromStack, 0)) < 0 ||
				MoveTile(w, fromStack, fromPosition, toStack,
					DOUBLE) < 0) {
			char *buf1, *buf2;

			intCat(&buf1, "Move from ", fromStack);
			stringCat(&buf2, buf1, " to ");
			free(buf1);
			intCat(&buf1, buf2, toStack);
			free(buf2);
			stringCat(&buf2, buf1, " can not be made");
			free(buf1);
			DISPLAY_WARNING(buf2);
			free(buf2);
		} else {
			SetPanex(w, PANEX_UNDO);
		}
	}
}

static void
SolveTiles(PanexWidget w)
{
	/* Cheat and Reset To Start */
	EraseFrame(w, 0, w->panex.focus);
	ResetTiles(w);
	DrawSlots(w, 0);
	DrawAllTiles(w);
	SetPanex(w, PANEX_RESET);
	SolveTilesFromStart(w);
}

static void
SpeedTiles(PanexWidget w)
{
	w->panex.delay -= 5;
	if (w->panex.delay < 0)
		w->panex.delay = 0;
#ifdef HAVE_MOTIF
	SetPanex(w, PANEX_SPEED);
#endif
}

static void
SlowTiles(PanexWidget w)
{
	w->panex.delay += 5;
#ifdef HAVE_MOTIF
	SetPanex(w, PANEX_SPEED);
#endif
}

static void
SoundTiles(PanexWidget w)
{
	w->panex.sound = !w->panex.sound;
}

static void
IncrementTiles(PanexWidget w)
{
	SetPanex(w, PANEX_INC);
}

static void
DecrementTiles(PanexWidget w)
{
	if (w->panex.tiles <= MINTILES)
		return;
	SetPanex(w, PANEX_DEC);
}

static void
ModeTiles(PanexWidget w)
{
	SetPanex(w, PANEX_MODE);
}

int
MovePanex(PanexWidget w, int fromStack, int fromPosition, int toStack,
		int fast)
{
	int	 toPosition;

	if ((toPosition = MoveTile(w, fromStack, fromPosition, toStack,
			fast)) >= 0) {
		SetPanex(w, PANEX_MOVED);
		PutMove(fromStack, toStack);
	}
	return toPosition;
}

#define FACTOR 0.7

#ifdef WINVER
#define MAXINTENSITY 0xFF
static int
brighter(const int light)
{
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);
	int temp = light;

	if (temp < i)
		temp = i;
	return MIN(temp / FACTOR, MAXINTENSITY);
}

static int
darker(const int light)
{
	 return (int) (light * FACTOR);
}


static void
SetValuesPanex(PanexWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80], buf[20], charbuf[2];
	int pyramid;

	w->panex.tiles = GetPrivateProfileInt(SECTION, "tiles",
		DEFAULTTILES, INIFILE);
	w->panex.mode = GetPrivateProfileInt(SECTION, "mode",
		DEFAULTMODE, INIFILE);
	w->panex.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->panex.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.frameGC = RGB(color.red, color.green, color.blue);
	/* gray75 */
	(void) GetPrivateProfileString(SECTION, "tileColor", "191 191 191",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.tileGC = RGB(color.red, color.green, color.blue);
	w->panex.tileBrighterGC = RGB(brighter(color.red),
		brighter(color.green), brighter(color.blue));
	w->panex.tileDarkerGC = RGB(darker(color.red),
		darker(color.green), darker(color.blue));
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "tileBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->panex.inverseGC = RGB(color.red, color.green, color.blue);
	for (pyramid = 0; pyramid < MAXSTACKS - 1; pyramid++) {
		(void) sprintf(buf, "pyramidColor%d", pyramid);
		(void) GetPrivateProfileString(SECTION,
			buf, pyramidColorString[pyramid],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->panex.pyramidGC[pyramid] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "pyramidChar%d", pyramid);
		charbuf[0] = pyramidColorChar[pyramid];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->panex.pyramidChar[pyramid] = szBuf[0];
	}
	w->panex.delay = GetPrivateProfileInt(SECTION, "delay",
		DEFAULTDELAY, INIFILE);
	w->panex.sound = (BOOL)
		GetPrivateProfileInt(SECTION, "sound", 0, INIFILE);
	(void) GetPrivateProfileString(SECTION, "moveSound", MOVESOUND,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->panex.moveSound, szBuf);
		w->panex.moveSound[80] = 0;
	(void) GetPrivateProfileString(SECTION, "userName", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->panex.userName, szBuf);
	 w->panex.userName[80] = 0;
	(void) GetPrivateProfileString(SECTION, "scoreFile", "",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->panex.scoreFile, szBuf);
	 w->panex.scoreFile[80] = 0;
}

void
DestroyPanex(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else
#define MAXINTENSITY 0xFFFF

static Pixel
brighter(PanexWidget w, Pixel pixel)
{
	XColor color;
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	if (color.red < i)
		color.red = i;
	if (color.green < i)
		color.green = i;
	if (color.blue < i)
		color.blue = i;
	color.red = (unsigned short) MIN(color.red / FACTOR, MAXINTENSITY);
	color.green = (unsigned short) MIN(color.green / FACTOR, MAXINTENSITY);
	color.blue = (unsigned short) MIN(color.blue / FACTOR, MAXINTENSITY);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static Pixel
darker(PanexWidget w, Pixel pixel)
{
	XColor color;

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	color.red = (unsigned short) (color.red * FACTOR);
	color.green = (unsigned short) (color.green * FACTOR);
	color.blue = (unsigned short) (color.blue * FACTOR);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static void
GetColor(PanexWidget w, int pyramid)
{
	XGCValues values;
	XtGCMask valueMask;
	XColor colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->panex.reverse) {
		values.background = w->panex.foreground;
	} else {
		values.background = w->panex.background;
	}
	if (!w->panex.mono) {
		if (XAllocNamedColor(XtDisplay(w),
				DefaultColormapOfScreen(XtScreen(w)),
				w->panex.pyramidName[pyramid], &colorCell, &rgb)) {
			values.foreground = w->panex.pyramidColor[pyramid] =
				colorCell.pixel;
			if (w->panex.pyramidGC[pyramid])
				XtReleaseGC((Widget) w,
					w->panex.pyramidGC[pyramid]);
			w->panex.pyramidGC[pyramid] = XtGetGC((Widget) w,
				valueMask, &values);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->panex.pyramidName[pyramid]);
			stringCat(&buf2, buf1,
				"\" is not defined for pyramid ");
			free(buf1);
			intCat(&buf1, buf2, pyramid);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->panex.reverse) {
		values.background = w->panex.foreground;
		values.foreground = w->panex.background;
	} else {
		values.background = w->panex.background;
		values.foreground = w->panex.foreground;
	}
	if (w->panex.pyramidGC[pyramid])
		XtReleaseGC((Widget) w, w->panex.pyramidGC[pyramid]);
	w->panex.pyramidGC[pyramid] = XtGetGC((Widget) w, valueMask, &values);
}

static void
SetAllColors(PanexWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int pyramid;

	valueMask = GCForeground | GCBackground;
	if (w->panex.reverse) {
		values.background = w->panex.foreground;
		values.foreground = w->panex.background;
	} else {
		values.foreground = w->panex.background;
		values.background = w->panex.foreground;
	}
	if (w->panex.inverseGC)
		XtReleaseGC((Widget) w, w->panex.inverseGC);
	w->panex.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->panex.mono) {
		if (w->panex.reverse) {
			values.foreground = w->panex.background;
			values.background = w->panex.foreground;
		} else {
			values.foreground = w->panex.foreground;
			values.background = w->panex.background;
		}
	} else {
		values.foreground = w->panex.frameColor;
		values.background = w->panex.borderColor;
	}
	if (w->panex.frameGC)
		XtReleaseGC((Widget) w, w->panex.frameGC);
	w->panex.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->panex.reverse) {
		if (w->panex.mono) {
			values.background = w->panex.foreground;
			values.foreground = w->panex.background;
		} else {
			values.background = w->panex.tileColor;
			values.foreground = w->panex.borderColor;
		}
	} else {
		if (w->panex.mono) {
			values.foreground = w->panex.foreground;
			values.background = w->panex.background;
		} else {
			values.foreground = w->panex.tileColor;
			values.background = w->panex.borderColor;
		}
	}
	if (w->panex.tileGC)
		XtReleaseGC((Widget) w, w->panex.tileGC);
	w->panex.tileGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->panex.mono) {
		values.foreground = brighter(w, w->panex.tileColor);
	}
	if (w->panex.tileBrighterGC)
		XtReleaseGC((Widget) w, w->panex.tileBrighterGC);
	w->panex.tileBrighterGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->panex.mono) {
		values.foreground = darker(w, w->panex.tileColor);
	}
	if (w->panex.tileDarkerGC)
		XtReleaseGC((Widget) w, w->panex.tileDarkerGC);
	w->panex.tileDarkerGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->panex.reverse) {
		if (w->panex.mono) {
			values.foreground = w->panex.foreground;
			values.background = w->panex.background;
		} else {
			values.foreground = w->panex.tileColor;
			values.background = w->panex.borderColor;
		}
	} else {
		if (w->panex.mono) {
			values.background = w->panex.foreground;
			values.foreground = w->panex.background;
		} else {
			values.background = w->panex.tileColor;
			values.foreground = w->panex.borderColor;
		}
	}
	if (w->panex.borderGC)
		XtReleaseGC((Widget) w, w->panex.borderGC);
	w->panex.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (pyramid = 0; pyramid <= w->panex.mode; pyramid++)
		GetColor(w, pyramid);
	if (w->panex.fontInfo)
		XSetFont(XtDisplay(w), w->panex.borderGC,
			w->panex.fontInfo->fid);
}

static Boolean
SetValuesPanex(Widget current, Widget request, Widget renew)
{
	PanexWidget c = (PanexWidget) current, w = (PanexWidget) renew;
	Boolean redraw = False;
	Boolean redrawTiles = False, setColors = False;
	int pyramid;

	CheckTiles(w);
	for (pyramid = 0; pyramid <= w->panex.mode; pyramid++) {
		if (strcmp(w->panex.pyramidName[pyramid],
				c->panex.pyramidName[pyramid])) {
			setColors = True;
			break;
		}
	}
	if (w->panex.font != c->panex.font ||
			w->panex.borderColor != c->panex.borderColor ||
			w->panex.reverse != c->panex.reverse ||
			w->panex.mono != c->panex.mono ||
			setColors) {
		loadFont(w);
		SetAllColors(w);
		redrawTiles = True;
	} else if (w->panex.background != c->panex.background ||
			w->panex.foreground != c->panex.foreground ||
			w->panex.borderColor != c->panex.borderColor ||
			w->panex.tileColor != c->panex.tileColor ||
			w->panex.reverse != c->panex.reverse ||
			w->panex.mono != c->panex.mono ||
			setColors) {
		SetAllColors(w);
		redrawTiles = True;
	}
	if (w->panex.tiles != c->panex.tiles) {
		EraseFrame(c, 0, False);
		SizePanex(w);
		redraw = True;
	} else if (w->panex.mode != c->panex.mode) {
		ResetTiles(w);
		redraw = True;
	} else if (w->panex.puzzleOffset.x != c->panex.puzzleOffset.x ||
			w->panex.puzzleOffset.y != c->panex.puzzleOffset.y) {
		EraseFrame(c, 0, False);
		ResizePanex(w);
		redraw = True;
	}
	if (w->panex.delay != c->panex.delay) {
		w->panex.numSlices = ((w->panex.delay < MAXSLICES) ?
			w->panex.delay + 1 : MAXSLICES);
	}
	if (w->panex.menu != -1) {
		int menu = w->panex.menu;

		w->panex.menu = -1;
		switch (menu) {
		case 0:
			GetTiles(w);
			break;
		case 1:
			WriteTiles(w);
			break;
		case 3:
			ClearTiles(w);
			break;
		case 4:
			UndoTiles(w);
			break;
		case 5:
			SolveTiles(w);
			break;
		case 6:
			IncrementTiles(w);
			break;
		case 7:
			DecrementTiles(w);
			break;
		case 8:
			ModeTiles(w);
			break;
		case 9:
			SpeedTiles(w);
			break;
		case 10:
			SlowTiles(w);
			break;
		case 11:
			SoundTiles(w);
			break;

		default:
			break;
		}
	}
	if (redrawTiles && !redraw && XtIsRealized(renew) &&
			renew->core.visible) {
		EraseFrame(w, 0, w->panex.focus);
		DrawSlots(w, 0);
		DrawAllTiles(w);
	}
	return (redraw);
}

static void
QuitPanex(PanexWidget w, XEvent * event, char **args, int nArgs)
{
	Display *display = XtDisplay(w);

	if (w->panex.fontInfo) {
		XUnloadFont(display, w->panex.fontInfo->fid);
		XFreeFont(display, w->panex.fontInfo);
	}
	XtCloseDisplay(display);
	exit(0);
}

static void
DestroyPanex(Widget old)
{
	PanexWidget w = (PanexWidget) old;
	int pyramid;

	for (pyramid = 0; pyramid <= w->panex.mode; pyramid++)
		XtReleaseGC(old, w->panex.pyramidGC[pyramid]);
	XtReleaseGC(old, w->panex.tileGC);
	XtReleaseGC(old, w->panex.borderGC);
	XtReleaseGC(old, w->panex.tileBrighterGC);
	XtReleaseGC(old, w->panex.tileDarkerGC);
	XtReleaseGC(old, w->panex.frameGC);
	XtReleaseGC(old, w->panex.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->panex.select);
}
#endif

static void
ResizeTiles(PanexWidget w)
{
	int sel;
#ifdef WINVER
	if (w->core.memDC == NULL) {
		w->core.memDC = CreateCompatibleDC(w->core.hDC);
		if (w->core.memDC == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->panex.bufferTiles[sel] != NULL) {
			DeleteObject(w->panex.bufferTiles[sel]);
			w->panex.bufferTiles[sel] = NULL;
		}
		if ((w->panex.bufferTiles[sel] =
				CreateCompatibleBitmap(w->core.hDC,
				w->core.width, w->core.height)) == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
#else
	Display *display = XtDisplay(w);
	Window window = XtWindow(w);
	XWindowAttributes xgwa;

	(void) XGetWindowAttributes(display, window, &xgwa);
	if (w->panex.colormap == None) {
		w->panex.mono = (xgwa.depth < 2 || w->panex.mono);
		w->panex.colormap = xgwa.colormap;
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->panex.bufferTiles[sel] != None) {
			XFreePixmap(display, w->panex.bufferTiles[sel]);
			w->panex.bufferTiles[sel] = None;
		}
		if ((w->panex.bufferTiles[sel] = XCreatePixmap(display,
				window, w->core.width, w->core.height,
				xgwa.depth)) == None) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	SetAllColors(w);
#endif
	DrawAllBufferedTiles(w);
}

#ifndef WINVER
static
#endif
void
ResizePanex(PanexWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->panex.delta.x = 8;
	w->panex.delta.y = 2;
	w->panex.pos.x = MAX(((int) w->core.width - w->panex.delta.x) /
			MAXSTACKS, w->panex.delta.x);
	w->panex.pos.y = MAX(((int) w->core.height - 2 * w->panex.delta.y - 3) /
			(w->panex.tiles + 1), w->panex.delta.y);
	w->panex.width = w->panex.pos.x * MAXSTACKS + w->panex.delta.x + 2;
	w->panex.height = w->panex.pos.y * (w->panex.tiles + 1) +
		w->panex.delta.y + 3;
	w->panex.puzzleOffset.x = ((int) w->core.width - w->panex.width + 2) / 2;
	w->panex.puzzleOffset.y = ((int) w->core.height - w->panex.height + 2) / 2;
	/* Make the following odd */
	w->panex.tileSize.x = (((w->panex.pos.x - w->panex.delta.x) >> 1) << 1) + 1;
	w->panex.tileSize.y = w->panex.pos.y - w->panex.delta.y + 1;
	w->panex.letterOffset.x = 3;
	w->panex.letterOffset.y = 4;
}

#ifndef WINVER
static
#endif
void
SizePanex(PanexWidget w)
{
	ResetTiles(w);
	ResizePanex(w);
}

#ifndef WINVER
static
#endif
void
InitializePanex(
#ifdef WINVER
PanexWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int stack;
#ifdef WINVER
	SetValuesPanex(w);
	brush = CreateSolidBrush(w->panex.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
	w->panex.bufferTiles[0] = NULL;
	w->panex.bufferTiles[1] = NULL;
#else
	PanexWidget w = (PanexWidget) renew;

	(void) SRAND(getpid());
	for (stack = 0; stack <= w->panex.mode; stack++)
		w->panex.pyramidGC[stack] = NULL;
	w->panex.bufferTiles[0] = None;
	w->panex.bufferTiles[1] = None;
	w->panex.colormap = None;
	w->panex.fontInfo = NULL;
	w->panex.tileGC = NULL;
	w->panex.borderGC = NULL;
	w->panex.tileBrighterGC = NULL;
	w->panex.tileDarkerGC = NULL;
	w->panex.frameGC = NULL;
	w->panex.inverseGC = NULL;
#endif
	w->panex.focus = False;
	loadFont(w);
	for (stack = 0; stack < MAXSTACKS; stack++) {
		w->panex.tileOfPosition[stack] = NULL;
		w->panex.positionOfTile[stack] = NULL;
	}
	CheckTiles(w);
	InitMoves();
	w->panex.numSlices = ((w->panex.delay < MAXSLICES) ?
		w->panex.delay + 1 : MAXSLICES);
	SizePanex(w);
}

void
ExposePanex(
#ifdef WINVER
PanexWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	PanexWidget w = (PanexWidget) renew;

	if (!w->core.visible)
		return;
#endif
	ResizeTiles(w);
	EraseFrame(w, 0, w->panex.focus);
	DrawSlots(w, 0);
	DrawAllTiles(w);
}

#ifndef WINVER
static
#endif
void
HidePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetPanex(w, PANEX_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectPanex(PanexWidget w
#ifdef WINVER
, const int x
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int stack;
#ifndef WINVER
	int x = event->xbutton.x;
#endif

	w->panex.previousStack = -1;
	if (CheckSolved(w)) {
		MoveNoTiles(w);
	} else if (((stack = SelectTile(w, x)) >= 0) &&
		   ((w->panex.currentPosition = TopOfStack(w, stack, 0)) >= 0)) {
		w->panex.currentStack = stack;
		if (MoveTileToLimbo(w, w->panex.currentStack,
		    w->panex.currentPosition, w->panex.currentStack, False)) {
			w->panex.currentPosition = 0;
			w->panex.previousStack = w->panex.currentStack;
		} else if (MoveTileToLimbo(w, w->panex.currentStack,
		    w->panex.currentPosition, w->panex.currentStack, True)) {
			w->panex.currentPosition = 0;
			w->panex.previousStack = w->panex.currentStack;
		}
		DrawTile(w, stack, w->panex.currentPosition, False,
			TRUE, 0, 0);
	} else
		w->panex.currentStack = -1;
}

#ifndef WINVER
static
#endif
void
MotionPanex(PanexWidget w
#ifdef WINVER
, const int x
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int toStack;
#ifndef WINVER
	int x = event->xbutton.x;
#endif

	if (w->panex.currentStack < 0)
		return;
	
	if ((toStack = SelectTile(w, x)) >= 0 &&
			toStack != w->panex.currentStack) {
		if (MoveTileToLimbo(w, w->panex.currentStack,
				w->panex.currentPosition, toStack, True)) {
#if 0
	DrawTile(w, w->panex.currentStack, w->panex.currentPosition,
		True, TRUE, 0, 0);
#endif
			if (w->panex.previousStack < 0) {
				w->panex.previousStack = w->panex.currentStack;
			}
			w->panex.currentStack = toStack;
			w->panex.currentPosition = 0;
			DrawTile(w, toStack, w->panex.currentPosition, False,
				TRUE, 0, 0);
		} else if (MoveTileToLimbo(w, w->panex.currentStack,
				w->panex.currentPosition, toStack, False)) {
#if 0
	DrawTile(w, w->panex.currentStack, w->panex.currentPosition,
		True, TRUE, 0, 0);
#endif
			w->panex.currentStack = toStack;
			w->panex.currentPosition = 0;
			DrawTile(w, toStack, w->panex.currentPosition, False,
				TRUE, 0, 0);
		}
	}
}

#ifndef WINVER
static
#endif
void
ReleasePanex(PanexWidget w
#ifdef WINVER
, const int x
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int toStack, toPosition;
#ifndef WINVER
	int x = event->xbutton.x;
#endif

	if (w->panex.currentStack < 0)
		return;
	DrawTile(w, w->panex.currentStack, w->panex.currentPosition,
		True, TRUE, 0, 0);
	DrawTile(w, w->panex.currentStack, w->panex.currentPosition,
		False, FALSE, 0, 0);
	if ((toStack = SelectTile(w, x)) >= 0 &&
			((w->panex.previousStack < 0 &&
			toStack != w->panex.currentStack) ||
			(w->panex.previousStack >= 0))) {
		if ((toPosition = MoveTile(w, w->panex.currentStack,
				w->panex.currentPosition, toStack,
				NORMAL)) >= 0) {
			if (w->panex.previousStack != toStack) {
				SetPanex(w, PANEX_MOVED);
				if (w->panex.previousStack >= 0) {
				  PutMove(w->panex.previousStack, toStack);
				} else {
				  PutMove(w->panex.currentStack, toStack);
				}
				if (CheckSolved(w)) {
					SetPanex(w, PANEX_SOLVED);
				}
			}
		} else if (toPosition == -1) {
		  SetPanex(w, PANEX_BLOCKED);
		  if (w->panex.previousStack != -1) {
		    if ((toPosition = MoveTile(w, w->panex.currentStack,
			w->panex.currentPosition,
			w->panex.currentStack, NORMAL)) >= 0) {
		      if (w->panex.previousStack != toStack &&
			  w->panex.previousStack != w->panex.currentStack) {
				SetPanex(w, PANEX_MOVED);
				if (w->panex.previousStack >= 0) {
				  PutMove(w->panex.previousStack, toStack);
				} else {
				  PutMove(w->panex.currentStack, toStack);
				}
				if (CheckSolved(w)) {
					SetPanex(w, PANEX_SOLVED);
				}
			}
		    }
		  }
		} else {
		  SetPanex(w, PANEX_ILLEGAL);
		  if (w->panex.previousStack != -1) {
		    toPosition = MoveTile(w, w->panex.currentStack,
			w->panex.currentPosition,
			w->panex.previousStack, NORMAL);
		  }
		}
	}
	w->panex.previousStack = -1;
	w->panex.currentStack = -1;
}

void
GetPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetTiles(w);
}

void
WritePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WriteTiles(w);
}

void
ClearPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearTiles(w);
}

#ifndef WINVER
void
ClearPanexMaybe(PanexWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->panex.started) {
		ClearTiles(w);
	}
#ifdef HAVE_MOTIF
	else {
		/* Check if one really wants to destroy current state. */
		SetPanex(w, PANEX_RESET_QUERY);
	}
#endif
}

void
ClearPanex2(PanexWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifndef HAVE_MOTIF
	ClearTiles(w);
#endif
}
#endif

void
UndoPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoTiles(w);
}

void
SolvePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolveTiles(w);
}

void
ModePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ModeTiles(w);
}

#ifndef WINVER
static
#endif
void
SpeedPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SpeedTiles(w);
}

#ifndef WINVER
static
#endif
void
SlowPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SlowTiles(w);
}

#ifndef WINVER
static
#endif
void
SoundPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SoundTiles(w);
}

void
IncrementPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	IncrementTiles(w);
}

void
DecrementPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	DecrementTiles(w);
}

void
EnterPanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->panex.focus = True;
	DrawFrame(w, 0, w->panex.focus);
}

void
LeavePanex(PanexWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->panex.focus = False;
	DrawFrame(w, 0, w->panex.focus);
}
