/*-
# X-BASED PYRAMINX(tm)
#
#  Pyraminx.c
#
###
#
#  Copyright (c) 1994 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Pyraminx */

#include "file.h"
#include "rngs.h"
#include "PyraminxP.h"

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wpyraminx.ini"
#endif

#define SECTION "setup"

static const char *faceColorString[MAXFACES] =
{
	"0 0 255",
	"255 0 0",
	"255 255 0",
	"0 255 0"
};

static const char faceColorChar[MAXFACES] =
{'B', 'R', 'Y', 'G'};
#else

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static void ResizePyraminx(PyraminxWidget w);
static void SizePyraminx(PyraminxWidget w);
static Boolean SetValuesPyraminx(Widget current, Widget request, Widget renew);
static void DestroyPyraminx(Widget old);
static void QuitPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void InitializePyraminx(Widget request, Widget renew);
static void ExposePyraminx(Widget renew, XEvent * event, Region region);
static void HidePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void PracticePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void PracticePyraminxMaybe(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void PracticePyraminx2(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void RandomizePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void RandomizePyraminxMaybe(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void RandomizePyraminx2(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void GetPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void WritePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void ClearPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void UndoPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void SolvePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void IncrementPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void DecrementPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void OrientizePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void StickyModePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void Period2ModePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void Period3ModePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void BothModePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void EnterPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void LeavePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxTop(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxTr(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxLeft(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxCw(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxRight(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxBl(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxBottom(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxCcw(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void MovePyraminxInput(PyraminxWidget w, int x, int y, int direction, int shift, int control);
static void SelectPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);
static void ReleasePyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsPyraminx[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>r: Randomize()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>2: Period2()\n\
 <KeyPress>3: Period3()\n\
 <KeyPress>b: Both()\n\
 <KeyPress>y: Sticky()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPyraminx[] =
{
	{(char *) "Quit", (XtActionProc) QuitPyraminx},
	{(char *) "Hide", (XtActionProc) HidePyraminx},
	{(char *) "MoveCcw", (XtActionProc) MovePyraminxCcw},
	{(char *) "MoveTop", (XtActionProc) MovePyraminxTop},
	{(char *) "MoveTr", (XtActionProc) MovePyraminxTr},
	{(char *) "MoveLeft", (XtActionProc) MovePyraminxLeft},
	{(char *) "MoveCw", (XtActionProc) MovePyraminxCw},
	{(char *) "MoveRight", (XtActionProc) MovePyraminxRight},
	{(char *) "MoveBl", (XtActionProc) MovePyraminxBl},
	{(char *) "MoveBottom", (XtActionProc) MovePyraminxBottom},
	{(char *) "Select", (XtActionProc) SelectPyraminx},
	{(char *) "Release", (XtActionProc) ReleasePyraminx},
	{(char *) "Practice", (XtActionProc) PracticePyraminx},
	{(char *) "PracticeMaybe", (XtActionProc) PracticePyraminxMaybe},
	{(char *) "Practice2", (XtActionProc) PracticePyraminx2},
	{(char *) "Randomize", (XtActionProc) RandomizePyraminx},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizePyraminxMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizePyraminx2},
	{(char *) "Get", (XtActionProc) GetPyraminx},
	{(char *) "Write", (XtActionProc) WritePyraminx},
	{(char *) "Clear", (XtActionProc) ClearPyraminx},
	{(char *) "Undo", (XtActionProc) UndoPyraminx},
	{(char *) "Solve", (XtActionProc) SolvePyraminx},
	{(char *) "Increment", (XtActionProc) IncrementPyraminx},
	{(char *) "Decrement", (XtActionProc) DecrementPyraminx},
	{(char *) "Orientize", (XtActionProc) OrientizePyraminx},
	{(char *) "Period2", (XtActionProc) Period2ModePyraminx},
	{(char *) "Period3", (XtActionProc) Period3ModePyraminx},
	{(char *) "Both", (XtActionProc) BothModePyraminx},
	{(char *) "Sticky", (XtActionProc) StickyModePyraminx},
	{(char *) "Enter", (XtActionProc) EnterPyraminx},
	{(char *) "Leave", (XtActionProc) LeavePyraminx}
};

static XtResource resourcesPyraminx[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(PyraminxWidget, core.width),
	 XtRString, (caddr_t) "200"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(PyraminxWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
  /* Beware color values are swapped */
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[0]),
	 XtRString, (caddr_t) "blue"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[1]),
	 XtRString, (caddr_t) "red"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[2]),
	 XtRString, (caddr_t) "yellow"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[3]),
	 XtRString, (caddr_t) "green"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.delay),
	 XtRString, (caddr_t) "300"},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNsize, XtCSize, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.size),
	 XtRString, (caddr_t) "3"},	/*DEFAULTFACETS */
	{XtNsticky, XtCSticky, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.sticky),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULTSTICKY */
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.mode),
	 XtRString, (caddr_t) "3"},	/*DEFAULTMODE */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.orient),
	 XtRString, (caddr_t) "FALSE"},	/*DEFAULTORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.practice),
	 XtRString, (caddr_t) "FALSE"},	/*DEFAULTPRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(PyraminxWidget, pyraminx.select),
	 XtRCallback, (caddr_t) NULL}
};

PyraminxClassRec pyraminxClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Pyraminx",	/* class name */
		sizeof (PyraminxRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePyraminx,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPyraminx,	/* actions */
		XtNumber(actionsListPyraminx),	/* num actions */
		resourcesPyraminx,	/* resources */
		XtNumber(resourcesPyraminx),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyPyraminx,		/* destroy */
		(XtWidgetProc) ResizePyraminx,	/* resize */
		(XtExposeProc) ExposePyraminx,	/* expose */
		(XtSetValuesFunc) SetValuesPyraminx,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPyraminx,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass pyraminxWidgetClass = (WidgetClass) & pyraminxClassRec;

void
SetPyraminx(PyraminxWidget w, int reason)
{
	pyraminxCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(PyraminxWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->pyraminx.fontInfo) {
		XUnloadFont(XtDisplay(w), w->pyraminx.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->pyraminx.fontInfo);
	}
	if ((w->pyraminx.fontInfo = XLoadQueryFont(display,
			w->pyraminx.font)) == NULL) {
		(void) sprintf(buf,
			"Can not open %s font.\nAttempting %s font as alternate\n",
			w->pyraminx.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->pyraminx.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Can not open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->pyraminx.fontInfo) {
		w->pyraminx.letterOffset.x =
			XTextWidth(w->pyraminx.fontInfo, "8", 1) / 2;
		w->pyraminx.letterOffset.y =
			w->pyraminx.fontInfo->max_bounds.ascent / 2;
	} else
#endif
	{
		w->pyraminx.letterOffset.x = 3;
		w->pyraminx.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "pyraminx.log"
#endif

#define NOTDIR(x) ((x==CW)?CCW:CW)

typedef struct _CRD {
	int column, row, diagonal;
} CRD;

typedef struct _RowNextP3 {
	int viewChanged, face, direction, reverse;
} RowNextP3;
static PyraminxLoc slideNextRowP2[MAXSIDES][3] =
{
	{
		{2, 0},
		{1, 3},
		{2, 3}},
	{
		{2, 0},
		{0, 3},
		{2, 3}}
};
static RowNextP3 slideNextRowP3[MAXSIDES][MAXORIENT] =
{
	{
		{TRUE, UP, TR, FALSE},
		{TRUE, UP, TOP, FALSE},
		{FALSE, UP, BOTTOM, TRUE},
		{FALSE, UP, RIGHT, TRUE},
		{TRUE, DOWN, RIGHT, TRUE},
		{TRUE, DOWN, TR, TRUE}
	},
	{
		{FALSE, DOWN, LEFT, TRUE},
		{TRUE, UP, LEFT, TRUE},
		{TRUE, UP, BL, TRUE},
		{TRUE, DOWN, BL, FALSE},
		{TRUE, DOWN, BOTTOM, FALSE},
		{FALSE, DOWN, TOP, TRUE}
	}
};
static int rotOrientRowP3[3][MAXORIENT] =
/* current orient, rotation */
{
	{1, 5, 1, 5, 4, 2},
	{2, 0, 2, 0, 5, 3},
	{3, 1, 3, 1, 0, 4}
};

static Point triangleUnit[MAXSIDES][4] =
{
	{
		{0, 2},
		{1, 0},
		{-1, 1},
		{0, -1}},
	{
		{1, 3},
		{-1, 0},
		{1, -1},
		{0, 1}}
};
static Point triangleList[MAXSIDES][4], letterList[MAXSIDES];
static Point offsetList[MAXSIDES];

static Pixmap dr = 0; /* dummy for future double buffering */

static void
CheckPieces(PyraminxWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->pyraminx.size < MINFACETS) {
		intCat(&buf1,
			"Number of triangles on an edge out of bounds, use at least ",
			MINFACETS);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTFACETS);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->pyraminx.size = DEFAULTFACETS;
	}
	if (w->pyraminx.mode < PERIOD3 || w->pyraminx.mode > BOTH) {
		intCat(&buf1, "Mode is in error, use 2 for Period2, 3 for Period3, 4 for Both, defaulting to ",
			DEFAULTMODE);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->pyraminx.mode = DEFAULTMODE;
	}

}

Boolean
CheckSolved(PyraminxWidget w)
{
	int face, position;
	PyraminxLoc test;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			if (!position) {
				test.face = w->pyraminx.facetLoc[face][position].face;
				test.rotation = w->pyraminx.facetLoc[face][position].rotation;
			} else if (test.face !=		/* MAXSIDES * view + face */
					 w->pyraminx.facetLoc[face][position].face ||
					(w->pyraminx.orient && test.rotation !=
					w->pyraminx.facetLoc[face][position].rotation))
				return False;
		}
	return True;
}

#ifdef DEBUG

static void
PrintFacet(PyraminxWidget w)
{
	int face, position, square;

	for (face = 0; face < MAXSIDES; face++) {
		square = 1;
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			(void) printf("%d %d  ",
				w->pyraminx.facetLoc[face][position].face,
				w->pyraminx.facetLoc[face][position].rotation);
			if (position == square * square - 1) {
				(void) printf("\n");
				++square;
			}
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}

static void
PrintFace(PyraminxWidget w)
{
	int g, h, side;

	for (g = 0; g < w->pyraminx.size; g++) {
		for (h = 0; h < w->pyraminx.size - g; h++)
			for (side = 0; side < MAXSIDES; side++)
				if (!side || h < w->pyraminx.size - g - 1)
					(void) printf("%d %d  ", w->pyraminx.faceLoc[side][h +
						g * w->pyraminx.size].face,
						w->pyraminx.faceLoc[side][h + g * w->pyraminx.size].rotation);
		(void) printf("\n");
	}
	(void) printf("\n");

#if 0
	int position;
	int square = 1;

	for (position = 0; position < w->pyraminx.sizeSize; position++) {
		(void) printf("%d %d  ", w->pyraminx.faceLoc[position].face,
			w->pyraminx.faceLoc[position].rotation);
		if (position == square * square - 1) {
			(void) printf("\n");
			++square;
		}
	}
	(void) printf("\n");
#endif
}

static void
PrintRow2(PyraminxWidget w, int orient)
{
	int g, side;

	(void) printf("Row %d:\n", orient);
	for (g = 0; g < w->pyraminx.size; g++)
		for (side = 0; side < MAXSIDES; side++)
			(void) printf("%d %d  ", w->pyraminx.rowLoc[orient][side][g].face,
				w->pyraminx.rowLoc[orient][side][g].rotation);
	(void) printf("\n");
}

static void
PrintRow3(PyraminxWidget w, int len, int orient)
{
	int g, side;

	(void) printf("Row %d:\n", orient);
	for (g = 0; g <= len; g++)
		for (side = 0; side < MAXSIDES; side++)
			if (!side || g < len)
				(void) printf("%d %d  ", w->pyraminx.rowLoc[orient][side][g].face,
				w->pyraminx.rowLoc[orient][side][g].rotation);
	(void) printf("\n");
}

#endif

/* This is fast for small i, a -1 is returned for negative i. */
static int
Sqrt(int i)
{
	int j = 0;

	while (j * j <= i)
		j++;
	return (j - 1);
}
static void
ToCRD(PyraminxWidget w, int face, int position, CRD * crd)
{
	int diag, diag2;

	diag = Sqrt(position);
	diag2 = diag * diag;
	/* Passing diagonal so there is no sqrt calculation again */
	if ((face % 2) == UP) {
		crd->column = w->pyraminx.size - 1 - (position - diag2) / 2;
		crd->row = w->pyraminx.size - 1 - (diag2 + 2 * diag - position) / 2;
		crd->diagonal = 2 * w->pyraminx.size - 1 - diag;
	} else {
		crd->column = (position - diag2) / 2;
		crd->row = (diag2 + 2 * diag - position) / 2;
		crd->diagonal = diag;
	}
}

static int
ToPosition(PyraminxWidget w, CRD crd)
{
	int diag;

	if (crd.diagonal < w->pyraminx.size)
		return (crd.diagonal * crd.diagonal + crd.diagonal + crd.column - crd.row);
	diag = 2 * w->pyraminx.size - 1 - crd.diagonal;
	return (diag * diag + diag + crd.row - crd.column);
}

static int
Crd(int dir, int I, int J, int side)
{
	if (dir == TOP || dir == BOTTOM)
		return (I);
	else if (dir == RIGHT || dir == LEFT)
		return (J);
	else			/* dir == TR || dir == BL */
		return (I + J + side);
}

static int
Length(PyraminxWidget w, int face, int dir, int h)
{
	if (dir == TR || dir == BL)
		return ((face == UP) ? 2 * w->pyraminx.size - 1 - h : h);
	else
		return ((face == UP) ? h : w->pyraminx.size - 1 - h);
}

static void
DrawOrientLine(PyraminxWidget w, int orient, int dx, int dy, int side,
		GC borderGC)
{
	int x_1 = 0, x_2 = 0, y_1 = 0, y_2 = 0;
	int temp1 = w->pyraminx.facetLength / 2 + 1;
	int temp2 = w->pyraminx.facetLength + 1;
	int fix = (w->pyraminx.size == 1) ? 1 : 0;

	switch (orient) {
		case TOP:
			x_2 = x_1 = dx + temp1 + 2;
			y_1 = (side == UP) ? dy + temp1 + 1 : dy;
			y_2 = y_1 + w->pyraminx.orientLineLength;
			break;
		case TR:
			x_1 = (side == UP) ? dx + temp2 + 1 : dx + temp1;
			x_2 = x_1 - w->pyraminx.orientDiagLength;
			y_1 = (side == UP) ? dy + temp1 + 1 : dy;
			y_2 = y_1 + w->pyraminx.orientDiagLength;
			break;
		case RIGHT:
			x_1 = (side == UP) ? dx + temp2 + 1 : dx + temp1 - fix - 1;
			x_2 = x_1 - w->pyraminx.orientLineLength;
			y_2 = y_1 = dy + temp1 - 1;
			break;
		case BOTTOM:
			x_2 = x_1 = dx + temp1 - 1;
			y_1 = (side == UP) ? dy + temp2 + 1 : dy + temp1 - fix - 1;
			y_2 = y_1 - w->pyraminx.orientLineLength;
			break;
		case BL:
			x_1 = (side == UP) ? dx + temp1 : dx;
			x_2 = x_1 + w->pyraminx.orientDiagLength;
			y_1 = (side == UP) ? dy + temp2 + 1 : dy + temp1;
			y_2 = y_1 - w->pyraminx.orientDiagLength;
			break;
		case LEFT:
			x_1 = (side == UP) ? dx + temp1 + 1 : dx;
			x_2 = x_1 + w->pyraminx.orientLineLength;
			y_2 = y_1 = dy + temp1 + 2;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "DrawOrientLine: orient ",
					orient);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
	DRAWLINE(w, dr, borderGC, x_1, y_1, x_2, y_2);
}

static void
DrawTriangle(PyraminxWidget w, int face, int position, int offset)
{
	GC faceGC, borderGC;
	int dx, dy, faceOnView, orient, side, view;
	CRD crd;

	ToCRD(w, face, position, &crd);
	view = (face / MAXSIDES == UP) ? DOWN : UP;
	side = crd.diagonal - crd.column - crd.row;
	if (!w->pyraminx.vertical && view == DOWN) {
		faceOnView = !side;
		dx = (w->pyraminx.size - crd.column - 1) *
			(w->pyraminx.facetLength + w->pyraminx.delta);
		dy = (w->pyraminx.size - crd.row - 1) *
			(w->pyraminx.facetLength + w->pyraminx.delta);
		orient = (w->pyraminx.facetLoc[face][position].rotation +
			MAXORIENT / 2) % MAXORIENT;
		if (2 * w->pyraminx.size - crd.column - crd.row - 2 +
				faceOnView >= w->pyraminx.size) {
			dx += w->pyraminx.sideOffset;
			dy += w->pyraminx.sideOffset;
		}
	} else {
		faceOnView = side;
		dx = crd.column * (w->pyraminx.facetLength + w->pyraminx.delta);
		dy = crd.row * (w->pyraminx.facetLength + w->pyraminx.delta);
		orient = w->pyraminx.facetLoc[face][position].rotation;
		if (crd.column + crd.row + faceOnView >= w->pyraminx.size) {
			dx += w->pyraminx.sideOffset;
			dy += w->pyraminx.sideOffset;
		}
	}
	dx += w->pyraminx.puzzleOffset.x + w->pyraminx.delta;
	dy += w->pyraminx.puzzleOffset.y + w->pyraminx.delta;
	if (view == DOWN) {
		if (w->pyraminx.vertical)
			dy += w->pyraminx.viewLength - w->pyraminx.delta - 1;
		else
			dx += w->pyraminx.viewLength - w->pyraminx.delta - 1;
	}
	triangleList[faceOnView][0].x = offsetList[faceOnView].x + dx;
	triangleList[faceOnView][0].y = offsetList[faceOnView].y + dy;
	if (offset) {
		borderGC = w->pyraminx.faceGC[w->pyraminx.facetLoc[face][position].face];
		if (w->pyraminx.mono) {
			faceGC = w->pyraminx.inverseGC;
		} else {
			faceGC = w->pyraminx.borderGC;
		}
	} else {
		faceGC = w->pyraminx.faceGC[w->pyraminx.facetLoc[face][position].face];
		borderGC = w->pyraminx.borderGC;
	}

	POLYGON(w, dr, faceGC, borderGC, triangleList[faceOnView], 3,
		True, False);
	dx += -faceOnView;
	dy += -faceOnView;
	if (w->pyraminx.mono) {
		int letterX, letterY;
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->pyraminx.faceChar[w->pyraminx.facetLoc
				[face][position].face];
#else
			w->pyraminx.faceName[w->pyraminx.facetLoc
				[face][position].face][0];
#endif
		buf[1] = '\0';
		letterX = dx + letterList[faceOnView].x;
		letterY = dy + letterList[faceOnView].y;
		if (offset) {
			borderGC = w->pyraminx.borderGC;
		} else {
			borderGC = w->pyraminx.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC, letterX, letterY, buf, 1);
	}
	if (w->pyraminx.orient)
		DrawOrientLine(w, orient, dx, dy, faceOnView, borderGC);
}

void
DrawAllPieces(PyraminxWidget w)
{
	int face, position;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++)
			DrawTriangle(w, face, position, FALSE);
}

static void
EraseFrame(const PyraminxWidget w)
{
	FILLRECTANGLE(w, dr, w->pyraminx.inverseGC,
		0, 0, w->core.width, w->core.height);
}

static void
DrawFrame(PyraminxWidget w, Boolean focus)
{
	int startx, starty, lengthx, lengthy, longlength;
	GC gc = (focus) ? w->pyraminx.frameGC : w->pyraminx.borderGC;

	startx = 1 + w->pyraminx.puzzleOffset.x;
	starty = 1 + w->pyraminx.puzzleOffset.y;
	lengthx = w->pyraminx.viewLength - w->pyraminx.delta +
		w->pyraminx.puzzleOffset.x;
	lengthy = w->pyraminx.viewLength - w->pyraminx.delta +
		w->pyraminx.puzzleOffset.y;
	DRAWLINE(w, dr, gc, startx, starty, lengthx, starty);
	DRAWLINE(w, dr, gc, startx, starty, startx, lengthy);
	DRAWLINE(w, dr, gc, startx, lengthy, lengthx, starty);
	if (w->pyraminx.vertical) {
		longlength = 2 * w->pyraminx.viewLength - 2 * w->pyraminx.delta - 1 +
			w->pyraminx.puzzleOffset.y;
		DRAWLINE(w, dr, gc,
			startx, lengthy, startx, longlength);
		DRAWLINE(w, dr, gc,
			lengthx, lengthy, lengthx, longlength);
		DRAWLINE(w, dr, gc,
			startx, longlength, lengthx, longlength);
		DRAWLINE(w, dr, gc,
			startx, longlength, lengthx, lengthy);
		DRAWLINE(w, dr, gc,
			lengthx, starty, lengthx, lengthy);
		DRAWLINE(w, dr, w->pyraminx.frameGC,
			0, lengthy, (int) w->core.width, lengthy);
	} else {
		longlength = 2 * w->pyraminx.viewLength - 2 * w->pyraminx.delta - 1 +
			w->pyraminx.puzzleOffset.x;
		DRAWLINE(w, dr, gc,
			lengthx, starty, longlength, starty);
		DRAWLINE(w, dr, gc,
			lengthx, lengthy, longlength, lengthy);
		DRAWLINE(w, dr, gc,
			longlength, starty, longlength, lengthy);
		DRAWLINE(w, dr, gc,
			longlength, starty, lengthx, lengthy);
		DRAWLINE(w, dr, gc,
			startx, lengthy, lengthx, lengthy);
		DRAWLINE(w, dr, w->pyraminx.frameGC,
			lengthx, 0, lengthx, (int) w->core.height);
	}
}

static void
MoveNoPieces(PyraminxWidget w)
{
	SetPyraminx(w, PYRAMINX_ILLEGAL);
}

static void
RotateFace(PyraminxWidget w, int view, int faceOnView, int direction)
{
	int g, h, side, face, position;
	CRD crd;

	/* Read Face */
	for (g = 0; g < w->pyraminx.size; g++)
		for (h = 0; h < w->pyraminx.size - g; h++)
			for (side = 0; side < MAXSIDES; side++)
				if (g + h + side < w->pyraminx.size) {
					if (faceOnView == DOWN) {
						crd.column = h;
						crd.row = g;
						crd.diagonal = h + g + side;
						face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = ToPosition(w, crd);
						w->pyraminx.faceLoc[side][h + g * w->pyraminx.size] =
							w->pyraminx.facetLoc[face][position];
					} else {	/* faceOnView == UP */
						crd.column = w->pyraminx.size - 1 - h;
						crd.row = w->pyraminx.size - 1 - g;
						crd.diagonal = crd.column + crd.row + !side;
						face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = ToPosition(w, crd);
						w->pyraminx.faceLoc[side][h + g * w->pyraminx.size] =
							w->pyraminx.facetLoc[face][position];
					}
				}
	/* Write Face */
	if (faceOnView == DOWN) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (h = 0; h < w->pyraminx.size - g; h++)
				for (side = 0; side < MAXSIDES; side++)
					if (g + h + side < w->pyraminx.size) {
						crd.column = h;
						crd.row = g;
						crd.diagonal = h + g + side;
						face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = ToPosition(w, crd);
						if (direction == CCW)
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[side][w->pyraminx.size - 1 - g - h - side +
							h * w->pyraminx.size];
						else	/* direction == CW */
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[side][g +
											(w->pyraminx.size - 1 - g - h - side) * w->pyraminx.size];
						w->pyraminx.facetLoc[face][position].rotation =
							(direction == CW) ?
							(w->pyraminx.facetLoc[face][position].rotation + 2) %
							MAXORIENT :
							(w->pyraminx.facetLoc[face][position].rotation + 4) %
							MAXORIENT;
						DrawTriangle(w, face, position, FALSE);
					}
	} else {		/* faceOnView == UP */
		for (g = w->pyraminx.size - 1; g >= 0; g--)
			for (h = w->pyraminx.size - 1; h >= w->pyraminx.size - 1 - g; h--)
				for (side = 1; side >= 0; side--)
					if (g + h + side >= w->pyraminx.size) {
						crd.column = h;
						crd.row = g;
						crd.diagonal = h + g + side;
						face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
						position = ToPosition(w, crd);
						if (direction == CCW)
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[!side][g + h - w->pyraminx.size + 1 - !side
											+ (w->pyraminx.size - 1 - h) * w->pyraminx.size];
						else	/* (direction == CW) */
							w->pyraminx.facetLoc[face][position] =
								w->pyraminx.faceLoc[!side][w->pyraminx.size - 1 - g +
											(g + h - w->pyraminx.size + 1 - !side) * w->pyraminx.size];
						w->pyraminx.facetLoc[face][position].rotation =
							(direction == CW) ?
							(w->pyraminx.facetLoc[face][position].rotation + 2) %
							MAXORIENT :
							(w->pyraminx.facetLoc[face][position].rotation + 4) %
							MAXORIENT;
						DrawTriangle(w, face, position, FALSE);
					}
	}
}

static void
ReadCRD2(PyraminxWidget w, int view, int dir, int h, int orient)
{
	int g, i, j, side, faceOnView, s, face, position;
	CRD crd;

	if (dir == TOP || dir == BOTTOM)
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAXSIDES; side++) {
				crd.column = h;
				crd.row = g;
				crd.diagonal = h + g + side;
				face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = ToPosition(w, crd);
				w->pyraminx.rowLoc[orient][side][g] =
					w->pyraminx.facetLoc[face][position];
	} else if (dir == RIGHT || dir == LEFT)
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAXSIDES; side++) {
				crd.column = g;
				crd.row = h;
				crd.diagonal = h + g + side;
				face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = ToPosition(w, crd);
				w->pyraminx.rowLoc[orient][side][g] =
					w->pyraminx.facetLoc[face][position];
	} else {		/* dir == TR || dir == BL */
		faceOnView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceOnView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (g = 0; g < w->pyraminx.size; g++) {
			for (side = 0; side < MAXSIDES; side++) {
				s = (((side == UP) && (faceOnView == UP)) ||
					((side == DOWN) && (faceOnView == DOWN)))
					? DOWN : UP;
				crd.column = i;
				crd.row = j;
				crd.diagonal = i + j + s;
				face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = ToPosition(w, crd);
				w->pyraminx.rowLoc[orient][side][g] =
					w->pyraminx.facetLoc[face][position];
				if (side == DOWN) {
					if (faceOnView == UP) {
						if (j == w->pyraminx.size - 1)
							view = !view;
						j = (j + 1) % w->pyraminx.size;
					} else {	/* faceOnView == DOWN */
						if (!j)
							view = !view;
						j = (j - 1 + w->pyraminx.size) % w->pyraminx.size;
					}
				}
			}
			i = (faceOnView == UP) ? i - 1 : i + 1;
		}
	}
}

static void
ReadCRD3(PyraminxWidget w, int view, int faceOnView, int dir, int h, int len, int orient)
{
	int g, i, j, side, s, face, position;
	CRD crd;

	if (dir == TOP || dir == BOTTOM) {
		for (g = 0; g <= len; g++)
			for (side = 0; side < MAXSIDES; side++)
				if (!side || g < len) {
					crd.column = h;
					crd.row = (faceOnView == UP) ? w->pyraminx.size - 1 - g : g;
					crd.diagonal = h + crd.row + ((((side == UP) && (faceOnView == DOWN)) ||
						((side == DOWN) && (faceOnView == UP)))
						? UP : DOWN);
					face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = ToPosition(w, crd);
					w->pyraminx.rowLoc[orient][side][g] =
						w->pyraminx.facetLoc[face][position];
				}
	} else if (dir == RIGHT || dir == LEFT) {
		for (g = 0; g <= len; g++)
			for (side = 0; side < MAXSIDES; side++)
				if (!side || g < len) {
					crd.column = (faceOnView == UP) ? w->pyraminx.size - 1 - g : g;
					crd.row = h;
					crd.diagonal = h + crd.column +
						((((side == UP) && (faceOnView == DOWN)) ||
						((side == DOWN) && (faceOnView == UP)))
						? UP : DOWN);
					face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = ToPosition(w, crd);
					w->pyraminx.rowLoc[orient][side][g] =
						w->pyraminx.facetLoc[face][position];
				}
	} else {		/* dir == TR || dir == BL */
		faceOnView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceOnView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (g = 0; g <= len; g++) {
			for (side = 0; side < MAXSIDES; side++) {
				if (side == DOWN || g < len) {
					s = (((side == UP) && (faceOnView == UP)) ||
						((side == DOWN) && (faceOnView == DOWN)))
						? DOWN : UP;
					crd.column = i;
					crd.row = j;
					crd.diagonal = i + j + s;
					face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = ToPosition(w, crd);
					w->pyraminx.rowLoc[orient][side][g] =
						w->pyraminx.facetLoc[face][position];
					if (side == DOWN)
						j = (faceOnView == UP) ? j + 1 : j - 1;
				}
			}
			i = (faceOnView == UP) ? i - 1 : i + 1;
		}
	}
}

static void
RotateCRD2(PyraminxWidget w, int rotate, int orient)
{
	int g, side;

	for (g = 0; g < w->pyraminx.size; g++)
		for (side = 0; side < MAXSIDES; side++)
			w->pyraminx.rowLoc[orient][side][g].rotation =
				(w->pyraminx.rowLoc[orient][side][g].rotation + rotate) % MAXORIENT;
}

static void
RotateCRD3(PyraminxWidget w, int faceOnView, int dir, int len, int orient)
{
	int g, side, direction, facetOrient;

	for (g = 0; g <= len; g++)
		for (side = 0; side < MAXSIDES; side++)
			if (side == DOWN || g < len) {
				direction = (faceOnView == UP) ? (dir + 3) % MAXORIENT : dir;
				facetOrient = w->pyraminx.rowLoc[orient][side][g].rotation;
				w->pyraminx.rowLoc[orient][side][g].rotation =
					(facetOrient >= 3) ?
					(rotOrientRowP3[facetOrient - 3][direction] + 3) % MAXORIENT :
					rotOrientRowP3[facetOrient][direction];
			}
}

static void
ReverseCRD2(PyraminxWidget w, int orient)
{
	PyraminxLoc temp;
	int g;

	for (g = 0; g < w->pyraminx.size; g++) {
		temp = w->pyraminx.rowLoc[orient][g % 2][g / 2];
		w->pyraminx.rowLoc[orient][g % 2][g / 2] =
			w->pyraminx.rowLoc[orient][!(g % 2)][w->pyraminx.size - 1 - g / 2];
		w->pyraminx.rowLoc[orient][!(g % 2)][w->pyraminx.size - 1 - g / 2] =
			temp;
	}
}

static void
ReverseCRD3(PyraminxWidget w, int len, int orient)
{
	PyraminxLoc temp;
	int g;

	for (g = 0; g < len; g++) {
		temp = w->pyraminx.rowLoc[orient][g % 2][len - ((g + 1) / 2)];
		w->pyraminx.rowLoc[orient][g % 2][len - ((g + 1) / 2)] =
			w->pyraminx.rowLoc[orient][g % 2][g / 2];
		w->pyraminx.rowLoc[orient][g % 2][g / 2] = temp;
	}
}

static void
WriteCRD2(PyraminxWidget w, int view, int dir, int h, int orient)
{
	int g, side, i, j, s, faceOnView, face, position;
	CRD crd;

	if (dir == TOP || dir == BOTTOM) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAXSIDES; side++) {
				crd.column = h;
				crd.row = g;
				crd.diagonal = h + g + side;
				face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = ToPosition(w, crd);
				w->pyraminx.facetLoc[face][position] =
					w->pyraminx.rowLoc[orient][side][g];
				DrawTriangle(w,
					view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN),
					ToPosition(w, crd), FALSE);
			}
	} else if (dir == RIGHT || dir == LEFT) {
		for (g = 0; g < w->pyraminx.size; g++)
			for (side = 0; side < MAXSIDES; side++) {
				crd.column = g;
				crd.row = h;
				crd.diagonal = h + g + side;
				face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = ToPosition(w, crd);
				w->pyraminx.facetLoc[face][position] =
					w->pyraminx.rowLoc[orient][side][g];
				DrawTriangle(w, face, position, FALSE);
			}
	} else {		/* dir == TR || dir == BL */
		faceOnView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceOnView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (g = 0; g < w->pyraminx.size; g++) {
			for (side = 0; side < MAXSIDES; side++) {
				s = (((side == UP) && (faceOnView == UP)) ||
					((side == DOWN) && (faceOnView == DOWN)))
					? DOWN : UP;
				crd.column = i;
				crd.row = j;
				crd.diagonal = i + j + s;
				face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
				position = ToPosition(w, crd);
				w->pyraminx.facetLoc[face][position] =
					w->pyraminx.rowLoc[orient][side][g];
				DrawTriangle(w, face, position, FALSE);
				if (side == DOWN) {
					if (faceOnView == UP) {
						if (j == w->pyraminx.size - 1) {
							view = !view;
							j = 0;
						} else
							++j;
					} else {	/* FACE == DOWN */
						if (!j) {
							view = !view;
							j = w->pyraminx.size - 1;
						} else
							--j;
					}
				}
			}
			if (faceOnView == UP)
				--i;
			else	/* faceOnView == DOWN */
				++i;
		}
	}
}

static void
WriteCRD3(PyraminxWidget w, int view, int faceOnView, int dir, int h, int len, int orient)
{
	int g, side, i, j, k, s, face, position;
	CRD crd;

	if (dir == TOP || dir == BOTTOM) {
		for (k = 0; k <= len; k++)
			for (side = 0; side < MAXSIDES; side++)
				if ((side == DOWN) || k < len) {
					g = (faceOnView == UP) ? w->pyraminx.size - 1 - k : k;
					s = (((side == UP) && (faceOnView == DOWN)) ||
						((side == DOWN) && (faceOnView == UP)))
						? UP : DOWN;
					crd.column = h;
					crd.row = g;
					crd.diagonal = h + g + s;
					face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = ToPosition(w, crd);
					w->pyraminx.facetLoc[face][position] =
						w->pyraminx.rowLoc[orient][side][k];
					DrawTriangle(w, face, position, FALSE);
				}
	} else if (dir == RIGHT || dir == LEFT) {
		for (k = 0; k <= len; k++)
			for (side = 0; side < MAXSIDES; side++)
				if ((side == DOWN) || k < len) {
					g = (faceOnView == UP) ? w->pyraminx.size - 1 - k : k;
					s = (((side == UP) && (faceOnView == DOWN)) ||
						((side == DOWN) && (faceOnView == UP)))
						? UP : DOWN;
					crd.column = g;
					crd.row = h;
					crd.diagonal = h + g + s;
					face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = ToPosition(w, crd);
					w->pyraminx.facetLoc[face][position] =
						w->pyraminx.rowLoc[orient][side][k];
					DrawTriangle(w, face, position, FALSE);
				}
	} else {		/* dir == TR || dir == BL */
		faceOnView = (h < w->pyraminx.size) ? DOWN : UP;
		i = (faceOnView == UP) ? w->pyraminx.size - 1 : 0;
		j = h % w->pyraminx.size;
		for (k = 0; k <= len; k++) {
			for (side = 0; side < MAXSIDES; side++)
				if ((side == DOWN) || k < len) {
					s = (((side == UP) && (faceOnView == UP)) ||
						((side == DOWN) && (faceOnView == DOWN)))
						? DOWN : UP;
					crd.column = i;
					crd.row = j;
					crd.diagonal = i + j + s;
					face = view * MAXSIDES + ((crd.diagonal >= w->pyraminx.size) ? UP : DOWN);
					position = ToPosition(w, crd);
					w->pyraminx.facetLoc[face][position] =
						w->pyraminx.rowLoc[orient][side][k];
					DrawTriangle(w, face, position, FALSE);
					if (side == DOWN) {
						if (faceOnView == UP) {
							if (j == w->pyraminx.size - 1) {
								view = !view;
								j = 0;
							} else
								++j;
						} else {	/* FACE == DOWN */
							if (!j) {
								view = !view;
								j = w->pyraminx.size - 1;
							} else
								--j;
						}
					}
				}
			if (faceOnView == UP)
				--i;
			else	/* faceOnView == DOWN */
				++i;
		}
	}
}

static void
MovePieces(PyraminxWidget w, int face, CRD crd, int direction, int style)
{
	int view, side;
	int newSide, newView, rotate, reverse, h, k, newH, faceOnView, len;
	int newFace, newDirection, bound, l = 0;

	view = face / MAXSIDES;
	side = crd.diagonal - crd.column - crd.row;
	if (style == PERIOD2) {
		/* Period 2 Slide rows */
		h = Crd(direction, crd.column, crd.row, side);
		if (w->pyraminx.sticky && (h % 4 == 1 || h % 4 == 2)) {
			bound = TRUE;
			l = 0;
			if (h % 4 == 2)
				h = h - 1;
		} else
			bound = FALSE;
		do {
			ReadCRD2(w, view, direction, h, 0);
			for (k = 1; k <= 2; k++) {
				rotate = slideNextRowP2[side][direction % 3].rotation;
				if (direction == TR || direction == BL) {
					newView = view;
					newSide = !side;
					newH = 2 * w->pyraminx.size - 1 - h;
					reverse = FALSE;
				} else if (!rotate) {
					newView = !view;
					newSide = side;
					newH = h;
					reverse = FALSE;
				} else {	/* rotate == 3 */
					newView = !view;
					newSide = !side;
					newH = w->pyraminx.size - 1 - h;
					reverse = TRUE;
				}
				if (k != 2)
					ReadCRD2(w, newView, direction, newH, k);
				RotateCRD2(w, rotate, k - 1);
				if (reverse == TRUE)
					ReverseCRD2(w, k - 1);
				WriteCRD2(w, newView, direction, newH, k - 1);
				view = newView;
				h = newH;
				side = newSide;
			}
			l++;
			h++;
		} while (bound && l < 2);
	} else {		/* style == PERIOD3 */
		faceOnView = (crd.diagonal >= w->pyraminx.size);
		h = Crd(direction, crd.column, crd.row, side);
		bound = FALSE;
		if (direction == TR || direction == BL) {
			if (h == w->pyraminx.size)
				RotateFace(w, view, DOWN, (direction == TR) ? CCW : CW);
			else if (h == w->pyraminx.size - 1)
				RotateFace(w, view, UP, (direction == TR) ? CW : CCW);
			else if (w->pyraminx.sticky)
				bound = TRUE;
		} else if (!crd.column && faceOnView == DOWN &&
				(direction == TOP || direction == BOTTOM))
			RotateFace(w, !view, DOWN,
			 (direction == TOP || direction == LEFT) ? CCW : CW);
		else if (crd.column == w->pyraminx.size - 1 && faceOnView &&
			 (direction == TOP || direction == BOTTOM))
			RotateFace(w, !view, UP,
				(direction == BOTTOM || direction == RIGHT) ? CCW : CW);
		else if (!crd.row && faceOnView == DOWN &&
			 (direction == RIGHT || direction == LEFT))
			RotateFace(w, !view, UP,
				(direction == BOTTOM || direction == RIGHT) ? CCW : CW);
		else if (crd.row == w->pyraminx.size - 1 && faceOnView == UP &&
			 (direction == RIGHT || direction == LEFT))
			RotateFace(w, !view, DOWN,
			 (direction == TOP || direction == LEFT) ? CCW : CW);
		else if (w->pyraminx.sticky)
			bound = TRUE;
		/* Slide rows */
		if (bound == TRUE) {
			l = 0;
			if (direction == TR || direction == BL)
				h = (faceOnView == UP) ? w->pyraminx.size + 1 : 0;
			else
				h = (faceOnView == UP) ? 0 : 1;
		}
		do {
			len = Length(w, faceOnView, direction, h);
			ReadCRD3(w, view, faceOnView, direction, h, len, 0);
			for (k = 1; k <= 3; k++) {
				newView = (slideNextRowP3[faceOnView][direction].viewChanged)
					? !view : view;
				newFace = slideNextRowP3[faceOnView][direction].face;
				newDirection = slideNextRowP3[faceOnView][direction].direction;
				reverse = slideNextRowP3[faceOnView][direction].reverse;
				newH = w->pyraminx.size - 1 - h;
				if (faceOnView == UP) {
					if (direction == TR || direction == RIGHT)
						newH = 2 * w->pyraminx.size - 1 - h;
					else if (direction == BOTTOM)
						newH = h;
					else if (direction == BL)
						newH = h - w->pyraminx.size;
				} else {
					if (direction == TOP)
						newH = w->pyraminx.size + h;
					else if (direction == TR)
						newH = h;
				}
				if (k != 3)
					ReadCRD3(w, newView, newFace, newDirection, newH, len, k);
				RotateCRD3(w, faceOnView, direction, len, k - 1);
				if (reverse == TRUE)
					ReverseCRD3(w, len, k - 1);
				WriteCRD3(w, newView, newFace, newDirection, newH, len, k - 1);
				view = newView;
				faceOnView = newFace;
				direction = newDirection;
				h = newH;
			}
			h++;
			l++;
		} while (bound && l < w->pyraminx.size - 1);
	}
}

static void
MoveControlCb(PyraminxWidget w, int face, int direction, int style)
{
	int i, faceOnView;
	CRD crd;

	faceOnView = face % MAXSIDES;
	if (w->pyraminx.sticky) {
		if (style == PERIOD2)
			for (i = 0; i < 3; i++) {
				if (direction == TR || direction == BL) {
					crd.column = 0;
					crd.row = 3 * i / 2;
					crd.diagonal = crd.row + i % 2;
					MovePieces(w, face, crd, direction, style);
				} else {
					crd.column = 3 * i / 2;
					crd.row = 3 * i / 2;
					crd.diagonal = crd.row + crd.column;
					MovePieces(w, face, crd, direction, style);
				}
				SetPyraminx(w, PYRAMINX_CONTROL);
		} else		/* (style == PERIOD3) */
			for (i = 0; i < 2; i++) {
				if (direction == TR || direction == BL) {
					crd.column = 1 + faceOnView;
					crd.row = 1 + faceOnView;
					crd.diagonal = crd.row + crd.column + i;
					MovePieces(w, face, crd, direction, style);
				} else {
					crd.column = i + 2 * faceOnView;
					crd.row = i + 2 * faceOnView;
					crd.diagonal = crd.row + crd.column + faceOnView;
					MovePieces(w, face, crd, direction, style);
				}
				SetPyraminx(w, PYRAMINX_CONTROL);
			}
	} else {
		for (i = 0; i < w->pyraminx.size; i++) {
			if (direction == TR || direction == BL) {
				if (style == PERIOD2) {
					crd.column = 0;
					crd.row = i;
					crd.diagonal = crd.row;
					MovePieces(w, face, crd, direction, style);
				} else {
					crd.column = faceOnView * (w->pyraminx.size - 1);
					crd.row = i;
					crd.diagonal = crd.column + crd.row + faceOnView;
					MovePieces(w, face, crd, direction, style);
				}
			} else {
				crd.column = i;
				crd.row = w->pyraminx.size - 1 - i;
				crd.diagonal = crd.column + crd.row + faceOnView;
				MovePieces(w, face, crd, direction, style);
			}
			SetPyraminx(w, PYRAMINX_CONTROL);
		}
	}
}

void
MovePyraminx(PyraminxWidget w, int face, int position, int direction,
		int style, int control)
{
	if (control)
		MoveControlCb(w, face, direction, style);
	else {
		CRD crd;

		ToCRD(w, face, position, &crd);
		MovePieces(w, face, crd, direction, style);
		SetPyraminx(w, PYRAMINX_MOVED);
	}
	PutMove(face, position, direction, style, control);
}

void
MovePyraminxDelay(PyraminxWidget w, int face, int position, int direction,
		int style, int control)
{
	MovePyraminx(w, face, position, direction, style, control);
	Sleep((unsigned int) w->pyraminx.delay);
}

static Boolean
SelectPieces(PyraminxWidget w, int x, int y, int *face, CRD * crd)
{
	int offset, modI, modJ, side, view;

	x -= w->pyraminx.puzzleOffset.x;
	y -= w->pyraminx.puzzleOffset.y;
	if (w->pyraminx.vertical && y > w->pyraminx.viewLength - 1) {
		y -= (w->pyraminx.viewLength - 1);
		view = DOWN;
	} else if (!w->pyraminx.vertical && x > w->pyraminx.viewLength - 1) {
		x -= (w->pyraminx.viewLength - 1);
		view = DOWN;
	} else
		view = UP;
	if (x <= 0 || y <= 0 ||
			x >= w->pyraminx.faceLength + w->pyraminx.delta ||
			y >= w->pyraminx.faceLength + w->pyraminx.delta)
		return False;
	else if (x + y > w->pyraminx.faceLength)
		offset = 2 * w->pyraminx.delta + 1;
	else
		offset = w->pyraminx.delta;
	crd->column = (x - offset) / (w->pyraminx.facetLength + w->pyraminx.delta);
	crd->row = (y - offset) / (w->pyraminx.facetLength + w->pyraminx.delta);
	modI = (x - offset) % (w->pyraminx.facetLength + w->pyraminx.delta);
	modJ = (y - offset) % (w->pyraminx.facetLength + w->pyraminx.delta);
	side = (modI + modJ > w->pyraminx.facetLength + 1) ? UP : DOWN;
	if (!w->pyraminx.vertical && view == DOWN) {
		crd->row = w->pyraminx.size - crd->row - 1;
		crd->column = w->pyraminx.size - crd->column - 1;
		side = !side;
	}
	crd->diagonal = crd->row + crd->column + side;
	*face = ((view == UP) ? DOWN : UP) * MAXSIDES +
		((crd->diagonal >= w->pyraminx.size) ? UP : DOWN);
	return True;
}

static int
CheckMoveDir(PyraminxWidget w, CRD crd1, CRD crd2, int face, int *direction)
{
	int which = -1, count = 0;
	int i, *p1, *p2;

	p1 = &(crd1.column);
	p2 = &(crd2.column);
	for (i = 0; i < 3; i++, p1++, p2++)
		if (*p1 == *p2) {
			which = i;
			count++;
		}
	if (count == 1)
		switch (which) {
			case 0:	/* COLUMN */
				*direction = (crd2.row > crd1.row) ? BOTTOM : TOP;
				break;
			case 1:	/* ROW */
				*direction = (crd2.column > crd1.column) ? RIGHT : LEFT;
				break;
			case 2:	/* DIAGONAL */
				*direction = (crd2.column > crd1.column) ? TR : BL;
				break;
		}
	if (!w->pyraminx.vertical && face >= MAXSIDES && *direction > LEFT)
		*direction = (*direction + MAXSIDES) % MAXFACES;
	return count;
}

static void
NarrowSelection(PyraminxWidget w, int style, int *face, CRD * crd, int *direction)
{
	if (!w->pyraminx.vertical && *face >= MAXSIDES && *direction < MAXORIENT)
		*direction = (*direction + MAXORIENT / 2) % MAXORIENT;
	if (style == PERIOD2) {
		if (*direction == CW)
			*direction = TR;
		else if (*direction == CCW)
			*direction = BL;
	} else {		/* style == PERIOD3 */
		if (*direction == CW || *direction == CCW) {
			crd->diagonal = w->pyraminx.size -
				((crd->diagonal < w->pyraminx.size) ? UP : DOWN);
			*direction = ((*direction == CW && crd->diagonal == w->pyraminx.size) ||
				(*direction == CCW && crd->diagonal != w->pyraminx.size))
				? TR : BL;
			*face = !(*face % 2) + 2 * (*face / 2);
			crd->row = w->pyraminx.size - 1;
			crd->column = 0;
		}
	}
}

static Boolean
PositionPieces(PyraminxWidget w, int x, int y, int style, int *face, CRD * crd, int *direction)
{
	if (!SelectPieces(w, x, y, face, crd))
		return False;
	NarrowSelection(w, style, face, crd, direction);
	return True;
}

#ifndef WINVER
static
#endif
void
MovePyraminxInput(PyraminxWidget w, int x, int y, int direction, int shift,
		int control)
{
	int style, face;
	CRD crd;

	if (w->pyraminx.mode != BOTH) {
		if (control && shift)
			style = (w->pyraminx.mode == PERIOD3) ? PERIOD2 : PERIOD3;
		else
			style = (w->pyraminx.mode == PERIOD2) ? PERIOD2 : PERIOD3;
	} else
		style = (shift) ? PERIOD3 : PERIOD2;
	if (!w->pyraminx.practice && !control && CheckSolved(w)) {
		MoveNoPieces(w);
		return;
	}
	if (!PositionPieces(w, x, y, style, &face, &crd, &direction))
		return;
	control = (control) ? 1 : 0;
	MovePyraminx(w, face, ToPosition(w, crd), direction, style, control);
	if (!control && CheckSolved(w)) {
		SetPyraminx(w, PYRAMINX_SOLVED);
	}
}

static void
ResetPieces(PyraminxWidget w)
{
	int face, position, orient, side;

	w->pyraminx.sizeSize = w->pyraminx.size * w->pyraminx.size;
	for (face = 0; face < MAXFACES; face++) {
		if (w->pyraminx.facetLoc[face])
			free(w->pyraminx.facetLoc[face]);
		if (!(w->pyraminx.facetLoc[face] = (PyraminxLoc *)
				malloc(sizeof (PyraminxLoc) * w->pyraminx.sizeSize))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		if (startLoc[face])
			free(startLoc[face]);
		if (!(startLoc[face] = (PyraminxLoc *)
				malloc(sizeof (PyraminxLoc) * w->pyraminx.sizeSize))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (orient = 0; orient < 3; orient++)
		for (side = 0; side < MAXSIDES; side++) {
			if (w->pyraminx.rowLoc[orient][side])
				free(w->pyraminx.rowLoc[orient][side]);
			if (!(w->pyraminx.rowLoc[orient][side] = (PyraminxLoc *)
					malloc(sizeof (PyraminxLoc) * w->pyraminx.size))) {
				DISPLAY_ERROR("Not enough memory, exiting.");
			}
		}
	for (side = 0; side < MAXSIDES; side++) {
		if (w->pyraminx.faceLoc)
			free(w->pyraminx.faceLoc[side]);
		if (!(w->pyraminx.faceLoc[side] = (PyraminxLoc *)
				malloc(sizeof (PyraminxLoc) * w->pyraminx.sizeSize))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < w->pyraminx.sizeSize; position++) {
			w->pyraminx.facetLoc[face][position].face = face;
			w->pyraminx.facetLoc[face][position].rotation = TOP;
		}
	FlushMoves(w);
	w->pyraminx.currentFace = -1;
	w->pyraminx.started = False;
}

static void
ResizePieces(PyraminxWidget w)
{
	int i, j;

	w->pyraminx.facetLength = w->pyraminx.faceLength / w->pyraminx.size -
		w->pyraminx.delta - 1;
	for (i = 0; i <= 3; i++)
		for (j = 0; j < MAXSIDES; j++) {
			triangleList[j][i].x = triangleUnit[j][i].x *
				w->pyraminx.facetLength;
			triangleList[j][i].y = triangleUnit[j][i].y *
				w->pyraminx.facetLength;
		}
	offsetList[DOWN].x = 0;
	offsetList[UP].x = w->pyraminx.facetLength + 2;
	offsetList[DOWN].y = 0;
	offsetList[UP].y = w->pyraminx.facetLength + 2;
	letterList[DOWN].x = w->pyraminx.facetLength / 4 -
		w->pyraminx.letterOffset.x;
	letterList[UP].x = 3 * w->pyraminx.facetLength / 4 + 3 -
		w->pyraminx.letterOffset.x;
	letterList[DOWN].y = w->pyraminx.facetLength / 4 + 1 +
		w->pyraminx.letterOffset.y;
	letterList[UP].y = 3 * w->pyraminx.facetLength / 4 + 1 +
		w->pyraminx.letterOffset.y;
	w->pyraminx.sideOffset = 3 * w->pyraminx.size / 4;
	w->pyraminx.orientLineLength = w->pyraminx.facetLength / 4;
	w->pyraminx.orientDiagLength = MAX(w->pyraminx.orientLineLength - 3, 0);
}

static void
GetPieces(PyraminxWidget w)
{
	FILE *fp;
	int c, i, size, mode, sticky, orient, practice, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &size);
	if (size >= MINFACETS) {
		for (i = w->pyraminx.size; i < size; i++) {
			SetPyraminx(w, PYRAMINX_INC);
		}
		for (i = w->pyraminx.size; i > size; i--) {
			SetPyraminx(w, PYRAMINX_DEC);
		}
	} else {
		stringCat(&buf1, name, " corrupted: size ");
		intCat(&buf2, buf1, size);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINFACETS);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &mode);
	switch (mode) {
		case PERIOD2:
			SetPyraminx(w, PYRAMINX_PERIOD2);
			break;
		case PERIOD3:
			SetPyraminx(w, PYRAMINX_PERIOD3);
			break;
		case BOTH:
			SetPyraminx(w, PYRAMINX_BOTH);
			break;
		default:
			stringCat(&buf1, name, " corrupted: mode ");
			intCat(&buf2, buf1, mode);
			free(buf1);
			stringCat(&buf1, buf2, " should be between ");
			free(buf2);
			intCat(&buf2, buf1, PERIOD2);
			free(buf1);
			stringCat(&buf1, buf2, " and ");
			free(buf2);
			intCat(&buf2, buf1, BOTH);
			free(buf1);
			DISPLAY_WARNING(buf2);
			free(buf2);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sticky);
	if (w->pyraminx.sticky != (Boolean) sticky) {
		SetPyraminx(w, PYRAMINX_STICKY);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &orient);
	if (w->pyraminx.orient != (Boolean) orient) {
		SetPyraminx(w, PYRAMINX_ORIENT);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &practice);
	if (w->pyraminx.practice != (Boolean) practice) {
		SetPyraminx(w, PYRAMINX_PRACTICE);
	}
#ifdef WINVER
	ResetPieces(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetPyraminx(w, PYRAMINX_RESTORE);
	SetStartPosition(w);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: size %d, mode %d, sticky %d, orient %d",
		name, size, mode, sticky, orient);
	(void) printf(", practice %d, moves %d.\n", practice, moves);
	free(lname);
	free(fname);
	w->pyraminx.cheat = True; /* Assume the worst. */
}

static void
WritePieces(PyraminxWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "size%c %d\n", SYMBOL, w->pyraminx.size);
	(void) fprintf(fp, "mode%c %d\n", SYMBOL, w->pyraminx.mode);
	(void) fprintf(fp, "sticky%c %d\n", SYMBOL,
		(w->pyraminx.sticky) ? 1 : 0);
	(void) fprintf(fp, "orient%c %d\n", SYMBOL,
		(w->pyraminx.orient) ? 1 : 0);
	(void) fprintf(fp, "practice%c %d\n", SYMBOL,
		(w->pyraminx.practice) ? 1 : 0);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearPieces(PyraminxWidget w)
{
	if (w->pyraminx.currentFace >= 0)
		return;
	ResetPieces(w);
	DrawAllPieces(w);
	SetPyraminx(w, PYRAMINX_RESET);
}

static void
UndoPieces(PyraminxWidget w)
{
	if (MadeMoves() && w->pyraminx.currentFace < 0) {
		int face, position, direction, style, control;

		GetMove(&face, &position, &direction, &style, &control);
		direction = (direction < MAXORIENT) ? (direction + MAXORIENT / 2) %
			MAXORIENT : 3 * MAXORIENT - direction;
		if (control)
			MoveControlCb(w, face, direction, style);
		else {
			CRD crd;

			ToCRD(w, face, position, &crd);
			MovePieces(w, face, crd, direction, style);
			SetPyraminx(w, PYRAMINX_UNDO);
			if (CheckSolved(w)) {
				SetPyraminx(w, PYRAMINX_SOLVED);
			}
		}
	}
}

static void
PracticePieces(PyraminxWidget w)
{
	SetPyraminx(w, PYRAMINX_PRACTICE);
}

static void
RandomizePieces(PyraminxWidget w)
{
	int randomDirection, face, position, style;
	int big = w->pyraminx.sizeSize * 3 + NRAND(2);

	if (w->pyraminx.currentFace >= 0)
		return;
	w->pyraminx.cheat = False;
	if (big > 1000)
		big = 1000;
	if (w->pyraminx.practice)
		PracticePieces(w);
	if (w->pyraminx.sticky)
		big /= 3;
	SetPyraminx(w, PYRAMINX_RESET);

#ifdef DEBUG
	big = 3;
#endif

	while (big--) {
		face = NRAND(MAXFACES);
		if (w->pyraminx.mode == BOTH)
			style = NRAND(MAXMODES - 1) + PERIOD2;
		else
			style = w->pyraminx.mode;
		if (w->pyraminx.sticky) {
			if (style == PERIOD2) {
				if (NRAND(3) == 2) {
					position = (NRAND(2)) ? 9 : 6;
					randomDirection = (NRAND(2)) ? TR : BL;
				} else {
					position = (NRAND(2)) ? 6 : 0;
					if (NRAND(2))
						randomDirection = (NRAND(2)) ? LEFT : RIGHT;
					else
						randomDirection = (NRAND(2)) ? TOP : BOTTOM;
				}
			} else {	/* style == PERIOD3 */
				position = 6;
				randomDirection = NRAND(6);
			}
		} else {	/* (!w->pyraminx.sticky) */
			randomDirection = NRAND(MAXORIENT);
			position = NRAND(w->pyraminx.sizeSize);
			if (w->pyraminx.mode == BOTH)
				style = NRAND(BOTH);
			else
				style = w->pyraminx.mode;
		}
		MovePyraminx(w, face, position, randomDirection, style, FALSE);
		SetPyraminx(w, PYRAMINX_MOVED);
	}
	FlushMoves(w);
	SetPyraminx(w, PYRAMINX_RANDOMIZE);
	if (CheckSolved(w)) {
		SetPyraminx(w, PYRAMINX_SOLVED);
	}
}

static void
SolvePieces(PyraminxWidget w)
{
	if (CheckSolved(w) && w->pyraminx.currentFace >= 0)
		return;
	if ((w->pyraminx.size == 2 || w->pyraminx.size == 3) &&
			w->pyraminx.mode == PERIOD3)
		SolveSomePieces(w);
	else {
		SetPyraminx(w, PYRAMINX_SOLVE_MESSAGE);
	}
}

static void
IncrementPieces(PyraminxWidget w)
{
	SetPyraminx(w, PYRAMINX_INC);
}

static Boolean
DecrementPieces(PyraminxWidget w)
{
	if (w->pyraminx.size <= MINFACETS)
		return False;
	SetPyraminx(w, PYRAMINX_DEC);
	return True;
}

static void
OrientizePieces(PyraminxWidget w)
{
	SetPyraminx(w, PYRAMINX_ORIENT);
}

static void
StickyPieces(PyraminxWidget w)
{
	SetPyraminx(w, PYRAMINX_STICKY);
}

#ifdef WINVER
static void
SetValuesPyraminx(PyraminxWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80], buf[20], charbuf[2];
	int face;

	w->pyraminx.size = GetPrivateProfileInt(SECTION, "size",
		DEFAULTFACETS, INIFILE);
	w->pyraminx.mode = GetPrivateProfileInt(SECTION, "mode",
		DEFAULTMODE, INIFILE);
	w->pyraminx.sticky = (BOOL) GetPrivateProfileInt(SECTION, "sticky",
		DEFAULTSTICKY, INIFILE);
	w->pyraminx.orient = (BOOL) GetPrivateProfileInt(SECTION, "orient",
		DEFAULTORIENT, INIFILE);
	w->pyraminx.practice = (BOOL) GetPrivateProfileInt(SECTION, "practice",
		DEFAULTPRACTICE, INIFILE);
	w->pyraminx.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->pyraminx.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->pyraminx.frameGC = RGB(color.red, color.green, color.blue);
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "pieceBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->pyraminx.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->pyraminx.inverseGC = RGB(color.red, color.green, color.blue);
	for (face = 0; face < MAXFACES; face++) {
		(void) sprintf(buf, "faceColor%d", face);
		(void) GetPrivateProfileString(SECTION, buf,
			faceColorString[face],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->pyraminx.faceGC[face] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "faceChar%d", face);
		charbuf[0] = faceColorChar[face];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->pyraminx.faceChar[face] = szBuf[0];
	}
	(void) GetPrivateProfileString(SECTION, "userName", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->pyraminx.userName, szBuf);
		w->pyraminx.userName[80] = 0;
	(void) GetPrivateProfileString(SECTION, "scoreFile", "",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->pyraminx.scoreFile, szBuf);
		w->pyraminx.scoreFile[80] = 0;
}

void
DestroyPyraminx(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else
static void
GetColor(PyraminxWidget w, int face)
{
	XGCValues values;
	XtGCMask valueMask;
	XColor colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->pyraminx.reverse) {
		values.background = w->pyraminx.foreground;
	} else {
		values.background = w->pyraminx.background;
	}
	if (!w->pyraminx.mono) {
		if (XAllocNamedColor(XtDisplay(w),
				DefaultColormapOfScreen(XtScreen(w)),
				w->pyraminx.faceName[face], &colorCell, &rgb)) {
			values.foreground = w->pyraminx.faceColor[face] = colorCell.pixel;
			if (w->pyraminx.faceGC[face])
				XtReleaseGC((Widget) w,
					w->pyraminx.faceGC[face]);
			w->pyraminx.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->pyraminx.faceName[face]);
			stringCat(&buf2, buf1, "\" is not defined for face ");
			free(buf1);
			intCat(&buf1, buf2, face);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->pyraminx.reverse) {
		values.background = w->pyraminx.foreground;
		values.foreground = w->pyraminx.background;
	} else {
		values.background = w->pyraminx.background;
		values.foreground = w->pyraminx.foreground;
	}
	if (w->pyraminx.faceGC[face])
		XtReleaseGC((Widget) w, w->pyraminx.faceGC[face]);
	w->pyraminx.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
}

static void
SetAllColors(PyraminxWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int face;

	valueMask = GCForeground | GCBackground;

	if (w->pyraminx.reverse) {
		values.background = w->pyraminx.background;
		values.foreground = w->pyraminx.foreground;
	} else {
		values.foreground = w->pyraminx.background;
		values.background = w->pyraminx.foreground;
	}
	if (w->pyraminx.inverseGC)
		XtReleaseGC((Widget) w, w->pyraminx.inverseGC);
	w->pyraminx.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->pyraminx.mono) {
		if (w->pyraminx.reverse) {
			values.background = w->pyraminx.foreground;
			values.foreground = w->pyraminx.background;
		} else {
			values.foreground = w->pyraminx.foreground;
			values.background = w->pyraminx.background;
		}
	} else {
		values.foreground = w->pyraminx.frameColor;
		values.background = w->pyraminx.background;
	}
	if (w->pyraminx.frameGC)
		XtReleaseGC((Widget) w, w->pyraminx.frameGC);
	w->pyraminx.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->pyraminx.mono) {
		if (w->pyraminx.reverse) {
			values.background = w->pyraminx.foreground;
			values.foreground = w->pyraminx.background;
		} else {
			values.foreground = w->pyraminx.foreground;
			values.background = w->pyraminx.background;
		}
	} else {
		values.foreground = w->pyraminx.borderColor;
		values.background = w->pyraminx.background;
	}
	if (w->pyraminx.borderGC)
		XtReleaseGC((Widget) w, w->pyraminx.borderGC);
	w->pyraminx.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (face = 0; face < MAXFACES; face++)
		GetColor(w, face);
	if (w->pyraminx.fontInfo)
		XSetFont(XtDisplay(w), w->pyraminx.borderGC,
			w->pyraminx.fontInfo->fid);
}

static Boolean
SetValuesPyraminx(Widget current, Widget request, Widget renew)
{
	PyraminxWidget c = (PyraminxWidget) current, w = (PyraminxWidget) renew;
	Boolean redraw = False, setColors = False;
	int face;

	CheckPieces(w);
	for (face = 0; face < MAXFACES; face++) {
		if (strcmp(w->pyraminx.faceName[face], c->pyraminx.faceName[face])) {
			setColors = True;
			break;
		}
	}
	if (w->pyraminx.font != c->pyraminx.font ||
			w->pyraminx.borderColor != c->pyraminx.borderColor ||
			w->pyraminx.reverse != c->pyraminx.reverse ||
			w->pyraminx.mono != c->pyraminx.mono) {
		loadFont(w);
		SetAllColors(w);
		redraw = True;
	} else if (w->pyraminx.background != c->pyraminx.background ||
			w->pyraminx.foreground != c->pyraminx.foreground ||
			setColors) {
		SetAllColors(w);
		redraw = True;
	}
	if (w->pyraminx.orient != c->pyraminx.orient) {
		ResetPieces(w);
		redraw = True;
	} else if (w->pyraminx.practice != c->pyraminx.practice) {
		ResetPieces(w);
		redraw = True;
	}
	if (w->pyraminx.size != c->pyraminx.size ||
			w->pyraminx.mode != c->pyraminx.mode ||
			w->pyraminx.sticky != c->pyraminx.sticky) {
		SizePyraminx(w);
		redraw = True;
	}
	if (w->pyraminx.facetLength != c->pyraminx.facetLength) {
		ResizePyraminx(w);
		redraw = True;
	}
	if (w->pyraminx.menu != -1) {
		switch (w->pyraminx.menu) {
		case 0:
			w->pyraminx.menu = -1;
			GetPieces(w);
			break;
		case 1:
			w->pyraminx.menu = -1;
			WritePieces(w);
			break;
		case 3:
			w->pyraminx.menu = -1;
			ClearPieces(w);
			break;
		case 4:
			w->pyraminx.menu = -1;
			UndoPieces(w);
			break;
		case 5:
			w->pyraminx.menu = -1;
			RandomizePieces(w);
			break;
		case 6:
			w->pyraminx.menu = -1;
			SolvePieces(w);
			break;
		case 7:
			w->pyraminx.menu = -1;
			IncrementPieces(w);
			break;
		case 8:
			w->pyraminx.menu = -1;
			(void) DecrementPieces(w);
			break;
		case 9:
			w->pyraminx.menu = -1;
			OrientizePieces(w);
			break;
		case 10:
			w->pyraminx.menu = -1;
			StickyPieces(w);
			break;
		case 11:
			w->pyraminx.menu = -1;
			PracticePieces(w);
			break;
		default:
			w->pyraminx.menu = -1;
			break;
		}
	}
	return (redraw);
}

static void
QuitPyraminx(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	Display *display = XtDisplay(w);

	if (w->pyraminx.fontInfo) {
		XUnloadFont(display, w->pyraminx.fontInfo->fid);
		XFreeFont(display, w->pyraminx.fontInfo);
	}
	XtCloseDisplay(display);
	exit(0);
}

static void
DestroyPyraminx(Widget old)
{
	PyraminxWidget w = (PyraminxWidget) old;
	int face;

	for (face = 0; face < MAXFACES; face++)
		XtReleaseGC(old, w->pyraminx.faceGC[face]);
	XtReleaseGC(old, w->pyraminx.borderGC);
	XtReleaseGC(old, w->pyraminx.frameGC);
	XtReleaseGC(old, w->pyraminx.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->pyraminx.select);
}
#endif

#ifndef WINVER
static
#endif
void
ResizePyraminx(PyraminxWidget w)
{
	int tempLength;
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	w->pyraminx.delta = 4;
	w->pyraminx.vertical = (w->core.height >= w->core.width);
	if (w->pyraminx.vertical)
		tempLength = MIN(w->core.height / 2, w->core.width);
	else
		tempLength = MIN(w->core.height, w->core.width / 2);
	w->pyraminx.facetLength = MAX((tempLength - w->pyraminx.delta + 1) /
		w->pyraminx.size, 0);
	w->pyraminx.faceLength = w->pyraminx.size * w->pyraminx.facetLength;
	w->pyraminx.viewLength = w->pyraminx.faceLength + w->pyraminx.delta + 3;
	if (w->pyraminx.vertical) {
		w->pyraminx.puzzleSize.x = w->pyraminx.viewLength - 1;
		w->pyraminx.puzzleSize.y = 2 * w->pyraminx.viewLength -
			w->pyraminx.delta - 2;
	} else {
		w->pyraminx.puzzleSize.x = 2 * w->pyraminx.viewLength -
			w->pyraminx.delta - 2;
		w->pyraminx.puzzleSize.y = w->pyraminx.viewLength - 1;
	}
	w->pyraminx.puzzleOffset.x = ((int) w->core.width -
		w->pyraminx.puzzleSize.x) / 2;
	w->pyraminx.puzzleOffset.y = ((int) w->core.height -
		w->pyraminx.puzzleSize.y) / 2;
	ResizePieces(w);
}

#ifndef WINVER
static
#endif
void
SizePyraminx(PyraminxWidget w)
{
	ResetPieces(w);
	ResizePyraminx(w);
}

#ifndef WINVER
static
#endif
void
InitializePyraminx(
#ifdef WINVER
PyraminxWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int face, orient, side;
#ifdef WINVER
	SetValuesPyraminx(w);
#else
	PyraminxWidget w = (PyraminxWidget) renew;

	w->pyraminx.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->pyraminx.mono);
	w->pyraminx.fontInfo = NULL;
	for (face = 0; face < MAXFACES; face++)
		w->pyraminx.faceGC[face] = NULL;
	w->pyraminx.borderGC = NULL;
	w->pyraminx.frameGC = NULL;
	w->pyraminx.inverseGC = NULL;
#endif
	w->pyraminx.focus = False;
	loadFont(w);
	for (face = 0; face < MAXFACES; face++)
		w->pyraminx.facetLoc[face] = NULL;
	for (orient = 0; orient < 3; orient++)
		for (side = 0; side < MAXSIDES; side++)
			w->pyraminx.rowLoc[orient][side] = NULL;
	for (side = 0; side < MAXSIDES; side++)
		w->pyraminx.faceLoc[side] = NULL;
	CheckPieces(w);
	InitMoves();
	w->pyraminx.cheat = False;
	SizePyraminx(w);
#ifdef WINVER
	brush = CreateSolidBrush(w->pyraminx.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
#else
	(void) SRAND(getpid());
	SetAllColors(w);
#endif
}

#ifndef WINVER
static
#endif
void
ExposePyraminx(
#ifdef WINVER
PyraminxWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	PyraminxWidget w = (PyraminxWidget) renew;

	if (!w->core.visible)
		return;
#endif
	EraseFrame(w);
	DrawFrame(w, w->pyraminx.focus);
	DrawAllPieces(w);
}

#ifndef WINVER
static
#endif
void
HidePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetPyraminx(w, PYRAMINX_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectPyraminx(PyraminxWidget w
#ifdef WINVER
, const int x, const int y, const int control
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	CRD crd;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
#endif

	if (SelectPieces(w, x, y, &(w->pyraminx.currentFace), &crd)) {
		w->pyraminx.currentPosition = ToPosition(w, crd);
		if (control || w->pyraminx.practice || !CheckSolved(w))
			DrawTriangle(w, w->pyraminx.currentFace, w->pyraminx.currentPosition,
				TRUE);
	} else
		w->pyraminx.currentFace = -1;
}

#ifndef WINVER
static
#endif
void
ReleasePyraminx(PyraminxWidget w
#ifdef WINVER
, const int x, const int y, const int shift, const int control
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int style, face, count = -1, direction = 0;
	CRD crd;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int shift = (int) (event->xbutton.state & (ShiftMask | LockMask));
	int control = (int) (event->xkey.state & ControlMask);
#endif


	if (w->pyraminx.currentFace < 0)
		return;
	DrawTriangle(w, w->pyraminx.currentFace, w->pyraminx.currentPosition,
		FALSE);
	if (!control && !w->pyraminx.practice && CheckSolved(w))
		MoveNoPieces(w);
	else if (SelectPieces(w, x, y, &face, &crd)) {
		if (w->pyraminx.mode != BOTH) {
			if (control && shift)
				style = (w->pyraminx.mode == PERIOD3) ? PERIOD2 : PERIOD3;
			else
				style = (w->pyraminx.mode == PERIOD2) ? PERIOD2 : PERIOD3;
		} else
			style = (shift) ? PERIOD3 : PERIOD2;
		if (face == w->pyraminx.currentFace) {
			CRD crd0;

			ToCRD(w, face, w->pyraminx.currentPosition, &crd0);
			count = CheckMoveDir(w, crd0, crd, face, &direction);
		}
		if (count == 1) {
			NarrowSelection(w, style, &face, &crd, &direction);
			MovePyraminx(w, face, w->pyraminx.currentPosition,
				direction, style, (control) ? 1 : 0);
			if (!control && CheckSolved(w)) {
				SetPyraminx(w, PYRAMINX_SOLVED);
			}
		} else if (count == 2) {
			SetPyraminx(w, PYRAMINX_AMBIGUOUS);
		} else if (count == 0)
			MoveNoPieces(w);
	}
	w->pyraminx.currentFace = -1;
}

#ifndef WINVER
static
#endif
void
PracticePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	PracticePieces(w);
}

#ifndef WINVER
static void
PracticePyraminxMaybe(PyraminxWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->pyraminx.started)
		PracticePieces(w);
#ifdef HAVE_MOTIF
	else {
		SetPyraminx(w, PYRAMINX_PRACTICE_QUERY);
	}
#endif
}

static void
PracticePyraminx2(PyraminxWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->pyraminx.started)
#endif
		PracticePieces(w);
}
#endif

#ifndef WINVER
static
#endif
void
RandomizePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	RandomizePieces(w);
}

#ifndef WINVER
static void
RandomizePyraminxMaybe(PyraminxWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->pyraminx.started)
		RandomizePieces(w);
#ifdef HAVE_MOTIF
	else {
		SetPyraminx(w, PYRAMINX_RANDOMIZE_QUERY);
	}
#endif
}

static void
RandomizePyraminx2(PyraminxWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->pyraminx.started)
#endif
		RandomizePieces(w);
}
#endif

#ifndef WINVER
static
#endif
void
GetPyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetPieces(w);
}

#ifndef WINVER
static
#endif
void
WritePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WritePieces(w);
}

#ifndef WINVER
static
#endif
void
ClearPyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearPieces(w);
}

#ifndef WINVER
static
#endif
void
UndoPyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoPieces(w);
}

#ifndef WINVER
static
#endif
void
SolvePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolvePieces(w);
}

#ifndef WINVER
static
#endif
void
IncrementPyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	IncrementPieces(w);
}

#ifdef WINVER
Boolean
#else
static void
#endif
DecrementPyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifdef WINVER
	return
#else
	(void)
#endif
	DecrementPieces(w);
}

#ifndef WINVER
static
#endif
void
OrientizePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	OrientizePieces(w);
}

#ifndef WINVER
static
#endif
void
StickyModePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	StickyPieces(w);
}

#ifndef WINVER
static void
Period2ModePyraminx(PyraminxWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetPyraminx(w, PYRAMINX_PERIOD2);
}

static void
Period3ModePyraminx(PyraminxWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetPyraminx(w, PYRAMINX_PERIOD3);
}

static
void
BothModePyraminx(PyraminxWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetPyraminx(w, PYRAMINX_BOTH);
}
#endif

#ifndef WINVER
static
#endif
void
EnterPyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->pyraminx.focus = True;
	DrawFrame(w, w->pyraminx.focus);
}

#ifndef WINVER
static
#endif
void
LeavePyraminx(PyraminxWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->pyraminx.focus = False;
	DrawFrame(w, w->pyraminx.focus);
}

#ifdef WINVER
void
PeriodModePyraminx(PyraminxWidget w, const int mode)
{
	SetPyraminx(w, mode + PYRAMINX_PERIOD2);
}

#else

static void
MovePyraminxCcw(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, CCW,
		(int) (event->xbutton.state & (ShiftMask | LockMask)),
		(int) (event->xbutton.state & ControlMask));
}

static void
MovePyraminxTop(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, TOP,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
MovePyraminxTr(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, TR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
MovePyraminxLeft(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, LEFT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
MovePyraminxCw(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, CW,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
MovePyraminxRight(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, RIGHT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
MovePyraminxBl(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, BL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
MovePyraminxBottom(PyraminxWidget w, XEvent * event, char **args, int nArgs)
{
	MovePyraminxInput(w, event->xbutton.x, event->xbutton.y, BOTTOM,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}
#endif
